package org.eaglei.datatools.client.ui;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;

import org.eaglei.datatools.client.ApplicationState;
import org.eaglei.datatools.client.QueryTokenObject.Mode;
import org.eaglei.datatools.client.logging.GWTLogger;
import org.eaglei.datatools.client.rpc.ClientRepositoryToolsManager;
import org.eaglei.datatools.client.rpc.RootAsyncCallback;
import org.eaglei.datatools.client.ui.ButtonsPanel.ProviderRestrictionListener;
import org.eaglei.datatools.client.ui.listeners.NewInnerInstanceListener;
import org.eaglei.model.EIEntity;
import org.eaglei.model.EIInstance;

import com.google.gwt.event.dom.client.ClickEvent;
import com.google.gwt.event.dom.client.ClickHandler;
import com.google.gwt.user.client.History;
import com.google.gwt.user.client.Window;

public class EditFormsPanel extends DatatoolsInstancePanel implements NewInnerInstanceListener {

	static enum EditFormType {
		Edit, New, Dialog;
	}

	public interface PropertiesLoadedListener {

		void onPropertiesLoaded();
	}

	private final List<EIInstance> newInnerInstances = new ArrayList<EIInstance>();
	private String editToken;
	private final EditFormType formType;
	private final EditFormRedisplay formRedisplayCallbacks;

	private static final GWTLogger log = GWTLogger.getLogger( "EditFormsPanel" );

	public static EditFormsPanel createEditForm(final EIInstance instance, final EditFormRedisplay displayCallbacks) {
		return new EditFormsPanel( instance, displayCallbacks, EditFormType.Edit );
	}

	public static EditFormsPanel createNewForm(final EIInstance instance, final EditFormRedisplay displayCallbacks) {
		return new EditFormsPanel( instance, displayCallbacks, EditFormType.New );
	}

	private EditFormsPanel(final EIInstance instance, final EditFormRedisplay displayCallbacks, final EditFormType formType) {
		super( instance, displayCallbacks );
		ontologyPropRenderer = new OntologyPropEditRenderer( eiInstance, ontologyPanel, this );
		renderer = new NonOntlogyPropEditRenderer( eiInstance, nonOntologyOuterPanel );

		buttonPanel.setRootSuperClass( eiInstance.getRootSuperType() );
		shouldShowReadOnlyProperties = true;
		formRedisplayCallbacks = displayCallbacks;
		this.formType = formType;
		setButtonState(); // in order to get new/edit
		if ( !isNewForm() ) {
			getEditToken();
		}

		if ( ResourceProvider.isResourceProviderType( instance.getInstanceType() ) ) {
			log.debug( "setting app state resource provider to '" + instance.getEntity() + "'" ); // TODO: remove
			ApplicationState.getInstance().setResourceProviderEntity( instance.getEntity() ); // deliberately bypass other update mechanisms
		}

		if ( buttonPanel.getSaveButton().isEnabled() ) {
			buttonPanel.getSaveButton().addClickHandler( new ClickHandler() {

				@Override
				public void onClick(final ClickEvent event) {
					save();
				}
			} );
		}

		if ( buttonPanel.getCancelButton().isEnabled() ) {
			buttonPanel.getCancelButton().addClickHandler( new ClickHandler() {

				@Override
				public void onClick(final ClickEvent event) {
					cancel();
				}
			} );
		}

		buttonPanel.setProviderRestrictionSelector( new ProviderRestrictionListener() {

			@Override
			public void selectAll() {
				// TODO Auto-generated method stub

			}

			@Override
			public void selectOnlyProvider() {
				// TODO Auto-generated method stub

			}

		} );

	}

	public boolean isNewForm() {
		return formType == EditFormType.New;
	}

	protected void save() { // TODO: handle stale token correctly
		if ( !checkLabels() ) {
			Window.alert( UIMessages.LABEL_REQUIRED );
			return;
		}

		saveNewInnerInstances();

	}

	private void saveMainInstance() {
		if ( isNewForm() ) {
			saveNewForm();
		} else {
			if ( eiInstance.getNonOntologyLiteralProperties().containsKey( DatatoolsUIConstants.isStubEntity ) ) {
				eiInstance.replaceNonOntologyLiteralPropertyAllValues( DatatoolsUIConstants.isStubEntity, new HashSet<String>() );
			}
			ClientRepositoryToolsManager.INSTANCE.updateInstance( eiInstance, editToken, new RootAsyncCallback<Void>() {

				@Override
				public void onSuccess(final Void result) {
					formRedisplayCallbacks.drawAfterSave( eiInstance );
				}

				@Override
				public void onFailure(Throwable caught) {
					super.onFailure( caught );
					formRedisplayCallbacks.drawAfterCancel( eiInstance );
				}
			} );
		}
	}

	private void saveNewInnerInstances() {
		if ( newInnerInstances.size() == 0 ) {
			saveMainInstance();
		} else {
			deduplicate();
			log.info( "saving new inner instances" ); // TODO: remove
			ClientRepositoryToolsManager.INSTANCE.createInstances( newInnerInstances, new RootAsyncCallback<List<EIInstance>>() {

				@Override
				public void onSuccess(List<EIInstance> result) {
					log.info( "saved inner instances; now saving main" ); // TODO: remove
					saveMainInstance();

				}
			} );
		}
	}

	private void deduplicate() {
		List<EIInstance> ignore = new ArrayList<EIInstance>();

		for (EIInstance instance : newInnerInstances) {
			if ( ignore.contains( instance ) ) {
				continue;
			}
			for (EIInstance otherInstance : newInnerInstances) {
				if ( ignore.contains( otherInstance ) ) {
					continue;
				}

				if ( isDuplicate( instance, otherInstance ) ) {
					EIEntity property = eiInstance.findPropertyForInstance( otherInstance.getEntity() );
					eiInstance.replaceObjectPropertyValue( property, otherInstance.getInstanceURI(), instance.getEntity() );
					ignore.add( otherInstance );
				}
			}
		}

		newInnerInstances.removeAll( ignore );
	}

	private boolean isDuplicate(EIInstance instance, EIInstance otherInstance) {
		if ( instance == otherInstance || instance.getEntity().equals( otherInstance.getEntity() ) ) {
			return false; // not a duplicate of itself
		}

		// duplicate if type & label are identical
		return instance.getInstanceType().equals( otherInstance.getInstanceType() ) && instance.getInstanceLabel().equals( otherInstance.getInstanceLabel() );
	}

	private boolean checkLabels() {
		if ( isMissingLabel( eiInstance ) ) {
			return false;
		}

		for (EIInstance inner : newInnerInstances) {
			if ( isMissingLabel( inner ) ) {
				return false;
			}
		}

		for (EIInstance inner : eiInstance.getEmbeddedInstanceList()) {
			if ( isMissingLabel( inner ) && inner.getObjectProperties().isEmpty() && inner.getDatatypeProperties().isEmpty() && !inner.getInstanceClass().equals( inner.getRootSuperType() ) ) {
				return false;
			}
		}

		return true;
	}

	private boolean isMissingLabel(EIInstance instance) {
		return instance.getInstanceLabel() == null || instance.getInstanceLabel().trim().equals( "" );
	}

	private void saveNewForm() {
		ClientRepositoryToolsManager.INSTANCE.createInstance( eiInstance, new RootAsyncCallback<EIInstance>() {

			@Override
			public void onSuccess(final EIInstance saved) {
				formRedisplayCallbacks.drawAfterSave( saved );
			}
		} );
	}

	private void cancel() {

		if ( formType == EditFormType.New ) {
			if ( ResourceProvider.isResourceProviderType( eiInstance.getInstanceType() ) ) {
				ApplicationState.getInstance().updateApplicationState( Mode.workbench, EIEntity.NULL_ENTITY, EIEntity.NULL_ENTITY, EIEntity.NULL_ENTITY );
			} else if ( ApplicationState.getInstance().getMode() == Mode.duplicate ) {
				History.back();
			} else {
				EIEntity typeEntity = eiInstance.getRootSuperType().getEntity();
				ApplicationState.getInstance().updateApplicationState( Mode.list, EIEntity.NULL_ENTITY, typeEntity, ApplicationState.getInstance().getResourceProviderEntity() );
			}
		} else {
			ClientRepositoryToolsManager.INSTANCE.getInstance( eiInstance.getInstanceURI(), new RootAsyncCallback<EIInstance>() {

				@Override
				public void onSuccess(final EIInstance fetched) {
					formRedisplayCallbacks.drawAfterCancel( fetched );
				}

				@Override
				public void onFailure(Throwable caught) {
					formRedisplayCallbacks.drawAfterCancel( eiInstance );
				}
			} );
		}

	}

	@Override
	protected void setButtonState() {
		buttonPanel.saveButton.setVisible( true );
		if ( formType == EditFormType.New ) {
			buttonPanel.copyResourceButton.setVisible( false );
			buttonPanel.cancelButton.setVisible( true );

		} else {
			buttonPanel.copyResourceButton.setVisible( true );
			buttonPanel.cancelButton.setVisible( true );
			buttonPanel.hideAllWorkflowButtons();
		}
		buttonPanel.editButton.setVisible( false );
		buttonPanel.claimReleaseButton.setVisible( false );
		buttonPanel.deleteButton.setVisible( false );
	}

	private void getEditToken() {
		ClientRepositoryToolsManager.INSTANCE.getToken( eiInstance, new RootAsyncCallback<String>() {

			@Override
			public void onSuccess(String token) {
				editToken = token;
			}
		} );

	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eaglei.datatools.client.ui.EIFormsPanel#drawReferencedBy()
	 * 
	 * @Override protected void drawReferencedBy() { // deliberate no-op // Edit form doesn't display referenced by link }
	 */

	@Override
	public void onInstanceAdded(EIInstance newInstance) {
		newInnerInstances.add( newInstance );
	}

	@Override
	public void onInstanceRemoved(EIInstance newInstance) {
		newInnerInstances.remove( newInstance );
	}

	@Override
	public boolean isAddedInstanceEntity(EIEntity entity) {
		for (EIInstance inner : newInnerInstances) {
			if ( inner.getEntity().equals( entity ) ) {
				return true;
			}
		}
		
		return false;
	}

	@Override
	public EIInstance getAddedInstance(EIEntity entity) {
		for (EIInstance inner : newInnerInstances) {
			if ( inner.getEntity().equals( entity ) ) {
				return inner;
			}
		}
		
		return EIInstance.NULL_INSTANCE;
	}
}
