package org.eaglei.datatools.client.ui;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.eaglei.datatools.client.AbstractEntryPoint;
import org.eaglei.datatools.client.ApplicationState;
import org.eaglei.datatools.client.QueryTokenObject.Mode;
import org.eaglei.datatools.client.logging.GWTLogger;
import org.eaglei.datatools.client.rpc.ClientRepositoryToolsManager;
import org.eaglei.datatools.client.rpc.RootAsyncCallback;
import org.eaglei.datatools.client.ui.ResourceProviderCache.ResourceProviderCacheListener;
import org.eaglei.datatools.client.ui.widgets.LeftListRowWidget;
import org.eaglei.model.EIClass;
import org.eaglei.model.EIEntity;
import org.eaglei.model.EIOntConstants;
import org.eaglei.model.EIURI;
import org.eaglei.model.gwt.rpc.ClientModelManager;
import org.eaglei.model.gwt.rpc.ClientModelManager.ClassCallback;
import org.eaglei.model.gwt.rpc.ClientModelManager.ClassDefinitionCallback;
import org.eaglei.model.gwt.rpc.ClientModelManager.ClassesInGroupCallback;

import com.google.gwt.event.dom.client.ChangeEvent;
import com.google.gwt.event.dom.client.ChangeHandler;
import com.google.gwt.event.dom.client.ClickEvent;
import com.google.gwt.event.dom.client.ClickHandler;
import com.google.gwt.user.client.ui.Anchor;
import com.google.gwt.user.client.ui.FlowPanel;
import com.google.gwt.user.client.ui.Label;
import com.google.gwt.user.client.ui.ListBox;

public class LeftListPanel extends FlowPanel implements ClientRepositoryToolsManager.SessionListener, ApplicationStateChangeListener, ResourceProviderCacheListener {

	private static final String PANEL_SELECTED = "panelSelected";
	private static final String PANEL_NOT_SELECTED = "panelNotSelected";
	private static final EIEntity allTypesEntity = EIEntity.create( EIURI.NULL_EIURI, "All Resource Types" );
	
	private boolean signedIn = false;

	private List<EIClass> resourceTypesForProvider;
	private List<EIClass> resourceTypesForBrowse = new ArrayList<EIClass>();
	private EIClass documentType;
	private Map<EIEntity, String> definitions;
	
	//Keeping track of row widgets; for setting "selected" item
	private Map<EIEntity, LeftListRowWidget> providerRows = new HashMap<EIEntity, LeftListRowWidget>();
	private Map<EIEntity, LeftListRowWidget> browseRows = new HashMap<EIEntity, LeftListRowWidget>();
	private final  LeftListRowWidget allTypesRow = new LeftListRowWidget( allTypesEntity, "", false ) ;	
	
	private final Anchor switchProvidersLink = new Anchor();
	private final ListBox providersList = new ListBox();
	private final Label providerNameLabel = new Label();
	private final Label resourcesLabel = new Label();
	private final Label peopleAndOrganizationsLabel = new Label();

	private final ResourceProviderCache resourceProviderCache;

	private static final GWTLogger log = GWTLogger.getLogger( "LeftListPanel" );

	public LeftListPanel(final ResourceProviderCache resourceProviderCache) {
		ClientRepositoryToolsManager.INSTANCE.addSessionListener( this );
		ApplicationState.getInstance().addApplicationStateListener( this );
		this.resourceProviderCache = resourceProviderCache;
		resourceProviderCache.addProviderLoadListener( this );
		signedIn = ApplicationState.getInstance().hasUser();
		if ( signedIn ) {
			getResourceTypes();
		}
	}


	@Override
	public void onApplicationStateChange() {
		drawFromApplicationState();
	}

	private void drawFromApplicationState() {
		if ( !ApplicationState.getInstance().hasUser() && !signedIn ) {
			clear();
			return;
		}

		if ( ApplicationState.getInstance().getMode() == Mode.workbench ) {
			this.setVisible( false );
		} else if ( ApplicationState.getInstance().getMode() == Mode.providers || 
				( ApplicationState.getInstance().getMode() == Mode.edit && ApplicationState.getInstance().hasResourceProviderType() ) ||
				( ApplicationState.getInstance().getMode() == Mode.filter && ApplicationState.getInstance().hasResourceProviderType() )) {
			this.setVisible( false );
		} else if ( ApplicationState.getInstance().hasResourceProvider() ) {
			this.setVisible( true );
			buildProviderPanel();
		} else {
			this.setVisible( true );
			buildBrowsePanel();
		}
	}


	/**
	 * Retrieves all the top level Resource Categories
	 * 
	 * @return
	 */
	//TODO: REFACTOR - create one single async call?
	private void getResourceTypes() {
		if(ApplicationState.getInstance().getResourceTypesForProvider() == null) {
			ClientModelManager.INSTANCE.getClassesInGroup( EIOntConstants.CG_DATA_MODEL_CREATE, new ClassesInGroupCallback() {
				@Override
				public void onSuccess(List<EIClass> classList) {
					AbstractEntryPoint.hideGlasspane();
					ApplicationState.getInstance().setResourceTypesForProvider( classList );
					resourceTypesForProvider = classList;
					getDocumentClass();
				}
			});
		} else {
			resourceTypesForProvider = ApplicationState.getInstance().getResourceTypesForProvider();
			getDocumentClass();
		}
	}

	private void getDocumentClass() {
		if(ApplicationState.getInstance().getResourceTypesForBrowse() == null ) {
		ClientModelManager.INSTANCE.getClass(DatatoolsUIConstants.documentEntity.getURI(), new ClassCallback() {

			@Override
			public void onSuccess(EIClass result) {
				resourceTypesForBrowse.addAll(resourceTypesForProvider);
				documentType = result;
				resourceTypesForBrowse.add(documentType);
				Collections.sort(resourceTypesForBrowse, new Comparator<EIClass>() {

					@Override
					public int compare(EIClass arg0, EIClass arg1) {
						return arg0.getEntity().compareTo( arg1.getEntity() );
					}
					
				});
				ApplicationState.getInstance().setResourceTypesForBrowse(resourceTypesForBrowse);
				getTypeDefinitions(resourceTypesForBrowse);

			}
			
		});
		} else {
			resourceTypesForBrowse = ApplicationState.getInstance().getResourceTypesForBrowse();
			//ASSUMPTION: resourceTypesForBrowse contains all of ResourceTypesForProviders
			getTypeDefinitions( resourceTypesForBrowse );
		}
	}
	
	private void getTypeDefinitions(List< EIClass > classes) {
		if(ApplicationState.getInstance().getClassDefinitions() == null ) {
		ClientModelManager.INSTANCE.getClassDefinitions( classes, new ClassDefinitionCallback() {

			@Override
			public void onSuccess(final List<EIClass> result) {
				definitions = new HashMap<EIEntity, String>();
				for (final EIClass eclass : result) {
					definitions.put( eclass.getEntity(), eclass.getDefinition() );
				}
				createPanel();
			}
		} );
		} else {
			definitions = ApplicationState.getInstance().getClassDefinitions();
			createPanel();
		}
	}

	/**
	 * Creates the left panel. 
	 */
	private void createPanel() {
		initializeMapOfRows();
		resourcesLabel.setText( "Browse Resources" );
		resourcesLabel.setStyleName( "leftListHeader" );
		peopleAndOrganizationsLabel.setText( "Browse People and Organizations" );
		peopleAndOrganizationsLabel.setStyleName( "leftListHeader" );
		
		switchProvidersLink.setHTML( "switch organizations" );
		switchProvidersLink.setStyleName( "gwt_add_item" ); 
		switchProvidersLink.addStyleName("liveLink");
		switchProvidersLink.addClickHandler( new ClickHandler() {
			@Override
			public void onClick(final ClickEvent event) {
				providersList.setVisible( true );
			}
		} );
		
		providerNameLabel.setStyleName( "labSelected" );
		providerNameLabel.addStyleName("liveLink");
		providerNameLabel.addClickHandler( new ClickHandler() {
			@Override
			public void onClick(final ClickEvent event) {
				ApplicationState.getInstance().updateApplicationState( Mode.view, ApplicationState.getInstance().getResourceProviderEntity(), ResourceProvider.BASE_RESOURCE_CONTAINER, ApplicationState.getInstance().getResourceProviderEntity() );
			}
		} );
		
		setStyleName( "leftPanel" );
		if ( signedIn ) {
			drawFromApplicationState();
		}
	}

	
	private void buildBrowsePanel() {
		clear();
		add( resourcesLabel );
		addAllTypesOption();
		populateTypeSelectors( resourceTypesForBrowse, browseRows, false );
		add( peopleAndOrganizationsLabel );
		populatePersonAndOrganizationSelectors( );
		selectType( browseRows );
	}
	

	private void buildProviderPanel() {
		clear();
		final String providerName = ApplicationState.getInstance().hasResourceProvider() ? ApplicationState.getInstance().getResourceProviderEntity().getLabel() : "";
		providerNameLabel.setText( providerName );
		add( providerNameLabel );
		add( switchProvidersLink );
		providersList.setVisible( false );
		if ( resourceProviderCache.isLoading() ) {
			providersList.addItem( "loading..." );
		} else {
			populateProvidersList();
		}
		add( providersList );
		addAllTypesOption( );
		populateTypeSelectors( resourceTypesForProvider, providerRows, true );
		selectType( providerRows );
	}

	private void populateProvidersList() {
		if ( resourceProviderCache.isLoading() ) {
			return; // TODO: user warning?
		}

		providersList.setStyleName( "leftListDropdown" );
		providersList.clear();

		providersList.addItem("select an organization");
		StringBuilder strBuild = new StringBuilder();
		for (final EIEntity provider : resourceProviderCache.getResourceProviderEntities()) {
			strBuild.append(provider.getLabel());
			strBuild.append(" <");
			strBuild.append(resourceProviderCache.getResourceProviderType(provider).getLabel());
			strBuild.append(">");
			providersList.addItem( strBuild.toString() , provider.getURI().toString() );
			strBuild.delete(0, strBuild.length());
		}

		providersList.addChangeHandler( new ChangeHandler() {

			@Override
			public void onChange(final ChangeEvent event) {
				EIURI selectedUri = EIURI.create( providersList.getValue( providersList.getSelectedIndex() ) );
				final EIEntity selectedEntity = resourceProviderCache.getResourceProvider( selectedUri );
				providersList.setVisible( false );
				if ( selectedEntity == null || selectedEntity.equals( EIEntity.NULL_ENTITY ) ) {
					return;
				}
				ApplicationState.getInstance().updateApplicationState( Mode.view, selectedEntity, ResourceProvider.BASE_RESOURCE_CONTAINER, selectedEntity );
			}
		} );

	}
	
	private void initializeMapOfRows() {
		initializeMapOfRowsForResources( resourceTypesForProvider, providerRows, true );		
		
		initializeMapOfRowsForResources( resourceTypesForBrowse, browseRows, false );
		final LeftListRowWidget personRow = new LeftListRowWidget( DatatoolsUIConstants.personEntity, DatatoolsUIConstants.personEntity.getLabel(), false );
		browseRows.put( DatatoolsUIConstants.personEntity, personRow );
		final LeftListRowWidget organizationRow = new LeftListRowWidget( DatatoolsUIConstants.organizationEntity, DatatoolsUIConstants.organizationEntity.getLabel(), false );
		browseRows.put( DatatoolsUIConstants.organizationEntity, organizationRow );
	}
	
	private void initializeMapOfRowsForResources( final List<EIClass> resourceTypesList, final Map<EIEntity, LeftListRowWidget> rows, final boolean canAdd ) {
		for (final EIClass eiClass : resourceTypesList) {
			final LeftListRowWidget row = new LeftListRowWidget( eiClass, definitions.get( eiClass.getEntity() ), canAdd);
			rows.put(eiClass.getEntity(), row);
		}
	}
	
	private void populateTypeSelectors(final List<EIClass> resourceTypesList, Map<EIEntity, LeftListRowWidget> rows, final boolean shouldHaveAddNew) {
		for (final EIClass eiClass : resourceTypesList) {
			add( rows.get( eiClass.getEntity() ) );
		}
	}

	private void populatePersonAndOrganizationSelectors() {
		add( browseRows.get( DatatoolsUIConstants.personEntity) );
		add( browseRows.get( DatatoolsUIConstants.organizationEntity) );
	}

	private void addAllTypesOption() {
		add( allTypesRow );
	}

	private void selectType(final Map<EIEntity, LeftListRowWidget> rows) {
		unselectAll( rows );
		/*if(!ApplicationState.getInstance().hasType() && Mode.isResourcesList( ApplicationState.getInstance().getMode() ) && ( ApplicationState.getInstance().getMode() != Mode.resources )) {
			return;
		}*/
		
		if(ApplicationState.getInstance().hasResourceProviderType()) {
			return;
		}
		if ( !ApplicationState.getInstance().hasType()) {
				allTypesRow.setStyleName( PANEL_SELECTED );
			} else {
				final EIEntity typeEntity = ApplicationState.getInstance().getTypeEntity();
				ClientRepositoryToolsManager.INSTANCE.getRootSuperClass( typeEntity, new  RootAsyncCallback<EIClass>() {
					
					@Override
					public void onSuccess(EIClass result) {
						if(rows.containsKey( result.getEntity() )) {
							rows.get( result.getEntity() ).setStyleName( PANEL_SELECTED );
						}
					}
				});	
			}
	}

	private void unselectAll(Map<EIEntity, LeftListRowWidget> rows) {
		for (LeftListRowWidget row : rows.values()) {
			row.setStyleName( PANEL_NOT_SELECTED );
		}
		allTypesRow.setStyleName( PANEL_NOT_SELECTED );
	}

	@Override
	public void onLogOut(boolean isSessionExpired) {
		signedIn = false;
		clear();
	}

	@Override
	public void onLogIn(final String username, final String userUri) {
		signedIn = true;
		if ( resourceTypesForProvider != null && resourceTypesForProvider.size() > 0 ) { // TODO: right?
			createPanel();
		} else {
			log.debug( "left list panel logging in: about to get resource types" );
			getResourceTypes();
		}
	}

	@Override
	public void onProvidersLoaded() {
		populateProvidersList();
	}

}
