package org.eaglei.datatools.client.rpc;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;

import org.eaglei.datatools.client.DatatoolsCookies;
import org.eaglei.datatools.client.ui.WorkFlowConstants;
import org.eaglei.datatools.model.DataToolsOntConstants;
import org.eaglei.model.EIEntity;
import org.eaglei.model.EIInstance;
import org.eaglei.model.EIURI;
import org.eaglei.model.gwt.rpc.LoggedException;

import com.allen_sauer.gwt.log.client.Log;
import com.google.gwt.core.client.GWT;
import com.google.gwt.user.client.Cookies;
import com.google.gwt.user.client.Window;
import com.google.gwt.user.client.rpc.AsyncCallback;
import com.google.gwt.user.client.rpc.IncompatibleRemoteServiceException;
import com.google.gwt.user.client.rpc.InvocationException;

/**
 * Maintains a client-side cache of EIInstance. Proxies all model RPC methods.
 * 
 * It is critical that all model RPC calls go through this class. All methods in
 * this class MUST call getCached(EIInstance) on all EIInstance objects it
 * receives from the server to ensure that there is only one instance of an
 * EIInstance per URI in the client.
 * 
 * 
 */
public class ClientRepositoryToolsManager {
	// Stupid, but GWT has problems referring to DataToolsOntConstants.
	public static final String DRAFT_STATE = "http://eagle-i.org/ont/repo/1.0/WFS_Draft";
	public static final String CURATION_STATE = "http://eagle-i.org/ont/repo/1.0/WFS_Curation";
	public static final String PUBLISHED_STATE = "http://eagle-i.org/ont/repo/1.0/WFS_Published";
	public static final String WITHDRAWN_STATE = "http://eagle-i.org/ont/repo/1.0/WFS_Withdrawn";

	public interface SessionListener {
		void onLogIn(String username, String userUri);

		void onLogOut(); // Notification that a logout occurred
	}

	public interface LoginRequiredCallback {
		void loginRequired();
	}

	public interface ResultsCallback extends LoginRequiredCallback {
		void onSuccess(String[] arg0);

		void onSuccess(String arg0);
	}

	public interface EIInstanceCallback extends LoginRequiredCallback {
		void onSuccess(EIInstance eiInstance);
	}

	public interface EIInstancesForLabCallback extends LoginRequiredCallback {
		void onSuccess(List<EIInstance> eiInstance);
	}

	public interface NewInstanceCallback extends LoginRequiredCallback {
		void onSuccess(Object obj);
	}

	public interface EIInstancesCallback extends LoginRequiredCallback {
		void onSuccess(List<EIInstance> result);
	}

	public interface QueryEIInstancesCallback extends LoginRequiredCallback {
		void onSuccess(List<EIInstance> result);
	}

	public interface FilterInstancesCallback extends LoginRequiredCallback {
		void onSuccess(List<EIInstance> result);
	}

	public interface IdCallback extends LoginRequiredCallback {
		void onSuccess(List<EIURI> list);
	}

	public interface UserCallback extends LoginRequiredCallback {
		void onSuccess(String[] userInfo);
	}

	public interface WFCallback {
		void onSuccess(String[] wfStates);
	}

	public interface DeleteInstanceCallback {
		void onSuccess(Object obj);
	}

	public static final ClientRepositoryToolsManager INSTANCE = new ClientRepositoryToolsManager();
	private static RepositoryToolsModelServiceAsync modelService;
	private final HashMap<EIURI, EIInstance> mapIdToClass = new HashMap<EIURI, EIInstance>();
	private String sessionId;
	private EIURI userURI; // TODO:
	// probably
	// get
	// rid
	// of
	// this
	// again
	private ArrayList<SessionListener> listeners;
	private List<String> editableStates;

	enum LoginFields {
		USERNAME(0), SESSION(1), USER_URI(2), ERROR_MESSAGE(3), WF_STATE(3);
		private final int value;

		private LoginFields(final int value) {
			this.value = value;
		}

		public int getValue() {
			return value;
		}
	}

	private ClientRepositoryToolsManager() {
		String userUriString = DatatoolsCookies.getUserUri();
		if (userUriString != null) {
			userURI = EIURI.create(userUriString);
			Log.info("got existing userUri: " + userURI);
		}
		sessionId = DatatoolsCookies.getSession();
		Log.info("got session ID " + sessionId);
		modelService = GWT.create(RepositoryToolsModelService.class);
	}

	public boolean isLoggedIn() {
		return sessionId != null;
	}

	// Shamelessly stolen from ClientSearchManager; TODO: refactor/merge
	private void handleLogOut() {
		sessionId = null;
		userURI = null;
		Cookies.removeCookie(DatatoolsCookies.DATATOOLS_USER_COOKIE_ID);
		Cookies.removeCookie(DatatoolsCookies.DATATOOLS_USER_URI_ID);
		Cookies.removeCookie(DatatoolsCookies.DATATOOLS_SESSION_ID);
		Log.info("removed cookies");
		if (listeners != null) {
			for (SessionListener listener : listeners) {
				listener.onLogOut();
			}
		}
	}

	public void addSessionListener(SessionListener listener) {
		if (listeners == null) {
			listeners = new ArrayList<SessionListener>();
		}
		this.listeners.add(listener);
		/*
		 * if (isLoggedIn()) { listener.onLogIn(""); }
		 */
	}

	public void logOut() {
		if (!isLoggedIn()) {
			// ?? Probably should fire logOut even in deferred command
			return;
		}
		// Need to wait for callback, or ok to just assume logged out?
		try {
			modelService.logout(sessionId, new AsyncCallback<Void>() {
				@Override
				public void onFailure(Throwable caught) {
					// Window.alert("Error logging out.");
					handleLogOut();
				}

				@Override
				public void onSuccess(Void result) {
					sessionId = null;
					userURI = null;
					handleLogOut();
				}
			});
		} catch (Exception e) {
			e.printStackTrace();
		}
	}

	public void logIn(final String username, final String password,
			final ResultsCallback callback) {
		final int USERNAME = 0;
		final int SESSION = 1;
		final int USER_URI = 2;
		if (isLoggedIn()) {
			// ?? I guess get a new session id
		}
		try {
			modelService.login(username, password,
					new AsyncCallback<String[]>() {
						@Override
						public void onFailure(Throwable caught) {
							userURI = null;
							if (caught instanceof Exception) {
								Window.alert(caught.getMessage());
							} else {
								Window.alert("Error logging in. ");
							}
						}

						@Override
						public void onSuccess(String[] result) {
							sessionId = result[SESSION];
							String userUri = result[USER_URI];
							userURI = EIURI.create(userUri);
							Cookies.setCookie(
									DatatoolsCookies.DATATOOLS_USER_COOKIE_ID,
									username);
							Cookies.setCookie(
									DatatoolsCookies.DATATOOLS_USER_URI_ID,
									userUri);
							Cookies.setCookie(
									DatatoolsCookies.DATATOOLS_SESSION_ID,
									sessionId);
							// Suggestion from GWT security is to use
							// Cookies.setCookie(DatatoolsCookies.DATATOOLS_USER_COOKIE_ID,
							// username, expires, null, "/", true);
							// and call checkValidSession() on server
							// --presumably call whoami & check it's valid not
							// invalid
							Log.info("set cookies; userUri = " + Cookies
													.getCookie(DatatoolsCookies.DATATOOLS_USER_URI_ID) + 
													" session = " + Cookies.getCookie(DatatoolsCookies.DATATOOLS_SESSION_ID));
							initializeEditableStates(result);
							if (listeners != null) {
								for (SessionListener listener : listeners) {
									listener.onLogIn(result[USERNAME],
											result[USER_URI]);
								}
							}
							callback.onSuccess(result[USERNAME]);
						}
					});
		} catch (Exception e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		}
	}

	public void updateInstance(final EIInstance eiInstance, String token,
			final ResultsCallback callback) throws Exception {
		modelService.updateInstance(sessionId, eiInstance, token,
				new AsyncCallback<String>() {
					@Override
					public void onFailure(Throwable arg0) {
						handleFailure(callback, arg0);
					}

					@Override
					public void onSuccess(String arg0) {
						if (arg0 == null) // because GWT makes it hard to throw
						// exceptions; typically get null if
						// not logged in
						{
							callback.loginRequired();
							return;
						}
						callback.onSuccess(arg0);
					}
				});
	}

	public void getInstance(final EIURI eiURI, final EIInstanceCallback callback)
			throws Exception {
		modelService.getInstance(sessionId, eiURI,
				new AsyncCallback<EIInstance>() {
					@Override
					public void onFailure(Throwable arg0) {
						handleFailure(callback, arg0);
					}

					@Override
					public void onSuccess(EIInstance arg0) {
						if (arg0 == null) // because GWT makes it hard to throw
						// exceptions
						{
							
						}
						callback.onSuccess(getCached(arg0));
					}
				});
	}

	public void deleteInstance(EIInstance instance,
			final DeleteInstanceCallback callback) throws Exception {
		modelService.deleteInstance(sessionId, instance, new AsyncCallback() {
			@Override
			public void onFailure(Throwable arg0) {
				// TODO Auto-generated method stub
			}

			@Override
			public void onSuccess(Object arg0) {
				callback.onSuccess(arg0);
			}
		});
	}

	private void initializeEditableStates(String[] userInfo) {
		Log.info("initializing editable states with " + userInfo.length
				+ " states");
		WorkFlowConstants wfc = new WorkFlowConstants();
		editableStates = new ArrayList<String>();
		if (userInfo.length <= LoginFields.WF_STATE.getValue())
			return;
		for (int i = LoginFields.WF_STATE.getValue(); i < userInfo.length; i++) {
			String state = userInfo[i];
			editableStates.add(state);
			editableStates.add(wfc.getStatusType(state));
			Log.info("adding editable state " + userInfo[i]);
		}
	}

	public boolean canEdit(String workflowState) {
		if (editableStates == null) {
			Log.info("editable states null");
			return false; // TODO: throw? Shouldn't happen if logged in
		}
		if (workflowState == null) {
			return false;
		}
		Log.info("asking if can edit " + workflowState + "; answer "
				+ editableStates.contains(workflowState));
		return editableStates.contains(workflowState);
	}

	public List<String> getEditableStates() {
		return editableStates;
	}

	public void getResourcesOfClass(String rnav, EIURI classUri,
			final EIInstancesCallback callback) {
		modelService.getResourcesOfClass(sessionId, rnav, classUri,
				new AsyncCallback<List<EIInstance>>() {
					@Override
					public void onFailure(Throwable arg0) {
						handleFailure(callback, arg0);
					}

					@Override
					public void onSuccess(List<EIInstance> instances) {
						/*
						 * if (instances == null) // because GWT makes it hard
						 * to // throw the right error, and // provider returns
						 * null if not // logged in {
						 * //callback.loginRequired();
						 * 
						 * return; }
						 */
						callback.onSuccess(instances);
					}
				});
	}

	public void getFilterQuery(final String user, final EIURI classUri,
			final EIURI state, final EIURI lab,
			final FilterInstancesCallback callback) throws Exception {
		modelService.getFilterQuery(sessionId, user, classUri, state, lab,
				new AsyncCallback<List<EIInstance>>() {
					@Override
					public void onFailure(Throwable arg0) {
						// TODO Auto-generated method stub
					}

					@Override
					public void onSuccess(List<EIInstance> arg0) {
						callback.onSuccess(arg0);
					}
				});
	}

	public void getAllResources(String rnav, final EIInstancesCallback callback) {
		modelService.getAllResources(sessionId, rnav,
				new AsyncCallback<List<EIInstance>>() {
					@Override
					public void onFailure(Throwable arg0) {
						handleFailure(callback, arg0);
					}

					@Override
					public void onSuccess(List<EIInstance> instances) {
						/*
						 * if (instances == null) // because GWT makes it hard
						 * to // throw exceptions { //callback.loginRequired();
						 * Window.alert("no resources"); return; } else {
						 */
						callback.onSuccess(instances);
						// }
					}
				});
	}

	private EIInstance getCached(EIInstance c) {
		if (mapIdToClass.containsKey(c.getEntity().getURI())) {
			return mapIdToClass.get(c.getEntity().getURI());
		} else {
			mapIdToClass.put(c.getEntity().getURI(), c);
			return c;
		}
	}
	
	private void updateCached(EIInstance instance, EIURI workFlowState, EIURI owner)
	{
		EIInstance cached = getCached(instance);
		if (workFlowState != null)
			cached.setWFState(workFlowState);
		if (owner != null)
			cached.setWFOwner(owner);
	}

	public void getNewInstanceID(int count, final IdCallback callback) {
		try {
			modelService.getNewInstanceID(sessionId, count,
					new AsyncCallback<List<EIURI>>() {
						@Override
						public void onFailure(Throwable arg0) {
							handleFailure(callback, arg0);
						}

						@Override
						public void onSuccess(List<EIURI> list) {
							if (list == null) // because GWT makes it hard to
							// throw exceptions
							{
								callback.loginRequired();
								return;
							}
							callback.onSuccess(list);
						}
					});
		} catch (Exception e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		}
	}

	public void createInstance(final EIInstance instance, final NewInstanceCallback callback) {
		try {
			modelService.createInstance(sessionId, instance,
					new AsyncCallback() {
						@Override
						public void onFailure(Throwable arg0) {
							handleFailure(callback, arg0);
						}

						@Override
						public void onSuccess(Object arg0) {
							instance.setWFState(EIURI.create(DRAFT_STATE));
							callback.onSuccess(arg0);
						}
					});
		} catch (Exception e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		}
	}

	public void getEmptyEIInstance(EIURI classUri,
			final EIEntity instanceEntity, final EIInstanceCallback callback) {
		modelService.getEmptyEIInstance(sessionId, classUri, instanceEntity,
				new AsyncCallback<EIInstance>() {
					@Override
					public void onFailure(Throwable arg0) {
						handleFailure(callback, arg0);
					}

					@Override
					public void onSuccess(EIInstance instance) {
						if (instance == null) // because GWT makes it hard to
						// throw exceptions
						{
							callback.loginRequired();
							return;
						}
						callback.onSuccess(instance);
					}
				});
	}

	// TODO:
	public void query(String sparql, AsyncCallback<StringBuffer> callback)
			throws Exception {
	}

	// TODO:
	public void isOnline(AsyncCallback callback) throws Exception {
	}

	public void whoami(final UserCallback callback) throws Exception {
		if (modelService == null) {
			modelService = GWT.create(RepositoryToolsModelService.class);
		}
		
		String session = DatatoolsCookies.getSession();
		
		modelService.whoami(session, new AsyncCallback<String[]>() {
			@Override
			public void onFailure(Throwable arg0) {
				Log.warn("client whoami failure");
				if (arg0 != null)
					Log.warn("client whoami failure: " + arg0.getMessage());
				// TODO Auto-generated method stub
			}

			@Override
			public void onSuccess(String[] userInfo) {
				Log.debug("model service whoami succeeded with "
						+ (userInfo == null ? "null" : userInfo.length)
						+ " fields");
				initializeEditableStates(userInfo);
				callback.onSuccess(userInfo);
			}
		});
	}

	public void returnToDraft(final EIInstance instance,
			final ResultsCallback callback) throws Exception {
		promote(instance, DRAFT_STATE, callback);
	}

	public void sendToCuration(final EIInstance instance,
			final ResultsCallback callback) throws Exception {
		promote(instance, CURATION_STATE, callback);
	}

	public void publish(final EIInstance instance,
			final ResultsCallback callback) throws Exception {
		promote(instance, PUBLISHED_STATE, callback);
	}

	public void withdraw(final EIInstance instance,
			final ResultsCallback callback) throws Exception {
		Log.info("withdrawing " + instance.getInstanceLabel());
		promote(instance, WITHDRAWN_STATE, callback);
	}

	public void bulkPublish(final EIInstance[] instanceAry,
			final ResultsCallback callback) throws Exception {
		bulkPromote(instanceAry, PUBLISHED_STATE, callback);
	}

	public void bulkWithdraw(final EIInstance[] instanceAry,
			final ResultsCallback callback) throws Exception {
		bulkPromote(instanceAry, WITHDRAWN_STATE, callback);
	}

	public void bulkReturnToDraft(final EIInstance[] instanceAry,
			final ResultsCallback callback) throws Exception {
		bulkPromote(instanceAry, DRAFT_STATE, callback);
	}

	public void bulkSendToCuration(final EIInstance[] instanceAry,
			final ResultsCallback callback) throws Exception {
		bulkPromote(instanceAry, CURATION_STATE, callback);
	}

	public void promote(final EIInstance instance, final String newState, final ResultsCallback callback) throws Exception {
		if (isLoggedIn()) {
			Log.info("promoting '" + instance.getInstanceURI().toString() + "' to " + newState);
			modelService.promote(sessionId, instance.getInstanceURI().toString(), newState, new AsyncCallback<String>() {
				@Override
				public void onFailure(Throwable arg0) {
					// TODO Auto-generated method stub
				}

				@Override
				public void onSuccess(String arg0) {
					if (arg0 == null) {
						// what to do? Need to let everyone know it failed
						Window.alert("could not promote "
								+ instance.getInstanceLabel()
								+ "(null response)"); // TODO
						// remove
						return;
					}
					
					instance.setWFState(EIURI.create(newState));
					callback.onSuccess(arg0);
				}
			});
		}
	}

	public void bulkPromote(final EIInstance[] aryInstance, final String newState,
			final ResultsCallback callback) throws Exception {
		if (isLoggedIn()) {
			String[] uriAry = new String[aryInstance.length];
			int i = 0;
			for (EIInstance instance : aryInstance) {
				uriAry[i] = instance.getInstanceURI().toString();
				i++;
			}
			
			modelService.bulkPromote(sessionId, uriAry, newState,
					new AsyncCallback<String[]>() {
						@Override
						public void onFailure(Throwable arg0) {
							// TODO Auto-generated method stub
						}

						@Override
						public void onSuccess(String[] arg0) {
							if (arg0 == null) {
								Window.alert("sucess");
								return;
							}
							EIURI newStateUri = EIURI.create(newState);
							for (EIInstance instance : aryInstance)
							{
								updateCached(instance, newStateUri, null);
								instance.setWFState(newStateUri);
								Log.info("forced state of " + instance.getInstanceLabel() + " to " + newStateUri);
							}
							callback.onSuccess(arg0);
						}
					});
		}
	}

	public void claim(final EIInstance instance, final ResultsCallback callback)
			throws Exception {
		if (isLoggedIn()) {
			Log.info("Calling claim on " + instance.getInstanceLabel() + "; '"
					+ instance.getInstanceURI().toString() + "'");
			modelService.claim(sessionId, instance.getInstanceURI().toString(),
					"self", new AsyncCallback<String>() {
						@Override
						public void onFailure(Throwable arg0) {
							// TODO Auto-generated method stub
						}

						@Override
						public void onSuccess(String arg0) {
							if (arg0 == null) {
								Window.alert("could not claim " + instance.getInstanceLabel());
								return;
							}
							Log.info("claim result: " + arg0);
							updateCached(instance, null, userURI);
							callback.onSuccess(arg0);
						}
					});
		}
	}

	public void getInstancesForLab(String rnav, String labUri,
			final EIInstancesForLabCallback callback) throws Exception {
		modelService.getInstancesForLab(sessionId, rnav, labUri,
				new AsyncCallback<List<EIInstance>>() {
					@Override
					public void onFailure(Throwable arg0) {
						// TODO Auto-generated method stub
					}

					@Override
					public void onSuccess(List<EIInstance> arg0) {
						callback.onSuccess(arg0);
					}
				});
	}

	public void getWFStates(String user, final WFCallback callback)
			throws Exception {
		modelService.getWFStates(sessionId, user,
				new AsyncCallback<String[]>() {
					@Override
					public void onFailure(Throwable arg0) {
						// TODO Auto-generated method stub
					}

					@Override
					public void onSuccess(String[] arg0) {
						callback.onSuccess(arg0);
					}
				});
	}

	public void retrieveLabel(EIURI uri, final ResultsCallback callback) {
		modelService.retrieveLabel(sessionId, uri, new AsyncCallback<String>() {
			@Override
			public void onFailure(Throwable arg0) {
				// TODO Auto-generated method stub
			}

			@Override
			public void onSuccess(String label) {
				callback.onSuccess(label);
			}
		});
	}

	private void handleFailure(final LoginRequiredCallback callback,
			Throwable arg0) {
		if (arg0 instanceof IncompatibleRemoteServiceException) {
			// log.error(arg0.getMessage() + )
		} else if (arg0 instanceof InvocationException) {
		} else if (arg0 instanceof LoggedException) {
			Window.alert(arg0.getMessage());
		} else {
			callback.loginRequired();
		}
	}
}
