package org.eaglei.datatools.server;

import java.io.File;
import java.net.URL;
import java.util.List;

import javax.servlet.ServletException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eaglei.datatools.client.rpc.RepositoryToolsModelService;
import org.eaglei.datatools.config.DatatoolsConfiguration;
import org.eaglei.datatools.jena.EIInstanceFactory;
import org.eaglei.datatools.jena.RESTRepositoryProviderFactory;
import org.eaglei.datatools.model.DataToolsOntConstants;
import org.eaglei.datatools.provider.RepositoryProvider;
import org.eaglei.datatools.provider.RepositoryProviderFactory;
import org.eaglei.model.EIEntity;
import org.eaglei.model.EIInstance;
import org.eaglei.model.EIURI;
import org.eaglei.model.gwt.rpc.LoggedException;

import com.google.gwt.user.server.rpc.RemoteServiceServlet;

public class RepositoryToolsModelServlet extends RemoteServiceServlet implements RepositoryToolsModelService
{

    /**
     * 
     */
    private static final long serialVersionUID = 1L;
    private static final Log log = LogFactory.getLog(RepositoryToolsModelServlet.class);
    private RepositoryProvider provider = null;
    private DatatoolsConfiguration config;

    @Override
    public void init() throws ServletException
    {

        config = getConfiguration();
        try
        {
            if(config != null)
            {
            	if (log.isDebugEnabled()) log.debug("about to create provider");
                provider = createProvider();
            }
            else
            {
            	if (log.isDebugEnabled()) log.debug("about to load defaults (null config)");
                loadDefaults();
            }

        }
        catch(Exception e)
        {
            log.error("Error initializing RepositoryToolsModelServlet: " + e);
            throw new ServletException(e);
        }
        
        
        lazyLoadFactories();

        log.info("init succesful");

    }

    private RepositoryProvider createProvider() throws Exception
    {

        final String providerFactoryClass = config.getDatatoolsRepositoryProviderFactory();
        final Class factoryClass = Class.forName(providerFactoryClass);
        final RepositoryProviderFactory factory = (RepositoryProviderFactory) factoryClass.newInstance();
        return factory.createRepositoryProvider(config);

    }

    private void lazyLoadFactories()
    {
        EIInstanceFactory.getInstance();
    }

    private void loadDefaults() throws Exception
    {
        // default provider is the REST provider
        log.info("Loading default provider, RESTProvider: ");

        final RepositoryProviderFactory factory = new RESTRepositoryProviderFactory();
        provider = factory.createRepositoryProvider(config);
        log.info("Loading default provider, RESTProvider: " + factory.toString());

    }
    
	@Override
	public void createInstance(String session, EIInstance instance)
			throws Exception {
		try {
			provider.createInstance(session, instance);
		} catch (Exception e) {
			throw new LoggedException(e.getMessage());
		}
		
	}

	@Override
	public void deleteInstance(String session, EIInstance instance)
			throws Exception {
		try {
			provider.deleteInstance(session, instance);
		} catch (Exception e) {
			throw new LoggedException(e.getMessage());
		}
		
	}

    @Override
	public String updateInstance(String session, EIInstance instance, String token)
			throws Exception {
		try {
			return provider.updateInstance(session, instance, token);
		} catch (Exception e) {
			throw new LoggedException(e.getMessage());
		}
		
	}

    @Override
    public String[] login(final String user, final String password) throws Exception
    {
		try {
			return provider.login(user, password);
		} catch (Exception e) {
			throw new LoggedException(e.getMessage());
		}
    }

    @Override
    public void logout(String session) throws Exception
    {
		try {
			provider.logout(session);
		} catch (Exception e) {
			throw new LoggedException(e.getMessage());
		}
    }

    @Override
    public String[] whoami(String session) throws Exception
    {
    	try {
    		return provider.whoami(session);
    	} catch (Exception e) {
    		log.error("whoami exception " + e);
			throw new LoggedException(e.getMessage());
		}
    }


    /*
     * Loads the DatatoolsConfiguration.
     */
    private DatatoolsConfiguration getConfiguration() throws ServletException
    {
        // check if the system property is specified
        String propFile = System.getProperty(DataToolsOntConstants.DATATOOLS_CONFIG);

        if(propFile == null)
        {
            // use the default
            propFile = DataToolsOntConstants.DEFAULT_DATATOOLS_CONFIG;
        }
        log.debug("Using properties file " + propFile);

        final URL configURL = this.getClass().getClassLoader().getResource(propFile);
        if(configURL == null)
        {
            log.error("Could not locate " + propFile + " on classpath ");
            throw new ServletException("Could not locate " + propFile + " on classpath ");
        }
        try
        {
            final File configFile = new File(configURL.toURI());
            return new DatatoolsConfiguration(configFile);
        }
        catch(Exception e)
        {
            log.error("Error loading configuration from " + configURL + " " + e);
            throw new ServletException("Error loading configuration from " + configURL + " " + e);
        }
    }

    @Override
    public boolean isOnline()
    {
        
        return provider.isOnline();
    }

	@Override
	public List<EIInstance> getAllResources(String session, String rnav)
			throws Exception {
		try {
			return provider.getAllResources(session, rnav);
		} catch (Exception e) {
			throw new LoggedException(e.getMessage());
		}
	}

	@Override
	public List<EIInstance> getResourcesOfClass(String session, String rnav,
			EIURI classUri) throws Exception {
		try {
			return provider.getResourcesOfClass(session, rnav, classUri);
		} catch (Exception e) {
			throw new LoggedException(e.getMessage());
		}
	}
	
    @Override
    public List<EIInstance> EIQuery(final String session, final String query) throws Exception{
		try {
			return provider.EIQuery(session, query);
		} catch (Exception e) {
			throw new LoggedException(e.getMessage());
		}
    }

    @Override
    public List<EIInstance> getInstancesForLab(final String session, final String rnav, final String labURI) throws Exception{
    	
		try {
			return provider.getInstancesForLab(session, rnav, labURI);
		} catch (Exception e) {
			throw new LoggedException(e.getMessage());
		}
    }
    
	@Override
	public void uploadInstances(String session, String rdf) throws Exception {
		provider.uploadInstances(session, rdf);
		
	}

	@Override
	public EIInstance getEmptyEIInstance(String session, EIURI classUri,
			EIEntity instanceEntity) throws Exception {
		try {
			return provider.getEmptyEIInstance(session, classUri,
					instanceEntity);
		} catch (Exception e) {
			throw new LoggedException(e.getMessage());
		}
	}

	@Override
	public List<EIURI> getNewInstanceID(String session, int count)
			throws Exception {
		try {
			return provider.getNewInstanceID(session, count);
		} catch (Exception e) {
			throw new LoggedException(e.getMessage());
		}
	}

	@Override
	public String query(String session, String sparql) throws Exception {
		try {
			return provider.query(session, sparql);
		} catch (Exception e) {
			throw new LoggedException(e.getMessage());
		}
	}

	@Override
	public EIInstance getInstance(String session, EIURI instanceID)
			throws Exception {
		return provider.getInstance(session, instanceID);
	}

	@Override
	public String claim(String session, String uri, String claimant)
			throws Exception {
		try {
			return provider.claim(session, uri, claimant);
		} catch (Exception e) {
			throw new LoggedException(e.getMessage());
		}
	}

	@Override
	public String promote(String session, String uri, String newState)
			throws Exception {
		try
		{
			return provider.promote(session, uri, newState);
		}
		catch(Exception e)
		{
			throw new LoggedException(e.getMessage());
		}
	}
	
	

	public String[] bulkPromote(String session, String[] uri, String newState)
			throws Exception {
		try
		{
			String[] retStrAry=new String[uri.length];
			int i=0;
			for(String strUri:uri)
			{
				 retStrAry[i]= provider.promote(session, strUri, newState);
				 i++;
			}
			return retStrAry;
			
		}
		catch(Exception e)
		{
			throw new LoggedException(e.getMessage());
		}
	}
	
	
	

	/* (non-Javadoc)
	 * @see org.eaglei.datatools.provider.RepositoryProvider#getWFStates(java.lang.String)
	 */
	@Override
	public String[] getWFStates(String session, String user) throws Exception {
		try
		{
			return provider.getWFStates(session, user);
		}
		catch(Exception e)
		{
			throw new LoggedException(e.getMessage());
		}
	}

	@Override
	public List<EIInstance> getFilterQuery(final String session, String user, EIURI classUri,
			EIURI state, EIURI lab) throws Exception {
		try
		{
			return provider.getFilterQuery(session, user, classUri, state, lab);
		}
		catch(Exception e)
		{
			throw new LoggedException(e.getMessage());
		}	
	}
	
	@Override
	public String retrieveLabel(String session, EIURI uri) throws Exception {
		try
		{
			return provider.retrieveLabel(session, uri);
		}
		catch(Exception e)
		{
			throw new LoggedException(e.getMessage());
		}
	}
}
