package org.eaglei.datatools.client.ui;

import java.util.HashMap;
import java.util.Iterator;
import java.util.Set;

import org.eaglei.model.EIEntity;
import org.eaglei.model.EIURI;

public class QueryTokenObject {

	private static final String	MODE_VARIABLE	= "mode";

	public static enum Mode {
		edit, filter, labs, list, references, resources, view
	};

	protected static final String	INSTANCE_URI_VARIABLE		= "instanceUri";
	protected static final String	INSTANCE_LABEL_VARIABLE		= "instanceLabel";
	protected static final String	TYPE_URI_VARIABLE			= "typeUri";
	protected static final String	TYPE_LABEL_VARIABLE			= "typeLabel";
	protected static final String	LAB_URI_VARIABLE			= "labUri";
	protected static final String	LAB_LABEL_VARIABLE			= "labLabel";
	protected static final String	WORKSPACE_URI_VARIABLE		= "workspaceUri";
	protected static final String	WORKSPACE_LABEL_VARIABLE	= "workspaceLabel";

	protected static final String	FILTER_TYPE_LABEL_VARIABLE	= "filterTypeLabel";
	protected static final String	FILTER_LAB_LABEL_VARIABLE	= "filterLabLabel";
	protected static final String	FILTER_WF_LABEL_VARIABLE	= "filterWorkflowLabel";
	protected static final String	FILTER_TYPE_URI_VARIABLE	= "filterTypeUri";
	protected static final String	FILTER_LAB_URI_VARIABLE		= "filterLabUri";
	protected static final String	FILTER_WF_URI_VARIABLE		= "filterWorkflowUri";
	protected static final String	FILTER_BY_OWNER				= "strictOwnerFilter";

	private Mode					mode						= null;
	private EIEntity				instanceEntity				= EIEntity.NULL_ENTITY;
	private EIURI					instanceUri					= EIURI.NULL_EIURI;
	private String					instanceLabel				= null;
	private EIEntity				typeEntity					= EIEntity.NULL_ENTITY;
	private EIURI					typeUri						= EIURI.NULL_EIURI;
	private String					typeLabel					= null;
	private EIEntity				labEntity					= EIEntity.NULL_ENTITY;
	private EIURI					labUri						= EIURI.NULL_EIURI;
	private String					labLabel					= null;
	private EIEntity				filterLabEntity				= EIEntity.NULL_ENTITY;
	private EIEntity				filterTypeEntity			= EIEntity.NULL_ENTITY;
	private EIEntity				filterWorkflowEntity		= EIEntity.NULL_ENTITY;
	private EIEntity				workspaceEntity				= EIEntity.NULL_ENTITY;
	protected static final Mode		defaultMode					= Mode.resources;
	private boolean					filterByOwner;
	private HashMap<String, String>	variableMap					= new HashMap<String, String>();

	public QueryTokenObject() {
		super();
		setMode(defaultMode);
	}

	public QueryTokenObject(String tokenString) {
		fromString(tokenString);
	}

	public Mode getMode() {
		return mode == null ? defaultMode : mode;
	}

	public void setMode(Mode mode) {
		this.mode = mode;
		variableMap.put(MODE_VARIABLE, mode.toString());
	}

	public EIEntity getInstanceEntity() {
		return this.instanceEntity;
	}

	public void setInstanceEntity(EIEntity instanceEntity) {
		this.instanceEntity = instanceEntity;
		addToMap(instanceEntity, INSTANCE_URI_VARIABLE, INSTANCE_LABEL_VARIABLE);
	}

	public EIEntity getTypeEntity() {
		return this.typeEntity;
	}

	public void setTypeEntity(EIEntity typeEntity) {
		this.typeEntity = typeEntity;
		addToMap(typeEntity, TYPE_URI_VARIABLE, TYPE_LABEL_VARIABLE);
	}

	public EIEntity getLabEntity() {
		return this.labEntity;
	}

	public void setLabEntity(EIEntity labEntity) {
		this.labEntity = labEntity;
		addToMap(labEntity, LAB_URI_VARIABLE, LAB_LABEL_VARIABLE);
	}

	public EIEntity getWorkspaceEntity() {
		return workspaceEntity;
	}

	public void setWorkspaceEntity(EIEntity workspaceEntity) {
		this.workspaceEntity = workspaceEntity;
		addToMap(workspaceEntity, WORKSPACE_URI_VARIABLE, WORKSPACE_LABEL_VARIABLE);
	}

	public EIEntity getFilterLabEntity() {
		return filterLabEntity;
	}

	public void setFilterLabEntity(EIEntity filterLabEntity) {
		this.filterLabEntity = filterLabEntity;
		addToMap(filterLabEntity, FILTER_LAB_URI_VARIABLE, FILTER_LAB_LABEL_VARIABLE);
	}

	public EIEntity getFilterTypeEntity() {
		return filterTypeEntity;
	}

	public void setFilterTypeEntity(EIEntity filterTypeEntity) {
		this.filterTypeEntity = filterTypeEntity;
		addToMap(filterTypeEntity, FILTER_TYPE_URI_VARIABLE, FILTER_TYPE_LABEL_VARIABLE);
	}

	public EIEntity getFilterWorkflowEntity() {
		return filterWorkflowEntity;
	}

	public void setFilterWorkflowEntity(EIEntity filterWorkflowEntity) {
		this.filterWorkflowEntity = filterWorkflowEntity;
		addToMap(filterWorkflowEntity, FILTER_WF_URI_VARIABLE, FILTER_WF_LABEL_VARIABLE);
	}

	protected void addToMap(EIEntity newEntity, String uriVariable, String labelVariable) {
		if (hasUri(newEntity)) {
			variableMap.put(uriVariable, newEntity.getURI().toString());
			variableMap.put(labelVariable, newEntity.getLabel().toString());
		} else {
			variableMap.remove(uriVariable);
			variableMap.remove(labelVariable);
		}
	}

	protected void addToMap(String label, String value) {
		variableMap.put(label, value);
	}

	public boolean isStrictlyFilteredByOwner() {
		return filterByOwner;
	}

	public void setStrictFilterByOwner(boolean filterByOwner) {
		this.filterByOwner = filterByOwner;
		addToMap(FILTER_BY_OWNER, filterByOwner + "");
	}

	@Override
	public String toString() {
		Set<String> keySet = variableMap.keySet();
		if (keySet.size() == 0)
			return "";
		StringBuffer sbuf = new StringBuffer();
		sbuf.append("history?");
		Iterator<String> iter = keySet.iterator();
		while (iter.hasNext()) {
			String key = iter.next();
			sbuf.append(key + "=" + variableMap.get(key) + "&");
		}
		return sbuf.toString().replaceAll("&$", "");
	}

	/**
	 * parses the given token and fills the Member variables
	 * 
	 * @param token
	 * @return
	 */
	public QueryTokenObject fromString(String token) {
		variableMap.clear();
		String[] splitArray = token.split("\\?")[1].split("&");
		for (String splitedString : splitArray) {
			String[] variableFromSplitString = splitedString.split("=");
			if (variableFromSplitString != null && variableFromSplitString.length > 1) {
				variableMap.put(variableFromSplitString[0], variableFromSplitString[1]);
			}
		}
		initializeEntities();
		validate();
		return this;
	}

	private EIEntity getEntityByVariable(String uriName, String labelName) {
		String uriValue;
		String labelValue;
		if (variableMap != null) {
			uriValue = variableMap.get(uriName);
			if (uriValue == null || uriValue.equals("") || uriValue.equals(EIURI.NULL_EIURI.toString()))
				return EIEntity.NULL_ENTITY;
			labelValue = variableMap.get(labelName);
			return EIEntity.create(uriValue, labelValue);
		} else
			return EIEntity.NULL_ENTITY;
	}

	//Fills default info, checks that parameters are present in the right configurations
	private boolean validate() {
		//If there is an instanceUri and there is no mode, we should set the mode to view
		if (hasUri(instanceUri) && mode == null) {
			mode = Mode.view;
			setMode(mode);
			return true;
		}
		if (mode == Mode.edit || mode == Mode.view && !hasUri(instanceUri)) {
			//Window.alert(UIMessages.NO_INSTANCEURI_IN_VIEW_EDIT);
			return false;
		}
		if (hasLabel(typeLabel) && !hasUri(typeUri)) {
			//Window.alert(UIMessages.TYPELABEL_BUT_NO_TYPEURI);
			return false;
		}
		if (hasLabel(labLabel) && !hasUri(labUri)) {
			//	Window.alert(UIMessages.LABLABEL_BUT_NO_LABURI);
			return false;
		}
		return true;
	}
	
	protected boolean hasLabelButNoUri(String label, EIURI uri) {
		return hasLabel(label) && ! hasUri(uri);
	}
	
	protected boolean hasLabel(String label) {
		if (label == null) {
			return false;
		}
		return !label.equals("") && !label.equals(EIEntity.NO_LABEL);
	}

	private void initializeEntities() {
		filterTypeEntity = getEntityByVariable(FILTER_TYPE_URI_VARIABLE, FILTER_TYPE_LABEL_VARIABLE);
		filterLabEntity = getEntityByVariable(FILTER_LAB_URI_VARIABLE, FILTER_LAB_LABEL_VARIABLE);
		filterWorkflowEntity = getEntityByVariable(FILTER_WF_URI_VARIABLE, FILTER_WF_LABEL_VARIABLE);
		instanceEntity = getEntityByVariable(INSTANCE_URI_VARIABLE, INSTANCE_LABEL_VARIABLE);
		typeEntity = getEntityByVariable(TYPE_URI_VARIABLE, TYPE_LABEL_VARIABLE);
		if (hasUri(typeEntity.getURI()))
			setTypeEntity(typeEntity);
		labEntity = getEntityByVariable(LAB_URI_VARIABLE, LAB_LABEL_VARIABLE);
		if (hasUri(labEntity.getURI()))
			setLabEntity(labEntity);
		instanceUri = instanceEntity.getURI();
		instanceLabel = instanceEntity.getLabel();
		typeUri = typeEntity.getURI();
		typeLabel = typeEntity.getLabel();
		labUri = labEntity.getURI();
		labLabel = labEntity.getLabel();
		
		String modeValue = variableMap.get(MODE_VARIABLE);
		if (modeValue == null) {
			mode = null;
		}
		for (Mode whichMode : Mode.values()) {
			if (whichMode.toString().equals(modeValue)) {
				mode = whichMode;
			}
		}
	}

	protected boolean hasUri(EIEntity entity) {
		return hasUri(entity.getURI());
	}

	protected boolean hasUri(EIURI uri) {
		return uri != null && !uri.toString().equals("") && uri != EIURI.NULL_EIURI;
	}
}
