package org.eaglei.datatools.client.rpc;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import org.eaglei.datatools.User;
import org.eaglei.datatools.client.Datatools;
import org.eaglei.datatools.client.DatatoolsCookies;
import org.eaglei.datatools.client.WorkFlowConstants;
import org.eaglei.datatools.client.ui.ApplicationState;
import org.eaglei.datatools.client.ui.DatatoolsUIConstants;
import org.eaglei.model.EIClass;
import org.eaglei.model.EIEntity;
import org.eaglei.model.EIInstance;
import org.eaglei.model.EIInstanceMinimal;
import org.eaglei.model.EIProperty;
import org.eaglei.model.EIURI;
import org.eaglei.model.gwt.rpc.LoggedException;
import org.eaglei.model.gwt.rpc.ModelService;
import org.eaglei.model.gwt.rpc.ModelServiceAsync;
import org.eaglei.model.gwt.rpc.ClientModelManager.ClassCallback;
import org.eaglei.security.Session;

import com.allen_sauer.gwt.log.client.Log;
import com.google.gwt.core.client.GWT;
import com.google.gwt.user.client.Cookies;
import com.google.gwt.user.client.Window;
import com.google.gwt.user.client.rpc.AsyncCallback;
import com.google.gwt.user.client.rpc.IncompatibleRemoteServiceException;
import com.google.gwt.user.client.rpc.InvocationException;

/**
 * Maintains a client-side cache of EIInstance. Proxies all model RPC methods.
 * 
 * It is critical that all model RPC calls go through this class. All methods in
 * this class MUST call getCached(EIInstance) on all EIInstance objects it
 * receives from the server to ensure that there is only one instance of an
 * EIInstance per URI in the client.
 * 
 * 
 */
public class ClientRepositoryToolsManager {

	public interface SessionListener {

		void onLogIn(String username, String userUri);

		void onLogOut(); // Notification that a logout occurred
	}

	public interface LoginRequiredCallback {

		void loginRequired();
	}

	public interface ResultsCallback extends LoginRequiredCallback {

		void onSuccess(String[] arg0);

		void onSuccess(String arg0);
	}

	public interface SaveResultsCallback extends LoginRequiredCallback {

		void onSuccess(String arg0);

		void onFailure();
	}

	public interface EIInstanceCallback extends LoginRequiredCallback {

		void onSuccess(EIInstance eiInstance);

		void onFailure();
	}

	@Deprecated
	public interface EIInstancesForLabCallback extends LoginRequiredCallback {

		void onSuccess(List<EIInstanceMinimal> eiInstance);
	}

	public interface NewInstanceCallback extends LoginRequiredCallback {

		void onSuccess(Object obj);
	}

	public interface EIInstancesCallback extends LoginRequiredCallback {

		void onSuccess(List<EIInstanceMinimal> result);
	}

	@Deprecated
	public interface QueryEIInstancesCallback extends LoginRequiredCallback {

		void onSuccess(List<EIInstanceMinimal> result);
	}

	@Deprecated
	public interface FilterInstancesCallback extends LoginRequiredCallback {

		void onSuccess(List<EIInstanceMinimal> result);
	}

	public interface IdCallback extends LoginRequiredCallback {

		void onSuccess(List<EIURI> list);
	}

	public interface UserCallback extends LoginRequiredCallback {

		void onSuccess(User userInfo);
	}

	public interface WFCallback {

		void onSuccess(String[] wfStates);
	}

	public interface DeleteInstanceCallback {
		void onSuccess(Object obj);
	}

	public interface LabelsCallback {
		void onSuccess(Map<EIEntity, String> labelMap);
	}

	public interface UriLabelsCallback {
		void onSuccess(Map<EIURI, String> labelMap);
	}

	public interface EIClassesCallback {

		void onSuccess(List<EIClass> classes);
		void onFailure(String result);
	}
	
    public interface EquivalentClassesCallback {
        void onSuccess(List<EIProperty> populatedProperties);
		void onFailure(String result);
    }


	public static final ClientRepositoryToolsManager	INSTANCE		= new ClientRepositoryToolsManager();
	public static RepositoryToolsModelServiceAsync		repositoryService;
	public static ModelServiceAsync						modelService;
	private final HashMap<EIURI, EIInstanceMinimal>		mapIdInstance	= new HashMap<EIURI, EIInstanceMinimal>();
	private final HashMap<EIURI, EIClass>				mapIdToClass	= new HashMap<EIURI, EIClass>();
	private Session										session;
	private EIEntity									user;														// TODO: probably get rid of this again
	private ArrayList<SessionListener>					listeners;
	private List<EIURI>									editableStates;

	enum LoginFields {
		USERNAME(0), SESSION(1), USER_URI(2), ERROR_MESSAGE(3), WF_STATE(3);

		private final int	value;

		private LoginFields(final int value) {
			this.value = value;
		}

		public int getValue() {
			return value;
		}
	}

	private ClientRepositoryToolsManager() {
		String userUriString = DatatoolsCookies.getUserUri();
		if (userUriString != null) {
			user = EIEntity.create(EIURI.create(userUriString), DatatoolsCookies.getUserName());
			Log.info("got existing userUri: " + user);
		}
		session = DatatoolsCookies.getSession();
		if (session != null)
			Log.info("got session ID " + session.getSessionId());
		repositoryService = GWT.create(RepositoryToolsModelService.class);
		modelService = GWT.create(ModelService.class);
	}

	public boolean isLoggedIn() {
		return Session.isValid(session);
	}

	// Shamelessly stolen from ClientSearchManager; TODO: refactor/merge
	private void handleLogOut() {
		session = null;
		user = null;
		Cookies.removeCookie(DatatoolsCookies.DATATOOLS_USER_COOKIE_ID);
		Cookies.removeCookie(DatatoolsCookies.DATATOOLS_USER_URI_ID);
		Cookies.removeCookie(DatatoolsCookies.DATATOOLS_SESSION_ID);
		Log.info("removed cookies");
		if (listeners != null) {
			for (SessionListener listener : listeners) {
				listener.onLogOut();
			}
		}
	}

	public void addSessionListener(SessionListener listener) {
		if (listeners == null) {
			listeners = new ArrayList<SessionListener>();
		}
		this.listeners.add(listener);
		Log.info("added listener");
		/*
		 * if (isLoggedIn()) { listener.onLogIn(""); }
		 */
	}

	public void logOut() {
		if (!isLoggedIn()) {
			// ?? Probably should fire logOut even in deferred command
			return;
		}
		// Need to wait for callback, or ok to just assume logged out?
		try {
			repositoryService.logout(session, new AsyncCallback<Void>() {

				@Override
				public void onFailure(Throwable caught) {
					// Window.alert("Error logging out.");
					handleLogOut();
				}

				@Override
				public void onSuccess(Void result) {
					session = null;
					user = null;
					handleLogOut();
				}
			});
		} catch (Exception e) {
			e.printStackTrace();
		}
	}

	public void logIn(final String username, final String password, final ResultsCallback callback) {
		final int USERNAME = 0;
		if (isLoggedIn()) {
			try {
				repositoryService.whoami(session, new AsyncCallback<String[]>() {

					@Override
					public void onFailure(Throwable caught) {
						user = null;
						if (caught instanceof Exception) {
							Window.alert(caught.getMessage());
						} else {
							Window.alert("Error logging in. ");
						}
					}

					@Override
					public void onSuccess(String[] result) {
						if (username.equals(result[USERNAME])) {
							callback.onSuccess(username);
						} else {
							user = null;
							session = null;
							Window.alert("Error logging in: already logged in as different user.  Please log out first.");
						}
					}
				});
			} catch (Exception e) {
			}
		} else { //FIXME verify
			try {
				repositoryService.login(username, password, new AsyncCallback<User>() {

					@Override
					public void onFailure(Throwable caught) {
						user = null;
						session = null;

						if (caught instanceof Exception) {
							Window.alert(caught.getMessage());
						} else {
							Window.alert("Error logging in. ");
						}

					}

					@Override
					public void onSuccess(final User result) {
						Datatools.showWorkspaceChooser(result.getWorkspaces());
						Datatools.setUser(result);
						session = result.getSession();
						String userUri = result.getUserURI().toString();
						String userName = result.getUserName();
						user = EIEntity.create(EIURI.create(userUri), userName);
						Cookies.setCookie(DatatoolsCookies.DATATOOLS_USER_COOKIE_ID, username);
						Cookies.setCookie(DatatoolsCookies.DATATOOLS_USER_URI_ID, userUri);
						Cookies.setCookie(DatatoolsCookies.DATATOOLS_SESSION_ID, session.getSessionId());
						// Suggestion from GWT security is to use
						// Cookies.setCookie(DatatoolsCookies.DATATOOLS_USER_COOKIE_ID,
						// username, expires, null, "/", true);
						// and call checkValidSession() on server
						// --presumably call whoami & check it's valid not
						// invalid
						Log.info("set cookies; userUri = " + Cookies.getCookie(DatatoolsCookies.DATATOOLS_USER_URI_ID) + " session = " + Cookies.getCookie(DatatoolsCookies.DATATOOLS_SESSION_ID));
						initializeEditableStates(result);
						callback.onSuccess(result.getUserName());
						if (listeners != null) {
							for (SessionListener listener : listeners) {
								listener.onLogIn(result.getUserName(), result.getUserURI().toString());
							}
						}

					}
				});
			} catch (Exception e) {
				e.printStackTrace();
			}
		}
	}

	public void updateInstance(final EIInstance eiInstance, String token, final SaveResultsCallback callback) throws Exception {
		Log.info("trying to update " + eiInstance.getInstanceLabel());
		repositoryService.updateInstance(session, eiInstance, token, new AsyncCallback<String>() {

			@Override
			public void onFailure(Throwable arg0) {
				Log.warn("update failed for" + eiInstance.getInstanceLabel() + " " + arg0.getMessage());
				if (arg0 instanceof LoggedException && arg0.getMessage().contains("No session information was found")) {
					callback.loginRequired();
				} else
					callback.onFailure();
			}

			@Override
			public void onSuccess(String arg0) {
				if (arg0 == null) // because GWT makes it hard to throw
				// exceptions; typically get null if
				// not logged in
				{
					callback.loginRequired();
					return;
				}
				Log.info("update succeeded for " + eiInstance.getInstanceLabel());
				callback.onSuccess(arg0);
			}
		});
	}

	public void getInstance(final EIURI eiURI, final EIInstanceCallback callback) throws Exception {
		repositoryService.getInstance(session, eiURI, new AsyncCallback<EIInstance>() {

			@Override
			public void onFailure(Throwable arg0) {
				handleFailure(callback, arg0);
			}

			@Override
			public void onSuccess(EIInstance arg0) {
				if (arg0 == null) // because GWT makes it hard to throw
				// exceptions
				{
				}
				callback.onSuccess(arg0);
			}
		});
	}

	public void deleteInstance(EIURI instanceUri, final DeleteInstanceCallback callback) throws Exception {
		repositoryService.deleteInstance(session, instanceUri, new AsyncCallback() {

			@Override
			public void onFailure(Throwable arg0) {
			}

			@Override
			public void onSuccess(Object arg0) {
				callback.onSuccess(arg0);
			}
		});
	}

	private void initializeEditableStates(User userInfo) {
		Log.info("initializing editable states with " + userInfo.getWFSStateList().size() + " states");
		editableStates = new ArrayList<EIURI>();
		/*
		 * if (userInfo.length <= LoginFields.WF_STATE.getValue()) return;
		 */

		Iterator<EIURI> iter = userInfo.getWFSStateList().iterator();
		while (iter.hasNext()) {
			EIURI stateURI = iter.next();
			editableStates.add(stateURI);
			Log.info("adding editable state " + stateURI);
		}
	}

	public boolean canEdit(EIURI workflowState) {
		if (editableStates == null) {
			Log.info("editable states null");
			return false; // TODO: throw? Shouldn't happen if logged in
		}
		if (workflowState == null) {
			return false;
		}
		// Log.info("asking if can edit " + workflowState + "; answer "
		// + editableStates.contains(workflowState));
		return editableStates.contains(workflowState);
	}

	public List<EIURI> getEditableStates() {
		return editableStates;
	}

	public void getFilterQuery(final EIURI userUri, final EIURI classUri, final EIURI state, final EIURI lab, final EIInstancesCallback callback) throws Exception {
		getFilterQuery(userUri.toString(), classUri, state, lab, false, callback);
	}

	public void getFilterQuery(final EIURI userUri, final EIURI classUri, final EIURI state, final EIURI lab, final boolean strictOwnerFilter, final EIInstancesCallback callback) throws Exception {
		getFilterQuery(userUri.toString(), classUri, state, lab, strictOwnerFilter, callback);
	}

	public void getFilterQuery(final String user, final EIURI classUri, final EIURI state, final EIURI lab, boolean strictOwnerFilter, final EIInstancesCallback callback) throws Exception {
		repositoryService.getFilterQuery(session, classUri, state, lab, strictOwnerFilter, new AsyncCallback<List<EIInstanceMinimal>>() {

			// TODO deal with 400 from repo (callback.onFailure?)
			@Override
			public void onFailure(Throwable arg0) {
				handleFailure(callback, arg0);
			}

			@Override
			public void onSuccess(List<EIInstanceMinimal> arg0) {
				callback.onSuccess(arg0);
			}
		});
	}

	public void referencedByQuery(final EIURI userUri, final EIURI resourceUri, final boolean strictOwnerFilter, final EIInstancesCallback callback) throws Exception {
		repositoryService.referencedByQuery(session, resourceUri, strictOwnerFilter, new AsyncCallback<List<EIInstanceMinimal>>() {

			@Override
			public void onFailure(Throwable arg0) {
				handleFailure(callback, arg0);
			}

			@Override
			public void onSuccess(List<EIInstanceMinimal> arg0) {
				callback.onSuccess(arg0);
			}
		});
	}

	private EIInstanceMinimal getCached(EIInstanceMinimal c) {
		if (mapIdInstance.containsKey(c.getInstanceURI())) {
			return mapIdInstance.get(c.getInstanceURI());
		} else {
			mapIdInstance.put(c.getInstanceURI(), c);
			return c;
		}
	}

	private void updateCached(EIInstanceMinimal instance, EIEntity workFlowState, EIEntity owner) {
		WorkFlowConstants wfc = new WorkFlowConstants();
		EIInstanceMinimal cached = getCached(instance);
		if (workFlowState != null && !workFlowState.equals(EIEntity.NULL_ENTITY))
			cached.setWFState(workFlowState);
		// TODO: (EIEntity) do the owner right!
		if (owner != null && !owner.equals(EIEntity.NULL_ENTITY))
			cached.setWFOwner(owner);
	}

	public void getNewInstanceID(int count, final IdCallback callback) {
		try {
			repositoryService.getNewInstanceID(session, count, new AsyncCallback<List<EIURI>>() {

				@Override
				public void onFailure(Throwable arg0) {
					handleFailure(callback, arg0);
				}

				@Override
				public void onSuccess(List<EIURI> list) {
					if (list == null) // because GWT makes it hard to
					// throw exceptions
					{
						callback.loginRequired();
						return;
					}
					callback.onSuccess(list);
				}
			});
		} catch (Exception e) {
			e.printStackTrace();
		}
	}

	public void createInstance(final EIInstance instance, final EIInstanceCallback callback) {
		try {
			Log.info("creating instance " + instance.getEntity() + " with type " + instance.getInstanceClass());
			repositoryService.createInstance(session, instance, ApplicationState.getInstance().getWorkspaceEntity(), new AsyncCallback() {

				@Override
				public void onFailure(Throwable arg0) {
					Window.alert(arg0.getMessage());
					Log.warn("failed " + arg0);
					handleFailure(callback, arg0);
				}

				@Override
				public void onSuccess(Object arg0) {
					Log.info("creation succeeded");
					instance.setWFState(WorkFlowConstants.DRAFT_ENTITY);
					callback.onSuccess(instance);
				}
			});
		} catch (Exception e) {
			e.printStackTrace();
		}
	}

	public void getEmptyEIInstance(EIURI classUri, final EIEntity instanceEntity, final EIInstanceCallback callback) {
		repositoryService.getEmptyEIInstance(session, classUri, instanceEntity, new AsyncCallback<EIInstance>() {

			@Override
			public void onFailure(Throwable arg0) {
				handleFailure(callback, arg0);
			}

			@Override
			public void onSuccess(EIInstance instance) {
				if (instance == null) // because GWT makes it hard to
				// throw exceptions
				{
					callback.loginRequired();
					return;
				}
				callback.onSuccess(instance);
			}
		});
	}

	public void getEmptyEIInstance(final EIURI classUri, final EIInstanceCallback callback) {
		repositoryService.getEmptyEIInstance(session, classUri, new AsyncCallback<EIInstance>() {

			@Override
			public void onFailure(Throwable arg0) {
				Log.warn("failed to get empty instance with class " + classUri);
				handleFailure(callback, arg0);
			}

			@Override
			public void onSuccess(EIInstance instance) {
				if (instance == null) // because GWT makes it hard to
				// throw exceptions
				{
					Log.warn("login required for getting empty instance");
					callback.loginRequired();
					return;
				}
				Log.info("got empty instance with class " + classUri);
				callback.onSuccess(instance);
			}
		});
	}

	public void deepCopyInstance(final EIURI originalUri, final EIInstanceCallback callback) {
		try {
			repositoryService.deepCopy(session, originalUri, new AsyncCallback<EIInstance>() {

				@Override
				public void onFailure(Throwable caught) {
				}

				@Override
				public void onSuccess(EIInstance result) {
					callback.onSuccess(result);
				}
			});
		} catch (Exception e) {
			Log.error(e.getMessage());
		}
	}

	// TODO:
	public void query(String sparql, AsyncCallback<StringBuffer> callback) throws Exception {
	}

	// TODO:
	public void isOnline(AsyncCallback callback) throws Exception {
	}

	public void whoami(final UserCallback callback) throws Exception {
		if (repositoryService == null) {
			repositoryService = GWT.create(RepositoryToolsModelService.class);
		}
		Session session = DatatoolsCookies.getSession();
		repositoryService.whoami(session, new AsyncCallback<User>() {

			@Override
			public void onFailure(Throwable arg0) {
				Log.warn("client whoami failure");
				if (arg0 != null)
					Log.warn("client whoami failure: " + arg0.getMessage());
			}

			@Override
			public void onSuccess(User userInfo) {
				Log.debug("model service whoami succeeded with " + (userInfo == null ? "null" : userInfo.getWFSStateList().size()) + " fields");
				if (userInfo != null) {
					initializeEditableStates(userInfo);
				}
				callback.onSuccess(userInfo);
			}
		});
	}

	public void returnToDraft(final EIInstance instance, final ResultsCallback callback) throws Exception {
		promote(instance, WorkFlowConstants.DRAFT_ENTITY, callback);
	}

	public void sendToCuration(final EIInstance instance, final ResultsCallback callback) throws Exception {
		promote(instance, WorkFlowConstants.CURATION_ENTITY, callback);
	}

	public void publish(final EIInstance instance, final ResultsCallback callback) throws Exception {
		promote(instance, WorkFlowConstants.PUBLISH_ENTITY, callback);
	}

	public void withdraw(final EIInstance instance, final ResultsCallback callback) throws Exception {
		Log.info("withdrawing " + instance);
		promote(instance, WorkFlowConstants.WITHDRAW_ENTITY, callback);
	}

	public void withdraw(final EIInstanceMinimal instance, final ResultsCallback callback) throws Exception {
		Log.info("withdrawing " + instance);
		promote(instance, WorkFlowConstants.WITHDRAW_ENTITY, callback);
	}

	public void bulkPublish(final List<EIInstanceMinimal> instancesToPromote, final ResultsCallback callback) throws Exception {
		bulkPromote(instancesToPromote, WorkFlowConstants.PUBLISH_ENTITY, callback);
	}

	public void bulkWithdraw(final List<EIInstanceMinimal> instancesToPromote, final ResultsCallback callback) throws Exception {
		bulkPromote(instancesToPromote, WorkFlowConstants.WITHDRAW_ENTITY, callback);
	}

	public void bulkReturnToDraft(final List<EIInstanceMinimal> instancesToPromote, final ResultsCallback callback) throws Exception {
		bulkPromote(instancesToPromote, WorkFlowConstants.DRAFT_ENTITY, callback);
	}

	public void bulkSendToCuration(final List<EIInstanceMinimal> instancesToPromote, final ResultsCallback callback) throws Exception {
		bulkPromote(instancesToPromote, WorkFlowConstants.CURATION_ENTITY, callback);
	}

	public void promote(final EIInstance instance, final EIEntity newState, final ResultsCallback callback) throws Exception {
		if (isLoggedIn()) {
			Log.info("promoting '" + instance.getInstanceURI().toString() + "' to " + newState);
			repositoryService.promote(session, instance.getInstanceURI(), newState.getURI(), new AsyncCallback<String>() {

				@Override
				public void onFailure(Throwable arg0) {
				}

				@Override
				public void onSuccess(String arg0) {
					if (arg0 == null) {
						// what to do? Need to let everyone know it failed
						Window.alert("could not promote " + instance.getInstanceLabel() + "(null response)"); // TODO: is this right?
						return;
					}
					instance.setWFState(newState);
					callback.onSuccess(arg0);
				}
			});
		}
	}

	// TODO re-implement the two versions with common code
	public void promote(final EIInstanceMinimal instance, final EIEntity newState, final ResultsCallback callback) throws Exception {
		if (isLoggedIn()) {
			Log.info("promoting '" + instance.getInstanceURI().toString() + "' to " + newState);
			repositoryService.promote(session, instance.getInstanceURI(), newState.getURI(), new AsyncCallback<String>() {

				@Override
				public void onFailure(Throwable arg0) {
				}

				@Override
				public void onSuccess(String arg0) {
					if (arg0 == null) {
						// what to do? Need to let everyone know it failed
						Window.alert("could not promote " + instance.getInstanceLabel() + "(null response)"); // TODO: is this right?
						return;
					}
					instance.setWFState(newState);
					callback.onSuccess(arg0);
				}
			});
		}
	}

	public void bulkPromote(final List<EIInstanceMinimal> instancesToPromote, final EIEntity newState, final ResultsCallback callback) throws Exception {
		if (isLoggedIn()) {
			List<EIURI> urisToPromote = new ArrayList<EIURI>(instancesToPromote.size());
			for (EIInstanceMinimal instance : instancesToPromote) {
				urisToPromote.add(instance.getInstanceURI());
			}
			repositoryService.bulkPromote(session, urisToPromote, newState.getURI(), new AsyncCallback<String[]>() {

				@Override
				public void onFailure(Throwable arg0) {
				}

				@Override
				public void onSuccess(String[] arg0) {
					if (arg0 == null) {
						Window.alert("sucess");
						return;
					}
					for (EIInstanceMinimal instance : instancesToPromote) {
						updateCached(instance, newState, EIEntity.NULL_ENTITY);
						instance.setWFState(newState);
						Log.info("forced state of " + instance.getInstanceLabel() + " to " + newState);
					}
					callback.onSuccess(arg0);
				}
			});
		}
	}

	public void claim(final EIInstanceMinimal instance, final String claimant, final ResultsCallback callback) throws Exception {
		if (isLoggedIn()) {
			Log.info("Calling claim on " + instance.getInstanceLabel() + ": " + instance.getInstanceURI().toString());
			repositoryService.claim(session, instance.getInstanceURI(), claimant, new AsyncCallback<String>() {

				@Override
				public void onFailure(Throwable arg0) {
				}

				@Override
				public void onSuccess(String arg0) {
					if (arg0 == null) {
						Window.alert("could not claim " + instance.getInstanceLabel().toString());
						return;
					}
					Log.info("claim result: " + arg0);
					instance.setWFOwner(null);
					updateCached(instance, EIEntity.NULL_ENTITY, user);
					callback.onSuccess(arg0);
				}
			});
		}
	}

	public void claim(final EIInstanceMinimal[] instances, final String claimant, final ResultsCallback callback) throws Exception {
		EIURI[] eiuriArray = new EIURI[instances.length];
		int i = 0;
		for (EIInstanceMinimal instance : instances) {
			eiuriArray[i++] = instance.getInstanceURI();
		}
		if (isLoggedIn()) {
			repositoryService.claim(session, eiuriArray, claimant, new AsyncCallback<String[]>() {

				@Override
				public void onFailure(Throwable arg0) {
				}

				@Override
				public void onSuccess(String[] arg0) {
					if (arg0 == null) {
						StringBuilder sbuf = new StringBuilder();
						for (EIInstanceMinimal instance : instances) {
							sbuf.append(instance.getInstanceLabel().toString() + '\n');
						}
						Window.alert("could not claim " + sbuf.toString());
						return;
					}
					for (EIInstanceMinimal instance : instances) {
						instance.setWFOwner(null);
						updateCached(instance, null, user);
					}
					callback.onSuccess(arg0);
				}
			});
		}
	}

	public void getWFStates(String user, final WFCallback callback) throws Exception {
		repositoryService.getWFStates(session, new AsyncCallback<String[]>() {

			@Override
			public void onFailure(Throwable arg0) {
			}

			@Override
			public void onSuccess(String[] arg0) {
				callback.onSuccess(arg0);
			}
		});
	}

	public void retrieveLabel(EIURI uri, final ResultsCallback callback) {
		repositoryService.retrieveLabel(session, uri, new AsyncCallback<String>() {

			@Override
			public void onFailure(Throwable arg0) {
			}

			@Override
			public void onSuccess(String label) {
				callback.onSuccess(label);
			}
		});
	}

	public void retrieveLabels(List<EIEntity> entityList, final LabelsCallback callback) {
		try {
			repositoryService.retrieveLabels(session, entityList, new AsyncCallback<Map<EIEntity, String>>() {

				@Override
				public void onFailure(Throwable arg0) {
				}

				@Override
				public void onSuccess(Map<EIEntity, String> labels) {
					callback.onSuccess(labels);
				}
			});
		} catch (Exception e) {
			e.printStackTrace();
		}
	}

	public void retrieveUriLabels(List<EIURI> entityList, final UriLabelsCallback callback) {
		try {
			repositoryService.retrieveUriLabels(session, entityList, new AsyncCallback<Map<EIURI, String>>() {

				@Override
				public void onFailure(Throwable arg0) {
				}

				@Override
				public void onSuccess(Map<EIURI, String> labels) {
					callback.onSuccess(labels);
				}
			});
		} catch (Exception e) {
			e.printStackTrace();
		}
	}

	public void getRootSuperclassForInstanceUri(EIURI instanceUri, final ClassCallback callback) {
		try {
			repositoryService.getRootSuperclassForInstanceUri(session, instanceUri, new AsyncCallback<EIClass>() {

				@Override
				public void onFailure(Throwable caught) {
				}

				@Override
				public void onSuccess(EIClass result) {
					callback.onSuccess(result);
				}
			});
		} catch (Exception e) {
			Log.error("modelService.getRootSuperclassForInstanceUri threw exception", e);
			e.printStackTrace();
		}
	}

	public void getClassAndSuperclassesForInstanceUri(EIURI instanceUri, final EIClassesCallback callback) {
		try {
			repositoryService.getClassAndSuperclassesForInstanceUri(session, instanceUri, new AsyncCallback<List<EIClass>>() {

				@Override
				public void onFailure(Throwable caught) {
					callback.onFailure(caught.getMessage());
				}

				@Override
				public void onSuccess(List<EIClass> result) {
					callback.onSuccess(result);
				}
			});
		} catch (Exception e) {
			Log.error("modelService.getSuperclassesForInstanceUri threw exception", e);
			e.printStackTrace();
			callback.onFailure(e.toString());
		}
	}

	private void handleFailure(final LoginRequiredCallback callback, Throwable arg0) {
		if (arg0 instanceof IncompatibleRemoteServiceException) {
			// log.error(arg0.getMessage() + )
		} else if (arg0 instanceof InvocationException) {
		} else if (arg0 instanceof LoggedException) {
			if (arg0.getMessage().contains("No session information was found")) {
				callback.loginRequired();
			} else
				Window.alert(arg0.getMessage());
		} else {
			callback.loginRequired();
		}
	}

	/*
	 * ontology methods that don't belong here, but we have to have them because
	 * of services etc
	 */
	public void getSuperClass(final EIClass eclass, final ClassCallback callback) {
		repositoryService.getSuperClass(eclass, new AsyncCallback<EIClass>() {

			public void onFailure(Throwable caught) {
				// TODO Auto-generated method stub
			}

			public void onSuccess(EIClass result) {
				callback.onSuccess(result);
			}
		});
	}

	public void getRootSuperClass(final EIClass eclass, final ClassCallback callback) {
		repositoryService.getRootSuperClass(eclass, new AsyncCallback<EIClass>() {

			public void onFailure(Throwable caught) {
				// TODO Auto-generated method stub
			}

			public void onSuccess(EIClass result) {
				callback.onSuccess(result);
			}
		});
	}

	public void getLabRootSuperclass(final ClassCallback callback) {
		if (mapIdToClass.containsKey(DatatoolsUIConstants.EI_LAB_URI)) {
			callback.onSuccess(mapIdToClass.get(DatatoolsUIConstants.EI_LAB_URI));
		} else {
			modelService.getClass(DatatoolsUIConstants.EI_LAB_URI, new AsyncCallback<EIClass>() {

				public void onFailure(Throwable caught) {
					// TODO Auto-generated method stub
				}

				public void onSuccess(EIClass result) {
					mapIdToClass.put(DatatoolsUIConstants.EI_LAB_URI, result);
					callback.onSuccess(result);
				}
			});
		}
	}
	
	public void getEquivalentClasses(final List<EIProperty> propertiesToPopulate, final EquivalentClassesCallback callback) {
		repositoryService.getAllEquivalentClasses(propertiesToPopulate, new AsyncCallback<List<EIProperty>>() {
			@Override
			public void onFailure(Throwable caught) {
				callback.onFailure(caught.toString());
			}

			@Override
			public void onSuccess(List<EIProperty> result) {
				callback.onSuccess(result);
			}
		});
	}
}
