package org.eaglei.datatools.client.ui;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.eaglei.datatools.client.Datatools;
import org.eaglei.datatools.client.WorkFlowConstants;
import org.eaglei.datatools.client.rpc.ClientRepositoryToolsManager;
import org.eaglei.datatools.client.rpc.ClientRepositoryToolsManager.ResultsCallback;
import org.eaglei.datatools.client.ui.QueryTokenObject.Mode;
import org.eaglei.datatools.client.ui.listeners.RowDeletionListener;
import org.eaglei.datatools.client.ui.widgets.EIConfirmationPopup;
import org.eaglei.datatools.client.ui.widgets.GridRowWidget;
import org.eaglei.model.EIEntity;
import org.eaglei.model.EIInstanceMinimal;

import com.allen_sauer.gwt.log.client.Log;
import com.google.gwt.event.dom.client.ChangeEvent;
import com.google.gwt.event.dom.client.ChangeHandler;
import com.google.gwt.event.dom.client.ClickEvent;
import com.google.gwt.event.dom.client.ClickHandler;
import com.google.gwt.user.client.Window;
import com.google.gwt.user.client.ui.Anchor;
import com.google.gwt.user.client.ui.CheckBox;
import com.google.gwt.user.client.ui.Composite;
import com.google.gwt.user.client.ui.FlexTable;
import com.google.gwt.user.client.ui.FlowPanel;
import com.google.gwt.user.client.ui.HorizontalPanel;
import com.google.gwt.user.client.ui.Label;
import com.google.gwt.user.client.ui.ListBox;
import com.google.gwt.user.client.ui.PopupPanel;
import com.google.gwt.user.client.ui.ScrollPanel;
import com.google.gwt.user.client.ui.VerticalPanel;

public class EIResourcesGrid extends Composite implements RowDeletionListener {

	private static final String										RESOURCE_COLUMN_NAMES[]	= { "Resource Name", "Type", "Date Added", "Status" };	// deliberately leaving off a space so that the picklist can go there
	// Total width 733 px to match filter widget; first 4 colums account for 7 px padding
	public static final int[]										widths					= new int[] { 198, 113, 97, 97, 50, 50, 50, 50 };
	private static final Map<String, Comparator<EIInstanceMinimal>>	resourceSorters			= new HashMap<String, Comparator<EIInstanceMinimal>>();
	public static final String										DATATOOLS_FILTER		= "DatatoolsFilter";
	private List<EIInstanceMinimal>									instances				= new ArrayList<EIInstanceMinimal>();
	private Map<EIInstanceMinimal, GridRowWidget>					instanceRowMap			= new HashMap<EIInstanceMinimal, GridRowWidget>();
	FilterPanel														filterPanel;
	VerticalPanel													resources;
	VerticalPanel													innerPanel				= new VerticalPanel();
	private final FlowPanel											headerPanel				= new FlowPanel();

	public EIResourcesGrid() {
		initialize();
		drawFromApplicationState();
	}

	public EIResourcesGrid(List<EIInstanceMinimal> instances) {
		this();
		Log.info("Creating grid with " + instances.size() + " instances");
		resources.remove(headerPanel);
		this.instances = instances;
		drawFromApplicationState();
	}

	private void initialize() {
		initWidget(innerPanel);
		if (resourceSorters.size() == 0) {
			initializeSorters();
		}
		filterPanel = new FilterPanel(ApplicationState.getInstance().hasFilterTypeEntity() ? ApplicationState.getInstance().getFilterTypeEntity() : ApplicationState.getInstance().getTypeEntity(), 
				ApplicationState.getInstance().getFilterWorkflowEntity(), 
				ApplicationState.getInstance().getFilterLabEntity(), ApplicationState.getInstance().isStrictlyFilteredByOwner());
		resources = new VerticalPanel();
		resources.getElement().setClassName("listGrid");
		Label pageHeader = getHeaderLabel();
		innerPanel.add(filterPanel);
		innerPanel.add(pageHeader);
		innerPanel.add(resources);
		headerPanel.setStyleName("resourcesHeaderPanel");
		Label emptyHeader = new Label(getEmptyListMessage());
		headerPanel.add(emptyHeader);
		resources.add(headerPanel);
	}

	private String getEmptyListMessage() {
		if (ApplicationState.getInstance().getMode() == Mode.references) {
			return "No resources refer to " + ApplicationState.getInstance().getInstanceEntity().getLabel();
		}
		return "To see a list of resources that you can view or edit, click the resource type, such as Instrument, or filter by Status or Laboratory above.";
	}

	private void drawFromApplicationState() {
		if (ApplicationState.getInstance().getMode() == Mode.resources) {
			return; // Leave the explanation box, and just get out of here
		} else if (ApplicationState.getInstance().getMode() != Mode.list && ApplicationState.getInstance().getMode() != Mode.labs && ApplicationState.getInstance().getMode() != Mode.filter && ApplicationState.getInstance().getMode() != Mode.references) {
			resources.clear();
			return;
		}
		filterPanel.setCount(instances.size());
		if (ApplicationState.getInstance().getMode() == Mode.references) {
			filterPanel.disable();
		}
		if (instances.size() == 0) {
			resources.remove(headerPanel);
			resources.clear();
			Label noInstances = new Label(getNoResourcesMessage());
			resources.add(noInstances);
			return;
		}
		resources.clear();
		setGridHeaders();
		updateGrid(0, instances);
	}

	private String getNoResourcesMessage() {
		if (ApplicationState.getInstance().getMode() == Mode.references) {
			return "No resources refer to " + ApplicationState.getInstance().getInstanceEntity().getLabel();
		}
		return "No resources found";
	}

	private void setGridHeaders() {
		HorizontalPanel header = new HorizontalPanel();
		// Header columns
		int i;
		// Set only columns with a header
		for (i = 0; i < RESOURCE_COLUMN_NAMES.length; i++) {
			final String columnName = RESOURCE_COLUMN_NAMES[i];
			final Anchor label = new Anchor(columnName);

			// remove one pixel to set borders in stylesheet
			String width = Integer.toString((widths[i] - 1)) + "px";

			header.add(label);
			header.setCellWidth(label, width);
			label.addClickHandler(new ClickHandler() {

				@Override
				public void onClick(ClickEvent event) {
					Log.info("sorting by " + columnName);
					sortGrid(instances, resourceSorters.get(columnName));
				}
			});
		}
		if (!ApplicationState.getInstance().hasLabType()) {
			final ListBox actionList = makeListBoxOfActions();
			header.add(actionList);

			// Action selector sized as 3 columns in grid (action columns) minus padding for action list
			// TODO if the style is set on div removing padding might not be necessary
			header.setCellWidth(actionList, Integer.toString(widths[4] + widths[5] + widths[6] - 33) + "px");

			CheckBox selectAllCheckBox = makeCheckBoxToSelectAll();
			header.add(selectAllCheckBox);
			header.setCellWidth(selectAllCheckBox, Integer.toString(widths[7] - 1) + "px");
		} else {
			Label placeholder = new Label("");
			header.add(placeholder);
			header.setCellWidth(placeholder, Integer.toString(widths[4] + widths[5] + widths[6] - 33) + "px");
		}
		resources.add(header);
		header.setStyleName("dtListGridHeader");
	}

	private CheckBox makeCheckBoxToSelectAll() {
		CheckBox chkBox = new CheckBox("");
		chkBox.addClickHandler(new ClickHandler() {

			@Override
			public void onClick(ClickEvent event) {
				CheckBox selectAllCheckBox = (CheckBox) event.getSource();
				for (EIInstanceMinimal eiInstance : instances) {
					GridRowWidget row = instanceRowMap.get(eiInstance);
					CheckBox chkBox = row.getCheckBox();
					if (chkBox != null) {
						chkBox.setValue(selectAllCheckBox.getValue());
					}
				}
			}
		});
		return chkBox;
	}

	private void removeRow(EIInstanceMinimal toRemove) {
		if (instanceRowMap.containsKey(toRemove)) {
			resources.remove(instanceRowMap.get(toRemove));
			instanceRowMap.remove(toRemove);
			instances.remove(toRemove);
		}
		filterPanel.setCount(instances.size());
	}

	private void sortGrid(final List<EIInstanceMinimal> eiInstanceList, final Comparator<EIInstanceMinimal> sorter) {
		Collections.sort(instances, sorter);
		for (EIInstanceMinimal instance : eiInstanceList) {
			resources.remove(instanceRowMap.get(instance));
			resources.add(instanceRowMap.get(instance));
		}
	}

	private void updateGrid(final int initialRowNumber, final List<EIInstanceMinimal> eiInstanceList) {
		int i = initialRowNumber;
		for (EIInstanceMinimal instance : eiInstanceList) {
			final GridRowWidget gridRowWidget = new GridRowWidget(instance, this);
			instanceRowMap.put(instance, gridRowWidget);
			resources.add(gridRowWidget);
			i++;
		}
	}

	private Label getHeaderLabel() {
		// TODO: more to it than that? What if there's a lab, or a filter?
		String labelContent = "";
		if (ApplicationState.getInstance().hasType()) {
			labelContent = ApplicationState.getInstance().getTypeEntity().getLabel();
		} else if (ApplicationState.getInstance().getFilterTypeEntity() != EIEntity.NULL_ENTITY) {
			labelContent = ApplicationState.getInstance().getFilterTypeEntity().getLabel();
		} else if (ApplicationState.getInstance().getMode() == Mode.references) {
			labelContent = "Resources that reference " + ApplicationState.getInstance().getInstanceEntity().getLabel();
		}
		Label headerLabel = new Label(labelContent);
		headerLabel.setStyleName("pageHeader");
		return headerLabel;
	}

	private void initializeSorters() {
		resourceSorters.put(RESOURCE_COLUMN_NAMES[0], new Comparator<EIInstanceMinimal>() {

			@Override
			public int compare(EIInstanceMinimal arg0, EIInstanceMinimal arg1) {
				return arg0.getInstanceLabel().compareToIgnoreCase(arg1.getInstanceLabel());
			}
		});
		resourceSorters.put(RESOURCE_COLUMN_NAMES[1], new Comparator<EIInstanceMinimal>() {

			@Override
			public int compare(EIInstanceMinimal arg0, EIInstanceMinimal arg1) {
				return arg0.getInstanceType().getLabel().compareToIgnoreCase(arg1.getInstanceType().getLabel());
			}
		});
		resourceSorters.put(RESOURCE_COLUMN_NAMES[2], new Comparator<EIInstanceMinimal>() {

			@Override
			public int compare(EIInstanceMinimal arg0, EIInstanceMinimal arg1) {
				return arg0.getCreationDate().compareToIgnoreCase(arg1.getCreationDate());
			}
		});
		resourceSorters.put(RESOURCE_COLUMN_NAMES[3], new Comparator<EIInstanceMinimal>() {

			@Override
			public int compare(EIInstanceMinimal arg0, EIInstanceMinimal arg1) {
				return arg0.getWFState().getLabel().compareToIgnoreCase(arg1.getWFState().getLabel());
			}
		});
	}

	private ListBox makeListBoxOfActions() {
		final ListBox actionList = new ListBox();
		actionList.addItem("Actions");
		if (WorkflowUtils.canUserChangeState(WorkFlowAction.CurationAction.getFromState())) {
			actionList.addItem(WorkFlowAction.CurationAction.getName());
		}
		if (WorkflowUtils.canUserChangeState(WorkFlowAction.ReturnToDraftAction.getFromState())) {
			actionList.addItem(WorkFlowAction.ReturnToDraftAction.getName());
		}
		if (WorkflowUtils.canUserChangeState(WorkFlowAction.PublishAction.getFromState())) {
			actionList.addItem(WorkFlowAction.PublishAction.getName());
		}
		if (WorkflowUtils.canUserChangeState(WorkFlowAction.WithdrawAction.getFromState())) {
			actionList.addItem(WorkFlowAction.WithdrawAction.getName());
		}
		actionList.addItem(WorkFlowConstants.CLAIM_ALL_ACTION);
		actionList.addItem(WorkFlowConstants.RELEASE_ALL_ACTION);
		actionList.addChangeHandler(new ChangeHandler() {

			@Override
			public void onChange(ChangeEvent event) {
				/* Redisplay Action callback after bulk claim and release */
				ActionRedisplay redisplay = new ActionRedisplay() {

					@Override
					public void drawAfterClaim(EIInstanceMinimal instance) {
					}

					@Override
					public void drawAfterPromote(EIInstanceMinimal instance) {
					}

					@Override
					public void drawAfterClaim(EIInstanceMinimal[] instances) {
						for (EIInstanceMinimal eiInstance : instances) {
							if (instanceMeetsFilterCriteria(eiInstance)) {
								GridRowWidget row = instanceRowMap.get(eiInstance);
								row.redrawActions();
							} else {
								removeRow(eiInstance);
							}
							
						}
						Datatools.hideGlasspane();
					}
				};
				String selectedText = actionList.getItemText(actionList.getSelectedIndex());
				if (selectedText.equalsIgnoreCase(WorkFlowAction.CurationAction.getName())) {
					bulkOperation(WorkFlowAction.CurationAction);
				} else if (selectedText.equalsIgnoreCase(WorkFlowAction.PublishAction.getName())) {
					bulkOperation(WorkFlowAction.PublishAction);
				} else if (selectedText.equalsIgnoreCase(WorkFlowAction.WithdrawAction.getName())) {
					bulkOperation(WorkFlowAction.WithdrawAction);
				} else if (selectedText.equalsIgnoreCase(WorkFlowAction.ReturnToDraftAction.getName())) {
					bulkOperation(WorkFlowAction.ReturnToDraftAction);
				} else if (selectedText.equalsIgnoreCase(WorkFlowConstants.CLAIM_ALL_ACTION)) {
					bulkClaimOrRelease(redisplay, OwnershipAction.ClaimAction, WorkFlowConstants.CLAIM_ALL_ACTION);
				} else if (selectedText.equalsIgnoreCase(WorkFlowConstants.RELEASE_ALL_ACTION)) {
					bulkClaimOrRelease(redisplay, OwnershipAction.ReleaseAction, WorkFlowConstants.RELEASE_ALL_ACTION);
				}
				actionList.setSelectedIndex(0);
			}
		});
		return actionList;
	}

	private boolean isPublishedOrWithdrawn(EIInstanceMinimal instance) {
		return (WorkFlowConstants.PUBLISH_ENTITY.equals(instance.getWFState()) || WorkFlowConstants.WITHDRAW_ENTITY.equals(instance.getWFState()));
	}
	
	protected boolean instanceMeetsFilterCriteria(EIInstanceMinimal eiInstance) {
		if (ApplicationState.getInstance().isStrictlyFilteredByOwner()) {
			return WorkflowUtils.isNotNull(eiInstance.getWFOwner());
		}
		return true;
	}

	protected void bulkClaimOrRelease(ActionRedisplay redisplay, OwnershipAction ClaimOrReleaseaction, final String seletecChoice) {
		ArrayList<EIInstanceMinimal> instanceList = new ArrayList<EIInstanceMinimal>();
		VerticalPanel vpanel = makeVerticalpanelWithHeading("The following resources will be " + seletecChoice + "ed");
		WorkFlowConstants wfc = new WorkFlowConstants();
		ScrollPanel scrPanel = getScrollPanelWithDimensionsSet();
		FlexTable flexTable = getFlextableWithHeadingSet();
		int i = 0;
		for (EIInstanceMinimal eiInstance : instances) {
			EIEntity status = eiInstance.getWFState();
			GridRowWidget row = instanceRowMap.get(eiInstance);
			// Sravan:checking all possible condition ,no need to do all but just for good measure
			if (row.getCheckBox() != null && row.getCheckBox().isVisible() && row.getCheckBox().getValue()) {
				if (seletecChoice.equals(WorkFlowConstants.CLAIM_ALL_ACTION) && WorkflowUtils.canUserChangeState(eiInstance.getWFState().getURI()) && !isPublishedOrWithdrawn(eiInstance) && !row.instanceHasOwner()) {
					i = addRowToTableAndInstanceList(instanceList, wfc, flexTable, i, eiInstance, status);
				} else if (seletecChoice.equals(WorkFlowConstants.RELEASE_ALL_ACTION) && row.instanceHasOwner()) {
					i = addRowToTableAndInstanceList(instanceList, wfc, flexTable, i, eiInstance, status);
				}
			}
		}
		if (i == 0) {
			Window.alert(OwnershipAction.getErrorMsgOfOwnershipAction(seletecChoice));
			return;
		}

		if (instanceList.size() > 0) {
			final EIConfirmationPopup popup = new EIConfirmationPopup("100%", "100%");
			popup.addConfirmClickHandler(makeHandlerforBulkClaimOrRelease(redisplay, ClaimOrReleaseaction, instanceList, popup));
			vpanel.add(scrPanel);
			scrPanel.add(flexTable);
			popup.add(vpanel);
			popup.show();
		}
	}

	private int addRowToTableAndInstanceList(ArrayList<EIInstanceMinimal> instanceList, WorkFlowConstants wfc, FlexTable flexTable, int i, EIInstanceMinimal eiInstance, EIEntity status) {
		instanceList.add(eiInstance);
		flexTable.setText(i + 2, 0, eiInstance.getInstanceLabel());
		flexTable.setText(i + 2, 1, eiInstance.getInstanceType().getLabel());
		flexTable.setText(i + 2, 2, status.getLabel());
		i++;
		return i;
	}

	public void bulkOperation(final WorkFlowAction action) {
		try {
			final List<EIInstanceMinimal> instancesToChange = new ArrayList<EIInstanceMinimal>();
			WorkFlowConstants wfc = new WorkFlowConstants();
			VerticalPanel vpanel = makeVerticalpanelWithHeading("The following resources will be set to " + wfc.getStatusLabel(action.getToState()) + ":");
			ScrollPanel scrPanel = getScrollPanelWithDimensionsSet();
			FlexTable flexTable = getFlextableWithHeadingSet();
			boolean somethingSelected = false;
			boolean actionValidForAllItems = true;
			int unclaimedResourcesCount = 0;
			int loopCount = 0;
			int i = 0;
			for (EIInstanceMinimal eiInstance : instances) {
				GridRowWidget row = instanceRowMap.get(eiInstance);
				EIEntity status = eiInstance.getWFState();
				if (row.getCheckBox() != null && row.getCheckBox().isVisible() && row.getCheckBox().getValue()) {
					if (!WorkflowUtils.instanceHasAnyOwner(eiInstance)) {
						unclaimedResourcesCount++;
					}
					if (action.getFromState().equals(status.getURI()) && WorkflowUtils.instanceHasAnyOwner(eiInstance)) {
						instancesToChange.add(eiInstance);
						flexTable.setText(i + 2, 0, eiInstance.getInstanceLabel());
						flexTable.setText(i + 2, 1, eiInstance.getInstanceType().getLabel());
						flexTable.setText(i + 2, 2, status.getLabel());
						i++;
						somethingSelected = true;
					} else {
						actionValidForAllItems = false;
					}
					loopCount++;
				}
			}
			if (unclaimedResourcesCount == loopCount) {
				Window.alert(WorkFlowAction.getAlertMessageForUnclaimed(action));
				return;
			}
			if (!actionValidForAllItems && !somethingSelected) {
				Window.alert(WorkFlowAction.getAlertMessage(action));
				return;
			} else if (!somethingSelected && actionValidForAllItems) {
				Window.alert(UIMessages.SELECT_MESSAGE);
				return;
			}
			final EIConfirmationPopup popup = new EIConfirmationPopup("100%", "100%");
			popup.addConfirmClickHandler(makeHandlerForBulkWorkflow(action, instancesToChange, popup));

			scrPanel.add(flexTable);
			vpanel.add(scrPanel);

			popup.add(vpanel);
			popup.show();
		} catch (Exception e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		}
	}

	private VerticalPanel makeVerticalpanelWithHeading(final String headingString) {
		VerticalPanel vpanel = new VerticalPanel();
		Label heading = new Label(headingString);
		heading.setStyleName("dataPanelLabel");
		vpanel.add(heading);
		return vpanel;
	}

	private ClickHandler makeHandlerforBulkClaimOrRelease(final ActionRedisplay redisplay, final OwnershipAction ClaimOrReleaseaction, final List<EIInstanceMinimal> instanceList, final PopupPanel popup) {
		return new ClickHandler() {

			@Override
			public void onClick(ClickEvent event) {
				ClaimOrReleaseaction.perform(instanceList.toArray(new EIInstanceMinimal[instanceList.size()]), redisplay);
				popup.hide();
				Datatools.showGlasspane();
			}
		};
	}

	private ClickHandler makeHandlerForBulkWorkflow(final WorkFlowAction action, final List<EIInstanceMinimal> instancesToChange, final PopupPanel popup) {
		return new ClickHandler() {

			@Override
			public void onClick(ClickEvent event) {
				try {
					ResultsCallback resultCall = new ResultsCallback() {

						@Override
						public void onSuccess(String[] arg0) {
							redraw();
						}

						@Override
						public void onSuccess(String arg0) {
							redraw();
						}

						@Override
						public void loginRequired() {
							Datatools.handleLoginRequired();
						}

						public void redraw() {
							ApplicationState.getInstance().refresh();
							// TODO: something history/app state-ish here?
						}
					};
					popup.hide();
					// TODO: make a bulkAction(WorkFlowAction, EIInstance[]) on ClientRepositoryToolsManager
					if (action.equals(WorkFlowAction.CurationAction)) {
						ClientRepositoryToolsManager.INSTANCE.bulkSendToCuration(instancesToChange, resultCall);
					} else if (action.equals(WorkFlowAction.PublishAction)) {
						ClientRepositoryToolsManager.INSTANCE.bulkPublish(instancesToChange, resultCall);
					} else if (action.equals(WorkFlowAction.WithdrawAction)) {
						ClientRepositoryToolsManager.INSTANCE.bulkWithdraw(instancesToChange, resultCall);
					} else if (action.equals(WorkFlowAction.ReturnToDraftAction)) {
						ClientRepositoryToolsManager.INSTANCE.bulkReturnToDraft(instancesToChange, resultCall);
					}
				} catch (Exception e) {
					Window.alert(e.getMessage());
				}
			}
		};
	}

	private FlexTable getFlextableWithHeadingSet() {
		FlexTable flexTable = new FlexTable();
		flexTable.setStyleName("dataPanel");
		flexTable.setHTML(1, 0, "<b>" + "Resource Name" + "</b>");
		flexTable.setHTML(1, 1, "<b>" + "Type" + "</b>");
		flexTable.setHTML(1, 2, "<b>" + "Status" + "</b>");
		return flexTable;
	}

	private ScrollPanel getScrollPanelWithDimensionsSet() {
		ScrollPanel scrPanel = new ScrollPanel();
		scrPanel.setHeight("270px");
		scrPanel.setWidth("1000px");
		scrPanel.setHorizontalScrollPosition(-1);
		return scrPanel;
	}

	@Override
	public void onRowDeletion(EIInstanceMinimal instance) {
		removeRow(instance);
	}
}
