package org.eaglei.datatools.server;

import java.io.File;
import java.net.URL;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import javax.servlet.ServletException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eaglei.datatools.User;
import org.eaglei.datatools.Workspace;
import org.eaglei.datatools.client.rpc.RepositoryToolsModelService;
import org.eaglei.datatools.config.DatatoolsConfiguration;
import org.eaglei.datatools.jena.EIInstanceFactory;
import org.eaglei.datatools.jena.RESTRepositoryProviderFactory;
import org.eaglei.datatools.model.DataToolsOntConstants;
import org.eaglei.datatools.provider.RepositoryProvider;
import org.eaglei.datatools.provider.RepositoryProviderFactory;
import org.eaglei.datatools.provider.RepositoryProviderMessages;
import org.eaglei.model.EIClass;
import org.eaglei.model.EIEntity;
import org.eaglei.model.EIEquivalentClass;
import org.eaglei.model.EIInstance;
import org.eaglei.model.EIInstanceMinimal;
import org.eaglei.model.EIObjectProperty;
import org.eaglei.model.EIOntModel;
import org.eaglei.model.EIProperty;
import org.eaglei.model.EIURI;
import org.eaglei.model.gwt.rpc.LoggedException;
import org.eaglei.security.Session;
import org.springframework.context.ApplicationContext;
import org.springframework.web.context.WebApplicationContext;
import org.springframework.web.context.support.WebApplicationContextUtils;

import com.google.gwt.user.server.rpc.RemoteServiceServlet;

public class RepositoryToolsModelServlet extends RemoteServiceServlet implements RepositoryToolsModelService {

	/**
     * 
     */
	private static final long		serialVersionUID	= 1L;
	private static final Log		log					= LogFactory.getLog(RepositoryToolsModelServlet.class);
	private RepositoryProvider		provider			= null;
	private EIOntModel				ontModel;
	private DatatoolsConfiguration	config;

	@Override
	public void init() throws ServletException {

		config = getConfiguration();
		try {
			if (config != null) {
				if (log.isDebugEnabled())
					log.debug("about to create provider");
				provider = createProvider();

			} else {
				if (log.isDebugEnabled())
					log.debug("about to load defaults (null config)");
				loadDefaults();
			}

		} catch (Exception e) {
			log.error("Error initializing RepositoryToolsModelServlet: " + e);
			throw new ServletException(e);
		}

		initializeOntology();

		lazyLoadFactories();

		log.info("init succesful");

	}

	/*
	 * @Override protected void onBeforeRequestDeserialized(String
	 * serializedRequest) { // TODO Auto-generated method stub
	 * super.onBeforeRequestDeserialized(serializedRequest); }
	 */
	protected void initializeOntology() {
		WebApplicationContext ctx = WebApplicationContextUtils.getWebApplicationContext(getServletContext());

		doDependencyInjection(ctx);
	}

	/*
	 * Allow test subclasses to override servlet init and invoke this with their
	 * own ApplicationContext.
	 */
	protected void doDependencyInjection(ApplicationContext springAppContext) {
		this.ontModel = springAppContext.getBean(EIOntModel.class);
	}

	private RepositoryProvider createProvider() throws Exception {

		final String providerFactoryClass = config.getDatatoolsRepositoryProviderFactory();
		final Class factoryClass = Class.forName(providerFactoryClass);
		final RepositoryProviderFactory factory = (RepositoryProviderFactory) factoryClass.newInstance();
		return factory.createRepositoryProvider(config);

	}

	private void lazyLoadFactories() {
		EIInstanceFactory.getInstance();
	}

	private void loadDefaults() throws Exception {
		// default provider is the REST provider
		log.info("Loading default provider, RESTProvider: ");

		final RepositoryProviderFactory factory = new RESTRepositoryProviderFactory();
		provider = factory.createRepositoryProvider(config);
		log.info("Loading default provider, RESTProvider: " + factory.toString());

	}

	
	@Override
	public void createInstance(Session session, EIInstance instance,EIEntity workspaceEntity) throws Exception {
		try {

			if (instance == null) {
				log.error("trying to create a null instance");
			}
			handleNullSession(session);

			provider.createInstance(session, instance,workspaceEntity);
		} catch (Exception e) {
			log.error(e.getMessage());
			for (StackTraceElement element : e.getStackTrace())
				log.error(element.toString());
			throw new LoggedException(e.getMessage());
		}

	}

	@Override
	public void deleteInstance(Session session, EIURI instanceUri) throws Exception {
		try {
			handleNullSession(session);
			provider.deleteInstance(session, instanceUri);
		} catch (Exception e) {
			throw new LoggedException(e.getMessage());
		}

	}

	@Override
	public String updateInstance(Session session, EIInstance instance, String token) throws Exception {
		try {
			handleNullSession(session);
			return provider.updateInstance(session, instance, token);
		} catch (Exception e) {
			throw new LoggedException(e.getMessage());
		}

	}

	@Override
	public User login(final String user, final String password) throws Exception {
		try {

			User userObj = provider.login(user, password);
			return userObj;
		} catch (Exception e) {
			e.printStackTrace();
			throw new LoggedException("Repository error:" + e.getLocalizedMessage());
		}
	}

	@Override
	public void logout(Session session) throws Exception {
		try {
			handleNullSession(session);
			provider.logout(session);
		} catch (Exception e) {
			throw new LoggedException(e.getMessage());
		}
	}

	@Override
	public User whoami(Session session) throws Exception {
		try {
			handleNullSession(session);
			return provider.whoami(session);
		} catch (Exception e) {
			log.error("whoami exception " + e);
			throw new LoggedException(e.getMessage());
		}
	}

	/*
	 * Loads the DatatoolsConfiguration.
	 */
	private DatatoolsConfiguration getConfiguration() throws ServletException {
		// check if the system property is specified
		String propFile = System.getProperty(DataToolsOntConstants.DATATOOLS_CONFIG);

		if (propFile == null) {
			// use the default
			propFile = DataToolsOntConstants.DEFAULT_DATATOOLS_CONFIG;
		}
		log.debug("Using properties file " + propFile);

		final URL configURL = this.getClass().getClassLoader().getResource(propFile);
		if (configURL == null) {
			log.error("Could not locate " + propFile + " on classpath ");
			throw new ServletException("Could not locate " + propFile + " on classpath ");
		}
		try {
			final File configFile = new File(configURL.toURI());
			return new DatatoolsConfiguration(configFile);
		} catch (Exception e) {
			log.error("Error loading configuration from " + configURL + " " + e);
			throw new ServletException("Error loading configuration from " + configURL + " " + e);
		}
	}

	@Override
	public boolean isOnline() {

		return provider.isOnline();
	}

	@Override
	public List<EIInstanceMinimal> EIQuery(final Session session, final String query) throws Exception {
		try {
			handleNullSession(session);
			return provider.EIQuery(session, query);
		} catch (Exception e) {
			throw new LoggedException(e.getMessage());
		}
	}

	@Override
	public EIInstance getEmptyEIInstance(Session session, EIURI classUri, EIEntity instanceEntity) throws Exception {
		try {
			handleNullSession(session);
			return provider.getEmptyEIInstance(session, classUri, instanceEntity);
		} catch (Exception e) {
			throw new LoggedException(e.getMessage());
		}
	}

	@Override
	public EIInstance getEmptyEIInstance(Session session, EIURI classUri) throws Exception {
		try {
			handleNullSession(session);
			EIInstance instance = provider.getEmptyEIInstance(session, classUri);
			if (instance.getInstanceClass() == null) {
				instance.setInstanceClass(ontModel.getClass(classUri));
			}
			return instance;
		} catch (Exception e) {
			throw new LoggedException(e.getMessage());
		}
	}

	@Override
	public EIInstance deepCopy(Session session, EIURI originalUri) throws Exception {
		handleNullSession(session);
		return provider.deepCopy(session, originalUri);
	}

	@Override
	public List<EIURI> getNewInstanceID(Session session, int count) throws Exception {
		try {
			handleNullSession(session);
			return provider.getNewInstanceID(session, count);
		} catch (Exception e) {
			throw new LoggedException(e.getMessage());
		}
	}

	@Override
	public String query(Session session, String sparql) throws Exception {
		try {
			handleNullSession(session);
			return provider.query(session, sparql);
		} catch (Exception e) {
			throw new LoggedException(e.getMessage());
		}
	}

	@Override
	public EIInstance getInstance(Session session, EIURI instanceID) throws Exception {
		handleNullSession(session);
		return provider.getInstance(session, instanceID);
	}

	@Override
	public String claim(Session session, EIURI uri, String claimant) throws Exception {
		try {
			handleNullSession(session);
			return provider.claim(session, uri, claimant);
		} catch (Exception e) {
			throw new LoggedException(e.getMessage());
		}
	}

	public String[] claim(Session session, EIURI[] uriArray, String claimant) throws Exception {
		try {
			handleNullSession(session);
			String[] responses = new String[uriArray.length];
			int i = 0;
			for (EIURI uri : uriArray) {
				responses[i++] = provider.claim(session, uri, claimant);
			}
			return responses;
		} catch (Exception e) {
			throw new LoggedException(e.getMessage());
		}
	}

	@Override
	public String promote(Session session, EIURI uri, EIURI newState) throws Exception {
		try {
			handleNullSession(session);
			return provider.promote(session, uri, newState);
		} catch (Exception e) {
			throw new LoggedException(e.getMessage());
		}
	}

	public String[] bulkPromote(Session session, List<EIURI> urisToPromote, EIURI newState) throws Exception {
		try {
			handleNullSession(session);
			String[] responses = new String[urisToPromote.size()];
			int i = 0;
			for (EIURI uri : urisToPromote) {
				responses[i++] = provider.promote(session, uri, newState);
			}
			return responses;

		} catch (Exception e) {
			throw new LoggedException(e.getMessage());
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * org.eaglei.datatools.provider.RepositoryProvider#getWFStates(java.lang
	 * .String)
	 */
	@Override
	public String[] getWFStates(Session session) throws Exception {
		try {
			handleNullSession(session);
			return provider.getWFStates(session);
		} catch (Exception e) {
			throw new LoggedException(e.getMessage());
		}
	}

	@Override
	public List<EIInstanceMinimal> getFilterQuery(final Session session, EIURI classUri, EIURI state, EIURI lab) throws Exception {
		try {
			handleNullSession(session);
			return provider.getFilterQuery(session, classUri, state, lab);
		} catch (Exception e) {
			throw new LoggedException(e.getMessage());
		}
	}

	@Override
	public List<EIInstanceMinimal> getFilterQuery(final Session session, EIURI classUri, EIURI state, EIURI lab, boolean strictOwnerFilter) throws Exception {
		try {
			handleNullSession(session);
			return provider.getFilterQuery(session, classUri, state, lab, strictOwnerFilter);
		} catch (Exception e) {
			throw new LoggedException(e.getMessage());
		}
	}

	@Override
	public String retrieveLabel(Session session, EIURI uri) throws Exception {
		try {
			handleNullSession(session);
			return provider.retrieveLabel(session, uri);
		} catch (Exception e) {
			throw new LoggedException(e.getMessage());
		}
	}

	@Override
	public Map<EIEntity, String> retrieveLabels(Session session, List<EIEntity> entities) throws Exception {
		try {
			handleNullSession(session);
			return provider.retrieveLabels(session, entities);
		} catch (Exception e) {
			throw new LoggedException(e.getMessage());
		}
	}

	@Override
	public Map<EIURI, String> retrieveUriLabels(Session session, List<EIURI> uris) throws Exception {
		handleNullSession(session);
		return provider.retrieveUriLabels(session, uris);
	}

	@Override
	public EIClass getRootSuperclassForInstanceUri(Session session, EIURI instanceUri) throws Exception {
		handleNullSession(session);
		EIInstance instance = provider.getInstance(session, instanceUri);
		if (instance == null)
			return null;
		return getRootSuperClass(instance.getInstanceClass());
	}

	@Override
	public List<EIClass> getClassAndSuperclassesForInstanceUri(Session session, EIURI instanceUri) throws Exception {
		handleNullSession(session);
		EIInstance instance = provider.getInstance(session, instanceUri);
		if (instance == null)
			return null;
		List<EIClass> classes = ontModel.getSuperClasses(instance.getInstanceType().getURI());
		if (classes == null || classes.isEmpty())
			classes = new ArrayList<EIClass>(1);
		classes.add(ontModel.getClass(instance.getInstanceType().getURI()));
		return classes;
	}

	@Override
	public EIClass getRootSuperClass(EIClass eclass) throws LoggedException {
		while (eclass.hasSuperClass()) {
			eclass = ontModel.getSuperClass(eclass);
		}
		return eclass;
	}

	@Override
	public EIClass getSuperClass(EIClass eclass) throws LoggedException {
		eclass = ontModel.getSuperClass(eclass);
		return eclass;
	}

	public boolean isModelClassURI(String uri) {
		return ontModel.isModelClassURI(uri);
	}

	@Override
	public List<Workspace> getWorkspaces(Session session) throws Exception {
		handleNullSession(session);
		return provider.getWorkspaces(session);

	}

	@Override
	public List<EIInstanceMinimal> referencedByQuery(Session session, EIURI resourceUri, boolean strictOwnerFilter) throws Exception {
		handleNullSession(session);
		return provider.referencedByQuery(session, resourceUri, strictOwnerFilter);
	}
	
	@Override
	public List<EIProperty> getAllEquivalentClasses(final List<EIProperty> propertiesToPopulate) {
		for (EIProperty property : propertiesToPopulate) {
			if (property instanceof EIObjectProperty) {
				EIObjectProperty objectProperty = (EIObjectProperty) property;
				log.debug("equivalent classes for object property " + property);
				final List<EIClass> assertedRanges = (objectProperty.getRangeList());
				log.debug("   had " + (assertedRanges == null ? "0" : assertedRanges.size()) + " asserted ranges");
				for(final EIClass range : assertedRanges) {
					if (range != null && !range.equals(EIClass.NULL_CLASS) && range.getEntity() != null && !range.getEntity().equals(EIEntity.NULL_ENTITY)) {
											
						List<EIEquivalentClass> equivalentClasses = ontModel.getEquivalentClasses(range.getEntity().getURI());
						if (equivalentClasses != null) {
							range.setEquivalentClasses(equivalentClasses);
							for (EIEquivalentClass equivalence : equivalentClasses) {
								log.debug("    equivalent class " + (equivalence.getEquivalentTo() == null ? "empty" : equivalence.getEquivalentTo().size()));
							}
						}
						else { // TODO: remove!
							log.debug("null classes");
						}
					}
				}
			}
		}
		return propertiesToPopulate;
	}

	private boolean isValid(Session session) {
		return session != null && session.getSessionId() != null;

	}

	private void handleNullSession(Session session) throws Exception {
		if (!isValid(session))
			throw new LoggedException(RepositoryProviderMessages.NO_SESSION_MESSAGE);
	}

}
