package org.eaglei.datatools.client.ui;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.eaglei.datatools.client.Datatools;
import org.eaglei.datatools.client.logging.GWTLogger;
import org.eaglei.datatools.client.rpc.ClientRepositoryToolsManager;

import org.eaglei.datatools.client.rpc.ClientRepositoryToolsManager.TopLevelClassesCallback;

import org.eaglei.datatools.client.ui.LabCache.LabCacheListener;
import org.eaglei.datatools.client.ui.QueryTokenObject.Mode;
import org.eaglei.datatools.client.ui.widgets.LeftListRowWidget;
import org.eaglei.model.EIClass;
import org.eaglei.model.EIEntity;
import org.eaglei.model.EIURI;
import org.eaglei.model.gwt.rpc.ClientModelManager;
import org.eaglei.model.gwt.rpc.ClientModelManager.ClassDefinitionCallback;
import org.eaglei.ui.gwt.instance.EagleIEntityConstants;
import com.google.gwt.event.dom.client.ChangeEvent;
import com.google.gwt.event.dom.client.ChangeHandler;
import com.google.gwt.event.dom.client.ClickEvent;
import com.google.gwt.event.dom.client.ClickHandler;
import com.google.gwt.user.client.ui.Anchor;
import com.google.gwt.user.client.ui.FlexTable;
import com.google.gwt.user.client.ui.FlowPanel;
import com.google.gwt.user.client.ui.Label;
import com.google.gwt.user.client.ui.ListBox;
import com.google.gwt.user.client.ui.VerticalPanel;

public class LeftListPanel extends VerticalPanel implements ClientRepositoryToolsManager.SessionListener, ApplicationStateChangeListener, LabCacheListener {

	private static final String PANEL_SELECTED = "panelSelected";
	private static final String PANEL_NOT_SELECTED = "panelNotSelected";
	private static final int TYPE_COLUMN = 0;
	private static final int ALL_TYPES_ROW = 1;

	private boolean signedIn = false;

	private List<EIClass> resourceTypes;
	private EIClass personType;
	private EIClass organizationType;
	private final Map<EIEntity, String> definitions = new HashMap<EIEntity, String>();
	private FlexTable labsTable = new FlexTable();
	private FlexTable resourcesTable = new FlexTable();
	private FlexTable peopleTable = new FlexTable();
	private final Label labNameLabel = new Label( "" );
	private final Anchor switchLabsLink = new Anchor();
	private ListBox labsList = new ListBox();
	private Label resourcesLink = new Label();
	private Label peopleAndOrganizationLink = new Label();
	private final FlowPanel labsPanel = new FlowPanel();
	private final FlowPanel resourcesPanel = new FlowPanel();
	private final FlowPanel peoplePanel = new FlowPanel();

	private final LabCache labCache;

	private static final GWTLogger log = GWTLogger.getLogger( "LeftListPanel" );

	public LeftListPanel(final LabCache labCache) {
		ClientRepositoryToolsManager.INSTANCE.addSessionListener( this );
		ApplicationState.getInstance().addApplicationStateListener( this );
		this.labCache = labCache;
		labCache.addLabLoadListener( this );
		signedIn = ApplicationState.getInstance().hasUser();
		if ( signedIn ) {
			getResourceTypes();
		}
	}

	@Override
	public void onApplicationStateChange() {
		drawFromApplicationState();
	}

	private void drawFromApplicationState() {
		if ( !ApplicationState.getInstance().hasUser() && !signedIn ) {
			labsTable.setVisible( false );
			resourcesTable.setVisible( false );
			return;
		}

		if ( ApplicationState.getInstance().getMode() == Mode.workbench ) {
			this.setVisible( false );
		} else if ( ApplicationState.getInstance().getMode() == Mode.labs || 
				( ApplicationState.getInstance().getMode() == Mode.edit && ApplicationState.getInstance().hasLabType() ) ||
				( ApplicationState.getInstance().getMode() == Mode.filter && ApplicationState.getInstance().hasLabType() )) {
			this.setVisible( false );
			labNameLabel.setText( "" );
			labsTable.setVisible( true );
			labsPanel.setVisible( true );
			resourcesTable.setVisible( false );
			resourcesPanel.setVisible( false );
			peoplePanel.setVisible( false );
			peopleTable.setVisible( false );
			selectType( labsTable );
			unselectAll( resourcesTable );
			unselectAll( peopleTable );
			resourcesPanel.setStyleName( PANEL_NOT_SELECTED );
		} else if ( ApplicationState.getInstance().hasLab() ) {
			this.setVisible( true );
			labNameLabel.setText( ApplicationState.getInstance().getLabEntity().getLabel() );
			labsTable.setVisible( true );
			labsPanel.setVisible( true );
			resourcesTable.setVisible( false );
			resourcesPanel.setVisible( false );
			peoplePanel.setVisible( false );
			peopleTable.setVisible( false );
			if ( !ApplicationState.getInstance().hasType() || ApplicationState.getInstance().hasLabType() ) {
				unselectAll( labsTable );
				if ( Mode.isResourcesList( ApplicationState.getInstance().getMode() ) ) {
					labsTable.getRowFormatter().setStyleName( ALL_TYPES_ROW, PANEL_SELECTED );
				}
			} else {
				selectType( labsTable );
			}
			unselectAll( resourcesTable );
			unselectAll( peopleTable );
			resourcesPanel.setStyleName( PANEL_NOT_SELECTED );
			peoplePanel.setStyleName( PANEL_NOT_SELECTED );
		} else {
			this.setVisible( true );
			labNameLabel.setText( "[No lab selected]" );
			labsTable.setVisible( false );
			labsPanel.setVisible( false );
			resourcesTable.setVisible( true );
			resourcesPanel.setVisible( true );
			peoplePanel.setVisible( true );
			peopleTable.setVisible( true );
			selectType( resourcesTable );
			selectType( peopleTable );
			unselectAll( labsTable );
		}
	}

	private boolean shouldShowResourcesList() {
		return ApplicationState.getInstance().getMode() == Mode.resources || ApplicationState.getInstance().hasType();
	}

	/**
	 * Retrieves all the top level Resource Categories
	 * 
	 * @return
	 */
	private void getResourceTypes() {

		ClientRepositoryToolsManager.INSTANCE.getTopClassesAnotatedByDataModelCreate( new TopLevelClassesCallback() {

			@Override
			public void onSuccess(final List<EIClass> classList) {
				Datatools.hideGlasspane();
				log.debug( "left list panel got top level classes: " + classList.size() );
				resourceTypes = classList;
				getTypeDefinitions();
			}

		} );
	}

	private void getTypeDefinitions() {
		log.debug( "Left panel getting definitions" );
		ClientModelManager.INSTANCE.getClassDefinitions( resourceTypes, new ClassDefinitionCallback() {

			@Override
			public void onSuccess(final List<EIClass> result) {
				for (final EIClass eclass : result) {
					definitions.put( eclass.getEntity(), eclass.getDefinition() );
				}
				createPanel();
			}
		} );
	}

	/**
	 * Creates the left panel. If the user is not logged in, it will display the general view. For a valid user it displays an authenticated view.
	 */
	private void createPanel() {
		log.debug( "left list panel creating sub-panel; logged in? " + signedIn );
		resourcesLink.setText( "Browse Resources" );
		resourcesLink.setStyleName( "leftListHeader" );
//		resourcesLink.addStyleName("liveLink");
		switchLabsLink.setHTML( "switch labs" );
		switchLabsLink.setStyleName( "gwt_add_item" ); //gwt_add_item or leftPanelSubAddNew?
		switchLabsLink.addStyleName("liveLink");
		setStyleName( "leftPanel" );

		peopleAndOrganizationLink.setText( "Browse People and Organizations" );
		peopleAndOrganizationLink.setStyleName( "leftListHeader" );
//		peopleAndOrganizationLink.addStyleName("liveLink");

		if ( signedIn ) {
			// Build labs section

			createLabsSection();

			add( labsPanel );
			add( labsTable );

			addAllTypesOption( labsTable );
			populateTypeTable( labsTable );

			// Build resources section
			resourcesPanel.add( resourcesLink );
			peoplePanel.add( peopleAndOrganizationLink );
			add( resourcesPanel );
			add( resourcesTable );
			add( peoplePanel );
			add( peopleTable );
			addAllTypesOption( resourcesTable );
			populateTypeTable( resourcesTable );
			populatePersonAndOrganizationTable( peopleTable );
			resourcesTable.setVisible( shouldShowResourcesList() );
			peopleTable.setVisible( shouldShowResourcesList() );
			drawFromApplicationState();
		}
	}

	private void createLabsSection() {
		final String labName = ApplicationState.getInstance().hasLab() ? ApplicationState.getInstance().getLabEntity().getLabel() : "";
		labNameLabel.setText( labName );
		labNameLabel.setStyleName( "labSelected" );
		labNameLabel.addStyleName("liveLink");
		labNameLabel.addClickHandler( new ClickHandler() {

			@Override
			public void onClick(final ClickEvent event) {
				ApplicationState.getInstance().updateApplicationState( Mode.view, ApplicationState.getInstance().getLabEntity(), EagleIEntityConstants.EI_LAB_ENTITY, ApplicationState.getInstance().getLabEntity() );
			}
		} );

		labsPanel.add( labNameLabel );
		labsPanel.add( switchLabsLink );
		labsList.setVisible( false );
		if ( labCache.isLoading() ) {
			labsList.addItem( "loading..." );
		} else {
			populateLabsList();
		}
		labsPanel.add( labsList );

		switchLabsLink.addClickHandler( new ClickHandler() {

			@Override
			public void onClick(final ClickEvent event) {
				labsList.setVisible( true );
			}
		} );
	}

	private void populateLabsList() {
		if ( labCache.isLoading() ) {
			return; // TODO: user warning?
		}

		labsList.setStyleName( "leftListDropdown" );
		labsList.clear();

		labsList.addItem("select a lab");
		for (final EIEntity lab : labCache.getLabEntities()) {
			labsList.addItem( lab.getLabel(), lab.getURI().toString() );
		}

		labsList.addChangeHandler( new ChangeHandler() {

			@Override
			public void onChange(final ChangeEvent event) {
				EIURI selectedUri = EIURI.create( labsList.getValue( labsList.getSelectedIndex() ) );
				final EIEntity selectedEntity = labCache.getLab( selectedUri );
				labsList.setVisible( false );
				if ( selectedEntity == null || selectedEntity.equals( EIEntity.NULL_ENTITY ) ) {
					return;
				}
				ApplicationState.getInstance().updateApplicationState( Mode.view, selectedEntity, EagleIEntityConstants.EI_LAB_ENTITY, selectedEntity );
			}
		} );

	}

	private void populateTypeTable(final FlexTable table) {
		for (final EIClass eiClass : resourceTypes) {
			table.setWidget( table.getRowCount(), TYPE_COLUMN, new LeftListRowWidget( eiClass, definitions.get( eiClass.getEntity() ), table == labsTable ) );
		}
	}

	private void populatePersonAndOrganizationTable(final FlexTable table) {
		table.setWidget( peopleTable.getRowCount(), TYPE_COLUMN, new LeftListRowWidget( DatatoolsUIConstants.personEntity, DatatoolsUIConstants.personEntity.getLabel(), false ) );
		table.setWidget( peopleTable.getRowCount(), TYPE_COLUMN, new LeftListRowWidget( DatatoolsUIConstants.organizationEntity, DatatoolsUIConstants.organizationEntity.getLabel(), false ) );

	}

	private void addAllTypesOption(final FlexTable typesTable) {
		EIEntity allTypesEntity = EIEntity.create( EIURI.NULL_EIURI, "All Resource Types" );
		typesTable.setWidget( ALL_TYPES_ROW, TYPE_COLUMN, new LeftListRowWidget( allTypesEntity, "", false ) );
	}

	private void selectType(final FlexTable typesTable) {
		if ( typesTable == labsTable || typesTable == resourcesTable ) {
			if ( !ApplicationState.getInstance().hasType() && Mode.isResourcesList( ApplicationState.getInstance().getMode() ) && ( ApplicationState.getInstance().getMode() != Mode.resources ) ) {
				typesTable.getRowFormatter().setStyleName( ALL_TYPES_ROW, PANEL_SELECTED );
			} else {
				typesTable.getRowFormatter().setStyleName( ALL_TYPES_ROW, PANEL_NOT_SELECTED );
			}
		}

		final int start = ( typesTable == labsTable || typesTable == resourcesTable ) ? ALL_TYPES_ROW + 1 : 0;
		final EIEntity typeEntity = ApplicationState.getInstance().getTypeEntity();

		for (int i = start; i < typesTable.getRowCount(); i++) {
			LeftListRowWidget rowWidget = (LeftListRowWidget)typesTable.getWidget( i, TYPE_COLUMN );
			if ( ApplicationState.getInstance().hasType() && rowWidget.getEntity().equals( typeEntity ) ) {
				typesTable.getRowFormatter().setStyleName( i, PANEL_SELECTED );
			} else {
				typesTable.getRowFormatter().setStyleName( i, PANEL_NOT_SELECTED );
			}
		}
	}

	private void unselectAll(final FlexTable typesTable) {
		for (int i = 0; i < typesTable.getRowCount(); i++) {
			typesTable.getRowFormatter().setStyleName( i, PANEL_NOT_SELECTED );
		}
	}

	@Override
	public void onLogOut() {
		signedIn = false;
		labsTable = new FlexTable();
		labsPanel.clear();
		resourcesTable = new FlexTable();
		resourcesPanel.clear();
		labsList = new ListBox();
		resourcesLink = new Label();
		peopleAndOrganizationLink = new Label();
		peoplePanel.clear();
		peopleTable = new FlexTable();
		clear();
	}

	@Override
	public void onLogIn(final String username, final String userUri) {
		signedIn = true;
		if ( resourceTypes != null && resourceTypes.size() > 0 ) { // TODO: right?
			createPanel();
		} else {
			log.debug( "left list panel logging in: about to get resource types" );
			getResourceTypes();
		}
	}

	@Override
	public void onLabsLoaded() {
		populateLabsList();
	}

}
