package org.eaglei.datatools.client.ui;

import java.util.HashMap;

import org.eaglei.datatools.SortByProperties;
import org.eaglei.datatools.client.logging.GWTLogger;
import org.eaglei.model.EIEntity;
import org.eaglei.model.EIURI;
import static org.eaglei.datatools.client.ui.DatatoolsUIConstants.*;

public class QueryTokenObject {
	private static final GWTLogger log = GWTLogger.getLogger("QueryTokenObject");

	private static final String MODE_VARIABLE = "mode";

	public static enum Mode {
		edit, filter, labs, list, references, resources, workbench, view;
		
		public static boolean isResourcesList(Mode mode) {
			return mode != Mode.edit && mode != Mode.view && mode != Mode.workbench;
		}
	};
	//FIXME at some point this value will come fro the UI and be selectable by user
	protected static final String INSTANCE_URI_VARIABLE = "instanceUri";
	protected static final String INSTANCE_LABEL_VARIABLE = "instanceLabel";
	protected static final String TYPE_URI_VARIABLE = "typeUri";
	protected static final String TYPE_LABEL_VARIABLE = "typeLabel";
	protected static final String LAB_URI_VARIABLE = "labUri";
	protected static final String LAB_LABEL_VARIABLE = "labLabel";
	protected static final String WORKSPACE_URI_VARIABLE = "workspaceUri";
	protected static final String WORKSPACE_LABEL_VARIABLE = "workspaceLabel";
	protected static final String PAGE_OFFSET_VARIABLE = "offset";
	protected static final String PAGE_LIMIT_VARIABLE = "limit";
	protected static final String SORT_VARIABLE = "sortBy";

	protected static final String FILTER_TYPE_LABEL_VARIABLE = "filterTypeLabel";
	protected static final String FILTER_LAB_LABEL_VARIABLE = "filterLabLabel";
	protected static final String FILTER_WF_LABEL_VARIABLE = "filterWorkflowLabel";
	protected static final String FILTER_TYPE_URI_VARIABLE = "filterTypeUri";
	protected static final String FILTER_LAB_URI_VARIABLE = "filterLabUri";
	protected static final String FILTER_WF_URI_VARIABLE = "filterWorkflowUri";
	protected static final String FILTER_BY_OWNER = "strictOwnerFilter";

	private Mode mode = null;
	private EIEntity instanceEntity = EIEntity.NULL_ENTITY;
	private EIURI instanceUri = EIURI.NULL_EIURI;
	private EIEntity typeEntity = EIEntity.NULL_ENTITY;
	private EIURI typeUri = EIURI.NULL_EIURI;
	private String typeLabel = null;
	private EIEntity labEntity = EIEntity.NULL_ENTITY;
	private EIURI labUri = EIURI.NULL_EIURI;
	private String labLabel = null;
	private EIEntity filterLabEntity = EIEntity.NULL_ENTITY;
	private EIEntity filterTypeEntity = EIEntity.NULL_ENTITY;
	private EIEntity filterWorkflowEntity = EIEntity.NULL_ENTITY;
	private EIEntity workspaceEntity = EIEntity.NULL_ENTITY;
	private Integer offset = 0;
	private Integer pageLimit = DEFAULT_PAGE_SIZE;
	private String sortBy = SortByProperties.label.getParameterName();
	
	protected static final Mode defaultMode = Mode.workbench;
	private boolean filterByOwner;
	private final HashMap<String, String> variableMap = new HashMap<String, String>();

	public QueryTokenObject() {
		super();
		setMode( defaultMode );
	}

	public QueryTokenObject(final String tokenString) {
		fromString( tokenString );
	}

	public Mode getMode() {
		return mode == null ? defaultMode : mode;
	}

	public void setMode(final Mode mode) {
		if(!Mode.isResourcesList( mode ))
		{
			removePagination();
		} else {
			setDefaultPagination();
		}
		this.mode = mode;
		variableMap.put( MODE_VARIABLE, mode.toString() );	
	}

	public void removePagination() {
		setOffset(null);
		setLimit(null);
		setSortBy( null );
	}
	
	public void setDefaultPagination()  {
		setOffset(0);
		setLimit(DEFAULT_PAGE_SIZE);
		setSortBy(SortByProperties.label.getParameterName());
	}
	
	public EIEntity getInstanceEntity() {
		return instanceEntity;
	}

	public void setInstanceEntity(final EIEntity instanceEntity) {
		this.instanceEntity = instanceEntity;
		addToMap( instanceEntity, INSTANCE_URI_VARIABLE, INSTANCE_LABEL_VARIABLE );
	}

	public EIEntity getTypeEntity() {
		return typeEntity;
	}

	public void setTypeEntity(final EIEntity typeEntity) {
		this.typeEntity = typeEntity;
		addToMap( typeEntity, TYPE_URI_VARIABLE, TYPE_LABEL_VARIABLE );
	}

	public EIEntity getLabEntity() {
		return labEntity;
	}

	public void setLabEntity(final EIEntity labEntity) {
		this.labEntity = labEntity;
		addToMap( labEntity, LAB_URI_VARIABLE, LAB_LABEL_VARIABLE );
	}

	public EIEntity getWorkspaceEntity() {
		return workspaceEntity;
	}

	public void setWorkspaceEntity(final EIEntity workspaceEntity) {
		this.workspaceEntity = workspaceEntity;
		addToMap( workspaceEntity, WORKSPACE_URI_VARIABLE, WORKSPACE_LABEL_VARIABLE );
	}

	public EIEntity getFilterLabEntity() {
		return filterLabEntity;
	}

	public void setFilterLabEntity(final EIEntity filterLabEntity) {
		this.filterLabEntity = filterLabEntity;
		addToMap( filterLabEntity, FILTER_LAB_URI_VARIABLE, FILTER_LAB_LABEL_VARIABLE );
	}

	public EIEntity getFilterTypeEntity() {
		return filterTypeEntity;
	}

	public void setFilterTypeEntity(final EIEntity filterTypeEntity) {
		this.filterTypeEntity = filterTypeEntity;
		addToMap( filterTypeEntity, FILTER_TYPE_URI_VARIABLE, FILTER_TYPE_LABEL_VARIABLE );
	}

	public EIEntity getFilterWorkflowEntity() {
		return filterWorkflowEntity;
	}

	public void setFilterWorkflowEntity(final EIEntity filterWorkflowEntity) {
		this.filterWorkflowEntity = filterWorkflowEntity;
		addToMap( filterWorkflowEntity, FILTER_WF_URI_VARIABLE, FILTER_WF_LABEL_VARIABLE );
	}

	public Integer getOffset() {
		return offset;
	}

	public void setOffset(Integer offset) {
		this.offset = offset;
		if (offset == null ) {
			variableMap.remove(PAGE_OFFSET_VARIABLE);
		} else {
			addToMap( PAGE_OFFSET_VARIABLE, offset.toString() );
		}
	}

	public Integer getLimit() {
		return pageLimit;
	}

	public void setLimit(Integer limit) {
		if (limit == null) {
			variableMap.remove(PAGE_LIMIT_VARIABLE);
		} else {
			//ensure that offset is reset every time we change page size
			if(this.pageLimit != limit) {
				setOffset( 0 );
			}
			addToMap( PAGE_LIMIT_VARIABLE, limit.toString() );
		}
		this.pageLimit = limit;

	}

	public String getSortBy() {
		return sortBy;
	}

	public void setSortBy(String sortBy) {
		if ( (this.sortBy == null && sortBy != null) || (this.sortBy != null && !this.sortBy.equals(sortBy)) ) {
			setLimit(DEFAULT_PAGE_SIZE);
			setOffset(0);
		}
		addToMap( SORT_VARIABLE, sortBy );
		this.sortBy = sortBy;
	}

	protected void addToMap(final EIEntity newEntity, final String uriVariable, final String labelVariable) {
		if ( hasUri( newEntity ) ) {
			variableMap.put( uriVariable, newEntity.getURI().toString() );
			variableMap.put( labelVariable, newEntity.getLabel().toString() );
		} else {
			variableMap.remove( uriVariable );
			variableMap.remove( labelVariable );
		}
	}

	protected void addToMap(final String label, final String value) {
		if ( value == null ) {
			variableMap.remove( label );
		} else {
			variableMap.put( label, value );
		}
	}

	public boolean isStrictlyFilteredByOwner() {
		return filterByOwner;
	}

	public void setStrictFilterByOwner(final boolean filterByOwner) {
		this.filterByOwner = filterByOwner;
		addToMap( FILTER_BY_OWNER, filterByOwner + "" );
	}

	@Override
	public String toString() {
		if ( variableMap.size() == 0 ) {
			return "";
		}
		
		final StringBuffer sbuf = new StringBuffer();
		sbuf.append( "history?" );
		
		for ( String key: variableMap.keySet() ) {
			sbuf.append( key + "=" + variableMap.get( key ) + "&" );
		}
		
		return sbuf.toString().replaceAll( "&$", "" );
	}

	/**
	 * parses the given token and fills the Member variables
	 * 
	 * @param token
	 * @return
	 */
	public QueryTokenObject fromString(final String token) {
		variableMap.clear();
		final String[] splitArray = token.split( "\\?" )[1].split( "&" );
		for (final String splitedString : splitArray) {
			final String[] variableFromSplitString = splitedString.split( "=" );
			if ( variableFromSplitString != null && variableFromSplitString.length > 1 ) {
				variableMap.put( variableFromSplitString[0], variableFromSplitString[1] );
			}
		}
		initializeEntities();
		validate();
		return this;
	}

	private EIEntity getEntityByVariable(final String uriName, final String labelName) {
		String uriValue;
		String labelValue;
		if ( variableMap != null ) {
			uriValue = variableMap.get( uriName );
			if ( uriValue == null || uriValue.equals( "" ) || uriValue.equals( EIURI.NULL_EIURI.toString() ) ) {
				return EIEntity.NULL_ENTITY;
			}
			labelValue = variableMap.get( labelName );
			return EIEntity.create( uriValue, labelValue );
		} else {
			return EIEntity.NULL_ENTITY;
		}
	}

	// Fills default info, checks that parameters are present in the right configurations
	private boolean validate() {
		// If there is an instanceUri and there is no mode, we should set the mode to view
		if ( hasUri( instanceUri ) && mode == null ) {
			mode = Mode.view;
			setMode( mode );
			return true;
		}
		if ( mode == Mode.edit || mode == Mode.view && !hasUri( instanceUri ) ) {
			// Window.alert(UIMessages.NO_INSTANCEURI_IN_VIEW_EDIT);
			return false;
		}
		if ( hasLabel( typeLabel ) && !hasUri( typeUri ) ) {
			// Window.alert(UIMessages.TYPELABEL_BUT_NO_TYPEURI);
			return false;
		}
		if ( hasLabel( labLabel ) && !hasUri( labUri ) ) {
			// Window.alert(UIMessages.LABLABEL_BUT_NO_LABURI);
			return false;
		}
		return true;
	}

	protected boolean hasLabelButNoUri(final String label, final EIURI uri) {
		return hasLabel( label ) && !hasUri( uri );
	}

	protected boolean hasLabel(final String label) {
		if ( label == null ) {
			return false;
		}
		return !label.equals( "" ) && !label.equals( EIEntity.NO_LABEL );
	}

	private void initializeEntities() {
		filterTypeEntity = getEntityByVariable( FILTER_TYPE_URI_VARIABLE, FILTER_TYPE_LABEL_VARIABLE );
		filterLabEntity = getEntityByVariable( FILTER_LAB_URI_VARIABLE, FILTER_LAB_LABEL_VARIABLE );
		filterWorkflowEntity = getEntityByVariable( FILTER_WF_URI_VARIABLE, FILTER_WF_LABEL_VARIABLE );
		instanceEntity = getEntityByVariable( INSTANCE_URI_VARIABLE, INSTANCE_LABEL_VARIABLE );
		typeEntity = getEntityByVariable( TYPE_URI_VARIABLE, TYPE_LABEL_VARIABLE );
		if ( hasUri( typeEntity.getURI() ) ) {
			setTypeEntity( typeEntity );
		}
		labEntity = getEntityByVariable( LAB_URI_VARIABLE, LAB_LABEL_VARIABLE );
		if ( hasUri( labEntity.getURI() ) ) {
			setLabEntity( labEntity );
		}
		instanceUri = instanceEntity.getURI();
		typeUri = typeEntity.getURI();
		typeLabel = typeEntity.getLabel();
		labUri = labEntity.getURI();
		labLabel = labEntity.getLabel();

		final String modeValue = variableMap.get( MODE_VARIABLE );
		if ( modeValue == null ) {
			mode = null;
		}
		
		for (final Mode whichMode : Mode.values()) {
			if ( whichMode.toString().equals( modeValue ) ) {
				mode = whichMode;
			}
		}
		
		final String offsetString = variableMap.get( PAGE_OFFSET_VARIABLE );
		if ( offsetString != null ) {
			offset = Integer.decode( offsetString );
		}
		
		final String pageLimitString = variableMap.get( PAGE_LIMIT_VARIABLE );
		if ( pageLimitString != null ) {
			pageLimit = Integer.decode( pageLimitString );
		}
		
		if (variableMap.containsKey( SORT_VARIABLE )) {
			sortBy = variableMap.get( SORT_VARIABLE );
		}
	}

	protected boolean hasUri(final EIEntity entity) {
		return hasUri( entity.getURI() );
	}

	protected boolean hasUri(final EIURI uri) {
		return uri != null && !uri.toString().equals( "" ) && uri != EIURI.NULL_EIURI;
	}

	public boolean isPaginated() {
		return getOffset() != null && getLimit() !=  null;
	}
}
