package org.eaglei.datatools.server;

import java.io.File;
import java.net.URL;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import javax.servlet.ServletException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eaglei.datatools.SortByProperties;
import org.eaglei.datatools.User;
import org.eaglei.datatools.WorkFlowTransition;
import org.eaglei.datatools.Workspace;
import org.eaglei.datatools.client.rpc.RepositoryToolsModelService;
import org.eaglei.datatools.config.DatatoolsConfiguration;
import org.eaglei.datatools.jena.RESTRepositoryProvider;
import org.eaglei.datatools.model.DataToolsOntConstants;
import org.eaglei.datatools.provider.RepositoryProvider;
import org.eaglei.datatools.provider.RepositoryProviderMessages;
import org.eaglei.model.EIClass;
import org.eaglei.model.EIEntity;
import org.eaglei.model.EIEquivalentClass;
import org.eaglei.model.EIInstance;
import org.eaglei.model.EIInstanceMinimal;
import org.eaglei.model.EIInstance;
import org.eaglei.model.EIObjectProperty;
import org.eaglei.model.EIOntModel;
import org.eaglei.model.EIProperty;
import org.eaglei.model.EIURI;
import org.eaglei.model.gwt.rpc.LoggedException;
import org.eaglei.model.jena.JenaEIInstanceFactory;
import org.eaglei.search.provider.AuthSearchRequest;
import org.eaglei.security.Session;
import org.springframework.context.ApplicationContext;
import org.springframework.web.context.WebApplicationContext;
import org.springframework.web.context.support.WebApplicationContextUtils;

import com.google.gwt.user.server.rpc.RemoteServiceServlet;

public class RepositoryToolsModelServlet extends RemoteServiceServlet implements RepositoryToolsModelService {

	/**
     * 
     */
	private static final long serialVersionUID = 1L;
	private static final Log log = LogFactory.getLog( RepositoryToolsModelServlet.class );
	private RepositoryProvider provider = null;
	private EIOntModel ontModel;
	private DatatoolsConfiguration config;
	private static String DATAMODELCREATE_ANNOTATION_URI = "http://eagle-i.org/ont/app/1.0/ClassGroup_DataModelCreate";

	@Override
	// TODO clean this up - no need to distinguish between createProvider and loadDwefaults
	public void init() throws ServletException {

		config = getConfiguration();
		try {
			initializeOntology();
		} catch (final Exception e) {
			log.error( "Error initializing RepositoryToolsModelServlet: " + e );
			throw new ServletException( e );
		}
		// TODO initialize ontology was here
		log.info( "init succesful" );

	}

	/*
	 * @Override protected void onBeforeRequestDeserialized(String serializedRequest) { // TODO Auto-generated method stub super.onBeforeRequestDeserialized(serializedRequest); }
	 */
	protected void initializeOntology() throws Exception {
		final WebApplicationContext ctx = WebApplicationContextUtils.getWebApplicationContext( getServletContext() );

		doDependencyInjection( ctx );
	}

	/*
	 * Allow test subclasses to override servlet init and invoke this with their own ApplicationContext.
	 */
	protected void doDependencyInjection(final ApplicationContext springAppContext) throws Exception {
		ontModel = springAppContext.getBean( EIOntModel.class );
		provider = new RESTRepositoryProvider( config, springAppContext.getBean( JenaEIInstanceFactory.class ) );
		
	}

	@Override
	public void createInstance(final Session session, final EIInstance instance, final EIEntity workspaceEntity) throws Exception {
		try {

			if ( isNull( instance )) {
				log.error( "trying to create a null instance" );
			}
			handleNullSession( session );

			provider.createInstance( session, instance, workspaceEntity );
		} catch (final Exception e) {
			log.error( e.getMessage() );
			for (final StackTraceElement element : e.getStackTrace()) {
				log.error( element.toString() );
			}
			throw new LoggedException( e.getMessage() );
		}

	}
	
	@Override
	public void createInstances(final Session session, final List<EIInstance> instances, final EIEntity workspaceEntity) throws Exception {
		try {

			if ( instances == null ) {
				log.error( "trying to create null instances" );
			}
			handleNullSession( session );

			provider.createInstances( session, instances, workspaceEntity );
		} catch (final Exception e) {
			log.error( e.getMessage() );
			for (final StackTraceElement element : e.getStackTrace()) {
				log.error( element.toString() );
			}
			throw new LoggedException( e.getMessage() );
		}

	}


	@Override
	public void deleteInstance(final Session session, final EIURI instanceUri) throws Exception {
		try {
			handleNullSession( session );
			provider.deleteInstance( session, instanceUri );
		} catch (final Exception e) {
			throw new LoggedException( e.getMessage() );
		}

	}

	@Override
	public void deleteInstances(final Session session, final List<EIURI> instanceUris) throws Exception {
		try {
			handleNullSession( session );
			provider.deleteInstances( session, instanceUris );
		} catch (final Exception e) {
			throw new LoggedException( e.getMessage() );
		}

	}
	
	@Override
	public String getToken(Session session, EIInstance instance) throws Exception {
		try {
			handleNullSession( session );
			return provider.getToken( session, instance );
		} catch (final Exception e) {
			throw new LoggedException( e.getMessage() );
		}
	}
	
	@Override
	public Map<EIInstance, String> getTokens(Session session, List<EIInstance> instances) throws Exception {
		try {
			handleNullSession( session );
			return provider.getTokens( session, instances );
		} catch (final Exception e) {
			throw new LoggedException( e.getMessage() );
		}
	}
	
	@Override
	public void updateInstance(final Session session, final EIInstance instance, final String token) throws Exception {
		try {
			handleNullSession( session );
			provider.updateInstance( session, instance, token );
		} catch (final Exception e) {
			throw new LoggedException( e.getMessage() );
		}

	}

	@Override
	public void updateInstances(final Session session, Map<EIInstance, String> instancesWithTokens) throws Exception {
		try {
			handleNullSession( session );
			provider.updateInstances( session, instancesWithTokens );
		} catch (final Exception e) {
			throw new LoggedException( e.getMessage() );
		}

	}

	@Override
	public User login(final String user, final String password) throws Exception {
		try {
			return provider.login( user, password );
		} catch (final Exception e) {
			throw new LoggedException( "Repository error:" + e.getLocalizedMessage() );
		}
	}

	@Override
	public void logout(final Session session) throws Exception {
		try {
			handleNullSession( session );
			provider.logout( session );
		} catch (final Exception e) {
			throw new LoggedException( e.getMessage() );
		}
	}

	@Override
	public User whoami(final Session session) throws Exception {
		try {
			handleNullSession( session );
			return provider.whoami( session );
		} catch (final Exception e) {
			log.error( "whoami exception " + e );
			throw new LoggedException( e.getMessage() );
		}
	}

	/*
	 * Loads the DatatoolsConfiguration.
	 */
	private DatatoolsConfiguration getConfiguration() throws ServletException {
		// check if the system property is specified
		String propFile = System.getProperty( DataToolsOntConstants.DATATOOLS_CONFIG );

		if ( propFile == null ) {
			// use the default
			propFile = DataToolsOntConstants.DEFAULT_DATATOOLS_CONFIG;
		}
		log.debug( "Using properties file " + propFile );

		final URL configURL = this.getClass().getClassLoader().getResource( propFile );
		if ( configURL == null ) {
			log.error( "Could not locate " + propFile + " on classpath " );
			throw new ServletException( "Could not locate " + propFile + " on classpath " );
		}
		try {
			final File configFile = new File( configURL.toURI() );
			return new DatatoolsConfiguration( configFile );
		} catch (final Exception e) {
			log.error( "Error loading configuration from " + configURL + " " + e );
			throw new ServletException( "Error loading configuration from " + configURL + " " + e );
		}
	}

	@Override
	public boolean isOnline() {

		return provider.isOnline();
	}

	@Override
	public List<EIInstanceMinimal> EIQuery(final Session session, final String query) throws Exception {
		try {
			handleNullSession( session );
			return provider.EIQuery( session, query );
		} catch (final Exception e) {
			throw new LoggedException( e.getMessage() );
		}
	}

	@Override
	public EIInstance getEmptyEIInstance(final Session session, final EIURI classUri, final EIEntity instanceEntity) throws Exception {
		try {
			handleNullSession( session );
			return provider.getEmptyEIInstance( session, classUri, instanceEntity );
		} catch (final Exception e) {
			throw new LoggedException( e.getMessage() );
		}
	}

	@Override
	public EIInstance getEmptyEIInstance(final Session session, final EIURI classUri) throws Exception {
		try {
			handleNullSession( session );
			final EIInstance instance = provider.getEmptyEIInstance( session, classUri );
			if ( instance.getInstanceClass() == null || instance.getInstanceClass().equals( EIClass.NULL_CLASS )) {
				instance.setInstanceClass( ontModel.getClass( classUri ) );
			}
			return instance;
		} catch (final Exception e) {
			throw new LoggedException( e.getMessage() );
		}
	}

	@Override
	public EIInstance deepCopy(final Session session, final EIURI originalUri) throws Exception {
		handleNullSession( session );
		return provider.deepCopy( session, originalUri );
	}

	@Override
	public List<EIURI> getNewInstanceID(final Session session, final int count) throws Exception {
		try {
			handleNullSession( session );
			return provider.getNewInstanceID( session, count );
		} catch (final Exception e) {
			throw new LoggedException( e.getMessage() );
		}
	}

	@Override
	public String query(final Session session, final String sparql) throws Exception {
		try {
			handleNullSession( session );
			return provider.query( session, sparql );
		} catch (final Exception e) {
			throw new LoggedException( e.getMessage() );
		}
	}

	@Override
	public EIInstance getInstance(final Session session, final EIURI instanceID) throws Exception {
		handleNullSession( session );
		return provider.getInstance( session, instanceID );
	}

	@Override
	public List<EIInstanceMinimal> listResourcesForObjectPropertyValue(final Session session, final EIURI classUri, final EIURI lab, final EIURI state, final boolean onlyLab) throws Exception {
		try {
			handleNullSession( session );
			return provider.listResourcesForObjectPropertyValue( session, classUri, lab, state, onlyLab );
		} catch (final Exception e) {
			throw new LoggedException( e.getMessage() );
		}
	}
	

	@Override
	public String retrieveLabel(final Session session, final EIURI uri) throws Exception {
		try {
			handleNullSession( session );
			return provider.retrieveLabel( session, uri );
		} catch (final Exception e) {
			throw new LoggedException( e.getMessage() );
		}
	}

	@Override
	public EIClass getRootSuperclassForInstanceUri(final Session session, final EIURI instanceUri) throws Exception {
		handleNullSession( session );
		final EIInstance instance = provider.getInstance( session, instanceUri );
		if ( isNull( instance )) {
			return null;
		}
		return getRootSuperClass( instance.getInstanceClass() );
	}

	private boolean isNull(final EIInstance instance) {
		return instance == null || instance.getInstanceURI() == null || EIInstance.NULL_INSTANCE.equals(instance);
	}

	@Override
	public List<EIClass> getClassAndSuperclassesForInstanceUri(final Session session, final EIURI instanceUri) throws Exception {
		handleNullSession( session );
		final EIInstance instance = provider.getInstance( session, instanceUri );
		if ( isNull( instance )) {
			return null;
		}
		List<EIClass> classes = ontModel.getSuperClasses( instance.getInstanceType().getURI() );
		if ( classes == null || classes.isEmpty() ) {
			classes = new ArrayList<EIClass>( 1 );
		}
		classes.add( ontModel.getClass( instance.getInstanceType().getURI() ) );
		return classes;
	}

	@Override
	public EIClass getRootSuperClass(final EIClass eclass) throws LoggedException {
		EIClass found = eclass;
		while ( found.hasSuperClass() ) {
			found = ontModel.getSuperClass( found );
		}
		return found;
	}

	@Override
	public EIClass getSuperClass(final EIClass eclass) throws LoggedException {
		return ontModel.getSuperClass( eclass );
	}

	public boolean isModelClassURI(final String uri) {
		return ontModel.isModelClassURI( uri );
	}

	@Override
	public List<Workspace> getWorkspaces(final Session session) throws Exception {
		handleNullSession( session );
		return provider.getWorkspaces( session );

	}

	@Override
	public List<EIProperty> getAllEquivalentClasses(final List<EIProperty> propertiesToPopulate) {
		for (final EIProperty property : propertiesToPopulate) {
			if ( property instanceof EIObjectProperty ) {
				final EIObjectProperty objectProperty = (EIObjectProperty)property;
				log.debug( "equivalent classes for object property " + property );
				final List<EIClass> assertedRanges = objectProperty.getRangeList();
				log.debug( "   had " + ( assertedRanges == null ? "0" : assertedRanges.size() ) + " asserted ranges" );
				for (final EIClass range : assertedRanges) {
					if ( range != null && !range.equals( EIClass.NULL_CLASS ) && range.getEntity() != null && !range.getEntity().equals( EIEntity.NULL_ENTITY ) ) {

						final List<EIEquivalentClass> equivalentClasses = ontModel.getEquivalentClasses( range.getEntity().getURI() );
						if ( equivalentClasses != null ) {
							range.setEquivalentClasses( equivalentClasses );
							for (final EIEquivalentClass equivalence : equivalentClasses) {
								log.debug( "    equivalent class " + ( equivalence.getEquivalentTo() == null ? "empty" : equivalence.getEquivalentTo().size() ) );
							}
						} else { // TODO: remove!
							log.debug( "null classes" );
						}
					}
				}
			}
		}
		return propertiesToPopulate;
	}

	/**
	 * This method get the top level classes annotated by ClassGroup_DataModelCreate
	 * 
	 */
	@Override
	public List<EIClass> getTopClassesAnotatedByDataModelCreate() {

		try {
			final List<EIClass> listOfDataModelCreateAnnotation;
			/*getClassesWithClassGroupAnnotation returns all classes annotated by the passed annotation argument*/
			listOfDataModelCreateAnnotation=	ontModel.getClassesInGroup(DATAMODELCREATE_ANNOTATION_URI);
			log.info( "the size of the eiclasses returned for Data model create sub class is  " + listOfDataModelCreateAnnotation.size());
			return listOfDataModelCreateAnnotation;
		} catch (Exception e) {
			//FIXME huh!!!!!
			System.out.println( e.toString() );
			return null;
		}

	}

	private boolean isValid(final Session session) {
		return session != null && session.getSessionId() != null;

	}

	private void handleNullSession(final Session session) throws Exception {
		if ( !isValid( session ) ) {
			throw new LoggedException( RepositoryProviderMessages.NO_SESSION_MESSAGE );
		}
	}
	
	
	

	@Override
	public List<EIURI> claim(final Session session, final List<EIURI> uris) throws Exception {
		return provider.claim( session, uris );
	}

	@Override
	public List<EIInstanceMinimal> listResources(final Session session, final AuthSearchRequest queryRequest, final SortByProperties orderBy, final boolean strictOwnerFilter) throws Exception {
		return provider.listResources( session, queryRequest, orderBy, strictOwnerFilter );
	}
	
	@Override
	public List<EIInstanceMinimal> listResources(final Session session, final AuthSearchRequest queryRequest, final SortByProperties orderBy, final boolean strictOwnerFilter, final boolean stubsOnly) throws Exception {
		return provider.listResources( session, queryRequest, orderBy, strictOwnerFilter, stubsOnly );
	}
	

	@Override
	public List<EIInstanceMinimal> listReferencingResources(final Session session, final EIURI resourceUri, final AuthSearchRequest queryRequest, final SortByProperties orderBy, final boolean strictOwnerFilter) throws Exception {
		return provider.listReferencingResources( session, resourceUri, queryRequest, orderBy, strictOwnerFilter );
	}

	@Override
	public List<WorkFlowTransition> listWorkFlowTransitions(final Session session, final EIEntity workspaceEntity) throws Exception {
		return provider.listWorkFlowTransitions( session, workspaceEntity );
	}

	@Override
	public List<EIURI> release(final Session session, final List<EIURI> uris) throws Exception {
		return provider.release( session, uris );
	}

	@Override
	public List<EIURI> transition(final Session session, final List<EIURI> uris, final EIEntity transitionEntity) throws Exception {
		return provider.transition( session, uris, transitionEntity );
	}

	@Override
	public Map<EIURI, String> getModifiedDates(Session session, List<EIURI> uris) throws Exception {
		return provider.getModifiedDates( session, uris );
	}

	@Override
	public boolean addLabToUser(Session session, EIURI labUri) throws Exception {
		// TODO Auto-generated method stub
		return false;
	}

	/* (non-Javadoc)
	 * @see org.eaglei.datatools.client.rpc.RepositoryToolsModelService#isModelClassURI(org.eaglei.model.EIURI)
	 */
	@Override
	public boolean isModelClassURI(EIURI uri) {
		return ontModel.isModelClassURI( uri.toString() );
	}

}
