package org.eaglei.datatools.client.rpc;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.eaglei.datatools.SortByProperties;
import org.eaglei.datatools.User;
import org.eaglei.datatools.WorkFlowTransition;
import org.eaglei.datatools.client.ApplicationState;
import org.eaglei.datatools.client.DatatoolsCookies;
import org.eaglei.datatools.client.WorkFlowConstants;
import org.eaglei.datatools.client.logging.GWTLogger;
import org.eaglei.model.EIClass;
import org.eaglei.model.EIEntity;
import org.eaglei.model.EIInstanceMinimal;
import org.eaglei.model.EIInstance;
import org.eaglei.model.EIProperty;
import org.eaglei.model.EIURI;
import org.eaglei.model.gwt.rpc.LoggedException;
import org.eaglei.model.gwt.rpc.ModelService;
import org.eaglei.model.gwt.rpc.ModelServiceAsync;
import org.eaglei.model.gwt.rpc.ClientModelManager.ClassCallback;
import org.eaglei.search.provider.AuthSearchRequest;
import org.eaglei.security.Session;
import org.eaglei.ui.gwt.instance.EagleIEntityConstants;

import com.google.gwt.core.client.GWT;
import com.google.gwt.i18n.client.DateTimeFormat;
import com.google.gwt.user.client.Cookies;
import com.google.gwt.user.client.Window;
import com.google.gwt.user.client.rpc.AsyncCallback;

/**
 * Maintains a client-side cache of EIInstance. Proxies all model RPC methods.
 * 
 * It is critical that all model RPC calls go through this class. All methods in this class MUST call getCached(EIInstance) on all EIInstance objects it receives from the server to ensure that there is only one instance of an EIInstance per URI in
 * the client.
 * 
 * 
 */
public class ClientRepositoryToolsManager {

	public interface SessionListener {

		void onLogIn(String username, String userUri);

		void onLogOut(); // Notification that a logout occurred
	}

	public interface LoginRequiredCallback {

		void loginRequired();
	}

	public interface ResultsCallback extends LoginRequiredCallback {

		void onSuccess(String[] arg0);

		void onSuccess(String arg0);

		void onFailure(String error);
	}

	public interface SaveResultsCallback extends LoginRequiredCallback {

		void onSuccess();

		void onFailure();
	}

	public interface TokenCallback extends LoginRequiredCallback {

		void onSuccess(String token);

		void onFailure(String error);
	}

	public interface TokensCallback extends LoginRequiredCallback {

		void onSuccess(Map<EIInstance, String> instancesAndTokens);

		void onFailure(String error);
	}

	public interface EIInstanceCallback extends LoginRequiredCallback {

		void onSuccess(EIInstance eiInstance);

		void onFailure(String error);

		// void onFailure();
	}

	public interface EIInstancesCallback extends LoginRequiredCallback {

		void onSuccess(List<EIInstance> result);

		void onFailure(String error);
	}

	public interface NewInstanceCallback extends LoginRequiredCallback {

		void onSuccess(Object obj);

		void onFailure(String error);
	}

	public interface MinimalEIInstancesCallback extends LoginRequiredCallback {

		void onSuccess(List<EIInstanceMinimal> result);

		void onFailure(String error);
	}

	public interface IdCallback extends LoginRequiredCallback {

		void onSuccess(List<EIURI> list);

		void onFailure(String error);
	}

	public interface UserCallback extends LoginRequiredCallback {

		void onSuccess(User userInfo);

		void onFailure();
	}

	public interface WFCallback {

		void onSuccess(String[] wfStates);
	}

	public interface BulkWorkflowCallback {
		void onSuccess(List<EIInstanceMinimal> successes);

		void onFailure(String error);

		// void onFailure();

		void needsRefresh(String message);
	}

	public interface DeleteInstanceCallback {
		void onSuccess(Object obj);

		void onFailure(String error);
	}

	public interface LabelsCallback {
		void onSuccess(Map<EIEntity, String> labelMap);

		void onFailure(String error);
	}

	public interface UriLabelsCallback {
		void onSuccess(Map<EIURI, String> labelMap);

		void onFailure(String error);
	}

	public interface EIClassesCallback {

		void onSuccess(List<EIClass> classes);

		void onFailure(String result);
	}

	public interface DatatoolsClassCallback extends ClassCallback {
		void onFailure(String error);
	}

	public interface EquivalentClassesCallback {
		void onSuccess(List<EIProperty> populatedProperties);

		void onFailure(String result);
	}

	public interface TopLevelClassesCallback {
		void onSuccess(List<EIClass> result);
	}

	public static final ClientRepositoryToolsManager INSTANCE = new ClientRepositoryToolsManager();
	public static RepositoryToolsModelServiceAsync repositoryService;
	public static ModelServiceAsync modelService;
	private final HashMap<EIURI, EIClass> mapIdToClass = new HashMap<EIURI, EIClass>();
	private Session session;
	private User user;
	private ArrayList<SessionListener> listeners;
	private static final GWTLogger log = GWTLogger.getLogger( "ClientRepositoryToolsManager" );

	enum LoginFields {
		USERNAME(0), SESSION(1), USER_URI(2), ERROR_MESSAGE(3), WF_STATE(3);

		private final int value;

		private LoginFields(final int value) {
			this.value = value;
		}

		public int getValue() {
			return value;
		}
	}

	private ClientRepositoryToolsManager() {
		repositoryService = GWT.create( RepositoryToolsModelService.class );
		modelService = GWT.create( ModelService.class );
	}

	public boolean isLoggedIn() {
		return Session.isValid( session );
	}

	public void initializeAfterRefresh() {
		session = DatatoolsCookies.getSession();
	}

	private void handleLogOut() {
		session = null;
		user = null;
		Cookies.removeCookie( DatatoolsCookies.DATATOOLS_USER_COOKIE_ID );
		Cookies.removeCookie( DatatoolsCookies.DATATOOLS_USER_URI_ID );
		Cookies.removeCookie( DatatoolsCookies.DATATOOLS_SESSION_ID );
		log.info( "removed cookies" );
		if ( listeners != null ) {
			for (final SessionListener listener : listeners) {
				listener.onLogOut();
			}
		}
	}

	public void addSessionListener(final SessionListener listener) {
		if ( listeners == null ) {
			listeners = new ArrayList<SessionListener>();
		}
		listeners.add( listener );
		/*
		 * if (isLoggedIn()) { listener.onLogIn(""); }
		 */
	}

	public void logOut() {
		if ( !isLoggedIn() ) {
			// ?? Probably should fire logOut even in deferred command
			return;
		}
		// Need to wait for callback, or ok to just assume logged out?
		try {
			repositoryService.logout( session, new AsyncCallback<Void>() {

				@Override
				public void onFailure(final Throwable caught) {
					// Window.alert("Error logging out.");
					handleLogOut();
				}

				@Override
				public void onSuccess(final Void result) {
					session = null;
					user = null;
					handleLogOut();
				}
			} );
		} catch (final Exception e) {
			e.printStackTrace();
		}
	}

	public void logIn(final String username, final String password, final UserCallback callback) {
		if ( isLoggedIn() ) {
			try {
				repositoryService.whoami( session, new AsyncCallback<User>() {

					@Override
					public void onFailure(final Throwable caught) {
						user = null;
						if ( caught instanceof Exception ) {
							Window.alert( caught.getMessage() );
						} else {
							Window.alert( "Error logging in. " );
						}
					}

					@Override
					public void onSuccess(final User retrievedUser) {
						if ( username.equals( retrievedUser.getUserName() ) ) {
							callback.onSuccess( retrievedUser );
						} else {
							user = null;
							session = null;
							Window.alert( "Error logging in: already logged in as different user.  Please log out first." );
						}
					}
				} );
			} catch (final Exception e) {
			}
		} else { // FIXME verify
			try {
				repositoryService.login( username, password, new AsyncCallback<User>() {

					@Override
					public void onFailure(final Throwable caught) {
						user = null;
						session = null;

						if ( caught instanceof Exception ) {
							Window.alert( caught.getMessage() );
						} else {
							Window.alert( "Error logging in. " );
						}

					}

					@Override
					public void onSuccess(final User result) {
						session = result.getSession();
						final String userUri = result.getUserURI().toString();
						user = result;
						Cookies.setCookie( DatatoolsCookies.DATATOOLS_USER_COOKIE_ID, username );
						Cookies.setCookie( DatatoolsCookies.DATATOOLS_USER_URI_ID, userUri );
						Cookies.setCookie( DatatoolsCookies.DATATOOLS_SESSION_ID, session.getSessionId() );
						// Suggestion from GWT security is to use
						// Cookies.setCookie(DatatoolsCookies.DATATOOLS_USER_COOKIE_ID,
						// username, expires, null, "/", true);
						// and call checkValidSession() on server
						// --presumably call whoami & check it's valid not
						// invalid
						log.info( "set cookies; userUri = " + Cookies.getCookie( DatatoolsCookies.DATATOOLS_USER_URI_ID ) + " session = " + Cookies.getCookie( DatatoolsCookies.DATATOOLS_SESSION_ID ) );
						// initializeEditableStates( result );
						if ( listeners != null ) {
							for (final SessionListener listener : listeners) {
								listener.onLogIn( result.getUserName(), result.getUserURI().toString() );
							}
						}

						callback.onSuccess( result );
					}
				} );
			} catch (final Exception e) {
				e.printStackTrace();
			}
		}
	}

	public void getToken(final EIInstance instance, final TokenCallback callback) throws Exception {
		repositoryService.getToken( session, instance, new AsyncCallback<String>() {

			@Override
			public void onFailure(Throwable caught) {
				callback.onFailure( caught.getMessage() );
			}

			@Override
			public void onSuccess(String token) {
				callback.onSuccess( token );
			}
		} );
	}

	public void getTokens(final List<EIInstance> instances, final TokensCallback callback) throws Exception {
		repositoryService.getTokens( session, instances, new AsyncCallback<Map<EIInstance, String>>() {

			@Override
			public void onFailure(Throwable caught) {
				callback.onFailure( caught.getMessage() );
			}

			@Override
			public void onSuccess(Map<EIInstance, String> instanceTokenMap) {
				callback.onSuccess( instanceTokenMap );
			}
		} );
	}

	public void updateInstance(final EIInstance eiInstance, final String token, final SaveResultsCallback callback) throws Exception {
		repositoryService.updateInstance( session, eiInstance, token, new AsyncCallback() {

			@Override
			public void onFailure(final Throwable arg0) {
				log.warn( "update failed for" + eiInstance.getInstanceLabel() + " " + arg0.getMessage() );
				if ( arg0 instanceof LoggedException && arg0.getMessage().contains( "No session information was found" ) ) {
					callback.loginRequired();
				} else {
					callback.onFailure();
				}
			}

			@Override
			public void onSuccess(final Object arg0) {
				log.info( "update succeeded for " + eiInstance.getInstanceLabel() );
				callback.onSuccess();
			}
		} );
	}

	public void updateInstances(final Map<EIInstance, String> instancesWithTokens, final SaveResultsCallback callback) throws Exception {
		repositoryService.updateInstances( session, instancesWithTokens, new AsyncCallback() {

			@Override
			public void onFailure(final Throwable arg0) {
				log.warn( "update failed for instances " + arg0.getMessage() );
				if ( arg0 instanceof LoggedException && arg0.getMessage().contains( "No session information was found" ) ) {
					callback.loginRequired();
				} else {
					callback.onFailure();
				}
			}

			@Override
			public void onSuccess(final Object arg0) {
				callback.onSuccess();
			}
		} );
	}

	public void getInstance(final EIURI eiURI, final EIInstanceCallback callback) throws Exception {
		repositoryService.getInstance( session, eiURI, new AsyncCallback<EIInstance>() {

			@Override
			public void onFailure(final Throwable arg0) {
				handleFailure( callback, arg0 );
			}

			@Override
			public void onSuccess(final EIInstance arg0) {
				if ( arg0 == null ) {
					callback.onFailure( "null instance" );
				}
				callback.onSuccess( arg0 );
			}
		} );
	}

	public void deleteInstance(final EIURI instanceUri, final DeleteInstanceCallback callback) throws Exception {
		repositoryService.deleteInstance( session, instanceUri, new AsyncCallback() {

			@Override
			public void onFailure(final Throwable arg0) {
			}

			@Override
			public void onSuccess(final Object arg0) {
				callback.onSuccess( arg0 );
			}
		} );
	}

	public boolean canEdit(final EIURI workflowState) {
		return user.canEdit( workflowState );
	}

	public boolean canClaim(final EIInstance instance) {
		// If instance has no state it means it's a new instance; by definition user owns it
		if ( isNotNull( instance.getWFState() ) ) {
			return !instanceHasAnyOwner( instance ) && canEdit( instance.getWFState().getURI() );
		} else {
			return true;
		}
	}

	public boolean canClaim(final EIInstanceMinimal instance) {
		return !instanceHasAnyOwner( instance ) && canEdit( instance.getWFStateUri() );
	}

	public boolean canEdit(final EIInstanceMinimal instance) {
		return instanceHasCurrentOwner( instance ) && canEdit( instance.getWFStateUri() );
	}

	public boolean canEdit(final EIInstance instance) {
		// If instance has no state it means it's a new instance; by definition user owns it
		if ( isNotNull( instance.getWFState() ) ) {
			return instanceHasCurrentOwner( instance ) && canEdit( instance.getWFState().getURI() );
		} else {
			return true;
		}
	}

	public void listResourcesForObjectPropertyValue(final EIURI classUri, final EIURI lab, final EIURI state, final boolean onlyLab, final MinimalEIInstancesCallback callback) throws Exception {
		repositoryService.listResourcesForObjectPropertyValue( session, classUri, lab, state, onlyLab, new AsyncCallback<List<EIInstanceMinimal>>() {
			@Override
			public void onFailure(final Throwable arg0) {
				callback.onFailure( arg0.toString() );
			}

			@Override
			public void onSuccess(final List<EIInstanceMinimal> fetchedInstances) {
				callback.onSuccess( fetchedInstances );
			}
		});
	}
	
	public void listResources(final AuthSearchRequest queryRequest, final SortByProperties orderBy, final boolean strictOwnerFilter, final MinimalEIInstancesCallback callback) throws Exception {
		listResources( queryRequest, orderBy, strictOwnerFilter, false, callback );
	}
	
	public void listResources(final AuthSearchRequest queryRequest, final SortByProperties orderBy, final boolean strictOwnerFilter, final boolean stubsOnly, final MinimalEIInstancesCallback callback) throws Exception {
		repositoryService.listResources( session, queryRequest, orderBy, strictOwnerFilter, stubsOnly, new AsyncCallback<List<EIInstanceMinimal>>() {
			@Override
			public void onFailure(final Throwable arg0) {
				callback.onFailure( arg0.toString() );
			}

			@Override
			public void onSuccess(final List<EIInstanceMinimal> fetchedInstances) {
				callback.onSuccess( fetchedInstances );
			}
		} );
	}

	public void listReferencingResources(final EIURI resourceUri, final AuthSearchRequest queryRequest, final SortByProperties orderBy, final boolean strictOwnerFilter, final MinimalEIInstancesCallback callback) throws Exception {
		repositoryService.listReferencingResources( session, resourceUri, queryRequest, orderBy, strictOwnerFilter, new AsyncCallback<List<EIInstanceMinimal>>() {

			@Override
			public void onFailure(final Throwable arg0) {
				handleFailure( callback, arg0 );
			}

			@Override
			public void onSuccess(final List<EIInstanceMinimal> arg0) {
				callback.onSuccess( arg0 );
			}
		} );
	}

	public void getNewInstanceID(final int count, final IdCallback callback) {
		try {
			repositoryService.getNewInstanceID( session, count, new AsyncCallback<List<EIURI>>() {

				@Override
				public void onFailure(final Throwable arg0) {
					handleFailure( callback, arg0 );
				}

				@Override
				public void onSuccess(final List<EIURI> list) {
					if ( list == null ) // because GWT makes it hard to
					// throw exceptions
					{
						callback.loginRequired();
						return;
					}
					callback.onSuccess( list );
				}
			} );
		} catch (final Exception e) {
			e.printStackTrace();
		}
	}

	public void createInstances(final List<EIInstance> instances, final EIInstancesCallback callback) {
		try {
			repositoryService.createInstances( session, instances, ApplicationState.getInstance().getWorkspaceEntity(), new AsyncCallback() {
				@Override
				public void onFailure(final Throwable arg0) {
					Window.alert( arg0.getMessage() );
					log.warn( "failed " + arg0 );
					callback.onFailure( arg0.toString() );
				}

				@Override
				public void onSuccess(final Object arg0) {
					log.info( "creation succeeded" );
					for (EIInstance instance : instances) {
						instance.setWFState( WorkFlowConstants.DRAFT_ENTITY );
					}
					callback.onSuccess( instances );
				}
			} );
		} catch (final Exception e) {
			e.printStackTrace();
		}
	}

	public void createInstance(final EIInstance instance, final EIInstanceCallback callback) {
		try {
			log.info( "creating instance " + instance.getEntity() + " with type " + instance.getInstanceClass() );
			repositoryService.createInstance( session, instance, ApplicationState.getInstance().getWorkspaceEntity(), new AsyncCallback() {

				@Override
				public void onFailure(final Throwable arg0) {
					Window.alert( arg0.getMessage() );
					log.warn( "failed " + arg0 );
					callback.onFailure( arg0.toString() );
				}

				@Override
				public void onSuccess(final Object arg0) {
					log.info( "creation succeeded" );
					instance.setWFState( WorkFlowConstants.DRAFT_ENTITY );
					callback.onSuccess( instance );
				}
			} );
		} catch (final Exception e) {
			e.printStackTrace();
		}
	}

	public void getEmptyEIInstance(final EIURI classUri, final EIEntity instanceEntity, final EIInstanceCallback callback) {
		repositoryService.getEmptyEIInstance( session, classUri, instanceEntity, new AsyncCallback<EIInstance>() {

			@Override
			public void onFailure(final Throwable arg0) {
				handleFailure( callback, arg0 );
			}

			@Override
			public void onSuccess(final EIInstance instance) {
				if ( instance == null || EIInstance.NULL_INSTANCE.equals( instance )) // because GWT makes it hard to
				// throw exceptions
				{
					callback.loginRequired();
					return;
				}
				instance.setWFOwner( user.getUserEntity() );
				// FIXME can we get away from hardcoding this first state?
				instance.setWFState( WorkFlowConstants.DRAFT_ENTITY );
				callback.onSuccess( instance );
			}
		} );
	}

	public void getEmptyEIInstance(final EIURI classUri, final EIInstanceCallback callback) {
		repositoryService.getEmptyEIInstance( session, classUri, new AsyncCallback<EIInstance>() {

			@Override
			public void onFailure(final Throwable arg0) {
				log.warn( "failed to get empty instance with class " + classUri );
				handleFailure( callback, arg0 );
			}

			@Override
			public void onSuccess(final EIInstance instance) {
				if ( instance == null || EIInstance.NULL_INSTANCE.equals( instance )) // because GWT makes it hard to
				// throw exceptions
				{
					log.warn( "login required for getting empty instance" );
					callback.loginRequired();
					return;
				}
				log.info( "got empty instance with class " + classUri );
				callback.onSuccess( instance );
			}
		} );
	}

	public void deepCopyInstance(final EIURI originalUri, final EIInstanceCallback callback) {
		try {
			repositoryService.deepCopy( session, originalUri, new AsyncCallback<EIInstance>() {

				@Override
				public void onFailure(final Throwable caught) {
				}

				@Override
				public void onSuccess(final EIInstance result) {
					callback.onSuccess( result );
				}
			} );
		} catch (final Exception e) {
			log.error( e.getMessage() );
		}
	}

	public void whoami(final UserCallback callback) throws Exception {
		if ( repositoryService == null ) {
			repositoryService = GWT.create( RepositoryToolsModelService.class );
		}
		final Session session = DatatoolsCookies.getSession();
		repositoryService.whoami( session, new AsyncCallback<User>() {

			@Override
			public void onFailure(final Throwable arg0) {
				log.warn( "client whoami failure" );
				if ( arg0 != null ) {
					log.warn( "client whoami failure: " + arg0.getMessage() );
				}
			}

			@Override
			public void onSuccess(final User userInfo) {
				if ( userInfo != null && !userInfo.equals( user ) ) {
					user = userInfo;
				}
				callback.onSuccess( userInfo );
			}
		} );
	}

	public void claim(final EIInstanceMinimal instance, final EIInstanceCallback callback) {
		claim( new EIInstanceMinimal[] { instance }, new BulkWorkflowCallback() {

			@Override
			public void onSuccess(final List<EIInstanceMinimal> successes) {
				if ( successes == null || successes.size() == 0 ) {
					callback.onFailure( "claim failed for " + instance.getInstanceLabel() );
				}

				try {
					getInstance( successes.get( 0 ).getInstanceURI(), callback );
				} catch (final Exception e) {
					// e.printStackTrace(); // TODO: meaningless in this context
					callback.onFailure( e.toString() );
				}
			}

			@Override
			public void onFailure(final String error) {
				callback.onFailure( error );
			}

			@Override
			public void needsRefresh(final String message) {
				callback.onFailure( message ); // TODO: add a needsRefresh?

			}
		} );
	}

	public void claim(final EIInstanceMinimal[] instances, final BulkWorkflowCallback callback) {
		try {
			final Map<EIURI, EIInstanceMinimal> instanceUriMap = new HashMap<EIURI, EIInstanceMinimal>();
			for (final EIInstanceMinimal instance : instances) {
				instanceUriMap.put( instance.getInstanceURI(), instance );
			}

			repositoryService.getModifiedDates( session, new ArrayList<EIURI>( instanceUriMap.keySet() ), new AsyncCallback<Map<EIURI, String>>() {

				@Override
				public void onFailure(final Throwable caught) {
					callback.onFailure( "Failed to get modified dates " + caught.toString() );
				}

				@Override
				public void onSuccess(final Map<EIURI, String> result) {

					final List<EIURI> unchangedUris = new ArrayList<EIURI>();
					final List<EIURI> outdatedUris = new ArrayList<EIURI>();

					partitionOKUris( instanceUriMap, result, unchangedUris, outdatedUris );

					if ( outdatedUris.size() > 0 ) {
						callback.needsRefresh( outdatedUris.size() + " resources have been edited since first load" );
						return;
					}

					// TODO: need to warn about outdated uris
					try {
						repositoryService.claim( session, unchangedUris, new AsyncCallback<List<EIURI>>() {

							@Override
							public void onFailure(final Throwable caught) {
								callback.onFailure( caught.toString() );
							}

							@Override
							public void onSuccess(final List<EIURI> result) {
								final List<EIInstanceMinimal> successes = new ArrayList<EIInstanceMinimal>();
								for (final EIURI succeededUri : result) {
									final EIInstanceMinimal succeeded = instanceUriMap.get( succeededUri );
									succeeded.setWFOwner( user.getUserEntity() );
									successes.add( succeeded );
								}
								callback.onSuccess( successes );
							}
						} );
					} catch (final Exception e) {
						// e.printStackTrace();
						callback.onFailure( e.toString() );
					}
				}
			} );

		} catch (final Exception e) {
			// e.printStackTrace();
			callback.onFailure( e.toString() );
		}
	}

	protected void partitionOKUris(final Map<EIURI, EIInstanceMinimal> instanceUriMap, final Map<EIURI, String> uriModificationDateMap, final List<EIURI> unchangedUris, final List<EIURI> outdatedUris) {
		final DateTimeFormat dateFormat = DateTimeFormat.getFormat( "yyyy-MM-dd'T'HH:mm:ss.SSSZ" );

		for (final EIURI uri : instanceUriMap.keySet()) {
			final String repositoryDateString = uriModificationDateMap.get( uri );
			if ( repositoryDateString == null ) {
				log.warn( "no modification date from repository for " + uri );
				continue;
			}

			String instanceDateString = instanceUriMap.get( uri ).getModifiedDate();
			if ( instanceDateString == null ) {
				outdatedUris.add( uri );
				return;
			}
			if ( instanceDateString.indexOf( "^" ) != -1 ) {
				instanceDateString = instanceDateString.substring( 0, instanceDateString.indexOf( "^" ) );
			}

			try {
				final Date repositoryModification = dateFormat.parse( repositoryDateString );
				final Date instanceModification = dateFormat.parse( instanceDateString );

				if ( repositoryModification.after( instanceModification ) ) {
					outdatedUris.add( uri );
				} else {
					unchangedUris.add( uri );
				}
			} catch (final IllegalArgumentException e) {
				log.info( "could not convert date " + e );
				e.printStackTrace();
			}

		}
	}

	public void release(final EIInstanceMinimal instance, final EIInstanceCallback callback) {
		release( new EIInstanceMinimal[] { instance }, new BulkWorkflowCallback() {

			@Override
			public void onSuccess(final List<EIInstanceMinimal> successes) {
				if ( successes == null || successes.size() == 0 ) {
					callback.onFailure( "release failed for " + instance.getInstanceLabel() );
				}

				try {
					getInstance( successes.get( 0 ).getInstanceURI(), callback );
				} catch (final Exception e) {
					e.printStackTrace(); // TODO: meaningless in this context
					callback.onFailure( e.toString() );
				}
			}

			@Override
			public void onFailure(final String error) {
				callback.onFailure( error );
			}

			@Override
			public void needsRefresh(final String message) {
				callback.onFailure( message ); // TODO: add a needsRefresh here too?
			}
		} );
	}

	public void release(final EIInstanceMinimal[] instances, final BulkWorkflowCallback callback) {
		try {
			final Map<EIURI, EIInstanceMinimal> instanceUriMap = new HashMap<EIURI, EIInstanceMinimal>();
			for (final EIInstanceMinimal instance : instances) {
				instanceUriMap.put( instance.getInstanceURI(), instance );
			}

			repositoryService.release( session, new ArrayList<EIURI>( instanceUriMap.keySet() ), new AsyncCallback<List<EIURI>>() {

				@Override
				public void onFailure(final Throwable caught) {
					callback.onFailure( caught.toString() );
				}

				@Override
				public void onSuccess(final List<EIURI> result) {
					final List<EIInstanceMinimal> successes = new ArrayList<EIInstanceMinimal>();
					for (final EIURI succeededUri : result) {
						final EIInstanceMinimal succeeded = instanceUriMap.get( succeededUri );
						succeeded.setWFOwner( EIEntity.NULL_ENTITY );
						successes.add( succeeded );
					}
					callback.onSuccess( successes );
				}
			} );
		} catch (final Exception e) {
			callback.onFailure( e.toString() );
			e.printStackTrace();
		}
	}

	public static boolean isNotNull(final EIEntity entity) {
		return entity != null && !EIEntity.NULL_ENTITY.equals( entity );
	}
	
	public static boolean isNotNull(final EIInstance instance) {
		return instance != null && !EIInstance.NULL_INSTANCE.equals( instance );
	}

	public boolean instanceHasCurrentOwner(final EIInstance instance) {
		return isNotNull( instance.getWFOwner() ) && instance.getWFOwner().equals( user.getUserEntity() );
	}

	public boolean instanceHasCurrentOwner(final EIInstanceMinimal instance) {
		return isNotNull( instance.getWFOwner() ) && instance.getWFOwner().equals( user.getUserEntity() );
	}

	public boolean instanceHasAnyOwner(final EIInstance instance) {
		return isNotNull( instance.getWFOwner() );
	}

	public boolean instanceHasAnyOwner(final EIInstanceMinimal instance) {
		return isNotNull( instance.getWFOwner() );
	}

	public void transition(final EIInstance instance, final WorkFlowTransition transition, final EIInstanceCallback callback) {
		transition( new EIInstanceMinimal[] { EIInstanceMinimal.create( instance ) }, transition, new BulkWorkflowCallback() {

			@Override
			public void onSuccess(final List<EIInstanceMinimal> successes) {
				if ( successes == null || successes.size() == 0 ) {
					callback.onFailure( "claim failed for " + instance.getInstanceLabel() );
				}

				try {
					getInstance( successes.get( 0 ).getInstanceURI(), callback );
				} catch (final Exception e) {
					e.printStackTrace(); // TODO: meaningless in this context
					callback.onFailure( e.toString() );
				}
			}

			@Override
			public void onFailure(final String error) {
				callback.onFailure( error );
			}

			@Override
			public void needsRefresh(final String message) {
				callback.onFailure( message );
			}
		} );
	}

	public void transition(final EIInstanceMinimal[] eiInstanceMinimals, final WorkFlowTransition transition, final BulkWorkflowCallback callback) {
		try {
			if ( transition == null ) {
				log.error( "null transition!" );
				callback.onFailure( null );
			}

			final Map<EIURI, EIInstanceMinimal> instanceUriMap = new HashMap<EIURI, EIInstanceMinimal>();
			for (final EIInstanceMinimal instance : eiInstanceMinimals) {
				instanceUriMap.put( instance.getInstanceURI(), instance );
			}

			repositoryService.transition( session, new ArrayList<EIURI>( instanceUriMap.keySet() ), transition.getEntity(), new AsyncCallback<List<EIURI>>() {

				@Override
				public void onFailure(final Throwable caught) {
					callback.onFailure( caught.toString() );
				}

				@Override
				public void onSuccess(final List<EIURI> result) {
					final List<EIInstanceMinimal> successes = new ArrayList<EIInstanceMinimal>();
					for (final EIURI succeededUri : result) {
						final EIInstanceMinimal succeeded = instanceUriMap.get( succeededUri );
						succeeded.setWFState( WorkFlowConstants.WORKFLOW_URI_MAP.get( transition.getToStateURI() ) );
						succeeded.setWFOwner( EIEntity.NULL_ENTITY ); // Force this: we know it's true because that's what transitioning does
						successes.add( succeeded );
					}
					callback.onSuccess( successes );
				}
			} );
		} catch (final Exception e) {
			callback.onFailure( e.toString() );
			e.printStackTrace();
		}
	}

	public void retrieveLabel(final EIURI uri, final ResultsCallback callback) {
		repositoryService.retrieveLabel( session, uri, new AsyncCallback<String>() {

			@Override
			public void onFailure(final Throwable arg0) {
				callback.onFailure( arg0.toString() );
			}

			@Override
			public void onSuccess(final String label) {
				callback.onSuccess( label );
			}
		} );
	}

	public void getRootSuperclassForInstanceUri(final EIURI instanceUri, final ClassCallback callback) {
		try {
			repositoryService.getRootSuperclassForInstanceUri( session, instanceUri, new AsyncCallback<EIClass>() {

				@Override
				public void onFailure(final Throwable caught) {
				}

				@Override
				public void onSuccess(final EIClass result) {
					callback.onSuccess( result );
				}
			} );
		} catch (final Exception e) {
			log.error( "modelService.getRootSuperclassForInstanceUri threw exception " + e.toString() );
			e.printStackTrace();
		}
	}

	public void getClassAndSuperclassesForInstanceUri(final EIURI instanceUri, final EIClassesCallback callback) {
		try {
			repositoryService.getClassAndSuperclassesForInstanceUri( session, instanceUri, new AsyncCallback<List<EIClass>>() {

				@Override
				public void onFailure(final Throwable caught) {
					callback.onFailure( caught.getMessage() );
				}

				@Override
				public void onSuccess(final List<EIClass> result) {
					callback.onSuccess( result );
				}
			} );
		} catch (final Exception e) {
			log.error( "modelService.getSuperclassesForInstanceUri threw exception " + e.toString() );
			e.printStackTrace();
			callback.onFailure( e.toString() );
		}
	}

	public List<WorkFlowTransition> getAllowedTransitions() {
		return user.getAllowedTransitions();
	}

	public List<WorkFlowTransition> getAllowedTransitions(final EIInstance instance) {
		return getAllowedTransitions( instance.getWFState() );
	}

	public List<WorkFlowTransition> getAllowedTransitions(final EIInstanceMinimal instance) {
		return getAllowedTransitions( instance.getWFState() );
	}

	private List<WorkFlowTransition> getAllowedTransitions(final EIEntity workflowState) {
		if ( isNotNull( workflowState ) ) {
			return user.getAllowedTransitionsForState( workflowState.getURI() );
		} else { // state may be null if instance is new and hasn't been pushed to repo
			// FIXME can we move from hardcoding this here?
			return user.getAllowedTransitionsForState( WorkFlowConstants.DRAFT_URI );
		}
	}

	private void handleFailure(final LoginRequiredCallback callback, final Throwable arg0) {
		if ( arg0 instanceof LoggedException ) {
			if ( arg0.getMessage().contains( "No session information was found" ) ) {
				callback.loginRequired();
			} else {
				Window.alert( arg0.getMessage() );
			}
		} else {
			callback.loginRequired();
		}
	}

	/*
	 * ontology methods that don't belong here, but we have to have them because of services etc
	 */
	public void getSuperClass(final EIClass eclass, final DatatoolsClassCallback callback) {
		repositoryService.getSuperClass( eclass, new AsyncCallback<EIClass>() {

			public void onFailure(final Throwable caught) {
				callback.onFailure( caught.toString() );
			}

			public void onSuccess(final EIClass result) {
				callback.onSuccess( result );
			}
		} );
	}

	public void getRootSuperClass(final EIClass eclass, final DatatoolsClassCallback callback) {
		repositoryService.getRootSuperClass( eclass, new AsyncCallback<EIClass>() {

			public void onFailure(final Throwable caught) {
				callback.onFailure( caught.toString() );
			}

			public void onSuccess(final EIClass result) {
				callback.onSuccess( result );
			}
		} );
	}

	public void getLabRootSuperclass(final DatatoolsClassCallback callback) {
		if ( mapIdToClass.containsKey( EagleIEntityConstants.EI_LAB_URI ) ) {
			callback.onSuccess( mapIdToClass.get( EagleIEntityConstants.EI_LAB_URI ) );
		} else {
			modelService.getClass( EagleIEntityConstants.EI_LAB_URI, new AsyncCallback<EIClass>() {

				public void onFailure(final Throwable caught) {
					callback.onFailure( caught.toString() );
				}

				public void onSuccess(final EIClass result) {
					mapIdToClass.put( EagleIEntityConstants.EI_LAB_URI, result );
					callback.onSuccess( result );
				}
			} );
		}
	}

	public void getEquivalentClasses(final List<EIProperty> propertiesToPopulate, final EquivalentClassesCallback callback) {
		repositoryService.getAllEquivalentClasses( propertiesToPopulate, new AsyncCallback<List<EIProperty>>() {
			@Override
			public void onFailure(final Throwable caught) {
				callback.onFailure( caught.toString() );
			}

			@Override
			public void onSuccess(final List<EIProperty> result) {
				callback.onSuccess( result );
			}
		} );
	}

	public void getTopClassesAnotatedByDataModelCreate(final TopLevelClassesCallback callback) {
		repositoryService.getTopClassesAnotatedByDataModelCreate( new AsyncCallback<List<EIClass>>() {

			@Override
			public void onFailure(Throwable arg0) {
				log.error( "Async request failed in org.eaglei.datatools.client.rpc.ClientRepositoryToolsManager#getTopClassesAnotatedByDataModelCreate : the reason is " + arg0.getMessage() );
			}

			@Override
			public void onSuccess(List<EIClass> result) {
				Collections.sort( result, new Comparator<EIClass>() {

					@Override
					public int compare(EIClass o1, EIClass o2) {
						return o1.getEntity().compareTo( o2.getEntity() );
					}

				} );
				callback.onSuccess( result );
			}

		} );
	}

	public void isModelClassURI(final EIURI eiuri, final AsyncCallback<Boolean> callback) {
		repositoryService.isModelClassURI( eiuri, callback );
	}

}
