package org.eaglei.datatools.client.ui;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.eaglei.datatools.SortByProperties;
import org.eaglei.datatools.client.logging.GWTLogger;
import org.eaglei.datatools.client.rpc.ClientRepositoryToolsManager;
import org.eaglei.datatools.client.rpc.ClientRepositoryToolsManager.MinimalEIInstancesCallback;
import org.eaglei.datatools.client.rpc.ClientRepositoryToolsManager.SessionListener;
import org.eaglei.model.EIEntity;
import org.eaglei.model.EIInstanceMinimal;
import org.eaglei.model.EIURI;
import org.eaglei.search.provider.AuthSearchRequest;
import org.eaglei.ui.gwt.instance.EagleIEntityConstants;

import com.google.gwt.user.client.Window;

public class LabCache implements SessionListener {

	public interface LabCacheListener {

		void onLabsLoaded();
	}

	private final Map<String, EIEntity> labs = new HashMap<String, EIEntity>();
	private final Map<EIURI, EIEntity> labUris = new HashMap<EIURI, EIEntity>();
	private boolean loading = true;
	private boolean loadStarted = false;
	private final List<LabCacheListener> listeners;

	private static final GWTLogger log = GWTLogger.getLogger( "LabCache" );

	public LabCache() {
		log.debug( "making new lab cache" );
		listeners = new ArrayList<LabCacheListener>();
		ClientRepositoryToolsManager.INSTANCE.addSessionListener( this );
	}

	public void update() {
		if ( loadStarted ) {
			return;
		}
		
		loading = true;
		loadStarted = true;
		log.debug( "lab cache getting labs" );
		try {
			final AuthSearchRequest request = new AuthSearchRequest();
			request.setType( EagleIEntityConstants.EI_LAB_URI );
			request.setPaginated( false ); // TODO: figure out what *other* filtering we need to do, and enable that
			ClientRepositoryToolsManager.INSTANCE.listResources(request, SortByProperties.label, false, new MinimalEIInstancesCallback() {

				@Override
				public void onFailure(String error) {
					Window.alert( "Unable to retrieve lisit of labs.  Error was " + error );
				}

				@Override
				public void onSuccess(List<EIInstanceMinimal> result) {
					labs.clear();

					for (final EIInstanceMinimal instance : result) {
						labs.put( instance.getInstanceLabel(), instance.getEntity() );
						labUris.put( instance.getInstanceURI(), instance.getEntity() );
					}
					
					loading = false;
					loadStarted = false;
					notifyListeners();
				}

				@Override
				public void loginRequired() {
					Window.alert( "(in login required)" ); // TODO: remove
				}

			} );
		} catch (Exception e) {
			e.printStackTrace();
			Window.alert( "Unable to retrieve lisit of labs.  Error was " + e );
		}
	}

	public List<String> getLabLabels() {
		List<String> labels = new ArrayList<String>( labs.keySet() );
		Collections.sort( labels ); // TODO: ignore case!
				
		return labels;
	}

	public List<EIEntity> getLabEntities() {
		List<EIEntity> labEntities = new ArrayList<EIEntity>();

		for (final String label : getLabLabels()) {
			labEntities.add( labs.get( label ) );
		}

		return labEntities;
	}

	public EIEntity getLab(String labLabel) {
		if ( !labs.containsKey( labLabel ) ) {
			log.warn( "trying to get nonexistent entry '" + labLabel + "'" );
		}
		return labs.get( labLabel );
	}
	
	public EIEntity getLab(EIURI labUri) {
		if ( !labUris.containsKey( labUri ) ) {
			log.warn( "trying to get nonexistent uri '" + labUri + "'" );
		}
		return labUris.get( labUri );
	}

	public boolean isLoading() {
		return loading;
	}

	public void addLabLoadListener(LabCacheListener listener) {
		listeners.add( listener );
	}

	private void notifyListeners() {
		for (final LabCacheListener listener : listeners) {
			listener.onLabsLoaded();
		}
	}

	@Override
	public void onLogIn(String username, String userUri) {
		update();
	}

	@Override
	public void onLogOut() {
		listeners.clear();
		// labs.clear();
	}
}
