/**
 * The eagle-i consortium
 * Harvard University
 * Aug 24, 2010
 */
package org.eaglei.datatools.client.ui.widgets;

import java.util.List;

import org.eaglei.datatools.client.ApplicationState;
import org.eaglei.datatools.client.QueryTokenObject.Mode;
import org.eaglei.datatools.client.logging.GWTLogger;
import org.eaglei.datatools.client.rpc.ClientRepositoryToolsManager;
import org.eaglei.datatools.client.rpc.ClientRepositoryToolsManager.DeleteInstanceCallback;
import org.eaglei.datatools.client.ui.ActionRedisplay;
import org.eaglei.datatools.client.ui.OwnershipAction;
import org.eaglei.datatools.client.ui.UIMessages;
import org.eaglei.datatools.client.ui.WidgetUtils;
import org.eaglei.datatools.client.ui.WorkflowUtils;
import org.eaglei.datatools.client.ui.listeners.RowDeletionListener;
import org.eaglei.model.EIEntity;
import org.eaglei.model.EIInstanceMinimal;
import org.eaglei.ui.gwt.instance.EagleIEntityConstants;

import com.google.gwt.event.dom.client.ClickEvent;
import com.google.gwt.event.dom.client.ClickHandler;
import com.google.gwt.user.client.Window;
import com.google.gwt.user.client.ui.Anchor;
import com.google.gwt.user.client.ui.CheckBox;
import com.google.gwt.user.client.ui.FlowPanel;
import com.google.gwt.user.client.ui.HTML;
import com.google.gwt.user.client.ui.Label;
import com.google.gwt.user.client.ui.Widget;

/**
 * @author Daniela Bourges-Waldegg
 * @author Lucy Hadden
 * 
 */
public class GridRowWidget extends FlowPanel {


	
	private static final int CHECKBOX_COLUMN = 7;
	private static final int ACTIONS_FIRST_COLUMN = 4; //3 action columns computed by adding relative action position
	private static final int STATUS_COLUMN = 3;
	private static final int DATE_COLUMN = 2;
	private static final int TYPE_COLUMN = 1;
	private static final int NAME_COLUMN = 0;
	private static final String LIVE_LINK = "liveLink";

	private final static String UNCLAIM_LABEL = "<a>share</a>";
	private final static String CLAIM_LABEL = "<a>claim</a>";
	private final static int CLAIM_OR_SHARE_POSITION = 0;
	private final static String EDIT_LABEL = "<a>edit</a>";
	private final static int EDIT_POSITION = 1;
	private final static String DELETE_LABEL = "<a>delete</a>";
	private final static int DELETE_POSITION = 2;

	private EIInstanceMinimal instance;
	private int rowIndex;
	private final ActionRedisplay redisplay;
	private final RowDeletionListener listener;
	// for now we have only three actions per widget
	// TODO html String for now; migrate to a new FormAction class
	private final HTML formActions[] = { new HTML(), new HTML(), new HTML() };

	private CheckBox checkBox;
	
	private static final GWTLogger log = GWTLogger.getLogger("GridRowWidget");

	public GridRowWidget(final EIInstanceMinimal instance, final RowDeletionListener deletionListener) {
		super();
		listener = deletionListener;
		redisplay = new ActionRedisplay() {

			public void drawAfterClaim(final EIInstanceMinimal instance) {
				if (ApplicationState.getInstance().isStrictlyFilteredByOwner() && !instanceHasOwner(instance)) {
					listener.onRowDeletion(instance);
				}
				redrawActions();
			}

			public void drawAfterPromote(final EIInstanceMinimal instance) {
				redrawActions();
			}

			@Override
			public void drawAfterClaim(final EIInstanceMinimal[] instances, final List<EIInstanceMinimal> successes) {
				if (successes == null || successes.size() == 0) {
					final String failMessage = UIMessages.CLAIM_FAILURE.replace("$RESOURCES$", instance.getInstanceLabel()).replace("they", "it");
					Window.alert(failMessage);
				} else {
					final EIInstanceMinimal claimedMinimal = successes.get(0);
					instance.setWFOwner(claimedMinimal.getWFOwner());
					drawAfterClaim(instance);
				}
			}

			@Override
			public void handleFailedTransition(final String error) {
				Window.alert("Action failed.");
			}

			@Override
			public void drawAfterPromote(final EIInstanceMinimal[] instances, final List<EIInstanceMinimal> successes) {
				if (successes == null || successes.size() == 0) {
					final String failMessage = UIMessages.TRANSITION_FAILURE.replace("$TRANSITION$", "").replace("$RESOURCES$", instance.getInstanceLabel()).replace("they", "it");
					Window.alert(failMessage);
				} else {
					final EIInstanceMinimal claimedMinimal = successes.get(0);
					instance.setWFState(claimedMinimal.getWFState());
					drawAfterPromote(instance);
				}
			}

			@Override
			public void needsRefresh(final String message) {
				Window.alert(UIMessages.REFRESH_SINGLE_MESSAGE);
			}
		};
		init(instance);
	}

	private void init(final EIInstanceMinimal instance) {
		this.instance = instance;
		setStyleName("dtListGridRow");
		//FIXME deal better with the column numbers!
		final HTML instanceLabel = new HTML("<a>" + instance.getInstanceLabel() + "<a>" );
		makeClickable(instance.getEntity(), instance.getInstanceType(), instanceLabel);
		instanceLabel.setStyleName(LIVE_LINK);
		if ( !isNull(instance.getLab()) ) {
			final FlowPanel firstCell = new FlowPanel();
			final HTML labLabel = new HTML("<i><a>" + instance.getLab().getLabel() + "<a></i>" );
			labLabel.setStyleName("dtListGridRowLab");
			makeClickable(instance.getLab(), EagleIEntityConstants.EI_LAB_ENTITY, labLabel);
			labLabel.setStyleName(LIVE_LINK);
			firstCell.add(instanceLabel);
			firstCell.add(labLabel);
			addWithStyle(firstCell, NAME_COLUMN);

		} else {
			addWithStyle( instanceLabel, NAME_COLUMN );
		}
		addWithStyle(new Label(instance.getInstanceType().getLabel()), TYPE_COLUMN);
		final String date = instance.getCreationDate();
		Label dateLabel = new Label(" ");
		if (date != null) {
			dateLabel = new Label(WidgetUtils.formatDate(date));
		}
		addWithStyle(dateLabel, DATE_COLUMN);
		final Label status = new Label(isNull(instance.getWFState()) ? " " : instance.getWFState().getLabel());
		addWithStyle(status, STATUS_COLUMN);
		for(Widget widget : formActions) {
			widget.setStyleName(LIVE_LINK);
		}
		if(ClientRepositoryToolsManager.INSTANCE.canClaim( instance ) || ClientRepositoryToolsManager.INSTANCE.canEdit( instance )) {
			addWithStyle(formActions[CLAIM_OR_SHARE_POSITION], ACTIONS_FIRST_COLUMN + CLAIM_OR_SHARE_POSITION);
			addWithStyle(formActions[EDIT_POSITION], ACTIONS_FIRST_COLUMN + EDIT_POSITION);
			addWithStyle(formActions[DELETE_POSITION], ACTIONS_FIRST_COLUMN + DELETE_POSITION);

			checkBox = new CheckBox();
			checkBox.setVisible(shouldDisplayCheckBox());
			addWithStyle(checkBox, CHECKBOX_COLUMN);
			redrawActions();
		} else if (instance.getWFOwner() != null && !EIEntity.NULL_ENTITY.equals( instance.getWFOwner() )){
			log.debug(instance.getInstanceLabel()+" is already claimed by differnt owner : the owner is  "+instance.getWFOwner().getLabel());
			//Per agreement with Seth style is gridRowColumn456
			addWithStyle(new Label("current editor is " + instance.getWFOwner().getLabel()), 456);
		}

	}

	private void makeClickable(final EIEntity entityToDisplay, final EIEntity typeOfEntityToDisplay, final HTML label) {
		label.addClickHandler(new ClickHandler() {
			@Override
			public void onClick(final ClickEvent event) {
				final EIEntity updatedLabEntity = getUpdatedLabEntity(entityToDisplay);
				log.info( "updated lab entity: " + updatedLabEntity );
				ApplicationState.getInstance().updateApplicationState(Mode.view, entityToDisplay, typeOfEntityToDisplay, updatedLabEntity);
			}
		});
	}

	public EIInstanceMinimal getInstance() {
		return instance;
	}

	public CheckBox getCheckBox() {
		return checkBox;
	}

	public HTML[] getFormActions() {
		return formActions;
	}

	private boolean shouldDisplayCheckBox() {
		if (ApplicationState.getInstance().hasLabType()) {
			return false;
		}
		return ClientRepositoryToolsManager.INSTANCE.canEdit(instance) || ClientRepositoryToolsManager.INSTANCE.canClaim(instance);
	}

	private void setClaimOrRelease() {
		if (ClientRepositoryToolsManager.INSTANCE.canEdit(instance)) {
			formActions[CLAIM_OR_SHARE_POSITION] = new HTML(UNCLAIM_LABEL);
			formActions[CLAIM_OR_SHARE_POSITION].setStyleName(LIVE_LINK);
			formActions[CLAIM_OR_SHARE_POSITION].addClickHandler(new ClickHandler() {

				@Override
				public void onClick(final ClickEvent event) {
					releaseAction();
				}
			});
		} else if (ClientRepositoryToolsManager.INSTANCE.canClaim(instance)){
			formActions[CLAIM_OR_SHARE_POSITION] = new HTML(CLAIM_LABEL);
			formActions[CLAIM_OR_SHARE_POSITION].setStyleName(LIVE_LINK);
			formActions[CLAIM_OR_SHARE_POSITION].addClickHandler(new ClickHandler() {

				@Override
				public void onClick(final ClickEvent event) {
					claimAction();
				}
			});
		} // this is called from redrawActions, which implies user can edit or claim; 
	}

	private void setEdit() {
		if (ClientRepositoryToolsManager.INSTANCE.canEdit(instance)) {
			formActions[EDIT_POSITION] = new HTML(EDIT_LABEL);
			formActions[EDIT_POSITION].setStyleName(LIVE_LINK);
			formActions[EDIT_POSITION].addClickHandler(new ClickHandler() {

				@Override
				public void onClick(final ClickEvent event) {
					ApplicationState.getInstance().updateApplicationState(Mode.edit, instance.getEntity(), instance.getInstanceType(), getUpdatedLabEntity(instance.getEntity()));
				}
			});
		} else {
			formActions[EDIT_POSITION] = new HTML();
		}
	}

	private void setDelete() {
		if (ClientRepositoryToolsManager.INSTANCE.canEdit(instance)) {
			formActions[DELETE_POSITION] = new HTML(DELETE_LABEL);
			formActions[DELETE_POSITION].setStyleName(LIVE_LINK);
			formActions[DELETE_POSITION].addClickHandler(new ClickHandler() {

				@Override
				public void onClick(final ClickEvent event) {
					if (Window.confirm(UIMessages.DELETE_CONFIRM_MESSAGE)) {
						try {

							ClientRepositoryToolsManager.INSTANCE.deleteInstance(instance.getInstanceURI(), new DeleteInstanceCallback() {

								@Override
								public void onSuccess(final Object obj) {
									if (listener != null) {
										listener.onRowDeletion(instance);
									}
								}

								@Override
								public void onFailure(final String error) {
									log.error("could not delete " + instance.getInstanceLabel());
									Window.alert("Could not delete " + instance.getInstanceLabel());
								}
							});
						} catch (final Exception e) {
							log.error("could not delete " + instance.getInstanceLabel());
						}
					}
				}
			});
		} else {
			formActions[DELETE_POSITION] = new HTML();
		}
	}

	public boolean instanceHasOwner(final EIInstanceMinimal minimal) {
		return WorkflowUtils.isNotNull(minimal.getWFOwner());
	}

	public boolean instanceHasOwner() {
		return instanceHasOwner(instance);
	}

	public int getRowIndex() {
		return rowIndex;
	}

	public void setRowIndex(final int rowIndex) {
		this.rowIndex = rowIndex;
	}


	private void claimAction() {
		OwnershipAction.ClaimAction.perform(instance, redisplay);
	}

	private void releaseAction() {
		OwnershipAction.ReleaseAction.perform(instance, redisplay);
	}
	
	public void updateStateData() {
		final String statusText = isNull(instance.getWFState()) ? " " : instance.getWFState().getLabel();
		((Label) getWidget(STATUS_COLUMN)).setText(statusText);
		redrawActions();
	}

	public void redrawActions() {
		remove(getFormActions()[CLAIM_OR_SHARE_POSITION]);
		remove(getFormActions()[EDIT_POSITION]);
		remove(getFormActions()[DELETE_POSITION]);
		if (checkBox != null) {
			remove(checkBox);
		}
		setEdit();
		setDelete();
		setClaimOrRelease();
		addWithStyle(getFormActions()[CLAIM_OR_SHARE_POSITION], ACTIONS_FIRST_COLUMN + CLAIM_OR_SHARE_POSITION);
		addWithStyle(getFormActions()[EDIT_POSITION], ACTIONS_FIRST_COLUMN + EDIT_POSITION);
		addWithStyle(getFormActions()[DELETE_POSITION], ACTIONS_FIRST_COLUMN + DELETE_POSITION);
		if (checkBox != null) {
			addWithStyle(checkBox, CHECKBOX_COLUMN);
		}
		if (checkBox != null) {
			checkBox.setVisible(shouldDisplayCheckBox());
		}
		// replace(editor, getEditor());
	}

	// TODO: test this!
	private EIEntity getUpdatedLabEntity(final EIEntity instanceEntity) {
		if (ApplicationState.getInstance().getMode() == Mode.filter && !ApplicationState.getInstance().getFilterLabEntity().equals( EIEntity.NULL_ENTITY )) {
			return ApplicationState.getInstance().hasLabType() ? instanceEntity : ApplicationState.getInstance().getLabEntity();
		} 
		
		if ( ApplicationState.getInstance().getMode() == Mode.labs ) {
			return instanceEntity;
		}
		
		log.debug("app has lab? " + ApplicationState.getInstance().hasLab() + "; app state lab: ApplicationState.getInstance().getLabEntity() " + "; instance lab " + instance.getLab());
		if ( ApplicationState.getInstance().hasLab() ) {
			return ApplicationState.getInstance().getLabEntity();
		}
		
		if ( instance.getLab().getLabel().equals( EIEntity.NO_LABEL ) ) {
			return EIEntity.NULL_ENTITY;
		}
		
		return instance.getLab();
	}

	private boolean isNull(final EIEntity entity) {
		return (entity == null || entity.equals(EIEntity.NULL_ENTITY));
	}

	private void addWithStyle(final Widget widget, final int column) {
		add(widget);
		final String style = "gridRowColumn" + column;
		widget.addStyleName("gridRowColumn");
		widget.addStyleName(style);
	}
}
