package org.eaglei.lexical.lucene;

import java.io.IOException;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.lucene.analysis.Analyzer;
import org.apache.lucene.index.IndexWriter;
import org.apache.lucene.store.Directory;
import org.eaglei.model.EIClass;
import org.eaglei.model.EIOntModel;
import org.eaglei.model.EIURI;
import org.eaglei.model.jena.JenaEIOntModel;

import com.hp.hpl.jena.ontology.OntClass;
import com.hp.hpl.jena.ontology.OntModel;
import com.hp.hpl.jena.rdf.model.Property;
import com.hp.hpl.jena.shared.Lock;
import com.hp.hpl.jena.vocabulary.OWL;

/**
 * Extension of LuceneIndexer that populates an index using the labels of all
 * for resource and non-resource classes in the eagle-i ontology.
 * 
 * TODO transition to EI*
 * 
 * @rfrost
 */
public class EIModelLuceneIndexer extends LuceneEntityExtractionIndexer {

    private static final Log logger = LogFactory.getLog(EIModelLuceneIndexer.class);
    private static final boolean DEBUG = logger.isDebugEnabled();
    
    private EIOntModel eagleiOntModel;
    private OntModel jenaOntModel;

    /**
     * Creates a new ModelLuceneIndexer that indexes the eagle-i ontology as
     * retrieved from JenaEIOntModel.
     * 
     * @param indexAnalyzer Analyzer to use for index creation.
     * @param directory Directory for the index 
     */
    public EIModelLuceneIndexer(final Analyzer indexAnalyzer,
            final Directory directory, EIOntModel eagleiOntModel, OntModel jenaOntModel) {
        super(indexAnalyzer, directory); 
        this.eagleiOntModel = eagleiOntModel;
        this.jenaOntModel = jenaOntModel;
    }
    

    @Override
    protected void addDocuments(final IndexWriter iwriter) throws IOException {
        jenaOntModel.enterCriticalSection(Lock.READ);
        try {
            final Set<String> processedURIs = new HashSet<String>();
            // process all top-level URIs
            for (EIClass eiClass: eagleiOntModel.getTopLevelClasses()) {                
                processSubTree(eiClass.getEntity().getURI().toString(), jenaOntModel, iwriter, processedURIs);
            }
            // process all non-resource URIs
            for (EIClass eiClass: eagleiOntModel.getNonResourceBaseClasses()) {
                processSubTree(eiClass.getEntity().getURI().toString(), jenaOntModel, iwriter, processedURIs);
            }            
        } finally {
            jenaOntModel.leaveCriticalSection();
        }
    }

    // TODO Update JenaEIOntModel to support model traversal from an arbitrary point
    // in the graph and then perform the indexing through the EI* abstractions
    private void processSubTree(final String classURI, final OntModel ontModel, 
            final IndexWriter iwriter, 
            final Set<String> processedURIs) throws IOException {
        // get the root class
        final OntClass parentClass = ontModel.getOntClass(classURI);
        // get the subclasses
        final List<OntClass> subClasses = parentClass.listSubClasses().toList();
        processClass(parentClass, iwriter, processedURIs);
        for (OntClass ontClass : subClasses) {
        	processClass(ontClass, iwriter, processedURIs);
        }
    }
    
    // TODO type inferencing only valid for single-parents
    private void processClass(final OntClass ontClass, final IndexWriter iwriter, 
            final Set<String> processedURIs) throws IOException {
        if (OWL.Nothing.getURI().equals(ontClass.getURI())) {
            // Some reasoners return owl#Nothing in subclass list.
            //logger.warn("OWL Nothing, skipping");
            return;
        }
        final String uri = ontClass.getURI();
        if (uri == null) {
        	return;
        }
        if (processedURIs.contains(uri)) {
        	return;
        }
        processedURIs.add(uri);

        final EIURI eiURI = EIURI.create(uri);
        final String prefLabel = eagleiOntModel.getPreferredLabel(eiURI);
        final List<String> typeURIs = new ArrayList<String>();
        typeURIs.add(uri);
        for (OntClass parent: ontClass.listSuperClasses(false).toList()) {
            if (parent != null && parent.getURI() != null) {
                typeURIs.add(parent.getURI());
            }
        }

        final List<String> labels  = eagleiOntModel.getLabels(eiURI);

        updateDocument(uri, prefLabel, labels, typeURIs, iwriter);
    }
}
