package org.eaglei.lexical.lucene;

import java.io.IOException;

import org.apache.lucene.analysis.Analyzer;
import org.apache.lucene.store.Directory;
import org.apache.lucene.store.RAMDirectory;
import org.eaglei.lexical.EntityExtractionProvider;
import org.eaglei.lexical.lucene.AutoSuggestIndexAnalyzer;
import org.eaglei.lexical.lucene.AutoSuggestQueryAnalyzer;
import org.eaglei.model.EIOntModel;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Lazy;

import com.hp.hpl.jena.ontology.OntModel;

/**
 * Contains some utility methods for creating common configurations
 * of LuceneEntityExtractionProvider and LuceneIndexer.
 * 
 * @author rfrost
 */
@Configuration
public class LuceneConfig {

    @Autowired
    private OntModel jenaOntModel;

    @Autowired
    private EIOntModel eagleiOntModel;

    /**
     * Creates an LuceneEntityExtractionProvider that uses a PorterStemAnalyzer for
     * indexing an querying over an RAMDirectory that is populated using the eagle-i ontology
     */
    public @Lazy @Bean EntityExtractionProvider eagleiModelEntityExtractor() throws IOException {
        final Directory directory = new RAMDirectory();
        final Analyzer analyzer = new PorterStemAnalyzer();
        final LuceneEntityExtractionIndexer indexer = new EIModelLuceneIndexer(analyzer, directory, eagleiOntModel, jenaOntModel);
        indexer.index();
        return new LuceneEntityExtractionProvider(analyzer, directory);
	}

    /**
     * Creates an LuceneEntityExtractionProvider that uses the AutoSuggest Analyzers
     * over an RAMDirectory that is populated using the eagle-i ontology. 
     */
    public @Bean EntityExtractionProvider eagleiModelSuggestionProvider() throws IOException {    
        final Directory directory = new RAMDirectory();
        final Analyzer queryAnalyzer = new AutoSuggestQueryAnalyzer();
        final Analyzer indexAnalyzer = new AutoSuggestIndexAnalyzer();
        final LuceneEntityExtractionIndexer indexer = new EIModelLuceneIndexer(indexAnalyzer, directory, eagleiOntModel, jenaOntModel);
        indexer.index();
        return new LuceneEntityExtractionProvider(queryAnalyzer, indexAnalyzer, directory);
    }    
}
