package org.eaglei.model;


import java.io.Serializable;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;


/**
 * @author Daniela Bourges-Waldegg
 * @author Ricardo De Lima
 * 
 *         April 1, 2010
 * 
 *         Center for Biomedical Informatics (CBMI)
 * @link https://cbmi.med.harvard.edu/
 * 
 * 
 * Wrapper around an eagle-i resource instance. A resource instance is identified by an instance URI
 * (contained in an EIEntity)is defined as a set of all triples
 * where subject .
 * This wrapper provides a stripped-down version of this set of triples (only the information
 * relevant to the UI)
 * Note that the instance class has to be an EIClass; most EIInstances will have only one; in some cases
 * there may be necessary to include more than one; they'll
 * be contained in the List otherEIClasses.
 * Other non-eagle-i types are kept in nonOntologyObject properties
 */

public class EIInstance extends EIResource implements Serializable, Comparable
{

    private static final long serialVersionUID = 1L;
	
	//Use our own labels for now
	private static final EIEntity WFStateEntity = EIEntity.create(
			EIURI.create("http://eagle-i.org/ont/repo/1.0/hasWorkflowState"), 
			"Workflow state");
	private static final EIEntity WFOwnerEntity = EIEntity.create(
			EIURI.create("http://eagle-i.org/ont/repo/1.0/hasWorkflowOwner"), 
			"Workflow owner");
	private static final EIEntity CreatedEntity = EIEntity.create(
			EIURI.create("http://purl.org/dc/terms/created"), "Creation date");
	
	//quick access to the eiClass
	//Most of the time there will only be one
	EIEntity instanceType;
	//There may be more than one ei types for the instance
    List<EIEntity> otherEITypes;

    Map<EIEntity, Set<EIURI>> objectProperties = new HashMap<EIEntity, Set<EIURI>>();
    Map<EIEntity, Set<String>> datatypeProperties = new HashMap<EIEntity, Set<String>>();
    
    // Hold the ID()s and values of properties that were not found in the ontology, but were attached to the data in the repo;
   
    Map<EIEntity, Set<EIURI>> nonOntologyResourceProperties = new HashMap<EIEntity, Set<EIURI>>();
    Map<EIEntity, Set<String>> nonOntologyLiteralProperties = new HashMap<EIEntity, Set<String>>();

    Map<EIEntity, EIURI> readOnlyResourceProperties;
    Map<EIEntity, String> readOnlyLiteralProperties;
    
    private EIInstance() 
    {
        // for GWT
    }
    
    //constructor with type
    private EIInstance(final EIEntity instanceType, final EIEntity instanceEntity)
    {
    	super(instanceEntity);
        this.instanceType = instanceType;
        //instance type can be null
    }    

    public static EIInstance create(final EIEntity instanceType, final EIEntity instanceEntity) {
    	return new EIInstance(instanceType, instanceEntity);
    }
    
    
    public void addObjectPropertyToInstance(final EIEntity property, final EIURI resource)
    {
    	Set<EIURI> pValues;
    	if (objectProperties != null) {
    		pValues = objectProperties.get(property);
    		if(pValues == null) 
    			objectProperties.put(property, pValues = new HashSet<EIURI>());
    	} else {
    		objectProperties = new HashMap<EIEntity, Set<EIURI>>();
    		objectProperties.put(property, pValues = new HashSet<EIURI>());
    	}
    	
		pValues.add(resource);	
    }

    public void addDatattypePropertyToInstance(final EIEntity property, final String literal)
    {
    	Set<String> pValues;
    	if(datatypeProperties != null) {    		
    		pValues = datatypeProperties.get(property);
    		if(pValues == null) 
    			datatypeProperties.put(property, pValues = new HashSet<String>());
    	} else {
    		datatypeProperties = new HashMap<EIEntity, Set<String>>();
    		datatypeProperties.put(property, pValues = new HashSet<String>());    		
    	}	
		pValues.add(literal);
    }
    
    public void replaceNonOntologyResourceProperty(final EIEntity property, final EIURI resource)
    {
    	Set<EIURI> pValues;
    	if(nonOntologyResourceProperties != null) {
    		 pValues = nonOntologyResourceProperties.get(property);
    		 if(pValues == null)
    			 nonOntologyResourceProperties.put(property, pValues = new HashSet<EIURI>());
    		 else
    			 pValues.clear();
    	} else {
    		nonOntologyResourceProperties = new HashMap<EIEntity, Set<EIURI>>();
    		nonOntologyResourceProperties.put(property, pValues = new HashSet<EIURI>());
    	}
    	pValues.add(resource);
    }
    
    
    public void addNonOntologyResourceProperty(final EIEntity property, final EIURI resource)
    {
    	Set<EIURI> pValues;
    	if(nonOntologyResourceProperties != null) {
    		pValues = nonOntologyResourceProperties.get(property);
    		if(pValues == null)
    			nonOntologyResourceProperties.put(property, pValues=new HashSet<EIURI>());
    	} else {
    		nonOntologyResourceProperties = new HashMap<EIEntity, Set<EIURI>>();
    		nonOntologyResourceProperties.put(property, pValues=new HashSet<EIURI>());
    	}
    	pValues.add(resource);
    }

    public void addNonOntologyLiteralProperty(final EIEntity property, final String literal)
    {
    	Set<String> pValues;
    	if(nonOntologyLiteralProperties != null) {
    		pValues = nonOntologyLiteralProperties.get(property);
    		if(pValues == null)
    			nonOntologyLiteralProperties.put(property, pValues = new HashSet<String>());
    	} else {
    		nonOntologyLiteralProperties = new HashMap<EIEntity, Set<String>>();
    		nonOntologyLiteralProperties.put(property, pValues = new HashSet<String>());
    	}
    	pValues.add(literal);
    }  
    
    public void addEIType(final EIEntity eiType) {
    	if(eiType == null) return;
    	if(otherEITypes == null) {
    		otherEITypes = new ArrayList<EIEntity>();
    	}
    	otherEITypes.add(eiType);
    }
    
    public void setObjectProperties(Map<EIEntity, Set<EIURI>> objectProperties)
    {
        this.objectProperties = objectProperties;
    }

    public void setDatatypeProperties(Map<EIEntity, Set<String>> datatypeProperties)
    {
        this.datatypeProperties = datatypeProperties;
    }
    
    public void setOtherEITypes(List<EIEntity> eiTypes) {
    	this.otherEITypes = eiTypes;
    }
    
    public Map<EIEntity, Set<EIURI>> getObjectProperties()
    {
    	if(objectProperties == null)
    		return Collections.emptyMap();
    	else
    		return objectProperties;    	
    }

    public Map<EIEntity, Set<String>> getDatatypeProperties()
    {
    	if(datatypeProperties == null)
    		return Collections.emptyMap();
    	else
        return datatypeProperties;
    }

    public List<EIEntity> getOtherEITypes() {
    	if(otherEITypes == null)
    		return Collections.emptyList();
    	else
    		return otherEITypes;
    }
    
    public EIURI getInstanceURI() 
    {
        return getEntity().getURI();
    }
    
    public String getInstanceLabel() 
    {
        return getEntity().getLabel();
    }
    
    public EIEntity getInstanceType() 
    {
        return instanceType;
    }
    
    public void setInstanceType(EIEntity instanceType) {
    	this.instanceType = instanceType;
    }
    
  


    @Override
    public String toString() {
    	//Display with the first asserted class
        return instanceType.toString() + " : " + getEntity().toString();
    }

	/**
	 * @return the nonOntologyResourceProperties
	 */
	public Map<EIEntity, Set<EIURI>> getNonOntologyResourceProperties() {
		if (nonOntologyResourceProperties == null)
			return Collections.emptyMap();
		else
			return nonOntologyResourceProperties;
	}

	/**
	 * @param nonOntologyResourceProperties the nonOntologyResourceProperties to set
	 */
	public void setNonOntologyResourceProperties(
			Map<EIEntity, Set<EIURI>> nonOntologyResourceProperties) {
		this.nonOntologyResourceProperties = nonOntologyResourceProperties;
	}

	/**
	 * @return the nonOntologyLiteralProperties
	 */
	public Map<EIEntity, Set<String>> getNonOntologyLiteralProperties() {
		if (nonOntologyLiteralProperties == null)
			return Collections.emptyMap();
		else
			return nonOntologyLiteralProperties;
	}

	/**
	 * @param nonOntologyLiteralProperties the nonOntologyLiteralProperties to set
	 */
	public void setNonOntologyLiteralProperties(
			Map<EIEntity, Set<String>> nonOntologyLiteralProperties) {
		this.nonOntologyLiteralProperties = nonOntologyLiteralProperties;
	}
	
	public EIURI getWFState() {
		if(readOnlyResourceProperties == null)
			return null;
		if (readOnlyResourceProperties.containsKey(WFStateEntity)) {
			EIURI state = readOnlyResourceProperties.get(WFStateEntity);
			return state;
			}
			else return null;
	}
	
	public void setWFState(EIURI state) {
		setReadOnlyResourceProperty(WFStateEntity, state);
	}
	
	public EIURI getWFOwner() {
		if(readOnlyResourceProperties == null)
			return null;
		if (readOnlyResourceProperties.containsKey(WFOwnerEntity)) {
			EIURI state = readOnlyResourceProperties.get(WFOwnerEntity);
			return state;
		} else return null;
	}
	
	public void setWFOwner(EIURI ownerUri) {
		setReadOnlyResourceProperty(WFOwnerEntity, ownerUri);
	}
	
	public String getCreationDate() {
		if(readOnlyLiteralProperties == null)
			return null;
		if (readOnlyLiteralProperties.containsKey(CreatedEntity)) {
			String created = readOnlyLiteralProperties.get(CreatedEntity);
			return created;
		} else return null;
	}
	
	public void setCreationDate(String date) {
		setReadOnlyLiteralProperty(CreatedEntity, date);
	}

	/* (non-Javadoc)
	 * @see java.lang.Comparable#compareTo(java.lang.Object)
	 */
	@Override
	public int compareTo(Object o) {
		// TODO Auto-generated method stub
		if(!(o instanceof EIInstance))
			return 0;
		return this.getEntity().getLabel().compareToIgnoreCase(((EIInstance)o).getEntity().getLabel());
	}

	/**
	 * @return the readOnlyResourceProperties
	 */
	public Map<EIEntity, EIURI> getReadOnlyResourceProperties() {
		if(readOnlyResourceProperties == null)
			return Collections.emptyMap();
		else
			return readOnlyResourceProperties;
	}

	/**
	 * @param readOnlyResourceProperties the readOnlyResourceProperties to set
	 */
	public void setReadOnlyResourceProperties(
			Map<EIEntity, EIURI> readOnlyResourceProperties) {
		this.readOnlyResourceProperties = readOnlyResourceProperties;
	}
	
	public void setReadOnlyResourceProperty(EIEntity property, EIURI value) {
		if(readOnlyResourceProperties == null)
			readOnlyResourceProperties = new HashMap<EIEntity, EIURI>();
		readOnlyResourceProperties.put(property, value);
	}

	/**
	 * @return the readOnlyLiteralProperties
	 */
	public Map<EIEntity, String> getReadOnlyLiteralProperties() {
		if(readOnlyLiteralProperties == null)
			return Collections.emptyMap();
		else
			return readOnlyLiteralProperties;
	}

	/**
	 * @param readOnlyLiteralProperties the readOnlyLiteralProperties to set
	 */
	public void setReadOnlyLiteralProperties(
			Map<EIEntity, String> readOnlyLiteralProperties) {
		this.readOnlyLiteralProperties = readOnlyLiteralProperties;
	}
	
	public void setReadOnlyLiteralProperty(EIEntity property, String value) {
		if(readOnlyLiteralProperties == null)
			readOnlyLiteralProperties = new HashMap<EIEntity, String>();
		readOnlyLiteralProperties.put(property, value);
	}
}
