package org.eaglei.model;


import java.io.Serializable;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;


/**
 * @author Daniela Bourges-Waldegg
 * @author Ricardo De Lima
 * 
 *         April 1, 2010
 * 
 *         Center for Biomedical Informatics (CBMI)
 * @link https://cbmi.med.harvard.edu/
 * 
 * 
 * Wrapper around an eagle-i resource instance. A resource instance is identified by an instance URI
 * (contained in an EIEntity)is defined as a set of all triples
 * where subject .
 * This wrapper provides a stripped-down version of this set of triples (only the information
 * relevant to the UI)
 * Note that the instance class has to be an EIClass; most EIInstances will have only one; in some cases
 * there may be necessary to include more than one; they'll
 * be contained in the List otherEIClasses.
 * Other non-eagle-i types are kept in nonOntologyObject properties
 */

public class EIInstanceMinimal extends EIResource implements Serializable, Comparable
{

    private static final long serialVersionUID = 1L;
	
	//Use our own labels for now
	private static final EIEntity WFStateEntity = EIEntity.create(
			EIURI.create("http://eagle-i.org/ont/repo/1.0/hasWorkflowState"), 
			"Workflow state");
	private static final EIEntity WFOwnerEntity = EIEntity.create(
			EIURI.create("http://eagle-i.org/ont/repo/1.0/hasWorkflowOwner"), 
			"Workflow owner");
	private static final EIEntity CreatedEntity = EIEntity.create(
			EIURI.create("http://purl.org/dc/terms/created"), "Creation date");
	
	//quick access to the eiClass
	//Most of the time there will only be one
	private EIEntity instanceType;
	//There may be more than one ei types for the instance
    private List<EIEntity> otherEITypes;
    private EIURI WFState;
    private String created;
    private EIURI owner;
    
    private EIInstanceMinimal() 
    {
        // for GWT
    }
    
    //constructor with type
    private EIInstanceMinimal(final EIEntity instanceType, final EIEntity instanceEntity)
    {
    	super(instanceEntity);
        this.instanceType = instanceType;
        //instance type can be null
    }    

    public static EIInstanceMinimal create(final EIEntity instanceType, final EIEntity instanceEntity) {
    	return new EIInstanceMinimal(instanceType, instanceEntity);
    }
    
    

    
    public void addEIType(final EIEntity eiType) {
    	if(eiType == null) return;
    	if(otherEITypes == null) {
    		otherEITypes = new ArrayList<EIEntity>();
    	}
    	otherEITypes.add(eiType);
    }
    
    
    public void setOtherEITypes(List<EIEntity> eiTypes) {
    	this.otherEITypes = eiTypes;
    }
    

    public List<EIEntity> getOtherEITypes() {
    	if(otherEITypes == null)
    		return Collections.emptyList();
    	else
    		return otherEITypes;
    }
    
    public EIURI getInstanceURI() 
    {
        return getEntity().getURI();
    }
    
    public String getInstanceLabel() 
    {
        return getEntity().getLabel();
    }
    
    public EIEntity getInstanceType() 
    {
        return instanceType;
    }
    
    public void setInstanceType(EIEntity instanceType) {
    	this.instanceType = instanceType;
    }
    
  


    @Override
    public String toString() {
    	//Display with the first asserted class
        return instanceType.toString() + " : " + getEntity().toString();
    }

	
	public EIURI getWFState() {
		return WFState;
	}
	
	public void setWFState(EIURI state) {
		this.WFState = state;
	}
	
	public EIURI getWFOwner() {
		return owner;
	}
	
	public void setWFOwner(EIURI ownerUri) {
		this.owner = ownerUri;
	}
	
	public String getCreationDate() {
		return this.created;
	}
	
	public void setCreationDate(String date) {
		this.created = date;
	}

	/* (non-Javadoc)
	 * @see java.lang.Comparable#compareTo(java.lang.Object)
	 */
	@Override
	public int compareTo(Object o) {
		// TODO Auto-generated method stub
		if(!(o instanceof EIInstanceMinimal))
			return 0;
		return this.getEntity().getLabel().compareToIgnoreCase(((EIInstanceMinimal)o).getEntity().getLabel());
	}
}
