package org.eaglei.model.gwt.rpc;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;

import org.eaglei.model.EIClass;
import org.eaglei.model.EIEntity;
import org.eaglei.model.EIEquivalentClass;
import org.eaglei.model.EIObjectProperty;
import org.eaglei.model.EIProperty;
import org.eaglei.model.EIURI;

import com.google.gwt.core.client.GWT;
import com.google.gwt.user.client.Command;
import com.google.gwt.user.client.DeferredCommand;
import com.google.gwt.user.client.rpc.AsyncCallback;

/**
 * Maintains a client-side cache of EIClasses.  
 * Proxies all model RPC methods.
 * 
 * It is critical that all model RPC calls go through this class.
 * All methods in this class MUST call getCached(EIClass) on
 * all EIClass objects it receives from the server to ensure
 * that there is only one instance of an EIClass per
 * URI in the client.
 * 
 * @author Ted
 *
 */
public class ClientModelManager {
    
    public interface VersionCallback {
        void onSuccess(String result);
    }
    
    public interface InstitutionsCallback {
        void onSuccess(List<EIEntity> result);
    }
    
    public interface TopLevelClassesCallback {
        void onSuccess(List<EIClass> result);
    }
    
    public interface NonResourceBaseClassesCallback {
        void onSuccess(List<EIClass> result);
    }
    
    public interface ClassCallback {
        void onSuccess(EIClass result);
    }
    
    public interface SubClassCallback {
        void onSuccess(EIClass result);
    }
    
    public interface SuperClassesCallback {
        void onSuccess(EIClass result);
    }
    
    public interface PropertyCallback {
        void onSuccess(EIClass result);
    }
    
    public interface EquivalentClassCallback {
        void onSuccess(EIClass result);
    }
    
    public interface ClassDefinitionCallback {
        void onSuccess(List<EIClass> result);
    }
    
    public interface LabelsCallback {
        void onSuccess(EIURI uri, List<String> result);
    }
    
    public static final ClientModelManager INSTANCE = new ClientModelManager();
    
    private static final ModelServiceAsync modelService = GWT.create(ModelService.class);
    private List<EIEntity> listInstitutions = null;
    private List<EIClass> listTopLevelClasses = null;
    private List<EIClass> listNonResourceClasses = null;
    private final HashMap<EIURI, EIClass> mapIdToClass = new HashMap<EIURI, EIClass>();
    
    private ClientModelManager() {
    }
    
    public void getVersion(final VersionCallback callback) {
        modelService.getVersion(new AsyncCallback<String>() {

            public void onFailure(Throwable caught) {
                // TODO Auto-generated method stub
                
            }

            public void onSuccess(String result) {
                callback.onSuccess(result);
            }
            
        });
    }
    
    public void getInstitutions(final InstitutionsCallback callback) {
        if (listInstitutions != null) {
            DeferredCommand.addCommand(new Command() {

                public void execute() {
                    callback.onSuccess(listInstitutions);
                }
                
            });
        } else {
            modelService.getInstitutions(new AsyncCallback<List<EIEntity>>() {

                public void onFailure(Throwable caught) {
                    // TODO Auto-generated method stub
                    
                }

                public void onSuccess(List<EIEntity> result) {
                    listInstitutions = result;
                    callback.onSuccess(listInstitutions);
                }
                
            });
        }
    }
    
    public void getTopLevelClasses(final TopLevelClassesCallback callback) {
        if (listTopLevelClasses != null) {
            DeferredCommand.addCommand(new Command() {

                public void execute() {
                    callback.onSuccess(listTopLevelClasses);
                }
                
            });
        } else {
            modelService.getTopLevelClasses(new AsyncCallback<List<EIClass>>() {

                public void onFailure(Throwable caught) {
                    // TODO Auto-generated method stub
                    
                }

                public void onSuccess(List<EIClass> result) {
                    // Ensure that we only have a single instance of the
                    // EIClass object in the client.
                    // Avoids losing cached info.
                    List<EIClass> cacheResults = new ArrayList<EIClass>(result.size());
                    for (EIClass c : result) {
                        cacheResults.add(getCachedClass(c));
                    }
                    listTopLevelClasses = cacheResults;
                    callback.onSuccess(listTopLevelClasses);
                }
                
            });
        }
    }
    
    public void getNonResourceClasses(final NonResourceBaseClassesCallback callback) {
        if (listNonResourceClasses != null) {
            DeferredCommand.addCommand(new Command() {

                public void execute() {
                    callback.onSuccess(listNonResourceClasses);
                }
                
            });
        } else {
            modelService.getNonResourceBaseClasses(new AsyncCallback<List<EIClass>>() {

                public void onFailure(Throwable caught) {
                    // TODO Auto-generated method stub
                    
                }

                public void onSuccess(List<EIClass> result) {
                    // Ensure that we only have a single instance of the
                    // EIClass object in the client.
                    // Avoids losing cached info.
                    List<EIClass> cacheResults = new ArrayList<EIClass>(result.size());
                    for (EIClass c : result) {
                        cacheResults.add(getCachedClass(c));
                    }
                    listNonResourceClasses = cacheResults;
                    callback.onSuccess(listNonResourceClasses);
                }
                
            });
        }
    }
    
    public void getClass(final EIURI id, final ClassCallback callback) {
        if (mapIdToClass.containsKey(id)) {
            DeferredCommand.addCommand(new Command() {

                public void execute() {
                    callback.onSuccess(mapIdToClass.get(id));
                }
                
            });
        } else {
            modelService.getClass(id, new AsyncCallback<EIClass>() {

                public void onFailure(Throwable caught) {
                    // TODO Auto-generated method stub
                    
                }

                public void onSuccess(EIClass result) {
                    callback.onSuccess(getCachedClass(result));
                }
                
            });
        }
    }
    
    /**
     * Call this to ensure that the subclass list of a given
     * EIClass has been populated.
     * 
     * Note that if EIClass.hasSubClass() is false,
     * the value of EIClass.getSubClasses() is undefined
     * (probably null).
     * 
     * @param resource
     * @param callback
     */
    public void getSubClasses(final EIClass resource, final SubClassCallback callback) {
        if (! resource.hasSubClass() || resource.getSubClasses() != null) {
            DeferredCommand.addCommand(new Command() {

                public void execute() {
                    callback.onSuccess(resource);
                }
                
            });
        } else {
            modelService.getSubClasses(resource.getEntity().getURI(), new AsyncCallback<List<EIClass>>() {

                public void onFailure(Throwable caught) {
                    // TODO Auto-generated method stub
                    
                }

                public void onSuccess(List<EIClass> result) {
                    // Ensure that we only have a single instance of the
                    // EIClass object in the client.
                    // Avoids losing cached info.
                    List<EIClass> cacheResults = new ArrayList<EIClass>(result.size());
                    for (EIClass c : result) {
                        cacheResults.add(getCachedClass(c));
                    }
                    resource.setSubClasses(cacheResults);
                    callback.onSuccess(resource);
                }
                
            });
        }
    }
    
    /**
     * Ensures that all superclasses of a given
     * EIURI are available.  In the callback, returns
     * an EIClass for the uri in which getSuperClass()
     * will be non-null if hasSuperClass() is true.
     * Likewise, all ancestor EIClasses will have non-null
     * superclasses if they have a superclass.
     * 
     * @param resourceURI
     * @param callback
     */
    public void getSuperClasses(final EIURI resourceURI, final SuperClassesCallback callback) {
        EIClass resourceClass = mapIdToClass.get(resourceURI);
        if (resourceClass != null) {
            getSuperClasses(resourceClass, callback);
        } else {
            // If the resource's class is not in cache, first
            // get it into cache and then get the supers
            getClass(resourceURI, new ClassCallback() {
                
                @Override
                public void onSuccess(EIClass result) {
                    getSuperClasses(getCachedClass(result), callback);
                }
            });
        }
    }
    
    /**
     * Ensures that all superclasses of a given
     * EIClass are available.
     * 
     * In the callback, getSuperClass()
     * will be non-null if hasSuperClass() is true.
     * Likewise, all ancestor EIClasses will have non-null
     * superclasses if they have a superclass.
     * 
     * Note that if EIClass.hasSuperClass() is false,
     * the value of EIClass.getSuperClass() is undefined
     * (probably null).
     * 
     * @param resource
     * @param callback
     */
    public void getSuperClasses(final EIClass resource, final SuperClassesCallback callback) {
        // Determine what, if any class in the ancestor heirarchy, we need
        // to get from the server.
        EIClass needsSuperClass = resource;
        while (needsSuperClass != null) {
            if (! needsSuperClass.hasSuperClass()) {
                // we've hit a top-level class, no supers were missing
                needsSuperClass = null;
            } else if (needsSuperClass.getSuperClass() != null) {
                // we're on a class that has a super, but it's not missing
                needsSuperClass = needsSuperClass.getSuperClass();
            } else {
                // we're on a class that has a super, but it's missing
                break;
            }
        }
        if (needsSuperClass == null) {
            DeferredCommand.addCommand(new Command() {

                public void execute() {
                    callback.onSuccess(resource);
                }
                
            });
        } else {
            final EIClass targetClass = needsSuperClass;
            modelService.getSuperClasses(targetClass.getEntity().getURI(), new AsyncCallback<List<EIClass>>() {

                public void onFailure(Throwable caught) {
                    // TODO Auto-generated method stub
                    
                }

                public void onSuccess(List<EIClass> result) {
                    EIClass cacheChild = targetClass;
                    for (EIClass parent : result) {
                        EIClass cacheParent = getCachedClass(parent);
                        cacheChild.setSuperClass(cacheParent);
                        cacheChild = cacheParent;
                    }
                    callback.onSuccess(resource);
                }
                
            });
        }
    }

    /**
     * Call this to ensure that the properties list of a given
     * EIClass has been populated.
     * 
     * Note that if EIClass.hasProperty() is false,
     * the value of EIClass.getProperties() is undefined
     * (probably null).
     * 
     * @param resource
     * @param callback
     */
    public void getProperties(final EIClass resource, final PropertyCallback callback) {
        if (! resource.hasProperty() || resource.getProperties() != null) {
            DeferredCommand.addCommand(new Command() {

                public void execute() {
                    callback.onSuccess(resource);
                }
                
            });
        } else {
            modelService.getProperties(resource.getEntity().getURI(), new AsyncCallback<List<EIProperty>>() {

                public void onFailure(Throwable caught) {
                    // TODO Auto-generated method stub
                    
                }

                public void onSuccess(List<EIProperty> result) {
                    // Ensure that we only have a single instance of the
                    // EIClass object in the client.
                    // Avoids losing cached info.
                    resource.setProperties(getCachedProperties(result));
                    callback.onSuccess(resource);
                }
                
            });
        }
    }
    
    /**
     * Call this to ensure that the equivalent class list of a given
     * EIClass has been populated.
     * 
     * Note that if EIClass.hasEquivalentClass() is false,
     * the value of EIClass.getEquivalentClasses() is undefined
     * (probably null).
     * 
     * @param resource
     * @param callback
     */
    public void getEquivalentClasses(final EIClass resource, final EquivalentClassCallback callback) {
        if (! resource.hasEquivalentClass() || resource.getEquivalentClasses() != null) {
            DeferredCommand.addCommand(new Command() {

                public void execute() {
                    callback.onSuccess(resource);
                }
                
            });
        } else {
            modelService.getEquivalentClasses(resource.getEntity().getURI(), new AsyncCallback<List<EIEquivalentClass>>() {

                public void onFailure(Throwable caught) {
                    // TODO Auto-generated method stub
                    
                }

                public void onSuccess(List<EIEquivalentClass> result) {
                    // Ensure that we only have a single instance of the
                    // EIClass object in the client.
                    // Avoids losing cached info.
                    resource.setEquivalentClasses(getCachedEquivalents(result));
                    callback.onSuccess(resource);
                }
                
            });
        }
    }
    
    /**
     * Ensures that class definitions are set for all the classes
     * in the given list.
     * 
     * @param classList
     */
    public void getClassDefinitions(final List<EIClass> classList, final ClassDefinitionCallback callback) {
        // Assert that theses are all in cache?
        ArrayList<EIURI> classURIList = new ArrayList<EIURI>(classList.size());
        for (EIClass clazz : classList) {
            classURIList.add(clazz.getEntity().getURI());
        }
        modelService.getClassDefinitions(classURIList, new AsyncCallback<List<String>>() {

            public void onFailure(Throwable caught) {
                // TODO Auto-generated method stub
                
            }

            public void onSuccess(List<String> result) {
                int index = 0;
                for (EIClass clazz : classList) {
                    clazz.setDefinition(result.get(index++));
                }
                callback.onSuccess(classList);
            }
            
        });
    }
    
    public void getLabels(final EIURI uri, final LabelsCallback callback) {
        modelService.getLabels(uri, new AsyncCallback<List<String>>() {

            public void onFailure(Throwable caught) {
                // TODO Auto-generated method stub
                
            }

            public void onSuccess(List<String> result) {
                callback.onSuccess(uri, result);
            }
            
        });
    }

    private List<EIProperty> getCachedProperties(List<EIProperty> propList) {
        for (EIProperty p : propList) {
            if (p instanceof EIObjectProperty) {
                EIObjectProperty op = (EIObjectProperty) p;
                List<EIClass> cList = op.getRangeList();
                for (int i=0; i<cList.size(); i++) {
                    // Replace each range class instance with cached one,
                    // if one is there.
                    cList.set(i, getCachedClass(cList.get(i)));
                }
            }
        }
        return propList;
    }
    
    private List<EIEquivalentClass> getCachedEquivalents(List<EIEquivalentClass> equivList) {
        for (EIEquivalentClass e : equivList) {
            List<EIClass> cList = e.getEquivalentTo();
            for (int i=0; i<cList.size(); i++) {
                // Replace each equiv to class instance with cached one,
                // if one is there.
                cList.set(i, getCachedClass(cList.get(i)));
            }
        }
        return equivList;
    }
    
    private EIClass getCachedClass(EIClass c) {
        if (mapIdToClass.containsKey(c.getEntity().getURI())) {
            return mapIdToClass.get(c.getEntity().getURI());
        } else {
            mapIdToClass.put(c.getEntity().getURI(), c);
            return c;
        }
    }
}
