package org.eaglei.model.jena;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.util.ArrayDeque;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Deque;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TreeMap;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eaglei.model.EIClass;
import org.eaglei.model.EIDatatypeProperty;
import org.eaglei.model.EIEntity;
import org.eaglei.model.EIEquivalentClass;
import org.eaglei.model.EIObjectProperty;
import org.eaglei.model.EIOntModel;
import org.eaglei.model.EIOntResource;
import org.eaglei.model.EIProperty;
import org.eaglei.model.EIURI;
import org.eaglei.model.EagleIOntConstants;
import org.eaglei.services.InstitutionRegistry;

import com.hp.hpl.jena.ontology.AnnotationProperty;
import com.hp.hpl.jena.ontology.ConversionException;
import com.hp.hpl.jena.ontology.IntersectionClass;
import com.hp.hpl.jena.ontology.OntClass;
import com.hp.hpl.jena.ontology.OntModel;
import com.hp.hpl.jena.ontology.OntProperty;
import com.hp.hpl.jena.ontology.OntResource;
import com.hp.hpl.jena.ontology.UnionClass;
import com.hp.hpl.jena.rdf.model.Literal;
import com.hp.hpl.jena.rdf.model.Property;
import com.hp.hpl.jena.rdf.model.RDFList;
import com.hp.hpl.jena.rdf.model.RDFNode;
import com.hp.hpl.jena.rdf.model.Resource;
import com.hp.hpl.jena.rdf.model.Statement;
import com.hp.hpl.jena.rdf.model.StmtIterator;
import com.hp.hpl.jena.util.iterator.ExtendedIterator;
import com.hp.hpl.jena.vocabulary.RDFS;
import com.hp.hpl.jena.vocabulary.OWL;

/**
 * Exposes the eagle-i ontology using GWT friendly model classes.
 */
public class JenaEIOntModel implements EIOntModel {

    protected static final Log logger = LogFactory.getLog(JenaEIOntModel.class);
    
    // Jena model of eagle-i ontology
    private final OntModel jenaOntModel;
    private final String version;
    private final AnnotationProperty inPropertyGroup;
    private final Resource dataModelExclude;
    private final Map<String, List<String>> mapPropURIToDomainConstraints;
    private final Map<String, List<String>> mapPropURIToRangeConstraints;
    
    // To be removed
    private InstitutionRegistry institutionRegistry;

    // Priority ordered list of properties that are used to compute preferred label
    private final List<Property> prefLabelProperties = new ArrayList<Property>();
    // Property used to specify synonyms 
    private final Property iaoAltTerm;
    // List of subtype labels of given type
    private final HashMap<EIURI, List<EIEntity>> mapClassIdToTypeLabels = new HashMap<EIURI, List<EIEntity>>();
    // List of top-level base classes
    private final List<OntClass> listTopLevelOntClasses;
    // Set of resource base class URI strings
    private final Set<String> setTopLevelURIs;
    // List of top-level base classes
    private final List<EIClass> listTopLevelEIClasses;
    // Set of non-resource base class URI strings
    private final Set<String> setNonResourceBaseURIs;
    // List of non-resource base classes
    private final List<EIClass> listNonResourceBaseEIClasses;
    
    // Cache of EIClasses
    // TODO Put non-base EIClasses in a weak reference cache.
    // It's kind of tricky to do so, because the base classes have hard
    // references to them, and they have hard refs to their subclass lists.
    // So nothing in your not-so-weak cache will actually ever be GC'd.
    private final HashMap<String, EIClass> mapURIStrToEIClass = new HashMap<String, EIClass>();
    
    /*
     * Should generally be using Spring config to construct
     */
    protected JenaEIOntModel(OntModel jenaOntModel) {
        this.jenaOntModel = jenaOntModel;
        
        // read the version number from a the root ontology file
        version = jenaOntModel.getOntology(JenaModelConfig.EAGLE_I_APP_EXT_URI).getVersionInfo();
        logger.info("eagle-i ontology version: " + version);
                
        inPropertyGroup = jenaOntModel.getAnnotationProperty(EagleIOntConstants.IN_PROPERTY_GROUP);
        dataModelExclude = jenaOntModel.getResource(EagleIOntConstants.DATA_MODEL_EXCLUDE);
        
        // Create a map of properties which have eagle-i specialized
        // domain constraints
        mapPropURIToDomainConstraints = new HashMap<String, List<String>>();
        Property domainConstraint = jenaOntModel.getProperty(EagleIOntConstants.DOMAIN_CONSTRAINT);
        List<Statement> domainConstrainedStatements = jenaOntModel.listStatements((Resource) null, domainConstraint, (RDFNode) null).toList();
        for (Statement stmt : domainConstrainedStatements) {
            Resource constraintSubject = stmt.getSubject();
            String propURI = constraintSubject.getURI();
            List<String> constraintList = mapPropURIToDomainConstraints.get(propURI);
            if (constraintList == null) {
                constraintList = new ArrayList<String>();
                mapPropURIToDomainConstraints.put(propURI, constraintList);
            }
            String domainURI = stmt.getString();
            constraintList.add(domainURI);
        }

        // Create a map of properties which have eagle-i specialized
        // range constraints
        mapPropURIToRangeConstraints = new HashMap<String, List<String>>();
        Property rangeConstraint = jenaOntModel.getProperty(EagleIOntConstants.RANGE_CONSTRAINT);
        List<Statement> rangeConstrainedStatements = jenaOntModel.listStatements((Resource) null, rangeConstraint, (RDFNode) null).toList();
        for (Statement stmt : rangeConstrainedStatements) {
            Resource constraintSubject = stmt.getSubject();
            String propURI = constraintSubject.getURI();
            List<String> constraintList = mapPropURIToRangeConstraints.get(propURI);
            if (constraintList == null) {
                constraintList = new ArrayList<String>();
                mapPropURIToRangeConstraints.put(propURI, constraintList);
            }
            String rangeURI = stmt.getString();
            constraintList.add(rangeURI);
        }

        // define the ordered lists of label properties
        Property eiPrefLabel = jenaOntModel.getProperty(EagleIOntConstants.EI_PREFERRED_LABEL);
        Property iaoPrefTerm = jenaOntModel.getProperty(EagleIOntConstants.IAO_PREFERRED_TERM_URI);

        prefLabelProperties.add(eiPrefLabel);
        prefLabelProperties.add(iaoPrefTerm);
        prefLabelProperties.add(RDFS.label);
        
        iaoAltTerm = jenaOntModel.getProperty(EagleIOntConstants.IAO_ALTERNATE_TERM_URI);        

        logger.info("initializing resource class lists...");
        long start = System.currentTimeMillis();
        
        List<List<OntClass>> listClassesInGroup = getResourceLists();
        
        // Resource base classes
        this.listTopLevelOntClasses = listClassesInGroup.get(0);
        this.setTopLevelURIs = new HashSet<String>(listTopLevelOntClasses.size());
        // Pre-compute an alpha sorted list of top-level EIClasses and cache them.
        TreeMap<String, EIClass> tmClasses = new TreeMap<String, EIClass>();
        for (OntClass ontClass: listTopLevelOntClasses) {
            // Note must put uri in set before calling createClass()
            // in order for hasSuperClass() to be set properly
            setTopLevelURIs.add(ontClass.getURI());
            final EIClass eiClass = createClass(ontClass); 
            tmClasses.put(eiClass.getEntity().getLabel().toLowerCase(), eiClass);
        }
        this.listTopLevelEIClasses = new ArrayList<EIClass>(tmClasses.values());
        
        // Non-resource base classes
        List<OntClass> listNonResourceOntClasses = listClassesInGroup.get(1);
        listNonResourceBaseEIClasses = new ArrayList<EIClass>(listNonResourceOntClasses.size());
        setNonResourceBaseURIs = new HashSet<String>(listNonResourceOntClasses.size());
        for (OntClass ontClass : listNonResourceOntClasses) {
            // Note must put uri in set before calling createClass()
            // in order for hasSuperClass() to be set properly
            setNonResourceBaseURIs.add(ontClass.getURI());
            final EIClass eiClass = createClass(ontClass); 
            listNonResourceBaseEIClasses.add(eiClass);
        }
        
        logger.info("resource class lists initialized: " + (System.currentTimeMillis()-start));
    }
    
    @Override
    public String getVersion() {
        return this.version;
    }
    
    /**
     * Retrieves the underlying Jena OntModel.
     * @return The underlying eagle-i OntModel.
     */
    public OntModel getOntModel() {
    	return this.jenaOntModel;
    }
    
    private List<List<OntClass>> getResourceLists() {
        List<List<OntClass>> result = new ArrayList<List<OntClass>>(2);
        
        List<OntClass> resourceList = EagleIOntUtils.getClassesInGroup(jenaOntModel, EagleIOntConstants.TOP_LEVEL_GROUP);
        if (resourceList != null) {
            result.add(resourceList);
        } else {
            assert(false) : "TODO";
        }
        
        resourceList = getResourceListFromCache("/rootNonResourceList.txt");
        if (resourceList != null) {
            result.add(resourceList);
        } else {
            //assert(false) : "TODO";
            result.add(new ArrayList<OntClass>());
        }
        
        return result;
    }
    
    private List<OntClass> getResourceListFromCache(String path) {
        //final URL cacheFileURL = this.getClass().getClassLoader().getResource(path);
        final InputStream is = this.getClass().getResourceAsStream(path);
        if (is == null) {
            return null;
        }
        BufferedReader br = null;
        try {
            br = new BufferedReader(new InputStreamReader(is));
            String list = br.readLine();
            if (list == null) {
                return null;
            }
            String[] uriStrings = list.split(",");
            List<OntClass> listOntClass = new ArrayList<OntClass>(uriStrings.length);
            for (String uri : uriStrings) {
                listOntClass.add(getOntologyClass(uri));
            }
            return listOntClass;
        } catch (Exception e) {
            logger.error(e);
            return null;
        } finally {
            try {
                if (br != null) br.close();
                if (is != null) is.close();
            } catch (IOException e) {
            }
        }
    }

    public List<EIEntity> getTypeEntities(EIURI classId) {
        assert classId != null;
        List<EIEntity> results = mapClassIdToTypeLabels.get(classId);
        if (results != null) {
            return results;
        }
        results = new ArrayList<EIEntity>();
        EIClass root = getClass(classId);
        getTypeLabelsInternal(root, results);
        mapClassIdToTypeLabels.put(classId, results);
        return results;
    }

    private void getTypeLabelsInternal(EIClass c, List<EIEntity> results) {
        results.add(c.getEntity());
        if (c.hasSubClass()) {
            List<EIClass> subclasses = getSubClasses(c.getEntity().getURI());
            for (EIClass subclass : subclasses) {
                getTypeLabelsInternal(subclass, results);
            }
        }
    }
    

    @Override
    public boolean isModelClassURI(String uri) {
        OntClass c = getOntologyClass(uri);
        if (c == null) {
            return false;
        }
        if (isResource(c)) {
            return true;
        }
        return isNonResource(c);
    }
    
    @Override
    public List<EIClass> getTopLevelClasses() {
        return listTopLevelEIClasses;
    }

    @Override
    public List<EIClass> getNonResourceBaseClasses() {
        return listNonResourceBaseEIClasses;
    }
    
    /**
     * Retrieves the classes in the specified group. See EagleIOntUtils.getClassesInGroup().
     * 
     * @param groupURI URI of the group.
     * 
     * @return Alpha sorted (by label) list of classes in the group.
     */
    /*
    private List<EIClass> getClassesInGroup(final String groupURI) {
        // TreeMap for alpha sort
        TreeMap<String, EIClass> tmClasses = new TreeMap<String, EIClass>();
        List<EIClass> eiClasses = new ArrayList<EIClass>();
        for (OntClass ontClass: EagleIOntUtils.getClassesInGroup(groupURI)) {
            final EIClass eiClass = createClass(ontClass); 
            tmClasses.put(eiClass.getEntity().getLabel(), eiClass);
        }
        return new ArrayList<EIClass>(tmClasses.values());
    }
    */
    
    // To be removed.
    public void setInstitutionRegistry(InstitutionRegistry institutionRegistry) {
        this.institutionRegistry = institutionRegistry;
    }

    @Override
    // To be removed.
    public List<EIEntity> getInstitutions() {
        return institutionRegistry.getInstitutions();
    }

    public EIClass getClass(EIURI uri) {
        assert (uri != null);
        return getClass(uri.toString());
    }
    
    protected EIClass getClass(String uri) {
        OntClass ontClass = getOntologyClass(uri);
        if (ontClass == null) {
            // could change this to an assert.
            return null;
        }
        return createClass(ontClass);
    }

    /*
     * Only call with a non-null ont class
     * that is known to be a data model resource or non-resource class.
     */
    private EIClass createClass(OntClass ontClass) {
        assert (ontClass != null);
        EIClass result = mapURIStrToEIClass.get(ontClass.getURI());
        if (result != null) {
            return result;
        }
        final EIURI uri = EIURI.create(ontClass.getURI());
        final String label = getPreferredLabel(uri);
        // All resource base classes are flagged as having no super
        boolean hasSuperClass = hasSuperClass(ontClass);
        boolean hasSubClass = hasSubClass(ontClass);
        /* TODO should figure out how to get this into sync
         *      with the getProperties implementation.
         *      Note that simply calling getProperties as it is
         *      currently implemented is bad because it currently
         *      creates an EIClass, so circular instantiation in the
         *      case where this method is called from createClass.
         */
        boolean hasProperties = true; //hasProperty(ontClass);
        boolean isEagleIResource = isResource(ontClass);
        List<EIEquivalentClass> equivalentClasses = getEquivalentClasses(ontClass);
        boolean hasEquivalentClass = equivalentClasses != null ? equivalentClasses.size() > 0 : false;
        if (!isEagleIResource) {
            // Check that this uri is a valid member of the
            // data model.
            // Can uncomment when http://jira.eagle-i.net:8080/browse/DTOOLS-215 is resolved.
            //assert(isNonResource(ontClass));
        }
        EIEntity entity = EIEntity.create(uri, label);
        result = new EIClass(entity, hasProperties, hasSubClass, hasSuperClass, hasEquivalentClass, isEagleIResource);
        result.setEquivalentClasses(equivalentClasses);
        mapURIStrToEIClass.put(ontClass.getURI(), result);
        return result;
    }
    
    private List<EIEquivalentClass> getEquivalentClasses(OntClass ontClass) {
        List<OntClass> listEquivOntClasses = ontClass.listEquivalentClasses().toList();
        // Working off the assumption that every class is equivalent 
        // to itself, so a size greater than 1 indicates the presence
        // of other classes.
        if (listEquivOntClasses.size() == 1) {
            return null;
        }
        // This is definitely not general purpose handling yet!
        // This is custom quick hack code to handle our one current
        // equivalent class case.
        // There are a bunch of asserts that will try to detect
        // the addition to the ontology of any new equivalentClass 
        // constructs that this code doesn't handle.
        if (listEquivOntClasses.size() != 2) {
            logger.warn("Unexpected equivalent class list in: " + ontClass.getURI() + ", listEquivOntClasses.size() == " + listEquivOntClasses.size());
            return null;
        }
        String uri = null;
        List<EIEquivalentClass> listEquivalentClasses = new ArrayList<EIEquivalentClass>(listEquivOntClasses.size()-1);
        for (OntClass equivOntClass : listEquivOntClasses) {
            if (ontClass.getURI().equals(equivOntClass.getURI())) {
                // the class itself will always appear in this list
                continue;
            } else {
                if (! equivOntClass.isIntersectionClass()) {
                    logger.warn("Unexpected equivalent class construct in: " + ontClass.getURI() + ", is not an intersection class");
                    return null;
                }
                uri = equivOntClass.getURI(); // Needed?
                List<EIClass> listEquivalentToClasses = new ArrayList<EIClass>();
                IntersectionClass intersectionClass = equivOntClass.asIntersectionClass();
                for (OntClass intersectionOperand : intersectionClass.listOperands().toList()) {
                    if (intersectionOperand.isUnionClass()) {
                        UnionClass uc = intersectionOperand.asUnionClass();
                        RDFList members = uc.getOperands();
                        for (int i=0; i<members.size(); i++) {
                            RDFNode n = members.get(i);
                            assert(n instanceof Resource);
                            String eiClassURI = ((Resource) n).getURI();
                            assert(!eiClassURI.equals(OWL.Thing.getURI()));
                            EIClass ec = getClass(eiClassURI);
                            listEquivalentToClasses.add(ec);
                        }
                    } else {
                        // TODO do something with Restriction operands
                    }
                }
                // Check that we came out of this with at least one equivalent to class.
                if(listEquivalentToClasses.size() == 0) {
                    logger.warn("Class has an equivalentClass that we aren't handling: " + ontClass.getURI());
                    continue;
                } else {
                    listEquivalentClasses.add(new EIEquivalentClass(uri, listEquivalentToClasses));
                }
            }
        }
        // Check that we came out of this with at least one equivalent to class.
        if(listEquivalentClasses.size() == 0) {
            logger.warn("Class has an equivalentClass that we aren't handling: " + ontClass.getURI());
            return null;
        } else {
            return listEquivalentClasses;
        }
    }

    private EIProperty createProperty(final OntProperty p) {
        EIProperty result = null;
        final EIURI uri = EIURI.create(p.getURI());
        final String label = getPreferredLabel(uri);
        final EIEntity entity = EIEntity.create(uri, label);
        if (p.isObjectProperty()) {
        	//logger.info("Obj : " + p.getURI() + " : " + p.getLabel(null));
            List<EIClass> listRangeClasses = new ArrayList<EIClass>();
            List<String> rangeConstraints = mapPropURIToRangeConstraints.get(p.getURI());
            if (rangeConstraints != null) {
                // This property has a fixed range list.
                for (String rangeURI : rangeConstraints) {
                    EIClass rangeClass = getClass(rangeURI); 
                    addObjectPropertyRangeClass(listRangeClasses, rangeClass);
                }
            } else {
                List<OntResource> listRange = (List<OntResource>) p.listRange().toList();
                for (OntResource rangeOntResource : listRange) {
                    OntClass rangeOntClass = (OntClass) rangeOntResource;
                    if (rangeOntClass.isUnionClass()) {
                        UnionClass uc = rangeOntClass.asUnionClass();
                        RDFList members = uc.getOperands();
                        for (int i=0; i<members.size(); i++) {
                            RDFNode rangeNode = members.get(i);
                            if (rangeNode instanceof Resource) {
                                String rangeURI = ((Resource) rangeNode).getURI();
                                if (!rangeURI.equals(OWL.Thing.getURI())) {
                                    EIClass rangeClass = getClass(rangeURI); 
                                    addObjectPropertyRangeClass(listRangeClasses, rangeClass);
                                }
                            }
                        }
                    } else if (!rangeOntClass.getURI().equals(OWL.Thing.getURI())) {
                        EIClass rangeClass = createClass(rangeOntClass); 
                        addObjectPropertyRangeClass(listRangeClasses, rangeClass);
                    }
                }
            }
            if (listRangeClasses.size() > 0) {
                result = new EIObjectProperty(entity, listRangeClasses);                                    
            } else {
                assert(false) : "Unable to determine range for object property " + entity.toString();
                result = new EIDatatypeProperty(entity, null);                    
            }
        } else if (p.isDatatypeProperty()) {
        	//logger.info("Data : " + p.getURI() + " : " + p.getLabel(null));
        	String typeLabel = null;
            OntResource rangeResource = p.getRange();
            if (rangeResource != null) {
                typeLabel = rangeResource.getLocalName();
            } else {
                //logger.warn("Datatype property has unknown range type: " + p.getURI());   
            }
            result = new EIDatatypeProperty(entity, typeLabel);
        } else {
            //logger.warn("createProperty called on property that is neither an object nor datatype property: " + p.getURI());   
        }
        return result;
    }
    
    private void addObjectPropertyRangeClass(List<EIClass> listRangeClasses, EIClass rangeClassToAdd) {
        if (rangeClassToAdd.hasEquivalentClass()) {
            // This is transitional code that checks if a range class
            // has an equivalent class, and if it does, puts the equivalent to
            // list into the range list rather than the original range class.
            // When all clients are ready to handle occurrences of 
            // EIEquivalentClass in the range list, then this code can be removed.
            for (EIEquivalentClass equivalentClass : rangeClassToAdd.getEquivalentClasses()) {
                for (EIClass equivalentToClass : equivalentClass.getEquivalentTo()) {
                    listRangeClasses.add(equivalentToClass); 
                }
            }
        } else {
            listRangeClasses.add(rangeClassToAdd); 
        }
    }
    
    private boolean hasSuperClass(OntClass ontClass) {
        boolean hasSuperClass =
            !isResourceBaseClass(ontClass) && !isNonResourceBaseClass(ontClass);
        return hasSuperClass;
    }

    @Override
    public List<EIClass> getSuperClasses(EIURI classId) {
        List<EIClass> result;
        
        EIClass eiClass = getClass(classId);
        if (eiClass == null) {
            return Collections.emptyList();            
        }
        if (!eiClass.hasSuperClass()) {
            return Collections.emptyList();            
        }
        
        result = eiClass.getSuperClasses();
        if (result != null) {
            return result;
        }
        
        result = new ArrayList<EIClass>();
        EIClass childClass = getClass(classId);
        while (childClass.hasSuperClass()) {
            EIClass parentClass = getSuperClass(childClass);
            assert(parentClass != null) : "getSuperClasses: " + classId + "   childClass has null parentClass: " + childClass.getEntity();
            result.add(parentClass);
            childClass = parentClass;
        }
        
        eiClass.setSuperClasses(result);
        return result;
    }
    
    @Override
    public EIClass getSuperClass(EIClass eiClass) {
        assert(eiClass != null);
        if (!eiClass.hasSuperClass()) {
            logger.debug("getSuperClass() called on class for which hasSuperClass() is false: "+eiClass.getEntity().toString());
            return null;
        }
        EIClass parentEIClass = eiClass.getSuperClass();
        if (parentEIClass != null) {
            return parentEIClass;
        }

        String uri = eiClass.getEntity().getURI().toString();
        if (isResourceBaseClass(uri)) {
            // Top-level classes have no super class
            return null;
        }
        if (isNonResourceBaseClass(uri)) {
            // Top-level classes have no super class
            return null;
        }
        
        OntClass childClass = getOntologyClass(uri);
        // Get from mapped class, if there is a mapping\
        /* ????
        if (resolveReference) {
            Resource resDefinedBy = parentClass.getIsDefinedBy();
            if (resDefinedBy != null) {
                String definedByURI = resDefinedBy.getURI();
                parentClass = getOntologyClass(definedByURI);
            }
        }
        */
        ExtendedIterator itrSuperClasses = childClass.listSuperClasses(true);
        while (itrSuperClasses.hasNext()) {
            OntClass parentClass = (OntClass) itrSuperClasses.next();
            String parentURI = parentClass.getURI();
            if (parentURI == null) {
                continue;
            }
            if (OWL.Thing.getURI().equals(parentURI)) {
                continue;
            }
            // Treat first non-null, non-THING parent class as the super
            parentEIClass = getClass(parentClass.getURI());   
            eiClass.setSuperClass(parentEIClass);
            return parentEIClass;
        }
        return null;
    }
    
    /*
     * Is the given ont class a resource or resource subtype.
     */
    private boolean isResource(OntClass ontClass) {
        if (isResourceBaseClass(ontClass)) {
            return true;
        }
        for (OntClass c : listTopLevelOntClasses) {
            if (c.hasSubClass(ontClass)) {
                return true;
            }
        }
        return false;
    }
    
    /*
     * Is the given ont class a non-resource or non-resource subtype.
     */
    private boolean isNonResource(OntClass ontClass) {
        if (isNonResourceBaseClass(ontClass)) {
            return true;
        }
        for (String strURI : setNonResourceBaseURIs) {
            OntClass c = getOntologyClass(strURI);
            if (c.hasSubClass(ontClass)) {
                return true;
            }
        }
        return false;
    }
    
    private boolean isResourceBaseClass(OntClass ontClass) {
    	return isResourceBaseClass(ontClass.getURI());
    }

    private boolean isResourceBaseClass(String uri) {
    	return setTopLevelURIs.contains(uri);
    }

    private boolean isNonResourceBaseClass(OntClass ontClass) {
        return isNonResourceBaseClass(ontClass.getURI());
    }

    private boolean isNonResourceBaseClass(String uri) {
        return setNonResourceBaseURIs.contains(uri);
    }

    private boolean hasSubClass(OntClass ontClass) {
        // Note: need to use listSubClasses() rather than hasSubClasses()
        // because that always returns
        // true when using a reasoner (since all OWL classes have owl:Nothing as
        // a subclass)
        ExtendedIterator<OntClass> itrSubClasses = ontClass.listSubClasses(true);
        while (itrSubClasses.hasNext()) {
            OntClass childClass = (OntClass) itrSubClasses.next();
            if (childClass == null || childClass.getURI() == null) {
                continue;
            }
            if (childClass.getURI().equals(OWL.Nothing.getURI())) {
                return false;
            } else {
                return true;
            }
        }
        return false;
    }
    
    public boolean isSubClass(EIURI ancestorURI, EIURI descendentURI) {
        OntClass ancestorOntClass = getOntologyClass(ancestorURI.toString());
        OntClass descendentOntClass = getOntologyClass(descendentURI.toString());
        return ancestorOntClass.hasSubClass(descendentOntClass);
    }

    @Override
    public List<EIClass> getSubClasses(EIURI classId) {
        List<EIClass> result;
        
        EIClass eiClass = getClass(classId);
        if (eiClass == null) {
            return Collections.emptyList();            
        }
        if (!eiClass.hasSubClass()) {
            logger.debug("getSubClasses() called on class for which hasSubClass() is false: "+eiClass.getEntity().toString());
            return new ArrayList<EIClass>();
        }
        
        result = eiClass.getSubClasses();
        if (result != null) {
            return result;
        }

        OntClass parentClass = getOntologyClass(classId.toString());

        result = getSubClasses(parentClass);
        eiClass.setSubClasses(result);
        return result;
    }

    OntClass getOntologyClass(String strURI) {
        return jenaOntModel.getOntClass(strURI);
    }

    private List<EIClass> getSubClasses(OntClass ontClass) {
        // TreeMap for alpha sort
        TreeMap<String, EIClass> tmSubClasses = new TreeMap<String, EIClass>();
        ExtendedIterator itrSubClasses = ontClass.listSubClasses(true);
        while (itrSubClasses.hasNext()) {
            OntClass childClass = (OntClass) itrSubClasses.next();
            if (childClass == null || childClass.getURI() == null) {
                continue;
            }
            if (childClass.getURI().equals(OWL.Nothing.getURI())) {
                continue;
            }
            EIClass eiChildClass = createClass(childClass);
            tmSubClasses.put(eiChildClass.getEntity().getLabel().toLowerCase(), eiChildClass);
        }
        List<EIClass> listSubClasses = new ArrayList<EIClass>(tmSubClasses.values());
        return listSubClasses;
    }

    /* TODO should figure out how to get this into sync
     *      with the getProperties implementation.
     *      Note that simply calling getProperties as it is
     *      currently implemented is bad because it currently
     *      creates an EIClass, so circular instantiation in the
     *      case where this method is called from createClass.
    private boolean hasProperty(OntClass ontClass) {
        ExtendedIterator itrProperties = ontClass.listDeclaredProperties();
        while (itrProperties.hasNext()) {
            try {
                Object o = itrProperties.next();
                if (o instanceof OntProperty) {
                    if (((OntProperty) o).isObjectProperty()) {
                        return true;
                    }
                }
            } catch (ConversionException ex) {
                //logger.warn(ex.getMessage());
                continue;
            }
        }
        return false;
    }
    */

    public List<EIEquivalentClass> getEquivalentClasses(EIURI classId) {
        EIClass eiClass = getClass(classId);
        assert(eiClass != null);
        // Should already be set.
        return eiClass.getEquivalentClasses();
    }
    
    public List<EIProperty> getProperties(EIURI classId) {
        return  getProperties(classId, null);
    }
    
    /**
     * Retrieves the properties with the inPropertyGroup annotation set to 
     * the given value.  Or all non-annotation properties if groupURI is null. 
     * @param classId ID of class.
     * @param groupURI URI of property group value to filter on.  May be null.
     * @return List of EIProperties that represent the properties.
     */
    public List<EIProperty> getProperties(final EIURI classId, final String groupURI) {
        List<EIProperty> result;
        EIClass eiClass = getClass(classId);
        assert(eiClass != null);
        if (!eiClass.hasProperty()) {
            logger.debug("getProperties() called on class for which hasProperty() is false: "+eiClass.getEntity().toString());
            return new ArrayList<EIProperty>();
        }
        
        result = eiClass.getProperties();
        if (result != null) {
            return result;
        }

        OntClass parentClass = getOntologyClass(classId.toString());

        result = getProperties(parentClass, groupURI);
        eiClass.setProperties(result);
        return result;
    }

    private List<EIProperty> getProperties(final OntClass ontClass, final String groupURI) {
        // TreeMap for alpha sort
        final TreeMap<String, EIProperty> tmProperties = new TreeMap<String, EIProperty>();
        final ExtendedIterator itrProperties = ontClass.listDeclaredProperties();
        while (itrProperties.hasNext()) {
            try {
                final Object o = itrProperties.next();
                if (o instanceof OntProperty) {
                    final OntProperty ontProperty = (OntProperty) o;
                    if (ontProperty.isAnnotationProperty()) {
                    	continue;
                    }
                    if (ontProperty.hasProperty(inPropertyGroup, dataModelExclude)) {
                        continue;
                    }
                    // Check that if there are domain constraints for this property,
                    // and if so, that the class is among the allowed domain classes.
                    List<String> domainConstraints = mapPropURIToDomainConstraints.get(ontProperty.getURI());
                    if (domainConstraints != null) {
                        boolean withinConstraint = false;
                        for (String domainRootURI : domainConstraints) {
                            if (domainRootURI.equals(ontClass.getURI())) {
                                // class is itself a domain constraint
                                withinConstraint = true;
                                break;
                            } else {
                                OntClass domainRootOntClass = getOntologyClass(domainRootURI);
                                if (domainRootOntClass.hasSubClass(ontClass)) {
                                    // class is a subclass of a domain constraint.
                                    withinConstraint = true;
                                    break;
                                }
                            }
                        }
                        // If class is not within the specified constraint,
                        // ignore the property.
                        if (!withinConstraint) continue;
                    }
                    final EIProperty property = createProperty(ontProperty);
                    if (property != null) {
                        tmProperties.put(property.getEntity().getLabel().toLowerCase(), property);
                    }
                }
            } catch (ConversionException ex) {
                //logger.warn(ex.getMessage());
                continue;
            }
        }

        final List<EIProperty> listProperties = new ArrayList<EIProperty>(tmProperties.values());
        return listProperties;
    }
    
    @Override
    public List<String> getClassDefinitions(List<EIURI> classURIs) {
        List<String> results = new ArrayList<String>(classURIs.size());
        for (EIURI classURI : classURIs) {
            String def = getClassDefinition(classURI);
            results.add(def);
        }
        return results;
    }
    
    @Override
    public String getClassDefinition(EIURI classURI) {
        OntClass ontClass = getOntologyClass(classURI.toString());
        if (ontClass == null) {
            return null;
        }
        String def = null;
        RDFNode defNode = ontClass.getPropertyValue(ontClass.getModel().getProperty(EagleIOntConstants.OBI_DEFINITION_URI));
        if (defNode != null) {
            def = ((Literal) defNode).getString();
        }
        if (def == null) {
            defNode = ontClass.getPropertyValue(RDFS.comment);
            if (defNode != null) {
                def = "COMMENT: " + ((Literal) defNode).getString();
            }
        }
        return def;        
    }
    
    @Override
    public String getPreferredLabel(EIURI uri) {
        assert uri != null;
        final Resource resource = this.jenaOntModel.getResource(uri.toString());
        if (resource == null) {
            return null;
        }
        final List<String> labels = new ArrayList<String>();
        // try each of the preferred label props, if one of them has a value, return it
        for (Property prop: prefLabelProperties) {
        	getLiteralsForProperty(resource, prop, labels, null);
        	if (labels.size() > 0) {
        		return labels.get(0);
        	}
        }
        return null;
    }    
    
    public List<Property> getPrefLabelProperties() {
    	return prefLabelProperties;
    }
    
    @Override
    public List<String> getLabels(EIURI uri) {
        assert uri != null;
        final Resource resource = this.jenaOntModel.getResource(uri.toString());
        if (resource == null) {
            return Collections.emptyList();
        }
        
        // Synonym list consists of the preferred labels plus
        // any synonyms specified using the IAO Alt Term property
        final List<String> labels = new ArrayList<String>();
        final List<String> labelsLower = new ArrayList<String>();
        for (Property prop: prefLabelProperties) {
            getLiteralsForProperty(resource, prop, labels, labelsLower);
        }
        getLiteralsForProperty(resource, iaoAltTerm, labels, labelsLower);
        
        return labels;
    }
    
    public static void getLiteralsForProperty(final Resource r, final Property p, final List<String> values, final List<String> valuesLower) {
        final StmtIterator it = r.listProperties(p);
        while (it.hasNext()) {
            final Statement stmt = it.next();
            final RDFNode node = stmt.getObject();
            if (node.isLiteral()) {
                String value = ((Literal) node).getLexicalForm();
                if (value == null) {
                    continue;
                }
                // Ignore empty string values
                value = value.trim();
                if (value.isEmpty()) {
                    continue;
                }              
                // For any label except the preferred label (which will be first in 
                // the list), replace any underscores.
                if (values.size() > 0) {
                    value = value.replace('_', ' ');
                }
                
                if (valuesLower != null) {
                    // If a lower case values list is passed in,
                    // then caller cares about checking for duplicates
                    String valueLower = value.toLowerCase();
                    if (!valuesLower.contains(valueLower)) {
                        values.add(value);
                        valuesLower.add(valueLower);
                    }
                } else {
                    values.add(value);
                }
            }
        }        
    }
    
    @Override
    public List<String> getPropertyDefinitions(List<EIURI> propertyURIs) {
        List<String> results = new ArrayList<String>(propertyURIs.size());
        for (EIURI propertyURI : propertyURIs) {
            String def = getClassDefinition(propertyURI);
            results.add(def);
        }
        return results;
    }
    
    @Override
    public String getPropertyDefinition(EIURI propertyURI) {
        String def = null;
        assert(false) : "Not yet implemented";
        /*
        OntClass ontClass = getOntologyClass(classURI.toString());
        RDFNode defNode = ontClass.getPropertyValue(ontClass.getModel().getProperty(EagleIOntConstants.OBI_DEFINITION_URI));
        if (defNode != null) {
            def = ((Literal) defNode).getString();
        }
        if (def == null) {
            defNode = ontClass.getPropertyValue(RDFS.comment);
            if (defNode != null) {
                def = "COMMENT: " + ((Literal) defNode).getString();
            }
        }
        if (def == null) {
            def = "NO DEFINITION";
            if (LOG != null) {
                LOG.addWarning(classURI.toString(), ontClass.getLabel(null), "No definition");
            }
        }
        */
        return def;        
    }

    @Override
    public void traverseDataModel(List<Visitor> visitors) {
        logger.info("Start traversal of data model");
        Set<EIClass> setChecked = new HashSet<EIClass>();
        Deque<String> stack = new ArrayDeque<String>();
        for (EIClass top : listTopLevelEIClasses) {
            traverseDataModel(top, setChecked, stack, visitors);
        }
        logger.info("Traversal of data model complete");
    }

    private void traverseDataModel(EIClass c, Set<EIClass> setChecked, Deque<String> stack, List<Visitor> visitors) {
        if (setChecked.contains(c)) {
            return;
        }
        stack.addLast(c.toString());
        for (Visitor v : visitors) {
            v.visit(c, stack);
        }
        setChecked.add(c);
        
        if (c.hasProperty()) {
            List<EIProperty> properties = getProperties(c.getEntity().getURI());
            for (EIProperty prop:properties) {
                stack.addLast(prop.toString());
                for (Visitor v : visitors) {
                    v.visit(prop, stack);
                }
                if (prop instanceof EIObjectProperty) {
                    List<EIClass> objPropClassList = ((EIObjectProperty) prop).getRangeList();
                    for (EIClass objPropClass : objPropClassList) {
                        //logger.debug("class: " + c.getEntity().getLabel() + "  prop: " + prop.getEntity().getLabel() + "  obj_prop: " + objPropClass.getEntity().getLabel());
                        if (objPropClass != null) {
                            traverseDataModel(objPropClass, setChecked, stack, visitors);
                        }
                    }
                }
                stack.removeLast();
            }
        }
        
        if (c.hasSubClass()) {
            List<EIClass> subclasses = getSubClasses(c.getEntity().getURI());
            for (EIClass subclass:subclasses) {
                //logger.debug("class: " + c.getEntity().getLabel() + " subclass " + subclass.getEntity().getLabel());
                stack.addLast("SUBCLASS");
                traverseDataModel(subclass, setChecked, stack, visitors);
                stack.removeLast();
            }
        }
        stack.removeLast();
    }

    @Override
    public String generateStackTrace(Deque<String> stack) {
        StringBuilder buf = new StringBuilder();
        for (String item : stack) {
            buf.append(item);
            buf.append("\r\n");
        }
        return buf.toString();
    }
}
