package org.eaglei.model.jena;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

import org.eaglei.model.EIOntConstants;

import com.hp.hpl.jena.ontology.AnnotationProperty;
import com.hp.hpl.jena.ontology.OntClass;
import com.hp.hpl.jena.ontology.OntModel;
import com.hp.hpl.jena.rdf.model.NodeIterator;
import com.hp.hpl.jena.rdf.model.Property;
import com.hp.hpl.jena.rdf.model.RDFNode;
import com.hp.hpl.jena.rdf.model.Statement;
import com.hp.hpl.jena.rdf.model.StmtIterator;
import com.hp.hpl.jena.util.iterator.ExtendedIterator;

/**
 * Utility methods for accessing common elements of the eagle-i ontology.
 */
public final class EagleIOntUtils {

    /**
     * Retrieves the classes in the eagle-i ontology that have been designated (via an special annotation
     * property) as belonging to the specified group. Group membership is determined by the URI of the 
     * resource instance that is bound to the annotation property http://purl.org/eagle-i/app/inClassGroup.
     *
     * @param classGroupURI URI of an instance of http://purl.org/eagle-i/app/ClassGroup that determines the
     * class group.
     * 
     * @return List of Jena OntClasses that belong to the specified group.
     */
    // TODO optimize 
    public static List<OntClass> getClassesInGroup(final OntModel jenaOntModel, final String classGroupURI) {
        return getClassesInGroup(jenaOntModel, Arrays.asList( classGroupURI )).get(0);
    }
    
    public static List<List<OntClass>> getClassesInGroup(final OntModel jenaOntModel, final List<String> listClassGroupURI) {
        assert listClassGroupURI != null;
        final AnnotationProperty inClassGroup = jenaOntModel.getAnnotationProperty(EIOntConstants.IN_CLASS_GROUP);
        final List<List<OntClass>> listClassesInGroup = new ArrayList<List<OntClass>>(listClassGroupURI.size());
        for (String uri : listClassGroupURI) {
            listClassesInGroup.add(new ArrayList<OntClass>());
        }
        // get all classes that declare inClassGroup
        // XXX listing the declaring classes for the annotation property is not working with Jena...
        //final ExtendedIterator exIt = inClassGroup.listDeclaringClasses(true);
        final ExtendedIterator exIt = jenaOntModel.listClasses();
        while (exIt.hasNext()) {
            final OntClass ontClass = (OntClass) exIt.next();
            final NodeIterator nodeIt = ontClass.listPropertyValues(inClassGroup);
            while (nodeIt.hasNext()) {
                final RDFNode node = nodeIt.next();
                String nodeURI = node.asNode().getURI();
                for (int i=0; i<listClassGroupURI.size(); i++) {
                    if (nodeURI.equals(listClassGroupURI.get(i))) {
                        listClassesInGroup.get(i).add(ontClass);
                    }
                }
            }
        }
        return listClassesInGroup;
    }
    
    /**
     * Determines if the specified property in the eagle-i ontology has been designated (via an special annotation
     * property) as belonging to the specified group. Group membership is determined by the URI of the 
     * resource instance that is bound to the annotation property http://purl.org/eagle-i/app/inPropertyGroup.
     *
     * @param propertyGroupURI URI of an instance of http://purl.org/eagle-i/app/PropertyGroup that determines the
     * property group.
     * @param propertyURI URI of the property whose membership in the group is being checked.
     * 
     * @return True if the specified property is in the specified property group, false otherwise.
     */
    public static boolean isPropertyInGroup(final OntModel jenaOntModel, final String propertyGroupURI, final String propertyURI) {
        assert propertyGroupURI != null;
        assert propertyURI != null;
        final AnnotationProperty inPropertyGroup = jenaOntModel.getAnnotationProperty(EIOntConstants.IN_PROPERTY_GROUP);
        final Property property = jenaOntModel.getProperty(propertyURI);
        final StmtIterator stmtIt = property.listProperties(inPropertyGroup);
        while (stmtIt.hasNext()) {
            Statement stmt = stmtIt.next();
            if (stmt.getObject().asNode().getURI().equals(propertyGroupURI)) {
                return true;
            }
        }
        return false;
    }

}