/**
 * eagle-i Project
 * Harvard University
 * Apr 23, 2010
 */
package org.eaglei.model.jena;

import static org.eaglei.model.jena.MetadataConstants.readOnlyLiterals;
import static org.eaglei.model.jena.MetadataConstants.readOnlyResources;

import java.io.IOException;
import java.io.StringReader;
import java.io.StringWriter;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eaglei.model.EIClass;
import org.eaglei.model.EIDatatypeProperty;
import org.eaglei.model.EIEntity;
import org.eaglei.model.EIInstance;
import org.eaglei.model.EIInstanceFactory;
import org.eaglei.model.EIObjectProperty;
import org.eaglei.model.EIOntConstants;
import org.eaglei.model.EIOntModel;
import org.eaglei.model.EIProperty;
import org.eaglei.model.EIURI;

import com.hp.hpl.jena.rdf.model.Model;
import com.hp.hpl.jena.rdf.model.ModelFactory;
import com.hp.hpl.jena.rdf.model.Property;
import com.hp.hpl.jena.rdf.model.RDFNode;
import com.hp.hpl.jena.rdf.model.Resource;
import com.hp.hpl.jena.rdf.model.SimpleSelector;
import com.hp.hpl.jena.rdf.model.Statement;
import com.hp.hpl.jena.vocabulary.RDF;
import com.hp.hpl.jena.vocabulary.RDFS;

/**
 * @author Daniela Bourges-Waldegg
 * @author Ricardo De Lima
 * 
 *         April 11, 2010
 * 
 *         Center for Biomedical Informatics (CBMI)
 * @link https://cbmi.med.harvard.edu/
 * 
 * 
 */

public class JenaEIInstanceFactory implements EIInstanceFactory {
	private static final Log logger = LogFactory.getLog( JenaEIInstanceFactory.class );

	private static final boolean isDebugEnabled = logger.isDebugEnabled();

	private final EIOntModel ontModel;

	public JenaEIInstanceFactory(final EIOntModel ontModel) {
		this.ontModel = ontModel;
	}

	public Model convertToJenaModel(final EIInstance instance) {
		if ( instance == null || EIInstance.NULL_INSTANCE.equals( instance ) ) {
			return null;
		}
		final Model model = ModelFactory.createDefaultModel();
		// Create the resource that will be the subject of all statements
		final Resource resourceInstance = model.createResource( instance.getInstanceURI().toString() );

		// Set its type and label
		final Resource eiType = model.createResource( instance.getInstanceType().getURI().toString() );
		model.add( model.createStatement( resourceInstance, RDF.type, eiType ) );
		model.add( model.createStatement( resourceInstance, RDFS.label, instance.getInstanceLabel() ) );

		// Set other types
		final List<EIEntity> otherEITypes = instance.getOtherEITypes();
		if ( otherEITypes != null ) {
			for (final EIEntity t : otherEITypes) {
				final Resource typeResource = model.createResource( t.getURI().toString() );
				model.add( model.createStatement( resourceInstance, RDF.type, typeResource ) );
			}
		}
		// Set all other properties
		final Map<EIEntity, Set<String>> dataProps = instance.getDatatypeProperties();
		final Map<EIEntity, Set<EIEntity>> objectProps = instance.getObjectProperties();
		addLiteralTriples( model, resourceInstance, dataProps );
		addResourceTriples( model, resourceInstance, objectProps );

		if ( instance.isExtendedInstance() ) {
			final Map<EIEntity, Set<String>> literalProps = instance.getNonOntologyLiteralProperties();
			final Map<EIEntity, Set<EIEntity>> resourceProps = instance.getNonOntologyResourceProperties();
			final Map<EIEntity, EIEntity> roResourceProps = instance.getReadOnlyResourceProperties();
			final Map<EIEntity, String> roLiteralProps = instance.getReadOnlyLiteralProperties();
			addLiteralTriples( model, resourceInstance, literalProps );
			addResourceTriples( model, resourceInstance, resourceProps );

			for (final Map.Entry<EIEntity, String> entry : roLiteralProps.entrySet()) {
				final Property p = model.createProperty( entry.getKey().getURI().toString() );
				if ( entry.getValue() != null && entry.getValue().length() > 0 ) {
					model.add( model.createStatement( resourceInstance, p, entry.getValue() ) );
				}
			}

			for (final Map.Entry<EIEntity, EIEntity> entry : roResourceProps.entrySet()) {
				final Property p = model.createProperty( entry.getKey().getURI().toString() );
				if ( entry.getValue() != null && entry.getValue().toString().length() > 0 ) {
					final Resource value = model.createResource( entry.getValue().getURI().toString() );
					model.add( model.createStatement( resourceInstance, p, value ) );
				}
			}
		}
		// Only enter recursion from main instance (assumption that embedded have no embedded)
		// FIXME uncomment when repo has moved to embedded-update
		
		  if(instance.hasEmbeddedInstances()) { for(EIInstance embedded : instance.getEmbeddedInstanceList()) { final Model embeddedModel = convertToJenaModel( embedded ); model.add( embeddedModel ); } }
		 
		return model;
	}

	public EIInstance create(final EIURI instanceUri, final Model model) {
		return commonCreate( instanceUri, model, false, false );
	}

	public EIInstance createEmbedded(final EIURI instanceUri, final Model model) {
		return commonCreate( instanceUri, model, false, true );
	}

	public EIInstance createExtended(final EIURI instanceUri, final Model model) {
		return commonCreate( instanceUri, model, true, false );
	}

	private EIInstance commonCreate(final EIURI instanceUri, final Model model, final boolean isExtended, final boolean isEmbedded) {
		final Resource subject = getInstanceSubject( instanceUri, model );
		if ( subject == null ) {
			return EIInstance.NULL_INSTANCE;
		}

		final List<EIClass> eiClasses = getInstanceClasses( model, subject );
		if ( eiClasses == null || eiClasses.isEmpty() || eiClasses.get( 0 ) == null ) {
			if ( isDebugEnabled ) {
				logger.debug( "No eagle-i types found in the model" );
			}
			return EIInstance.NULL_INSTANCE;
		}

		// FIXME for now set instance type to first in list;
		// The most common case is only one ei type; we can't really know what to put there otherwise
		// Ideally the most specific type (use reasoner's realize method?)
		final EIClass instanceClass = eiClasses.get( 0 );
		eiClasses.remove( 0 );

		final String instanceLabel = getInstanceLabel( subject );
		if ( instanceLabel == null ) {
			return EIInstance.NULL_INSTANCE;
		}

		if ( isDebugEnabled ) {
			logger.debug( "Creating an instance of class: " + instanceClass.getEntity().toString() + " with URI: " + instanceUri + " and label: " + instanceLabel );
		}

		// create EInstance
		final EIInstance ei = EIInstance.createEmptyInstance( instanceClass, EIEntity.create( instanceUri, instanceLabel ) );
		ei.setRootSuperType( getRootSuperType( instanceClass ));
		ei.setEmbedded( isEmbedded );
		setInstanceTypes( eiClasses, ei );
		// Process statements for main instance
		processStatements( model, subject, eiClasses, instanceClass, ei, isExtended );
		// Process embedded instances
		return ei;
	}

	private EIClass getRootSuperType(EIClass eiClass) {
		//FIXME hack so that lab appear as the root
		/*if(eiClass.getEntity().getURI().toString().equals( EIOntConstants.EI_LAB)) {
			return eiClass;
		}*/
		if( !eiClass.hasSuperClass() ) {
			return eiClass;
		}
		final List<EIClass> superclasses = ontModel.getSuperClasses( eiClass.getEntity().getURI() );
		if(superclasses == null || superclasses.isEmpty()) {
			return eiClass;
		}
		return superclasses.get(superclasses.size() - 1);
		//return superclasses.get( 0 );
	}

	@Override
	public EIInstance create(final EIURI instanceUri, final String rdf, final String lang) {
		return commonCreate( instanceUri, rdf, lang, false, false );
	}

	@Override
	public EIInstance createEmbedded(final EIURI instanceUri, final String rdf, final String lang) {
		return commonCreate( instanceUri, rdf, lang, false, true );
	}

	@Override
	public EIInstance createExtended(final EIURI instanceUri, final String rdf, final String lang) {
		return commonCreate( instanceUri, rdf, lang, true, false );
	}

	private EIInstance commonCreate(final EIURI instanceUri, final String rdf, final String lang, final boolean isExtended, final boolean isEmbedded) {
		if ( instanceUri == null || instanceUri.toString().length() == 0 || rdf == null || rdf.length() == 0 ) {
			return EIInstance.NULL_INSTANCE;
		}
		final Model model = ModelFactory.createDefaultModel();
		model.read( new StringReader( rdf ), null, lang );
		return commonCreate( instanceUri, model, isExtended, isEmbedded );
	}

	public List<EIInstance> create(final Model model) {
		return commonCreate( model, false, false );
	}

	public List<EIInstance> createEmbedded(final Model model) {
		return commonCreate( model, false, true );
	}

	public List<EIInstance> createExtended(final Model model) {
		return commonCreate( model, true, false );
	}

	private List<EIInstance> commonCreate(final Model model, final boolean isExtended, final boolean isEmbedded) {
		if ( model == null ) {
			return Collections.emptyList();
		}
		final List<EIInstance> instances = new ArrayList<EIInstance>();
		final Set<Resource> subjects = model.listSubjects().toSet();
		// create an EIInstance per subject
		for (final Resource r : subjects) {
			final Model results = model.query( new SimpleSelector( r, null, (RDFNode)null ) );
			final EIInstance ei = commonCreate( EIURI.create( r.getURI() ), results, isExtended, isEmbedded );
			if ( ei != null && !EIInstance.NULL_INSTANCE.equals( ei ) ) {
				instances.add( ei );
			}
		}
		return instances;
	}

	public EIInstance createEmpty(final EIURI typeUri, final EIEntity instanceEntity) {
		if ( instanceEntity == null ) {
			return EIInstance.NULL_INSTANCE;
		}
		final EIClass instanceClass = safeGetClass( typeUri );
		if ( instanceClass == null ) {
			return EIInstance.NULL_INSTANCE;
		}
		final String label = instanceClass.getEntity().getLabel();
		instanceClass.getEntity().setLabel( label );
		final EIInstance ei = EIInstance.createEmptyInstance( instanceClass, instanceEntity );
		ei.setRootSuperType( getRootSuperType( instanceClass ) );
		return ei;
	}

	public String serialize(final EIInstance instance, final String lang) {
		if ( instance == null || EIInstance.NULL_INSTANCE.equals( instance ) ) {
			return null;
		}

		final Model model = convertToJenaModel( instance );
		// Serialize model to String in specified format
		final StringWriter sw = new StringWriter();
		try {
			// TODO validate format
			model.write( sw, lang );
			final String s = sw.toString();
			sw.flush();
			return s;
		} finally {
			try {
				if ( sw != null ) {
					sw.close();
				}
			} catch (final IOException e) {/* not much to do */
			}
		}

	}

	private EIEntity addEIProperty(final EIInstance ei, final EIProperty predicate, final EIURI propertyUri, final RDFNode o, final boolean isExtended) {
		// returned value is used to process embedded instance
		final String preferredLabel = ontModel.getPreferredLabel( propertyUri );
		final EIEntity propEntity = EIEntity.create( propertyUri, preferredLabel );
		if ( predicate instanceof EIDatatypeProperty ) {
			if ( o.isLiteral() ) {
				final String value = o.toString();
				ei.addDatattypeProperty( propEntity, value );
			} else if ( isExtended ) {
				// property is an EIProperty but value doesn't fit Ontology (datatype expected)
				final String objectLabel = getObjectLabel( o );
				final EIEntity entity = EIEntity.create( EIURI.create( ( (Resource)o ).getURI() ), objectLabel );
				ei.addNonOntologyResourceProperty( propEntity, entity );
			} // ignore otherwise
		} else if ( predicate instanceof EIObjectProperty ) {
			if ( o.isResource() ) {
				// check if its range is embedded
				final String objectLabel = getObjectLabel( o );
				final EIEntity entity = EIEntity.create( EIURI.create( ( (Resource)o ).getURI() ), objectLabel );
				ei.addObjectProperty( propEntity, entity );
				return entity;
			} else if ( isExtended ) {
				// property is an EIProperty but value doesn't fit Ontology (object expected)
				ei.addNonOntologyLiteralProperty( propEntity, o.toString() );
			} // ignore otherwise
		}
		return EIEntity.NULL_ENTITY;
	}

	private void addLiteralTriples(final Model model, final Resource resourceInstance, final Map<EIEntity, Set<String>> literals) {
		for (final Map.Entry<EIEntity, Set<String>> entry : literals.entrySet()) {
			final Property p = model.createProperty( entry.getKey().getURI().toString() );
			final Set<String> values = entry.getValue();
			for (final String value : values) {
				if ( value != null && value.length() > 0 ) {
					model.add( model.createStatement( resourceInstance, p, value ) );
				}
			}
		}
	}

	private void addNonEIProperty(final EIInstance ei, final String propertyLabel, final EIURI propertyUri, final RDFNode o) {
		EIEntity propEntity;
		if ( readOnlyResources.containsKey( propertyUri ) ) {
			propEntity = readOnlyResources.get( propertyUri );
		} else if ( readOnlyLiterals.containsKey( propertyUri ) ) {
			propEntity = readOnlyLiterals.get( propertyUri );
		} else {
			propEntity = EIEntity.create( propertyUri, propertyLabel );
		}
		if ( o.isLiteral() ) {
			if ( readOnlyLiterals.containsKey( propertyUri ) ) {
				ei.setReadOnlyLiteralProperty( propEntity, o.toString() );
			} else {
				ei.addNonOntologyLiteralProperty( propEntity, o.toString() );
			}
		} else if ( o.isResource() ) {
			final String objectLabel = getObjectLabel( o );
			final EIEntity entity = EIEntity.create( EIURI.create( ( (Resource)o ).getURI() ), objectLabel );
			if ( readOnlyResources.containsKey( propertyUri ) ) {
				ei.setReadOnlyResourceProperty( propEntity, entity );
			} else {
				ei.addNonOntologyResourceProperty( propEntity, entity );
			}
		} else {
			logger.info( "Could not add property " + propertyUri );
		}
	}

	private void addResourceTriples(final Model model, final Resource resourceInstance, final Map<EIEntity, Set<EIEntity>> resources) {
		for (final Map.Entry<EIEntity, Set<EIEntity>> entry : resources.entrySet()) {
			final Property p = model.createProperty( entry.getKey().getURI().toString() );
			final Set<EIEntity> values = entry.getValue();
			if ( values != null ) {
				for (final EIEntity value : values) {
					if ( value != null && value.toString().length() > 0 ) {
						final Resource valueResource = model.createResource( value.getURI().toString() );
						model.add( model.createStatement( resourceInstance, p, valueResource ) );
					}
				}
			}
		}
	}

	private boolean containsUri(final List<EIClass> eiClasses, final String uri) {
		if ( eiClasses == null || uri == null ) {
			return false;
		}
		for (final EIClass eiClass : eiClasses) {
			if ( uri.equalsIgnoreCase( eiClass.getEntity().getURI().toString() ) ) {
				return true;
			}
		}
		return false;
	}

	/**
	 * Find a property for a class in the eagle-i ontology.
	 * 
	 * @param instanceClassUri
	 * @param propertyUri
	 * @return the property if found, null otherwise
	 */
	private EIProperty getEIOntProperty(final EIURI instanceClassUri, final EIURI propertyUri) {
		if ( instanceClassUri == null || propertyUri == null ) {
			return null;
		}

		final List<EIProperty> properties = ontModel.getProperties( instanceClassUri );
		if ( properties == null || properties.isEmpty() ) {
			return null;
		}
		for (final EIProperty p : properties) {
			final EIURI propUri = p.getEntity().getURI();
			if ( propUri.equals( propertyUri ) ) {
				return p;
			}
		}
		// If property wasn't found in domain
		return null;
	}

	private List<EIClass> getInstanceClasses(final Model model, final Resource subject) {
		// Instances can have multiple types
		final List<Statement> typeStatements = model.listStatements( subject, RDF.type, (Resource)null ).toList();
		if ( typeStatements == null || typeStatements.isEmpty() ) {
			return null;
		}
		final List<EIClass> eiClasses = new ArrayList<EIClass>();
		for (final Statement st : typeStatements) {
			final Resource type = (Resource)st.getObject();
			final EIURI typeUri = EIURI.create( type.getURI() );
			final EIClass eiClass = safeGetClass( typeUri );
			if ( eiClass != null ) {
				eiClasses.add( eiClass );
			}
		}
		return eiClasses;
	}

	private String getInstanceLabel(final Resource subject) {
		final Statement labelStatement = subject.getProperty( RDFS.label );
		if ( labelStatement == null ) {
			logger.info( "RDFS Label is not set for instance" );

			return null;
			// return EIInstance.NULL_INSTANCE;
		}
		return labelStatement.getString();
	}

	private Resource getInstanceSubject(final EIURI instanceUri, final Model model) {
		if ( instanceUri == null || model == null ) {
			logger.info( "null parameters; no EIInstance was created" );
			return null;
		}
		final Resource subject = model.getResource( instanceUri.toString() );
		// Get resource will create a resource if one doens't exist already, so need to check
		if ( !model.contains( subject, null, (RDFNode)null ) ) {
			if ( isDebugEnabled ) {
				logger.debug( "model doesn't contain subject; no EIInstance was created" );
			}
			return null;
		}
		return subject;
	}

	private String getObjectLabel(final RDFNode o) {
		String objectLabel;
		final EIClass classForObject = safeGetClass( EIURI.create( ( (Resource)o ).getURI() ) );
		if ( classForObject != null ) {
			// TODO reuse this entity, duh
			objectLabel = classForObject.getEntity().getLabel();
		} else { // object is not an eagle-i resource {
			final Statement objectLabelStatement = ( (Resource)o ).getProperty( RDFS.label );
			if ( objectLabelStatement != null ) {
				objectLabel = objectLabelStatement.getLiteral().getLexicalForm();
			} else {
				objectLabel = ( (Resource)o ).getURI().toString();
			}
		}
		return objectLabel;
	}

	/**
	 * Process one instance, i.e. all statements with the same subject
	 * 
	 * @param model
	 * @param subject
	 * @param eiClasses
	 * @param instanceClass
	 * @param ei
	 * @param isExtended
	 */
	private void processStatements(final Model model, final Resource subject, final List<EIClass> eiClasses, final EIClass instanceClass, final EIInstance ei, final boolean isExtended) {
		final List<Statement> statements = model.listStatements( subject, null, (RDFNode)null ).toList();
		for (final Statement statement : statements) {
			final Property predicate = statement.getPredicate();
			// No need to process label, as it was done previously
			if ( predicate.equals( RDFS.label ) ) {
				continue;
			}
			final EIURI propertyUri = EIURI.create( predicate.getURI() );
			final RDFNode o = statement.getObject();

			// No need to process type if it is instanceClass
			if ( predicate.equals( RDF.type ) && o.isResource() && instanceClass.getEntity().getURI().toString().equals( ( (Resource)o ).getURI() ) ) {
				continue;
			}

			// No need to process types already in eiClasses; other types will end up in resource map
			if ( predicate.equals( RDF.type ) && o.isResource() && containsUri( eiClasses, ( (Resource)o ).getURI() ) ) {
				continue;
			}

			// Check if property is in ontology and add accordingly
			final EIProperty p = getEIOntProperty( instanceClass.getEntity().getURI(), propertyUri );
			if ( p != null ) { // For ont properties we get labels from OntModel
				final EIEntity addedPropertyValue = addEIProperty( ei, p, propertyUri, o, isExtended );
				if ( !ei.isEmbeddedInstance() && propertyNeedsEmbeddedProcessing( p ) && !isNull( addedPropertyValue ) ) {
					ei.addEmbeddedInstance( p.getEntity(), createEmbedded( addedPropertyValue.getURI(), model ) );
				}
			} else if ( isExtended ) {// For non-ont properties we get labels from received data
				final String propertyLabel = getObjectLabel( predicate );
				addNonEIProperty( ei, propertyLabel, propertyUri, o );
			} // else ignore non ont
		}
	}

	private boolean isNull(final EIEntity entity) {
		return entity == null || EIEntity.NULL_ENTITY.equals( entity );
	}

	private boolean propertyNeedsEmbeddedProcessing(final EIProperty property) {
		if ( !( property instanceof EIObjectProperty ) ) {
			return false;
		}
		// We assume no multi-range for embedded classes
		final EIClass range = ( (EIObjectProperty)property ).getRangeList().get( 0 );
		final Set<String> annotations = ontModel.getClassAnnotations( range.getEntity().getURI() );
		return annotations.contains( EIOntConstants.CG_EMBEDDED_CLASS );
	}

	private EIClass safeGetClass(final EIURI typeUri) {
		if ( ontModel.isModelClassURI( typeUri.toString() ) ) {
			return ontModel.getClass( typeUri );
		} else {
			return null;
		}
	}

	private void setInstanceTypes(final List<EIClass> eiClasses, final EIInstance ei) {
		// If there were more than one ei types found previously
		if ( !eiClasses.isEmpty() ) {
			final List<EIEntity> eiTypes = new ArrayList<EIEntity>( eiClasses.size() );
			for (final EIClass c : eiClasses) {
				eiTypes.add( c.getEntity() );
			}
			ei.setOtherEITypes( eiTypes );
		}
	}

	/**
	 * @param mainInstance
	 * @return
	 */
	@Deprecated
	public Set<EIEntity> getEmbeddedUrisFromObjectProperties(final EIInstance mainInstance) {
		final Map<EIEntity, Set<EIEntity>> objectProperties = mainInstance.getObjectProperties();
		if ( objectProperties == null || objectProperties.isEmpty() ) {
			return Collections.emptySet();
		}
		final Set<EIEntity> embeddedInstanceUris = new HashSet<EIEntity>();
		for (final Map.Entry<EIEntity, Set<EIEntity>> entry : objectProperties.entrySet()) {
			final EIObjectProperty p = (EIObjectProperty)getEIOntProperty( mainInstance.getInstanceType().getURI(), entry.getKey().getURI() );
			if ( p != null ) {
				// FIXME for now we assume that embedded classes are not part of a multi-range;
				// may need to revisit
				final EIClass range = p.getRangeList().get( 0 );
				// FIXME is this a bug?: range.getAnnotations() returns null, asking the ontModel gets the lis
				final Set<String> annotations = ontModel.getClassAnnotations( range.getEntity().getURI() );
				if ( annotations.contains( EIOntConstants.CG_EMBEDDED_CLASS ) ) {
					// all the values of the property represent embedded instances
					embeddedInstanceUris.addAll( entry.getValue() );
				}
			}
		}
		return embeddedInstanceUris;
	}
}
