package org.eaglei.model.webapp.client.searchbar;

import java.io.Serializable;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.Set;

import org.eaglei.model.EIURI;

/**
 * Represents a search request.
 */
public class TermSearchRequest implements Serializable {

    public static final long serialVersionUID = 1L;

    public static class Term implements Serializable {

        public static final long serialVersionUID = 1L;

        /**
         * Optional field that holds the value of the query string for the
         * search term. Either query or entity must be specified.
         */
        private String query;

        /**
         * Optional field that holds the URI of the resolved or selected entity
         * from the eagle-i ontology used to constrain the search term. Either
         * query or entity must be specified.
         */
        private EIURI uri;

        /*
         * 
         */
        private Term() {
            // for GWT
        }

        /**
         * Creates a new search term with the specified query string and entity
         * URI. At least one of these must be non-null. If both are specified,
         * they are combined using an implicit OR.
         * 
         * @param query Query used for a full-text searching. Must not be a zero
         *            length string. TBD: support for advanced syntax in the
         *            query string
         * @param uri The URI of a class in the eagle-i ontology (or referenced
         *            ontology) that is used as the target for the search
         *            operation. This URI may have been resolved from the query
         *            string.
         */
        public Term(final String query, final EIURI uri) {
            assert (query != null || uri != null);
            assert (query == null || query.length() > 0);
            this.query = query;
            this.uri = uri;
        }

        /**
         * Creates a new search term with the specified query string.
         * 
         * @param query Query used for a full-text searching. TBD: support for
         *            advanced syntax in the query string
         */
        public Term(final String query) {
            this(query, null);
        }

        /**
         * Creates a new search term with the specified entity URI.
         * 
         * @param uri The URI of a class in the eagle-i ontology (or referenced
         *            ontology) that is used as the target for the search
         *            operation.
         */
        public Term(final EIURI uri) {
            this(null, uri);
        }

        /**
         * Creates a deep copy of the given Term.
         * 
         * @param term Term to copy
         */
        public Term(final Term term) {
            this(term.query, term.uri);
        }

        /**
         * Gets the search query
         * 
         * @return The search query. May be null.
         */
        public String getQuery() {
            return this.query;
        }

        /**
         * Get the URI for the eagle-i ontology entity used to constrain the
         * search term
         * 
         * @return Entity URI. May be null.
         */
        public EIURI getURI() {
            return this.uri;
        }

		@Override
		public int hashCode() {
			final int prime = 31;
			int result = 1;
			result = prime * result + ((query == null) ? 0 : query.hashCode());
			result = prime * result + ((uri == null) ? 0 : uri.hashCode());
			return result;
		}

		@Override
		public boolean equals(Object obj) {
			if (this == obj) {
				return true;
			}
			if (obj == null) {
				return false;
			}
			if (getClass() != obj.getClass()) {
				return false;
			}
			Term other = (Term) obj;
			if (query == null) {
				if (other.query != null) {
					return false;
				}
			} else if (!query.equals(other.query)) {
				return false;
			}
			if (uri == null) {
				if (other.uri != null) {
					return false;
				}
			} else if (!uri.equals(other.uri)) {
				return false;
			}
			return true;
		}
    }

    /**
     * Set of resource type-specific bindings for a search. These bindings are
     * specified in reference to the eagle-i ontology. NOT multi-thread safe.
     */
    public static class TypeBinding implements Serializable {

        public static final long serialVersionUID = 1L;

        /*
         * Type for search result resources. Will be the URI of a class in the
         * eagle-i ontology.
         */
        private EIURI type;
        /*
         * Bindings for various properties of the eagle-i ontology class
         * specified via the type field above.
         */
        private Map<EIURI, Object> properties;

        private TypeBinding() {
            // GWT
        }

        /**
         * Creates a new TypeBinding object for the specified eagle-i class.
         * 
         * @param type A type from the eagle-i ontology.
         */
        public TypeBinding(EIURI type) {
            assert type != null;
            this.type = type;
        }

        /**
         * Retrieves the type constraint for search result resources. A URI of a
         * class in the eagle-i ontology.
         * 
         * @return Type constraint.
         */
        public EIURI getType() {
            return this.type;
        }

        /**
         * Adds a property binding.
         * 
         * @param property URI of one of the type's properties.
         * @param value Value that the property must have.
         */
        public void addProperty(EIURI property, Object value) {
            assert property != null;
            assert value != null;
            // Not threadsafe
            if (properties == null) {
                properties = new HashMap<EIURI, Object>();
            }
            properties.put(property, value);
        }

        /**
         * Returns the URIs (as EIURI instances) for the property bindings
         * 
         * @return Set of EIURIs representing property bindings.
         */
        @SuppressWarnings("unchecked")
        public Set<EIURI> getProperties() {
            if (this.properties == null) {
                // GWT compiler doesn't like emptySet()
                // return Collections.emptySet();
                return Collections.EMPTY_SET;
            }
            return this.properties.keySet();
        }

        /**
         * Retrieves the specified property.
         * 
         * @param property Property URI as an EIURI.
         * @return Value of the property if it has been set or null.
         */
        public Object getProperty(EIURI property) {
            assert property != null;
            if (this.properties == null) {
                return null;
            }
            return this.properties.get(property);
        }

		@Override
		public int hashCode() {
			final int prime = 31;
			int result = 1;
			result = prime * result
					+ ((properties == null) ? 0 : properties.hashCode());
			result = prime * result + ((type == null) ? 0 : type.hashCode());
			return result;
		}

		@Override
		public boolean equals(Object obj) {
			if (this == obj) {
				return true;
			}
			if (obj == null) {
				return false;
			}
			if (getClass() != obj.getClass()) {
				return false;
			}
			TypeBinding other = (TypeBinding) obj;
			if (properties == null) {
				if (other.properties != null) {
					return false;
				}
			} else if (!properties.equals(other.properties)) {
				return false;
			}
			if (type == null) {
				if (other.type != null) {
					return false;
				}
			} else if (!type.equals(other.type)) {
				return false;
			}
			return true;
		}
    }

    public static final int DEFAULT_PAGE_SIZE = 10;

    private static final String PARAM_DELIMITER = "&";
    private static final String QUERY_KEY = "q";
    private static final String URI_KEY = "uri";
    private static final String INSTITUTION_KEY = "inst";
    private static final String TYPE_KEY = "t";
    private static final String START_KEY = "start";
    private static final String MAX_KEY = "max";
    private static final String RESOLVE_KEY = "r"; // default true

    /*
     * Optional. URI of the eagle-i institution instance used to constrain the
     * search.
     */
    private EIURI institution;

    /*
     * Optional. Search term to be matched against any resource property value.
     */
    private Term term;

    /*
     * Optional. Constrain results to the given type and property bindings.
     */
    private TypeBinding binding;

    /*
     * Start index, 0-based
     */
    private int startIndex = 0;

    /*
     * Max results.
     */
    private int maxResults = DEFAULT_PAGE_SIZE;

    /**
     * 
     */
    public TermSearchRequest() {
    }

    /**
     * 
     */
    public TermSearchRequest(String query) {
        this(new Term(query));
    }

    /**
     * 
     */
    public TermSearchRequest(EIURI uri) {
        this(new Term(uri));
    }

    /**
     * 
     */
    public TermSearchRequest(Term term) {
        this.term = term;
    }

    /**
     * Parses a param list into a search request. Invalid params will be
     * ignored. 
     * 
     * @param strParams
     * @return SearchRequest
     */
    public static TermSearchRequest parse(String strParams) {
        TermSearchRequest r = new TermSearchRequest();
        if (strParams != null && strParams.length() > 0) {
            String[] params = strParams.split(PARAM_DELIMITER);
            for (String param : params) {
                String[] parts = param.split("=");
                if (parts.length != 2) {
                    // ignore
                    continue;
                    // throw new Exception("Invalid param: " + param); // TODO
                    // offset...
                }
                if (parts[0].equals(QUERY_KEY)) {
                    if (r.term == null) {
                        r.term = new Term();
                    }
                    // validate?
                    r.term.query = parts[1];
                } else if (parts[0].equals(URI_KEY)) {
                    if (r.term == null) {
                        r.term = new Term();
                    }
                    // For now, uri string MUST omit the prefix.
                    EIURI uri = EIURI.create(parts[1]);
                    r.term.uri = uri;
                } else if (parts[0].equals(INSTITUTION_KEY)) {
                    EIURI uri = EIURI.create(parts[1]);
                    r.institution = uri;
                } else if (parts[0].equals(TYPE_KEY)) {
                    if (r.binding == null) {
                        r.binding = new TypeBinding();
                    }
                    // Support serializing the type segment of the URI
                    // TODO Factor out application-specific type lookup
                    // for now, just create a new URI
                    EIURI uri = EIURI.create(parts[1]);
                    r.binding.type = uri;
                } else if (parts[0].equals(START_KEY)) {
                    try {
                        r.setStartIndex(Integer.valueOf(parts[1]));
                    } catch (NumberFormatException ex) {
                        // ignore
                        // throw new Exception("Invalid page param: " +
                        // ex.getMessage()); // TODO
                        // offset...
                    }
                } else if (parts[0].equals(MAX_KEY)) {
                    try {
                        r.setMaxResults(Integer.valueOf(parts[1]));
                    } catch (NumberFormatException ex) {
                        // ignore
                        // throw new Exception("Invalid page size param: " +
                        // ex.getMessage()); // TODO
                        // offset...
                    }
                }
            }
        }
        return r;
    }

    /**
     * Get the URI for the eagle-i institution instance used to constrain the
     * search term
     * 
     * @return Institution URI. May be null.
     */
    public EIURI getInstitution() {
        return this.institution;
    }

    /**
     * Sets the URI for the eagle-i institution instance used to constrain the
     * search term.
     * 
     * @param institution Institution URI.
     */
    public void setInstitution(final EIURI institution) {
        this.institution = institution;
    }

    /**
     * Gets the search term, may be null.
     */
    public Term getTerm() {
        return this.term;
    }

    /**
     * Sets the search term.
     * 
     * @param term The search term
     */
    public void setTerm(final Term term) {
        this.term = term;
    }

    /**
     * Gets the search resource type binding, may be null.
     */
    public TypeBinding getBinding() {
        return this.binding;
    }

    /**
     * Sets the search resource type binding.
     * 
     * @param binding The search result resource type binding.
     */
    public void setBinding(final TypeBinding binding) {
        this.binding = binding;
    }

    /**
     * Gets the request start index.
     * 
     * @return The request start index.
     */
    public int getStartIndex() {
        return this.startIndex;
    }

    /**
     * Sets the search request start index.
     * 
     * @param startIndex The start index.
     */
    public void setStartIndex(final int startIndex) {
        assert startIndex >= 0;
        this.startIndex = startIndex;
    }

    /**
     * Gets the max number of results.
     * 
     * @return Maximum results to return.
     */
    public int getMaxResults() {
        return this.maxResults;
    }

    /**
     * Sets the maximum results to return.
     * 
     * @param maxResults Maximum number of results to return.
     */
    public void setMaxResults(final int maxResults) {
        assert maxResults >= 0;
        this.maxResults = maxResults;
    }

    /**
     * Returns a string representation of this request suitable for use as the
     * param list in a catalyst search URL.
     * 
     * @return param String
     */
    public String toURLParams() {
        StringBuilder buf = new StringBuilder();
        if (term != null) {
            if (term.query != null && term.query.length() > 0) {
                buf.append(QUERY_KEY);
                buf.append("=");
                buf.append(term.query);
            }
            if (term.uri != null) {
                if (buf.length() > 0) {
                    buf.append(PARAM_DELIMITER);
                }
                buf.append(URI_KEY);
                buf.append("=");
                buf.append(term.uri.toString());
            }
        }
        if (institution != null) {
            if (buf.length() > 0) {
                buf.append(PARAM_DELIMITER);
            }
            buf.append(INSTITUTION_KEY);
            buf.append("=");
            buf.append(institution.toString());
        }
        // Don't bother writing a page param if on page 1.
        if (startIndex > 0) {
            if (buf.length() > 0) {
                buf.append(PARAM_DELIMITER);
            }
            buf.append(START_KEY);
            buf.append("=");
            buf.append(startIndex);
        }
        if (maxResults != DEFAULT_PAGE_SIZE) {
            if (buf.length() > 0) {
                buf.append(PARAM_DELIMITER);
            }
            buf.append(MAX_KEY);
            buf.append("=");
            buf.append(maxResults);
        }
        if (binding != null) {
            if (binding.type != null) {
                if (buf.length() > 0) {
                    buf.append(PARAM_DELIMITER);
                }
                buf.append(TYPE_KEY);
                buf.append("=");
                // Support serializing just the type segment of the URI
                buf.append(binding.type.toString());
            }
        }
        /*
         * if (mapSubfilters != null) { for (String key :
         * mapSubfilters.keySet()) { if (buf.length() > 0) {
         * buf.append(PARAM_DELIMITER); } buf.append(key); buf.append("=");
         * buf.append(mapSubfilters.get(key)); } }
         */
        return buf.toString();
    }

    @Override
    public String toString() {
        return toURLParams();
    }

	public int hashCode() {
		final int prime = 31;
		int result = 1;
		result = prime * result + ((binding == null) ? 0 : binding.hashCode());
		result = prime * result + ((institution == null) ? 0 : institution.hashCode());
		result = prime * result + maxResults;
		result = prime * result + startIndex;
		result = prime * result + ((term == null) ? 0 : term.hashCode());
		return result;
	}

	public boolean equals(Object obj) {
		if (this == obj) {
			return true;
		}
		if (obj == null) {
			return false;
		}
		if (getClass() != obj.getClass()) {
			return false;
		}
		TermSearchRequest other = (TermSearchRequest) obj;
		if (binding == null) {
			if (other.binding != null) {
				return false;
			}
		} else if (!binding.equals(other.binding)) {
			return false;
		}
		if (institution == null) {
			if (other.institution != null) {
				return false;
			}
		} else if (!institution.equals(other.institution)) {
			return false;
		}
		if (maxResults != other.maxResults) {
			return false;
		}
		if (startIndex != other.startIndex) {
			return false;
		}
		if (term == null) {
			if (other.term != null) {
				return false;
			}
		} else if (!term.equals(other.term)) {
			return false;
		}
		return true;
	}
}
