package org.eaglei.model.webapp.client;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;

import org.eaglei.model.EIClass;
import org.eaglei.model.EIURI;
import org.eaglei.model.gwt.rpc.ClientModelManager;
import org.eaglei.model.gwt.rpc.ClientModelManager.ClassDefinitionCallback;
import org.eaglei.model.gwt.rpc.ClientModelManager.SubClassCallback;
import org.eaglei.model.gwt.rpc.ClientModelManager.SuperClassesCallback;
import org.eaglei.model.gwt.rpc.ClientModelManager.TopLevelClassesCallback;
import org.eaglei.model.webapp.client.searchbar.TermSearchRequest;

import com.google.gwt.event.dom.client.ClickEvent;
import com.google.gwt.event.dom.client.ClickHandler;
import com.google.gwt.event.dom.client.MouseOutEvent;
import com.google.gwt.event.dom.client.MouseOutHandler;
import com.google.gwt.event.dom.client.MouseOverEvent;
import com.google.gwt.event.dom.client.MouseOverHandler;
import com.google.gwt.user.client.DOM;
import com.google.gwt.user.client.ui.Composite;
import com.google.gwt.user.client.ui.Image;
import com.google.gwt.user.client.ui.Label;
import com.google.gwt.user.client.ui.UIObject;
import com.google.gwt.user.client.ui.VerticalPanel;

public abstract class AbstractNavigatorPanel extends Composite {

    private static final int MAX_TOPICS = 5;
    private static final int MARGIN_INCREMENT = 6; // 
    private static final String STANDARD_TEXT_INDENT = "-10px"; //
    private static final String ARROW_TEXT_INDENT = "-14px"; // extra indent for <
    private final VerticalPanel outer = new VerticalPanel();
    private int currentMargin;
    private Image loadingImage = new Image("images/loading.gif");
    private EIClass currentClass;
    
    AbstractNavigatorPanel() {
        super();
        initWidget(outer);
    }
    
    private UIObject addLabel(EIClass c, HashMap<EIClass,UIObject> mapNoDefClassToUIObject) {
        UIObject ui = addLabel(c.getEntity().getLabel(), c.isInferred());
        setClassDefinitionTooltip(c, ui, mapNoDefClassToUIObject);
        return ui;
    }
    
    private UIObject addLabel(String labelString, boolean isInferred) {
        Label l = new Label(labelString);
        DOM.setStyleAttribute(l.getElement(), "textIndent", STANDARD_TEXT_INDENT);
        DOM.setStyleAttribute(l.getElement(), "marginLeft", currentMargin+"px");
        DOM.setStyleAttribute(l.getElement(), "fontWeight", "bold");
        if (isInferred) {
            DOM.setStyleAttribute(l.getElement(), "fontStyle", "italic");
        }
        outer.add(l);
        return l;
    }

    /*
     * Adds a link for an ancestor class of the search request type binding
     */
    private UIObject addAncestorLink(EIClass c, HashMap<EIClass,UIObject> mapNoDefClassToUIObject) {
        UIObject ui = addLink(c, true, false, mapNoDefClassToUIObject);
        return ui;
    }
    
    /*
     * Adds a link for a subclass of the search request type binding
     */
    private UIObject addSubClassLink(EIClass c, HashMap<EIClass,UIObject> mapNoDefClassToUIObject) {
        UIObject ui = addLink(c, false, c.hasSubClass(), mapNoDefClassToUIObject);
        return ui;
    }
    
    /*
     * Adds a link for a class
     */
    private UIObject addLink(EIClass c, boolean displayArrow, boolean bold, HashMap<EIClass,UIObject> mapNoDefClassToUIObject) {
        UIObject ui = addLink(c.getEntity().getLabel(), c.getEntity().getURI(), c.isInferred(), displayArrow, bold);
        setClassDefinitionTooltip(c, ui, mapNoDefClassToUIObject);
        return ui;
    }
    
    private Label addLink(final String labelString, final EIURI typeBindingURI, boolean isInferred, boolean displayArrow, boolean bold) {
        final Label l;
        if (displayArrow) {
            l = new Label("< "+labelString);
            DOM.setStyleAttribute(l.getElement(), "textIndent", ARROW_TEXT_INDENT);
        } else {
            l = new Label(labelString);
            DOM.setStyleAttribute(l.getElement(), "textIndent", STANDARD_TEXT_INDENT);
        }
        DOM.setStyleAttribute(l.getElement(), "marginLeft", currentMargin+"px");
        if (bold) {
            DOM.setStyleAttribute(l.getElement(), "fontWeight", "bold");
        }
        l.setStyleName("link");
        if (isInferred) {
        	l.addStyleDependentName("inferred");
        }
        l.addMouseOverHandler(new MouseOverHandler() {
            
            @Override
            public void onMouseOver(MouseOverEvent event) {
                l.addStyleDependentName("hovering");
            }
            
        });
        l.addMouseOutHandler(new MouseOutHandler() {
            
            @Override
            public void onMouseOut(MouseOutEvent event) {
                l.removeStyleDependentName("hovering");
            }
            
        });
        l.addClickHandler(new ClickHandler() {
            
            @Override
            public void onClick(ClickEvent event) {
                TermSearchRequest newRequest = new TermSearchRequest();
                 if (typeBindingURI != null) {
                    newRequest.setBinding(new TermSearchRequest.TypeBinding(typeBindingURI));
                }
                Application.executeSearch(newRequest);
            }
        });
        outer.add(l);
        return l;
    }

    public void setClass(EIClass clazz) {
        if (clazz == null && currentClass == null && outer.getWidgetCount() > 0) {
            return; // no-op, we're already displaying the list of base classes
        }
        currentClass = clazz;
        outer.clear();
        currentMargin = 18;
        if (clazz == null) {
            // No type binding, display top-level resources
            //addLabel("Any Resource");
            addLoading();
            getRootClasses();
        } else {
            addLink(getAllLinkLabel(), null, false, true, false);
            setSuperClasses(clazz);
        }
    }
    
    protected abstract String getAllLinkLabel();
    
    /**
     * When called, subclass is expected to call setRootClasses
     * asynchronously with a list of root EIClasses for this panel.
     */
    protected abstract void getRootClasses();
    
    protected void setRootClasses(List<EIClass> result) {
        if (currentClass != null || !(outer.getWidget(outer.getWidgetCount()-1) == loadingImage)) {
            // Check that UI is still waiting for the list of root classes
            return;
        }
        removeLoading();
        currentMargin += MARGIN_INCREMENT;
        HashMap<EIClass,UIObject> mapNoDefClassToUIObject = new HashMap<EIClass,UIObject>();
        for (EIClass top : result) {
            addSubClassLink(top, mapNoDefClassToUIObject);
        }
        getClassDefinitions(mapNoDefClassToUIObject);
    }

    private void setSuperClasses(EIClass c) {
        if (!c.hasSuperClass()) {
            // Top-level class, no supers
            currentMargin += MARGIN_INCREMENT;
            HashMap<EIClass,UIObject> mapNoDefClassToUIObject = new HashMap<EIClass,UIObject>();
            addLabel(c, mapNoDefClassToUIObject);
            setSubClasses(c);
            getClassDefinitions(mapNoDefClassToUIObject);
        } else {
            addLoading();
            ClientModelManager.INSTANCE.getSuperClasses(c, new SuperClassesCallback() {

                @Override
                public void onSuccess(EIClass result) {
                    if (!result.equals(currentClass) || !(outer.getWidget(outer.getWidgetCount()-1) == loadingImage)) {
                        // Check that UI is still waiting for this class.
                        return;
                    }
                    removeLoading();
                    ArrayList<EIClass> listAncestors = new ArrayList<EIClass>();
                    EIClass parent = result.getSuperClass();
                    while (parent != null) {
                        listAncestors.add(parent);
                        parent = parent.getSuperClass();
                    }
                    HashMap<EIClass,UIObject> mapNoDefClassToUIObject = new HashMap<EIClass,UIObject>();
                    for (int i = listAncestors.size() - 1; i >= 0; i--) {
                        // Supers
                        currentMargin += MARGIN_INCREMENT;
                        addAncestorLink(listAncestors.get(i), mapNoDefClassToUIObject);
                    }
                    // The class
                    currentMargin += MARGIN_INCREMENT;
                    addLabel(result, mapNoDefClassToUIObject);
                    setSubClasses(result);
                    getClassDefinitions(mapNoDefClassToUIObject);
                }

            });
        }
    }

    private void setSubClasses(final EIClass c) {
        if (!c.hasSubClass()) {
            // No subclasses
        } else {
            addLoading();
            ClientModelManager.INSTANCE.getSubClasses(c, new SubClassCallback() {

                @Override
                public void onSuccess(EIClass result) {
                    if (!result.equals(currentClass) || !(outer.getWidget(outer.getWidgetCount()-1) == loadingImage)) {
                        // Check that UI is still waiting for this class.
                        return;
                    }
                    removeLoading();
                    // Subclasses
                    currentMargin += MARGIN_INCREMENT;
                    HashMap<EIClass,UIObject> mapNoDefClassToUIObject = new HashMap<EIClass,UIObject>();
                    for (EIClass child : result.getSubClasses()) {
                        addSubClassLink(child, mapNoDefClassToUIObject);
                    }
                    getClassDefinitions(mapNoDefClassToUIObject);
                }

            });
        }
    }
    
    /*
     * 
     */
    private void setClassDefinitionTooltip(EIClass clazz, UIObject ui, HashMap<EIClass,UIObject> mapNoDefClassToUIObject) {
        // Set tooltip with the definition.
        // If definition is not in cache, it will be fetched and
        // set async.
        if (clazz.getDefinition() == null) {
            mapNoDefClassToUIObject.put(clazz, ui);
        } else {
            ui.setTitle(clazz.getDefinition());
        }
    }

    private void getClassDefinitions(final HashMap<EIClass,UIObject> mapNoDefClassToUIObject) {
        // Asynchronously populate tooltips
        if (mapNoDefClassToUIObject.size() > 0) {
            final List<EIClass> listNoDefClasses = new ArrayList<EIClass>(mapNoDefClassToUIObject.keySet());
            ClientModelManager.INSTANCE.getClassDefinitions(listNoDefClasses, new ClassDefinitionCallback() {
                @Override
                public void onSuccess(List<EIClass> result) {
                    for (EIClass clazz : result) {
                        UIObject o = mapNoDefClassToUIObject.get(clazz);
                        o.setTitle(clazz.getDefinition());
                    }
                }
            });
        }            
    }

    private void addLoading() {
        outer.add(loadingImage);
        DOM.setStyleAttribute(loadingImage.getElement(), "marginLeft", currentMargin+"px");
    }
    
    private void removeLoading() {
        outer.remove(outer.getWidgetCount()-1);        
    }
}
