package org.eaglei.network.actions;

import static java.util.Collections.unmodifiableSet;
import static org.spin.tools.Util.asSet;

import java.util.Collection;
import java.util.Map;
import java.util.Set;

import org.eaglei.network.driver.Query;
import org.spin.node.DestroyableQueryActionMap;
import org.spin.node.UnknownQueryTypeException;
import org.spin.node.actions.QueryAction;
import org.spin.tools.ClassTools;
import org.spin.tools.DynamicLoadingException;
import org.spin.tools.Plugins;
import org.spin.tools.Util;

/**
 * 
 * @author Clint Gilbert
 * 
 * Nov 22, 2010
 *
 * Center for Biomedical Informatics (CBMI)
 * @link https://cbmi.med.harvard.edu/
 *
 */
public final class EagleIQueryActionMap extends DestroyableQueryActionMap
{
    private final Set<String> knownQueryTypes = unmodifiableSet(asSet(EagleIQueryActions.queryTypes));

    private final Map<Query, QueryAction<?>> queryTypesToActions = Util.makeEnumMap(Query.class);
    
    public EagleIQueryActionMap()
    {
        super();
    }

    /**
     * @param queryType
     * @return true if queryType is a known EagleI query type.  Query types are given by the
     * queryType fields of the org.eaglei.network.driver.Query enum constants.
     */
    @Override
    public boolean containsQueryType(final String queryType)
    {
        return knownQueryTypes.contains(queryType);
    }

    /**
     * @param queryType
     * @return a cached QueryAction instance corresponding to the passed queryType, or null if there's an error instantiating one.
     * (The correspondence between query types and implementing classes is set out in the EagleIQueryActions enum.) 
     * @throws UnknownQueryTypeException if an unknown queryType (as determined by containsQueryType()) is passed in.
     */
    @Override
    public QueryAction<?> getQueryAction(final String queryType) throws UnknownQueryTypeException
    {
        guardContainsQueryType(queryType);
        
        final QueryAction<?> action = queryTypesToActions.get(Query.fromQueryType(queryType));
        
        if(action != null)
        {
            return action;
        }
        
        final EagleIQueryActions queryActionDef = makeQueryActionDef(queryType);
        
        return createAndCacheQueryAction(queryActionDef);
    }

    private QueryAction<?> createAndCacheQueryAction(final EagleIQueryActions queryActionDef)
    {
        try
        {
            final QueryAction<?> newAction = createQueryAction(queryActionDef);
            
            queryTypesToActions.put(queryActionDef.query(), newAction);
            
            return newAction;
        }
        catch(DynamicLoadingException e)
        {
            throw new QueryActionInstantiationException("Couldn't instantiate query with type '" + queryActionDef.query().queryType + "' (tried to instantiate a " + queryActionDef.queryActionClass().getName() + ")", e);
        }
    }

    void guardContainsQueryType(final String queryType) throws UnknownQueryTypeException
    {
        if(!containsQueryType(queryType))
        {
            throw new UnknownQueryTypeException("Unknown query type '" + queryType + "': known values are " + knownQueryTypes);
        }
    }

    private EagleIQueryActions makeQueryActionDef(final String queryType)
    {
        return EagleIQueryActions.fromQueryType(queryType);
    }

    private QueryAction<?> createQueryAction(final EagleIQueryActions queryActionDef) throws DynamicLoadingException
    {
        return createQueryAction(queryActionDef.queryActionClass().getName());
    }

    /**
     * @return an unmodifiable collection of known query types
     */
    @Override
    public Collection<String> getQueryTypes()
    {
        return knownQueryTypes;
    }
    
    /**
     * Instantiates a QueryAction given a class name, ensuring that the plugin
     * ClassLoader is used.
     * 
     * @param queryClassName
     * @return
     * @throws DynamicLoadingException
     */
    //TODO: replace this with a call to QueryActionCreator.createQueryAction once that method is public in Spin 1.15
    static QueryAction<?> createQueryAction(final String queryClassName) throws DynamicLoadingException
    {
        try
        {
            return ClassTools.createInstance(queryClassName, QueryAction.class, Plugins.getPluginClassLoader());
        }
        catch(final Exception e)
        {
            throw new DynamicLoadingException(e);
        }
    }
}
