package org.eaglei.network.tools;

import java.io.File;
import java.util.Arrays;
import java.util.List;

import org.eaglei.network.actions.EagleIQueryActions;
import org.eaglei.network.actions.NoopQueryAction;
import org.eaglei.network.driver.Constants;
import org.spin.tools.Durations;
import org.spin.tools.JAXBUtils;
import org.spin.tools.config.EndpointConfig;
import org.spin.tools.config.EndpointType;
import org.spin.tools.config.Module;
import org.spin.tools.config.NodeConfig;
import org.spin.tools.config.PeerGroupConfig;
import org.spin.tools.config.QueryTypeConfig;
import org.spin.tools.config.RoutingTableConfig;

/**
 * 
 * @author Clint Gilbert
 * 
 * Feb 8, 2011
 *
 * Center for Biomedical Informatics (CBMI)
 * @link https://cbmi.med.harvard.edu/
 *
 */
public final class ConfigGenerator
{
    private ConfigGenerator()
    {
    }
    
    static enum NodeType
    {
        Institutional
        {
            public NodeConfig mungeNodeConfig(final NodeConfig original)
            {
                NodeConfig munged = original.withIsAggregator(false)
                                            .withIsBroadcaster(false)
                                            .withIsQueryable(true);
                
                for(final EagleIQueryActions queryActionDef : EagleIQueryActions.values())
                {
                    munged = munged.withQuery(queryActionDef.toQueryTypeConfig());
                }
                
                return munged;
            }
        }, 
        Central
        {
            public NodeConfig mungeNodeConfig(final NodeConfig original)
            {
                NodeConfig munged = original.withIsAggregator(true)
                                            .withIsBroadcaster(true)
                                            .withIsQueryable(true);
                
                for(final EagleIQueryActions queryActionDef : EagleIQueryActions.values())
                {
                    munged = munged.withQuery(new QueryTypeConfig(queryActionDef.query().queryType, NoopQueryAction.class.getName()));
                }
                     
                return munged;
            }
        };
        
        public abstract NodeConfig mungeNodeConfig(final NodeConfig original);
        
        public final boolean isCentral()
        {
            return this == Central;
        }
        
        public final boolean isInstitutional()
        {
            return this == Institutional;
        }
    }
    
    public static void main(String[] args) throws Exception
    {
        if(!checkUsage(args))
        {
            System.err.println("Usage: ");
            System.err.println("\tConfigGenerator Institutional <parent_url>");
            System.err.println("or");
            System.err.println("\tConfigGenerator Central <child_url1> <child_url2> ... <child_urlN>");
            
            return;
        }
        
        final NodeType nodeType;
        
        try
        {
            nodeType= NodeType.valueOf(args[0]);
        }
        catch(Exception e)
        {
            System.err.println("Unknown node type: allowed values are " + NodeType.values());
            
            return;
        }
        
        {
            final NodeConfig eagleIConfig = makeEagleINodeConfig(nodeType);
            
            final File nodeXMLFile = new File(Module.node.name() + ".xml");
            
            JAXBUtils.marshalToFile(eagleIConfig, nodeXMLFile);
            
            System.out.println("Node config written to " + nodeXMLFile.getCanonicalPath());
        }
        
        {
            final List<String> urls = Arrays.asList(Arrays.copyOfRange(args, 1, args.length));
            
            final PeerGroupConfig peerGroup = makePeerGroup(urls, nodeType);
            
            final File routingTableXMLFile = new File(Module.routingtable.name() + ".xml");
            
            JAXBUtils.marshalToFile(new RoutingTableConfig().withPeerGroup(peerGroup), routingTableXMLFile);
            
            System.out.println("Routing table config written to " + routingTableXMLFile.getCanonicalPath());
        }
    }

    static PeerGroupConfig makePeerGroup(List<String> urls, final NodeType nodeType)
    {
        PeerGroupConfig peerGroup = new PeerGroupConfig(Constants.DefaultPeerGroup);
        
        if(nodeType.isCentral())
        {
            peerGroup = peerGroup.withParent(null);
            
            
            
            for(final String childURL : urls)
            {
                peerGroup = peerGroup.withChild(soap(childURL));
            }
        }
        else if(nodeType.isInstitutional())
        {
            peerGroup = peerGroup.withParent(soap(urls.get(0)));
        }
        
        return peerGroup;
    }

    static boolean checkUsage(final String[] args)
    {
        return args.length >= 2;
    }

    static EndpointConfig soap(final String url)
    {
        return new EndpointConfig(EndpointType.SOAP, url);
    }

    static NodeConfig makeEagleINodeConfig(final NodeType nodeType)
    {
        return nodeType.mungeNodeConfig(NodeConfig.Default).withCacheTTL(30 * Durations.InMilliseconds.oneMinute)
                                                           .withIsAuthenticator(false);
    }
}
