#!/bin/sh

# Usage:  make-snapshot.sh  username  password  repo-URL  directory
#
# Command-line parameters:
#  1. username - username with which to authenticate to the repo
#  2. password - password with which to authenticate to the repo
#  3. repo-URL  - prefix of repository URL, e.g. "https://localhost/"
#  4. directory - directory in which to write the dump, will be created if necessary

# Creates a "snapshot" backup of a repository's RDF data content and
# user accounts, in a designated directory.
#
# Upon success, the directory will contain two files:
#   resources.trig  --  RDF resource data in TriG format, read by /graph
#   users.trig  --  user accounts, must be read by /import service
#
# Upon failure, it prints an explanatory messaeg and returns non-0 status.
# NO MESSAGE is printed upon success, which lets it run under cron.
# For example, your crontab might invoke this command to write a daily snapshot
# in a differently-named directory each day, rotating through a week:
#
#   make-snapshot.sh ADMIN PASSWORD http://localhost/ "daily_cron_`date +%u`"
#
# --------------
#
# To restore this dump on a newly-created, empty, repository, use these
# commands: (note login, DUMPDIR and REPOSITORY are as supplied to the
# snapshot command that created the dump)
#
# curl -D - -s -u ADMIN:PASSWORD -F type=user -F format=application/x-trig \
#   -F content=@${DUMPDIR}/users.trig -F duplicate=replace \
#   -F transform=no ${REPOSITORY}/repository/import
#
# curl -s -D - -u ADMIN:PASSWORD -F action=replace -F all= \
#   -F "content=@${DUMPDIR}/resources.trig;type=application/x-trig" \
#   ${REPOSITORY}/repository/graph
#
# Larry Stone - 22-Sep-10

usage="Usage:  $0   username  password  repo-URL  directory"

#--------- constants
resourceDump=resources.trig
userDump=users.trig
format=application/x-trig

#-------- command line
username=$1
password=$2
repo=$3
target=$4

if [ $# -lt 4 ]; then
    echo $usage
    exit 1
fi

# curl login arg
login="${username}:$password"

# sanity-check target dir
if [ -e "$target" ]; then
    if [ -d "$target" ]; then
        if [ ! -w "$target"  ]; then
            echo "No write permission to target directory: $target"
            exit 1
        fi
    else
        echo "Target must be a directory: $target"
        exit 2
    fi
else
    if mkdir -p "$target"; then
        true
    else
        echo "Failed to create target directory: $target"
        exit 3
    fi
fi

#------------- procedure

# clean out old files if they are in the way
test -f "${target}/$resourceDump" && rm -f "${target}/$resourceDump"
test -f "${target}/$userDump" && rm -f "${target}/$userDump"

# dump resources
status=`curl -s -G -u "${login}" -o "${target}/$resourceDump" -d all \
 --write-out '%{http_code}' -d "format=${format}" "${repo}/repository/graph"`
cmd=$?
if [ $cmd -ne 0 ]; then
    echo "curl failed to send HTTP request for resource dump, URL=${repo}, status=${cmd}"
    exit 5
fi
if [ "$status" != "200" ]; then
    echo "HTTP Request to get resource dump failed, status=${status}"
    if [ -f "${target}/$resourceDump" ]; then
        echo "Response document:"
        cat "${target}/$resourceDump"
        rm -f "${target}/$resourceDump"
    fi
    exit 6
fi

# dump users
status=`curl -s -G -u "${login}" -o "${target}/$userDump" -d type=user \
 --write-out '%{http_code}' -d "format=${format}" "${repo}/repository/export"`
cmd=$?
if [ $cmd -ne 0 ]; then
    echo "curl failed to send HTTP request for user account dump, URL=${repo}, status=${cmd}"
    exit 7
fi
if [ "$status" != "200" ]; then
    echo "HTTP Request to get user account dump failed, status=${status}"
    if [ -f "${target}/$userDump" ]; then
        echo "Response document:"
        cat "${target}/$userDump"
        rm -f "${target}/$userDump"
    fi
    exit 8
fi

if [ ! -f "${target}/$resourceDump" ]; then
    echo "No resource dump file found, something failed: ${target}/$resourceDump"
    exit 9
fi
if [ ! -f "${target}/$userDump" ]; then
    echo "No user account dump file found, something failed: ${target}/$userDump"
    exit 10
fi
