#!/bin/bash

# TEMPORARY HACKED MS4-only COPY OF
#  Repository Upgrade Script
#
# THIS IS ***ONLY*** for ***TEMPORARY*** use during the dev time
# to upgrade MS4 nodes which were FRESH-INSTALLED and then DATA COPIED
# from prod..  It ONLY upgrades USERS.
#
# Run this after installing a new release of the repository webapp
# *ONLY* WHEN PRESERVING THE SESAME RDF DATABASE CONTENTS OF AN OLDER RELEASE.
# It will automatically upgrade the repository contents.
#
# See "usage" variable below for synopsis of arguments.
#
# Note: All diagnostic output is prefaced with either INFO, WARNING, or ERROR
#  so a script running this can simply grep for WARNING|ERROR to see
#  if there are any messages worth forwarding to a human.
#
# $Id: upgrade.sh 5984 2010-12-07 01:50:31Z lcs14 $
# Author: Larry Stone

#------- Constants
usage="Usage: $0  username  password  repo-URL-root, e.g. $0  bigbird passwird https://repo.mysite.edu:8443"

# useful namespace prefixs
repo="http://eagle-i.org/ont/repo/1.0/"
owl="http://www.w3.org/2002/07/owl#"
rdfs="http://www.w3.org/2000/01/rdf-schema#"

# n-triples mime type
format=text/plain

# named graphs
NG_Internal=${repo}NG_Internal
NG_Query=${repo}NG_Query

# data model ontology graph URI:
dataModel=http://purl.obolibrary.org/obo/ero.owl


#-------- boilerplate
# keep files in /tmp subdir, cleanup on exit
tmpDir="/tmp/repo-upgrade-$$"
mkdir $tmpDir

cleanup () {
  ## XXX insert echo here to retain the tmp work dir
  rm -rf $tmpDir
}
trap cleanup EXIT


# ------------------ Function definitions ---------------------------------

# REturns the useful data from the /model query
# line contains 3 quoted strings: loaded, jar, and EI-model versions
getDataModelVersions () {
    local result="${tmpDir}/model"
    rm -f $result

    local status=`curl -k -s -S -G -u "$login" -o "$result" \
      -d "format=text/plain" --write-out "%{http_code}" ${repoURL}/repository/model`
    local cmd=$?
    if [ $cmd -ne 0 -o "$status" != "200" ]; then
        echo "/model: HTTP request failed: HTTP status=$status cmd=$cmd " 1>&2
        test -f $result && cat $result 1>&2
        echo "" 1>&2
        return 11
    fi
    test -f "$result" && tail -n +2 "$result"
    return 0
}

# Usage:  N  line   -- where N is 1-3, line is returned by /model
# Also cleans up residual "'s at start or end.
selectModelVersion () {
    local n=$1
    echo "$2" | awk -F '"[ \t]+"' "{print \$${n}}" | sed -e 's/^"//' -e 's/"$//'
}


# Usage:  $0  versionA  versionB
# compare two Data Model Ontology version strings, success if the first
# is "newer" (later) than the second.
# Each version string has the following internal format:
#    <maven-version>   <SVN-revision>  <date>  <time>
# The maven version is REQUIRED, the otehrs may be elided.
# So, this only compares maven versions.  Note that 0.6.1.3 is later
# than 0.6.1 in this format.
modelVersionNewer () {
    local a=`echo "$1" | awk '{print $1}'`
    local b=`echo "$2" | awk '{print $1}'`

    # if destination isn't a real version, and proposed one is, say it's newer
    # since destination may be an empty graph or something like that.
    if echo "$a"|grep -q '^[0-9]'; then
        if echo "$b"|grep -q '^[0-9]'; then
            mavenVersionLessThan $b $a
        else
            return 0
        fi
    else
        echo "ERROR: Cannot compare anything to target version \"${a}\""
        return 1
    fi
}

# compare two version strings of the form  0.4.9 and 0.4.3
# NOTE: This ASSUMES it is given conforming NUMERIC Maven version strings
#  so be sure there is no "SNAPSHOT" or other junk in there.
mavenVersionLessThan () {
    #echo "DEBUG: comparing \"${1}\" to \"${2}\" "
    local a=(`echo "$1" | tr '.' ' '`)
    local b=(`echo "$2" | tr '.' ' '`)
    for i in 0 1 2 3 4; do
        #echo "DEBUG: a[${i}]=\"${a[$i]}\", b[${i}]=\"${b[$i]}\""
        if [ -z "${b[$i]}" -a -n "${a[$i]}" ]; then
            return 1
        elif [ -z "${a[$i]}" -a -n "${b[$i]}" ]; then
            return 0
        # equal is not less than
        elif [ -z "${a[$i]}" -a -z "${b[$i]}" ]; then
            return 1
        elif [ "${a[$i]}" -gt "${b[$i]}" ]; then
            return 1
        elif [ "${a[$i]}" -lt "${b[$i]}" ]; then
            return 0
        fi
    done
    # they are equal
    return 1
}

# Usage: $0  jar|model
loadDataModelOntology () {
    local source=$1
    local result="${tmpDir}/model2"
    rm -f $result

    echo "INFO: Loading Data Model Ontology from $source"
    local status=""
    if status=`curl -k -s -S -u "$login" -o "$result" \
          -d action=load -d "source=$source" \
          --write-out '%{http_code}' "${repoURL}/repository/model"` ; then
        if [ "$status" = 200 -o "$status" = 201 ]; then
            echo "INFO: Done loading Data Model Ontology, HTTP status=$status"
        else
            echo "ERROR: Failed to load Data Model Ontology, HTTP status=$status"
            test -f $result && cat $result 1>&2
            commandStatus=1
        fi
    else
        echo "ERROR: Failed sendign HTTP request to load data model ontology, see output above."
        commandStatus=1
    fi
}

# usage:  view  query
#  returns text/plain results (ntriples for construct, TSV for select)
doQuery () {
    local view=$1
    local query=$2

    local expectStatus="200"
    local result="${tmpDir}/sparql"
    rm -f $result

    local status=`curl -k -s -S -u "$login" -o "$result" -F "format=$format"  \
      -F "query=$query" -F "view=$view" \
      --write-out "%{http_code}" ${repoURL}/repository/sparql`
    local cmd=$?
    if [ $cmd -ne 0 -o "$status" != "$expectStatus" ]; then
        echo "doQuery: SPARQL Protocol request failed: HTTP status=$status cmd=$cmd " 1>&2
        test -f $result && cat $result 1>&2
        echo "" 1>&2
        return 1
    elif [ "$expectStatus" != 200 ]; then
        return 0
    fi
    # skip first line if select??
    if [ -f "$result" ]; then
        if echo "$query"|grep -qi "^SELECT"; then
            tail -n +2 "$result"
        else
            cat "$result"
        fi
    fi
    return 0
}

# usage:  graph-name  action  contents
#  uploads contents to named graph, assume format=$format
doGraph () {
    local name=$1
    local action=$2
    local content=$3
    local expectStatus="200"

    local status=`curl -k -s -S -u "$login" -o "$result" -F "format=$format"  \
      -F "name=$name" -F "action=$action" --form-string "content=$content" \
      --write-out "%{http_code}" ${repoURL}/repository/graph`
    local cmd=$?
    if [ $cmd -ne 0 -o "$status" != "$expectStatus" ]; then
        echo "doGraph: SPARQL Protocol request failed: HTTP status=$status cmd=$cmd " 1>&2
        test -f $result && cat $result 1>&2
        return 1
    elif [ "$expectStatus" != 200 ]; then
        return 0
    fi
}

#---------------------------------------------------------------------------
# Compare loaded version of data model ontology against the version
# available in the webapp's jar -- load the one from the jar if newer,
# or if none appears to be loaded.
upgradeDataModelOntology () {
    # this will have versions 1 = loaded, 2 = jar, 3 = eiModel
    local versions=""
    if versions=`getDataModelVersions`; then
        local loaded=`selectModelVersion 1 "$versions"`
        local jar=`selectModelVersion 2 "$versions"`
        if echo "$loaded"|grep -q "not loaded"||modelVersionNewer "$jar" "$loaded"; then
            echo "INFO: Loading new Data Model Ontology from internal jar, version: $jar"
            if loadDataModelOntology jar; then
                true
            else
                echo "ERROR: Failed loading data model ontology."
                commandStatus=1
            fi
        else
            echo "INFO: Data Model Ontology is already up to date, version: $loaded"
        fi
        return 0
    else
        echo "ERROR: Failed getting data model ontology versions."
        commandStatus=1
        return 1
    fi
}

#---------------------------------------------------------------------------
#   Special for 1.1MS4.00: transform user account metadata
#     a. remove Superuser (Admin) role from users who have other roles
#     b. add implicit Anonymous and Authenticated roles to all users if needed
#       (this will be done automatically by the Admin UI anyway)
# NOTE: Even though this is a one-off task only required for the 1.1MS2->MS4
# upgrade, keep this code around to serve as an example of how to ues the
# Edit User JSP as a REST service.
special_1_1MS4 () {

    # ------- Part a. Remoev Superuser role from users who don't need it
    # Get list of usernames of such users; for each one get its roles
    # and invoke the Admin UI Edit User page to change them.  This is
    # necessary because removing the Superuser role also changes the RDBMS
    # and only that JSP (or the import-users service) can do both.
    local badAdminQuery="SELECT DISTINCT ?n WHERE
     {?s a <${repo}Person>; <${repo}hasPrincipalName> ?n;
      <${repo}hasRole> <${repo}Role_Superuser> ; <${repo}hasRole> ?r
      FILTER(!(?r = <${repo}Role_Superuser> ||
               ?r = <${repo}Role_Authenticated> ||
               ?r = <${repo}Role_Anonymous>))}"

    local count=0
    local userCount=0
    if fixUsers=`doQuery all "$badAdminQuery"` ; then
        if [ -z "$fixUsers" ]; then
            echo "INFO: All unecessary Superuser roles have already been revoked."
        else
            for rawUser in $fixUsers; do
                userCount=`expr "$userCount" + 1`
                user=`echo "$rawUser" | tr -d '"'`
                # Skip yourself, SHOULD NOT be necessary, but you never know..
                if [ "$user" = "$adminUser" ]; then
                    echo "WARNING: Skipping admin user $user - but IT SHOULD NOT HAVE OTHER ROLES!"
                else
                    # get all non-admin roles for a user
                    getUserRolesQuery="SELECT DISTINCT ?r WHERE
                     {?s <${repo}hasPrincipalName> \"${user}\"; <${repo}hasRole> ?r
                      filter(?r != <${repo}Role_Superuser>)}"
                    if roles=`doQuery all "$getUserRolesQuery"`; then
                        args="-d standalone=true -d username=${user} -d submit=true "
                        for ruri in $roles; do
                            args="${args} --data-urlencode role=${ruri} "
                        done
                        #
                        # Sneaky invocation of Admin UI servlet to mung user,
                        # because ONLY it can turn off Superuser in Derby DB as well:
                        echo "INFO: Revoking Superuser role for user: $user"
                        if hs=`curl -k -s -S -u "$login" -o /dev/null $args --write-out "%{http_code}" \
                             ${repoURL}/repository/admin/editUserPortlet.jsp`; then
                            if [ "$hs" != '200' ]; then
                                echo "FAIL: Failed in Admin UI req to set Roles on user=${user}, status=${hs}, args=$args"
                                commandStatus=1
                            fi
                        else
                            echo "FAIL: HTTP request to Admin UI failed, user=$user"
                            commandStatus=1
                        fi
                    else
                        echo "FAIL: Failed to get roles for user=$user"
                        commandStatus=1
                    fi
                    count=`expr "$count" + 1`
                fi
            done
            echo "INFO: Revoked $count unecessary Superuser roles on $userCount users."
        fi
    else
        echo "FAIL: Failed to get users with Superuser roles to revoke."
        commandStatus=1
    fi

    # -------  Part b. Add explicit Anonymous and Authenticated roles to each user
    local addRolesQuery="CONSTRUCT {?s <${repo}hasRole> <${repo}Role_Anonymous>,
                                    <${repo}Role_Authenticated>} WHERE
     {GRAPH <${repo}NG_Users> {?s a <${repo}Person>}}"

    if addRoles=`doQuery all "$addRolesQuery"`; then
        if doGraph "${repo}NG_Internal" add "$addRoles"; then
            count=`echo "$addRoles" | grep -c 'Role_Authenticated'`
            echo "INFO: Added implicit roles to $count users."
        else
            echo "FAIL: Failed adding the implicit role statements, though they were generated."
        fi
    else
        echo "FAIL: Failed generating the implicit role statements to add."
    fi
}

# ------------------ Main Thread ---------------------------------

# Sanity-check args
if [ "$1" = '--version' ]; then
    echo "$0 from release ${project.version} SCM revision ${buildNumber}"
    exit 0

elif true; then

echo "      * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *"
echo "      * * * * *                                         * * * * *"
echo "      * * * * *     THIS SCRIPT HAS BEEN DEPRECATED     * * * * *"
echo "      * * * * *       Use the real upgrade.sh now!      * * * * *"
echo "      * * * * *                                         * * * * *"
echo "      * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *"
exit 13

elif [ $# -lt 3 ]; then
    echo "ERROR: $usage"
    exit 1
fi



adminUser=$1
login="${1}:$2"
repoURL=$3

if echo "$repoURL"|egrep -qv '^https?:'; then
    echo "ERROR: Repository URL must be a valid http(s) URL: \"$repoURL\""
    exit 1
fi

commandStatus=0

# ------------ Upgrade Procedure Steps:

echo ""
echo "***** Checking and Upgrading Data Model Ontology"
echo ""
upgradeDataModelOntology


echo ""
echo "***** Upgrading User Account Metadata for 1.1MS4.00"
echo ""
special_1_1MS4


echo ""
echo "***** TODO: Upgrading Repo Ontology + Internal Metadata for 1.1MS4.00"
echo ""

exit $commandStatus
