#!/bin/bash

# Repository Upgrade Script
#
# Run this after installing a new release of the repository webapp
# *ONLY* WHEN PRESERVING THE SESAME RDF DATABASE CONTENTS OF AN OLDER RELEASE.
# It will automatically upgrade the repository contents.
#
# Any data migration operations include a test for whether there is anything
# needing migration, which means you can run this script again and it will
# simply skip the migration steps which have already successfully completed.
#
# See "usage" variable below for synopsis of arguments.
#
# Note: All diagnostic output is prefaced with either INFO, WARNING, or ERROR
#  so a script running this can simply grep for WARNING|ERROR to see
#  if there are any messages worth forwarding to a human.
#
# $Id: upgrade.sh 8369 2011-03-30 15:39:38Z rd67 $
# Author: Larry Stone

#------- Constants
usage="Usage: $0  username  password  repo-URL-root, e.g. $0  bigbird passwird https://repo.mysite.edu:8443"

# useful namespace prefixs
repo="http://eagle-i.org/ont/repo/1.0/"
owl="http://www.w3.org/2002/07/owl#"
rdfs="http://www.w3.org/2000/01/rdf-schema#"

# delete wildcard
wildcard="${repo}MatchAnything"

# named graphs
NG_Repo="${repo}"
NG_Internal=${repo}NG_Internal
NG_Metadata=${repo}NG_Metadata
NG_Query=${repo}NG_Query

# data model ontology graph URI:
dataModel=http://purl.obolibrary.org/obo/ero.owl

inClassGroup="http://eagle-i.org/ont/app/1.0/inClassGroup"
embeddedClass="http://eagle-i.org/ont/app/1.0/ClassGroup_embedded_class"

# troublesome inverse predicate
isPartConstruct="http://eagle-i.org/ont/data/is_part_construct"


#-------- boilerplate
# keep files in /tmp subdir, cleanup on exit
tmpDir="/tmp/repo-upgrade-$$"
mkdir $tmpDir

cleanup () {
  ## XXX insert echo here to retain the tmp work dir
  rm -rf $tmpDir
}
trap cleanup EXIT

Debug=false

# ------------------ Function definitions ---------------------------------

# return a new URI via stdout
# usage: $0 [ optional-suffix ]
newURI () {
    local suffix="$1"
    local status
    local uriFile="${tmpDir}/uri"
    rm -f $uriFile
    if status=`curl -k -s -u $login -o "$uriFile" -d format=text/plain --write-out "%{http_code}" ${repoURL}/repository/new`; then
        if [ "$status" != "200" ]; then
            echo "newURI FAILED to get new URI, HTTP status=$status" 1>&2
            test -f $uriFile && cat $uriFile 1>&2
            return 1
        fi
    else
        echo "newURI FAILED sending HTTP request (curl) to get new URI: shell status=$?" 1>&2
        return 1
    fi
    local result=`tail -1 < "$uriFile" | tr -d ' \r\n\t'`
    if [ -n "$suffix" ]; then
        echo "${result}-$suffix"
    else
        echo "$result"
    fi
}


# run /update with insert, delete..
# args: LOGIN, action, subject, inserts, removes, workspace [expectStatus]
# NOTE: since this gets called inside a `` quote, so careful with output!
doUpdate () {
    local login=$1
    local action="$2"
    local subject="$3"
    local inserts="$4"
    local removes="$5"
    local wkspace="$6"
    local expectStatus="$7"
    local bypassSanity="$8"
    local status
    local insArg="insert"
    local delArg="delete"

    if [ -z "$expectStatus" ]; then
        if [ "$action" = "update" ]; then
            expectStatus="200"
        else
            expectStatus="201"
        fi
    fi

    local insaneArg="x_preserveSanity="
    if [ -n "$bypassSanity" ]; then
        insaneArg="bypassSanity="
    fi

    # disable functions by renaming http args
    if [ -z "$inserts" ]; then
        insArg="x_insert"
    fi
    if [ -z "$removes" ]; then
        delArg="x_delete"
    fi
    local out="${tmpDir}/out"
    rm -rf "$out"
    local token=""
    local tokenArg="foo=bar"
    if [ "$action" = "update" ]; then
        if token=`getToken "$login" "$subject"`;then
            tokenArg="token=$token"
        else
            echo "Failed to get token!"
            return 1
        fi
    fi

    local wkspaceArg="xworkspace="
    if [ -n "${wkspace}" ]; then
        wkspaceArg="workspace=${wkspace}"
    fi

    if status=`curl -k -o $out -s -u $login -F "action=${action}" \
     --form-string "${tokenArg}" --form-string 'format=text/rdf+n3' \
     --form-string "${wkspaceArg}" \
     --form-string "${insaneArg}" \
     --form-string "uri=$subject" \
     --form-string "${insArg}=${inserts}" --form-string "${delArg}=${removes}" \
     --write-out '%{http_code}' ${repoURL}/repository/update`; then
        if [ "$status" = "${expectStatus}" ]; then
            $Debug && echo "DEBUG: doUpdate OK." 1>&2
            if [ "$expectStatus" -gt 201 -a -f "$out" ]; then
                echo "OK - got expected status = $status"
                fgrep -A 1 '<b>Description</b>' $out
            fi
        else
            echo "doUpdate FAILED to update Instance=${subject}, HTTP status=$status" 1>&2
            test -f $out && cat $out 1>&2
            return 1
        fi
    else
        echo "DoUpdate  FAILED sending HTTP request (curl) to mung Instance=${subject}: shell status=$?" 1>&2
        return 1
    fi
    return 0
}

# Usage: $0 login uri
# get edit token, returns it through stdout
getToken () {
  local login="$1"
  local uri="$2"

  local tFile="${tmpDir}/token"
  rm -f $tFile
  local status
  if status=`curl -k -s -u $login -o "$tFile" -F action=gettoken -F 'format=text/plain' -F "uri=$uri" --write-out "%{http_code}" ${repoURL}/repository/update`; then
      if [ "$status" != "200" ]; then
        echo "getToken: Failed to get Token: HTTP status=$status " 1>&2
        test -f $tFile && cat $tFile 1>&2
        return 1
      fi
  else
      echo "getToken: Failed sending HTTP request to get token, status=$?" 1>&2
      return 1
  fi
  awk 'FNR ==2 { print $1 }'  "$tFile"
}

# Usage: doImport FILE TYPE DUPE TRANSFORM
doImport () {
    local afile=$1
    local atype=$2
    local dupe=$3
    local xform=$4

    local status=`curl -k -s -S -q -u "$login" --write-out "%{http_code}"  \
     -F type=${atype} -F "content=@${afile};type=application/x-trig" \
     -F duplicate=$dupe -F transform=$xform ${repoURL}/repository/import`
    cmd=$?
    if [ $cmd -ne 0 -o "$status" != "200" ]; then
      echo "FAILED ${atype} import: HTTP status=$status cmd=$cmd " 1>&2
      return 1
    fi
}


# REturns the useful data from the /model query
# line contains 3 quoted strings: loaded, jar, and EI-model versions
getDataModelVersions () {
    local result="${tmpDir}/model"
    rm -f $result

    local status=`curl -k -s -S -G -u "$login" -o "$result" \
      -d "format=text/plain" --write-out "%{http_code}" ${repoURL}/repository/model`
    local cmd=$?
    if [ $cmd -ne 0 -o "$status" != "200" ]; then
        echo "/model: HTTP request failed: HTTP status=$status cmd=$cmd " 1>&2
        test -f $result && cat $result 1>&2
        echo "" 1>&2
        return 11
    fi
    test -f "$result" && tail -n +2 "$result"
    return 0
}

# Usage:  N  line   -- where N is 1-3, line is returned by /model
# Also cleans up residual "'s at start or end.
selectModelVersion () {
    local n=$1
    echo "$2" | awk -F '"[ \t]+"' "{print \$${n}}" | sed -e 's/^"//' -e 's/"$//'
}

# Usage:  $0  versionA  versionB
# compare two Data Model Ontology version strings, success if the first
# is "newer" (later) than the second.
# Each version string has the following internal format:
#    <maven-version>   <SVN-revision>  <date>  <time>
# The maven version is REQUIRED, the otehrs may be elided.
# So, this only compares maven versions.  Note that 0.6.1.3 is later
# than 0.6.1 in this format.
modelVersionNewer () {
    local a=`echo "$1" | awk '{print $1}'`
    local b=`echo "$2" | awk '{print $1}'`

    # if destination isn't a real version, and proposed one is, say it's newer
    # since destination may be an empty graph or something like that.
    if echo "$a"|grep -q '^[0-9]'; then
        if echo "$b"|grep -q '^[0-9]'; then
            mavenVersionLessThan $b $a
        else
            return 0
        fi
    else
        echo "WARNING: Cannot compare anything to target version \"${a}\"" 1>&2
        return 1
    fi
}

# compare two version strings of the form  0.4.9 and 0.4.3
# NOTE: This ASSUMES it is given conforming NUMERIC Maven version strings
#  so be sure there is no "SNAPSHOT" or other junk in there.
mavenVersionLessThan () {
    #echo "DEBUG: comparing \"${1}\" to \"${2}\" "
    local a=(`echo "$1" | tr '.' ' '`)
    local b=(`echo "$2" | tr '.' ' '`)
    for i in 0 1 2 3 4; do
        #echo "DEBUG: a[${i}]=\"${a[$i]}\", b[${i}]=\"${b[$i]}\""
        if [ -z "${b[$i]}" -a -n "${a[$i]}" ]; then
            return 1
        elif [ -z "${a[$i]}" -a -n "${b[$i]}" ]; then
            return 0
        # equal is not less than
        elif [ -z "${a[$i]}" -a -z "${b[$i]}" ]; then
            return 1
        elif [ "${a[$i]}" -gt "${b[$i]}" ]; then
            return 1
        elif [ "${a[$i]}" -lt "${b[$i]}" ]; then
            return 0
        fi
    done
    # they are equal
    return 1
}

# Usage: $0  jar|model
loadDataModelOntology () {
    local source=$1
    local result="${tmpDir}/model2"
    rm -f $result

    echo "INFO: Loading Data Model Ontology from $source"
    local status=""
    if status=`curl -k -s -S -u "$login" -o "$result" \
          -d action=load -d "source=$source" \
          --write-out '%{http_code}' "${repoURL}/repository/model"` ; then
        if [ "$status" = 200 -o "$status" = 201 ]; then
            echo "INFO: Done loading Data Model Ontology, HTTP status=$status"
        else
            echo "ERROR: Failed to load Data Model Ontology, HTTP status=$status"
            test -f $result && cat $result
            commandStatus=1
        fi
    else
        echo "ERROR: Failed sendign HTTP request to load data model ontology, see output above."
        commandStatus=1
    fi
}

# Usage: $0  uri
# Returns the useful data from the /internal service
# line contains URI and 2 quoted strings: loaded, and available (jar) versions
getInternalVersions () {
    local result="${tmpDir}/internal"
    rm -f $result

    if doInternal "$result" "${1}" ; then
        test -f "$result" && tail -n +2 "$result"
    else
        return 1
    fi
    return 0
}

# returns all internal graphs
# get first column of all retunred rows
getInternalGraphs () {
    local result="${tmpDir}/internal"
    rm -f $result

    if doInternal "$result" && test -f "$result" ; then
        awk "-F\t" 'NR > 1 {print $1}' < "$result" | tr '\n' ' '
    else
        return 1
    fi
}


# Usage: $0  filepath  [ uri ]
# writes output of /internal into given file paht.
doInternal () {
    local result=$1
    local uriArg="x=y"
    if [ $# -gt 1 ]; then
        uriArg="name=$2"
    fi
    local status=`curl -k -s -S -G -u "$login" -o "$result" -d "format=text/plain" \
      -d $uriArg --write-out "%{http_code}" ${repoURL}/repository/internal`
    local cmd=$?
    if [ $cmd -ne 0 -o "$status" != "200" ]; then
        echo "/internal: HTTP request failed: HTTP status=$status cmd=$cmd " 1>&2
        test -f $result && cat $result 1>&2
        echo "" 1>&2
        return 1
    fi
    return 0
}

#usage: $0  uri
# forceibly loads an Internal graph from its source resource file
loadInternalGraph () {
    local result="${tmpDir}/internal"
    rm -f $result

    local status=`curl -k -s -S -u "$login" -o "$result" -d action=load \
      --data-urlencode name=$1 --write-out "%{http_code}" ${repoURL}/repository/internal`
    local cmd=$?
    if [ $cmd -ne 0 -o "$status" != "201" ]; then
        echo "/internal: HTTP request failed: HTTP status=$status cmd=$cmd " 1>&2
        test -f $result && cat $result 1>&2
        echo "" 1>&2
        return 1
    fi
    return 0
}

# compare internal version markers
# Usage: $0 "$available" "$loaded"
#  Typeical format of version is "$Id: upgrade.sh 8369 2011-03-30 15:39:38Z rd67 $"
#  possibly including double-quotes
internalAvailableVersionIsNewer () {
    if vs=`getInternalVersions "$uri"`; then
        local loaded=`echo "$vs"|awk "-F\t" '{print $2}'`
        local available=`echo "$vs"|awk "-F\t" '{print $3}'`
        local lv=`getInternalRevision "$loaded"`
        local av=`getInternalRevision "$available"`
         
        # if second term is empty, first is always newer
        if [ -n "$av" -a -z "$lv" ]; then
            $Debug && echo "DEBUG: intVersion: No loaded version and avail=$av for $uri" 1>&2
            return 0
        elif [ -n "$lv" -a -n "$av" ]; then
            $Debug && echo "DEBUG: intVersion: Comparing loaded=$lv to avail=$av for $uri" 1>&2
            test "$lv" -lt "$av"
        else
            $Debug && echo "DEBUG: intVersion: Failed getting versions: loaded=$lv to avail=$av for $uri" 1>&2
            return 1
        fi
    else
        echo "WARNING: Failed to get version info for internal graph $uri" 1>&2
        return 1
    fi
}

# compare internal version markers
# Usage: $0 version-string
#  Typeical format of version is "$Id: upgrade.sh 8369 2011-03-30 15:39:38Z rd67 $"
#  possibly including double-quotes
getInternalRevision () {
    if echo "$1" | egrep -q '^"?\$Id: '; then
        echo "$1" | awk '{print $3}'
    else
        return 1
    fi
}

# usage:  $0  view  query
#  returns text/plain results (ntriples for construct, TSV for select)
doTupleQuery () {
    doQueryInternal "$1" text/plain "$2"
}

# usage:  $0  view  query
#  returns true | false
doBooleanQuery () {
    doQueryInternal "$1" text/boolean "$2"
}

# SPARQL query prefixes
nl=$'\n'
queryPrefix="PREFIX rdf: <http://www.w3.org/1999/02/22-rdf-syntax-ns#>
PREFIX rdfs: <http://www.w3.org/2000/01/rdf-schema#>
PREFIX owl: <http://www.w3.org/2002/07/owl#>
PREFIX dcterms: <http://purl.org/dc/terms/>
PREFIX : <http://eagle-i.org/ont/repo/1.0/>"

# usage: $0  view  response-format  query
#  guts of SPARQL query request, result on stdout
doQueryInternal () {
    local view=$1
    local format=$2
    local query="${queryPrefix}${nl}$3"

    local expectStatus="200"
    local result="${tmpDir}/sparql"
    rm -f $result
    local status=""

    if status=`curl -k -s -S -u "$login" -o "$result" -F "format=$format"  \
      -F "query=$query" -F "view=$view" \
      --write-out "%{http_code}" ${repoURL}/repository/sparql` ; then
        if [ "$status" != "$expectStatus" ]; then
            echo "doQueryInternal: SPARQL Protocol request failed: HTTP status=$status cmd=$cmd " 1>&2
            test -f $result && cat $result 1>&2
            echo "" 1>&2
            echo "Query=" 1>&2
            echo "$query"|cat -n 1>&2
            return 1
        fi
    else
        echo "Failed sending HTTP /sparql request, curl status=$?" 1>&2
    fi

    # skip first line if select??
    if [ -f "$result" ]; then
        if echo "$query"|grep -qi "^SELECT"; then
            tail -n +2 "$result"
        else
            cat "$result"
        fi
    fi
    return 0
}

# usage:  graph-name  action  format-mimeType contents
#  uploads contents to named graph, action should be add or replace
doGraph () {
    local name=$1
    local action=$2
    local format=$3
    local content=$4
    local expectStatus="200"
    local status=""
    local result="${tmpDir}/init"
    rm -f $result
    local contentOpt="--form-string"

    # if the content arg is @<filename> for a real file path, use -F
    # otherwise the @ might be part of @PREFIX..
    if echo "$content"|grep -q '^@' && test -f `echo "$content"|sed -e 's/^@//'`; then
        contentOpt="-F"
    fi

    if status=`curl -k -s -S -u "$login" -o "$result" -F "format=$format"  \
      -F "name=$name" -F "action=$action" "$contentOpt" "content=$content" \
      --write-out "%{http_code}" ${repoURL}/repository/graph` ; then
        if [ "$status" != "$expectStatus" ]; then
            echo "doGraph: SPARQL Protocol request failed: HTTP status=$status cmd=$cmd " 1>&2
            test -f $result && cat $result 1>&2
            return 1
        fi
    else
        echo "doGraph: Failed sending HTTP /graph request, curl status=$?" 1>&2
    fi
}

# Usage: $0  construct-query  uri ...
# queries the initializer for given internal graph with CONSTRUCT query
queryInitializer () {
    local query="${queryPrefix}${nl}$1"
    local nameArgs=""
    shift
    while [ $# -gt 0 ]; do
        nameArgs="${nameArgs} -d name=${1}"
        shift
    done

    local result="${tmpDir}/init"
    rm -f $result

    if status=`curl -k -s -S -u "$login" -o "$result" \
      $nameArgs -d "action=query" -d "query=$query" \
      --write-out "%{http_code}" ${repoURL}/repository/internal` ; then
        if [ "$status" != "200" ]; then
            echo "queryInitializer: /internal request failed: HTTP status=$status cmd=$cmd " 1>&2
            test -f $result && cat $result 1>&2
            echo "" 1>&2
            return 1
        fi
        if [ -s "$result" ]; then
            cat "$result"
        else
            $Debug && echo "[DEBUG: QueryInitializer got empty results.]" 1>&2
        fi
        return 0
    else
        echo "Failed sending HTTP request /internal, curl status=$?" 1>&2
        return 1
    fi
}

# Force the repo server to decache all its in-memory caches of RDF data
doDecache () {
    local expectStatus="200"

    local status=`curl -k -s -S -u "$login" -d "action=decache" --write-out "%{http_code}" ${repoURL}/repository/internal`
    local cmd=$?
    if [ $cmd -ne 0 -o "$status" != "$expectStatus" ]; then
        echo "doDecache: Internal decache request failed: HTTP status=$status cmd=$cmd " 1>&2
        return 1
    fi
}


#---------------------------------------------------------------------------
# Compare loaded version of data model ontology against the version
# available in the webapp's jar -- load the one from the jar if newer,
# or if none appears to be loaded.
upgradeDataModelOntology () {
    # this will have versions 1 = loaded, 2 = jar, 3 = eiModel
    local versions=""
    if versions=`getDataModelVersions`; then
        local loaded=`selectModelVersion 1 "$versions"`
        local jar=`selectModelVersion 2 "$versions"`
        echo "INFO:  Loaded version = ${loaded}"
        if echo "$loaded"|grep -q "not loaded"||modelVersionNewer "$jar" "$loaded"; then
            echo "INFO: Loading new Data Model Ontology from internal jar, version: $jar"
            if loadDataModelOntology jar; then
                true
            else
                echo "ERROR: Failed loading data model ontology."
                commandStatus=1
            fi
        else
            echo "INFO: Data Model Ontology is already up to date, version: $loaded"
        fi
    else
        echo "ERROR: Failed getting data model ontology versions."
        commandStatus=1
    fi
}

#---------------------------------------------------------------------------

# query to pick up Orphans: has EI type, and metadata (upgrade hasn't run)
# and NO parents..
# and NOT in withdrawn graph yet, which is where we put htem.
orphanQueryBody="{
  ?orphan a ?eiType . ?eiType <${inClassGroup}> <${embeddedClass}>
  optional{graph :NG_Metadata {?orphan :hasWorkflowState ?wfs}}
  optional{ ?parent ?v ?orphan } filter(!BOUND(?parent))
  graph ?hg { ?orphan a ?t } filter(?hg != :NG_Inferred && ?hg != :NG_Withdrawn)}"

# metadata query shared by driver and step 1
mdQuerySuffix="{ ?ei a ?eiType . ?eiType <${inClassGroup}> <${embeddedClass}>
      graph ?hg { ?ei a ?t } filter(?hg != :NG_Inferred && ?hg != :NG_Withdrawn)
      graph :NG_Metadata {?ei ?mdverb ?mdobj}}"

# body of query to find shared (poached) EIs..
sharedEIQueryBody="{
   ?parent ?pred ?ei . ?ei a ?eiType .
   ?eiType <${inClassGroup}> <${embeddedClass}> .
   ?poacher ?ppred ?ei FILTER(?poacher != ?parent)
   graph ?hg { ?ei a ?t } filter(?hg != :NG_Inferred && ?hg != :NG_Withdrawn)}"

# Query to find tricky EIs in separate graph from their parent.
# NOTE This is a crude optimization since the general case with a FILTER
# clause for pg != eg took *forever* on harvard.  This takes advantage of
# the fact we really only use DefaultWorkspace and publishd
#  should start with:   select distinct  ?parent ?ei where {
mixedGraphEIQueryBody="{
 ?ei a ?eiType .
 ?eiType <http://eagle-i.org/ont/app/1.0/inClassGroup>
  <http://eagle-i.org/ont/app/1.0/ClassGroup_embedded_class> .
 ?parent ?pp ?ei .
 {
  {graph :NG_DefaultWorkspace { ?ei a ?aType }
   graph :NG_Published { ?parent a ?pType }
  } UNION {
   graph :NG_Published { ?ei a ?aType }
   graph :NG_DefaultWorkspace { ?parent a ?pType }}
}}"


# migrate the WFTs by importing new ones
special_1_2MS2_wft () {
    local result=0
    local transImport="${tmpDir}/wft.txt"
    cat >"$transImport" <<EOF
<http://eagle-i.org/ont/repo/1.0/NG_Internal> {
        <http://eagle-i.org/ont/repo/1.0/WFT_0> a <http://eagle-i.org/ont/repo/1.0/WorkflowTransition> ;
                <http://www.w3.org/2000/01/rdf-schema#label> "Default Create" ;
                <http://www.w3.org/2000/01/rdf-schema#comment> "Create new resource instance" ;
                <http://eagle-i.org/ont/repo/1.0/hasReadAccess> <http://eagle-i.org/ont/repo/1.0/Role_RNAV> , <http://eagle-i.org/ont/repo/1.0/Role_Curator> , <http://eagle-i.org/ont/repo/1.0/Role_DraftEditor> ;
                <http://eagle-i.org/ont/repo/1.0/initial> <http://eagle-i.org/ont/repo/1.0/WFS_New> ;
                <http://eagle-i.org/ont/repo/1.0/final> <http://eagle-i.org/ont/repo/1.0/WFS_Draft> ;
                <http://eagle-i.org/ont/repo/1.0/order> "100"^^<http://www.w3.org/2001/XMLSchema#integer> ;
                <http://eagle-i.org/ont/repo/1.0/hasReadAccess> <http://eagle-i.org/ont/repo/1.0/Role_RNAV> , <http://eagle-i.org/ont/repo/1.0/Role_Curator> , <http://eagle-i.org/ont/repo/1.0/Role_DraftEditor> .
        <http://eagle-i.org/ont/repo/1.0/WFT_0_1> a <http://eagle-i.org/ont/repo/1.0/WorkflowTransition> ;
                <http://www.w3.org/2000/01/rdf-schema#label> "Edit Draft" ;
                <http://www.w3.org/2000/01/rdf-schema#comment> "Edit a Draft resource instance, keeping it in the same state.  NOTE: This transition allows e.g. Lab User to claim and edit a Draft resource but NOT promote it to curation." ;
                <http://eagle-i.org/ont/repo/1.0/hasReadAccess> <http://eagle-i.org/ont/repo/1.0/Role_RNAV> , <http://eagle-i.org/ont/repo/1.0/Role_Curator> , <http://eagle-i.org/ont/repo/1.0/Role_DraftEditor> ;
                <http://eagle-i.org/ont/repo/1.0/initial> <http://eagle-i.org/ont/repo/1.0/WFS_Draft> ;
                <http://eagle-i.org/ont/repo/1.0/final> <http://eagle-i.org/ont/repo/1.0/WFS_Draft> ;
                <http://eagle-i.org/ont/repo/1.0/order> "110"^^<http://www.w3.org/2001/XMLSchema#integer> ;
                <http://eagle-i.org/ont/repo/1.0/hasReadAccess> <http://eagle-i.org/ont/repo/1.0/Role_RNAV> , <http://eagle-i.org/ont/repo/1.0/Role_Curator> , <http://eagle-i.org/ont/repo/1.0/Role_DraftEditor> .
        <http://eagle-i.org/ont/repo/1.0/WFT_1> a <http://eagle-i.org/ont/repo/1.0/WorkflowTransition> ;
                <http://www.w3.org/2000/01/rdf-schema#label> "Send to Curation" ;
                <http://www.w3.org/2000/01/rdf-schema#comment> "Move from Draft to Curation" ;
                <http://eagle-i.org/ont/repo/1.0/hasReadAccess> <http://eagle-i.org/ont/repo/1.0/Role_RNAV> , <http://eagle-i.org/ont/repo/1.0/Role_Curator> ;
                <http://eagle-i.org/ont/repo/1.0/initial> <http://eagle-i.org/ont/repo/1.0/WFS_Draft> ;
                <http://eagle-i.org/ont/repo/1.0/final> <http://eagle-i.org/ont/repo/1.0/WFS_Curation> ;
                <http://eagle-i.org/ont/repo/1.0/order> "200"^^<http://www.w3.org/2001/XMLSchema#integer> ;
                <http://eagle-i.org/ont/repo/1.0/hasReadAccess> <http://eagle-i.org/ont/repo/1.0/Role_RNAV> , <http://eagle-i.org/ont/repo/1.0/Role_Curator> .
        <http://eagle-i.org/ont/repo/1.0/WFT_2> a <http://eagle-i.org/ont/repo/1.0/WorkflowTransition> ;
                <http://www.w3.org/2000/01/rdf-schema#label> "Publish" ;
                <http://www.w3.org/2000/01/rdf-schema#comment> "Move from Curation to Published on default workspaces" ;
                <http://eagle-i.org/ont/repo/1.0/hasReadAccess> <http://eagle-i.org/ont/repo/1.0/Role_Curator> ;
                <http://eagle-i.org/ont/repo/1.0/initial> <http://eagle-i.org/ont/repo/1.0/WFS_Curation> ;
                <http://eagle-i.org/ont/repo/1.0/final> <http://eagle-i.org/ont/repo/1.0/WFS_Published> ;
                <http://eagle-i.org/ont/repo/1.0/action> "org.eaglei.repository.model.workflow.ActionMoveGraph" ;
                <http://eagle-i.org/ont/repo/1.0/actionParameter> <http://eagle-i.org/ont/repo/1.0/NG_Published> ;
                <http://eagle-i.org/ont/repo/1.0/order> "300"^^<http://www.w3.org/2001/XMLSchema#integer> ;
                <http://eagle-i.org/ont/repo/1.0/hasReadAccess> <http://eagle-i.org/ont/repo/1.0/Role_Curator> .
        <http://eagle-i.org/ont/repo/1.0/WFT_5> a <http://eagle-i.org/ont/repo/1.0/WorkflowTransition> ;
                <http://www.w3.org/2000/01/rdf-schema#label> "Return to Draft" ;
                <http://www.w3.org/2000/01/rdf-schema#comment> "Move from Curation back to Draft status" ;
                <http://eagle-i.org/ont/repo/1.0/hasReadAccess> <http://eagle-i.org/ont/repo/1.0/Role_Curator> ;
                <http://eagle-i.org/ont/repo/1.0/initial> <http://eagle-i.org/ont/repo/1.0/WFS_Curation> ;
                <http://eagle-i.org/ont/repo/1.0/final> <http://eagle-i.org/ont/repo/1.0/WFS_Draft> ;
                <http://eagle-i.org/ont/repo/1.0/order> "350"^^<http://www.w3.org/2001/XMLSchema#integer> ;
                <http://eagle-i.org/ont/repo/1.0/hasReadAccess> <http://eagle-i.org/ont/repo/1.0/Role_Curator> .
        <http://eagle-i.org/ont/repo/1.0/WFT_7> a <http://eagle-i.org/ont/repo/1.0/WorkflowTransition> ;
                <http://www.w3.org/2000/01/rdf-schema#label> "Return to Draft" ;
                <http://www.w3.org/2000/01/rdf-schema#comment> "Send a Published resource back to Draft (ONLY for DEFAULT published and draft workspaces)" ;
                <http://eagle-i.org/ont/repo/1.0/hasReadAccess> <http://eagle-i.org/ont/repo/1.0/Role_Curator> ;
                <http://eagle-i.org/ont/repo/1.0/workspace> <http://eagle-i.org/ont/repo/1.0/NG_Published> ;
                <http://eagle-i.org/ont/repo/1.0/initial> <http://eagle-i.org/ont/repo/1.0/WFS_Published> ;
                <http://eagle-i.org/ont/repo/1.0/final> <http://eagle-i.org/ont/repo/1.0/WFS_Draft> ;
                <http://eagle-i.org/ont/repo/1.0/action> "org.eaglei.repository.model.workflow.ActionMoveGraph" ;
                <http://eagle-i.org/ont/repo/1.0/actionParameter> <http://eagle-i.org/ont/repo/1.0/NG_DefaultWorkspace> ;
                <http://eagle-i.org/ont/repo/1.0/order> "360"^^<http://www.w3.org/2001/XMLSchema#integer> ;
                <http://eagle-i.org/ont/repo/1.0/hasReadAccess> <http://eagle-i.org/ont/repo/1.0/Role_Curator> .
        <http://eagle-i.org/ont/repo/1.0/WFT_3> a <http://eagle-i.org/ont/repo/1.0/WorkflowTransition> ;
                <http://www.w3.org/2000/01/rdf-schema#label> "Withdraw" ;
                <http://www.w3.org/2000/01/rdf-schema#comment> "Withdraw a resource by moving from Published to Withdrawn graph" ;
                <http://eagle-i.org/ont/repo/1.0/hasReadAccess> <http://eagle-i.org/ont/repo/1.0/Role_Curator> ;
                <http://eagle-i.org/ont/repo/1.0/workspace> <http://eagle-i.org/ont/repo/1.0/NG_Published> ;
                <http://eagle-i.org/ont/repo/1.0/initial> <http://eagle-i.org/ont/repo/1.0/WFS_Published> ;
                <http://eagle-i.org/ont/repo/1.0/final> <http://eagle-i.org/ont/repo/1.0/WFS_Withdrawn> ;
                <http://eagle-i.org/ont/repo/1.0/action> "org.eaglei.repository.model.workflow.ActionMoveGraph" ;
                <http://eagle-i.org/ont/repo/1.0/actionParameter> <http://eagle-i.org/ont/repo/1.0/NG_Withdrawn> ;
                <http://eagle-i.org/ont/repo/1.0/order> "400"^^<http://www.w3.org/2001/XMLSchema#integer> ;
                <http://eagle-i.org/ont/repo/1.0/hasReadAccess> <http://eagle-i.org/ont/repo/1.0/Role_Curator> .
        <http://eagle-i.org/ont/repo/1.0/WFT_6> a <http://eagle-i.org/ont/repo/1.0/WorkflowTransition> ;
                <http://www.w3.org/2000/01/rdf-schema#label> "Return to Curation" ;
                <http://www.w3.org/2000/01/rdf-schema#comment> "Send a Published resource back to Curation (ONLY for DEFAULT published and draft workspaces)" ;
                <http://eagle-i.org/ont/repo/1.0/hasReadAccess> <http://eagle-i.org/ont/repo/1.0/Role_Curator> ;
                <http://eagle-i.org/ont/repo/1.0/workspace> <http://eagle-i.org/ont/repo/1.0/NG_Published> ;
                <http://eagle-i.org/ont/repo/1.0/initial> <http://eagle-i.org/ont/repo/1.0/WFS_Published> ;
                <http://eagle-i.org/ont/repo/1.0/final> <http://eagle-i.org/ont/repo/1.0/WFS_Curation> ;
                <http://eagle-i.org/ont/repo/1.0/action> "org.eaglei.repository.model.workflow.ActionMoveGraph" ;
                <http://eagle-i.org/ont/repo/1.0/actionParameter> <http://eagle-i.org/ont/repo/1.0/NG_DefaultWorkspace> ;
                <http://eagle-i.org/ont/repo/1.0/order> "450"^^<http://www.w3.org/2001/XMLSchema#integer> ;
                <http://eagle-i.org/ont/repo/1.0/hasReadAccess> <http://eagle-i.org/ont/repo/1.0/Role_Curator> .
        <http://eagle-i.org/ont/repo/1.0/WFT_8> a <http://eagle-i.org/ont/repo/1.0/WorkflowTransition> ;
                <http://www.w3.org/2000/01/rdf-schema#label> "Return to Draft" ;
                <http://www.w3.org/2000/01/rdf-schema#comment> "Send a Withdrawn resource back to Draft (ONLY for DEFAULT published and draft workspaces)" ;
                <http://eagle-i.org/ont/repo/1.0/hasReadAccess> <http://eagle-i.org/ont/repo/1.0/Role_Curator> ;
                <http://eagle-i.org/ont/repo/1.0/workspace> <http://eagle-i.org/ont/repo/1.0/NG_Withdrawn> ;
                <http://eagle-i.org/ont/repo/1.0/initial> <http://eagle-i.org/ont/repo/1.0/WFS_Withdrawn> ;
                <http://eagle-i.org/ont/repo/1.0/final> <http://eagle-i.org/ont/repo/1.0/WFS_Draft> ;
                <http://eagle-i.org/ont/repo/1.0/action> "org.eaglei.repository.model.workflow.ActionMoveGraph" ;
                <http://eagle-i.org/ont/repo/1.0/actionParameter> <http://eagle-i.org/ont/repo/1.0/NG_DefaultWorkspace> ;
                <http://eagle-i.org/ont/repo/1.0/order> "460"^^<http://www.w3.org/2001/XMLSchema#integer> ;
                <http://eagle-i.org/ont/repo/1.0/hasReadAccess> <http://eagle-i.org/ont/repo/1.0/Role_Curator> .
        <http://eagle-i.org/ont/repo/1.0/WFT_4> a <http://eagle-i.org/ont/repo/1.0/WorkflowTransition> ;
                <http://www.w3.org/2000/01/rdf-schema#label> "Reinstate" ;
                <http://www.w3.org/2000/01/rdf-schema#comment> "Return a Withdrawn resource instance to the default Published graph" ;
                <http://eagle-i.org/ont/repo/1.0/hasReadAccess> <http://eagle-i.org/ont/repo/1.0/Role_Curator> ;
                <http://eagle-i.org/ont/repo/1.0/workspace> <http://eagle-i.org/ont/repo/1.0/NG_Withdrawn> ;
                <http://eagle-i.org/ont/repo/1.0/initial> <http://eagle-i.org/ont/repo/1.0/WFS_Withdrawn> ;
                <http://eagle-i.org/ont/repo/1.0/final> <http://eagle-i.org/ont/repo/1.0/WFS_Published> ;
                <http://eagle-i.org/ont/repo/1.0/action> "org.eaglei.repository.model.workflow.ActionMoveGraph" ;
                <http://eagle-i.org/ont/repo/1.0/actionParameter> <http://eagle-i.org/ont/repo/1.0/NG_Published> ;
                <http://eagle-i.org/ont/repo/1.0/order> "500"^^<http://www.w3.org/2001/XMLSchema#integer> ;
                <http://eagle-i.org/ont/repo/1.0/hasReadAccess> <http://eagle-i.org/ont/repo/1.0/Role_Curator> .
}
EOF

    if doImport $transImport transition replace no; then
        echo "INFO: imported new WFTs."
    else
        echo "ERROR: FAIL: @$LINENO Failed importing WFTs"
        result=1
    fi
    return $result
}

# Dispatcher for special 1.2MS2 upgrades, i.e. EI data migration
#
# NOTE: it is CRUCIAL to return failed status if ANYTHING goes
# wrnog since it will probably require restoring the backup of the RDF DB.
# We can't wrap a transaction around this whole thing.
#
special_1_2MS2_all () {

    # 1 check whether WFTs need fixup - is action still old class?
    local oldWFA="org.eaglei.repository.workflow."
    local wft_query="select * where { :WFT_3 :action ?a }"
    local qr
    if qr=`doTupleQuery all "$wft_query"`; then
         if echo "$qr" | fgrep -q "$oldWFA" ; then
            echo "====>>> Step 1 - migrating old WFTs for 1.2MS2"
            if special_1_2MS2_wft; then
                true
            else
                result=1
            fi
        else
            echo "INFO: Step 1, WFTs are already migrated for 1.2MS2"
        fi
    else
        echo "ERROR: Failed in WFT query at line $LINENO"
        result=1
    fi
}

# Dispatcher for special 1.1MS5 upgrades, i.e. EI data migration
#
# NOTE: it is CRUCIAL to return failed status if ANYTHING goes
# wrnog since it will probably require restoring the backup of the RDF DB.
# We can't wrap a transaction around this whole thing.
#
special_1_1MS5_all () {
    local result=0
    # get "profile" of EIs and parents for before/after comparison
    local eiProfileQuery="SELECT DISTINCT ?parent ?eiLabel WHERE {
      ?parent ?rp ?embInstance .
      ?embInstance a ?eiType . ?eiType <${inClassGroup}> <${embeddedClass}>
      OPTIONAL{ ?embInstance rdfs:label ?eiLabel }} ORDER BY ?parent"

    local answer=""
    local before="${tmpDir}/before.txt"
    local after="${tmpDir}/after.txt"
    if doTupleQuery user-resources "$eiProfileQuery" | sort > $before ; then
        echo "Before migration: `wc -l < $before` EIs to check."
    else
        echo "ERROR: Failed in profile query at line $LINENO"
        result=1
    fi

    # ---- Step 1. Banish Orphan EIs
    echo ""
    if answer=`doBooleanQuery user-resources "ASK WHERE $orphanQueryBody"`; then
        if [ "$answer" = true ]; then
            echo "====>>> Step 1 - Rusticating Orphan Embedded Instances"
            if special_1_1MS5_rusticate_orphans; then
                true
            else
                result=1
            fi
        elif [ "$answer" = false ]; then
            echo "INFO: Step 1 - Already Rusticated all the Orphans"
        else
            echo "ERROR: Step 1 - Orphans, got unknown result = \"$answer\""
            result=1
        fi
    else
        echo "ERROR: FAILED evaluating query in SPECIAL 1.1MS5 step 1"
            result=1
    fi

    # ---- Step 2. Remove metadata on EIs
    echo ""
    if answer=`doBooleanQuery user-resources "ASK WHERE $mdQuerySuffix"`; then
        if [ "$answer" = true ]; then
            echo "====>>> Step 2 - Deleting Metadata statements on Emebedded Instances."
            if special_1_1MS5_remove_metadata; then
                true
            else
                result=1
            fi
        elif [ "$answer" = false ]; then
            echo "INFO: Step 2 - Already removed EI metadata"
        else
            echo "ERROR: Step 2 got unknown result = \"$answer\""
            result=1
        fi
    else
        echo "ERROR: FAILED evaluating query in SPECIAL 1.1MS5 step 2"
            result=1
    fi

    # ---- Step 3. fix shared EIs
    echo ""
    if answer=`doBooleanQuery user-resources "ASK WHERE $sharedEIQueryBody"`; then
        if [ "$answer" = true ]; then
            echo "====>>> Step 3 - Fix All Shared Emebedded Instances."
            if special_1_1MS5_fix_all_shared ; then
                true
            else
                result=1
            fi
        elif [ "$answer" = false ]; then
            echo "INFO: Step 3 - No Shared EIs found (or already fixed)."
        else
            echo "ERROR: Step 3 - fix EIs, got unknown result = \"$answer\""
            result=1
        fi
    else
        echo "ERROR: FAILED evaluating query in SPECIAL 1.1MS5 step 3"
            result=1
    fi

    # ---- Step 4 Fix any leftover mixed-graph EIs
    echo ""
    if answer=`doBooleanQuery user-resources "ASK WHERE $mixedGraphEIQueryBody"`; then
        if [ "$answer" = true ]; then
            echo "====>>> Step 4 - Fix mixed-graph EIs"
            if special_1_1MS5_mixed_graph ; then
                true
            else
                result=1
            fi
        elif [ "$answer" = false ]; then
            echo "INFO: Step 4 - Already fixed mixed graph EIs"
        else
            echo "ERROR: Step 4 got unknown result = \"$answer\""
            result=1
        fi
    else
        echo "ERROR: FAILED evaluating query in SPECIAL 1.1MS5 step 4"
            result=1
    fi

    # CHECK RESULTS: Run profile query again and compare
    echo ""
    echo "====>>> Comparing repo state with before migration..."
    if doTupleQuery user-resources "$eiProfileQuery" |sort > $after ; then
        echo "After migration: `wc -l < $after` EIs in repo now."
        local diffs="${tmpDir}/diffs.txt"
        if diff $before $after > $diffs ; then
            echo "INFO: OK - no difference in EI profiles before and after migration."
            echo ""
            echo "====>>> List of Resource Instances containing EIs, for spot checks:"
            cat $before $after | awk '{print $1}' | sort -u
            echo "====>>> END <<<===="
            echo ""
        else
            echo "INFO: ***** MAYBE PROBLEMS: ******"
            echo "INFO: Please examine these differences in EI profiles before and after migration:"
            cat $diffs
            echo ""
        fi
    else
        echo "ERROR: Failed in profile query at line $LINENO"
        result=1
    fi
    return $result
}

# Ratoinalize "mixed graph" EIs - where the to-be-embedded EI is in a DIFFERENT
# GRAPH from its parent.  Crazy, eh?  what will they think of next to make
# our lives miserable?
special_1_1MS5_mixed_graph () {
    local toFix="${tmpDir}/fixme.txt"
    local eiGuts="${tmpDir}/eiGuts.txt"
    local fixDebug=false
    local result=0
    local workQuery="SELECT DISTINCT ?parent ?ei WHERE $mixedGraphEIQueryBody"

    rm -f $toFix
    if doQueryInternal "all" text/plain "$workQuery" > $toFix; then
        local parent
        local ei

        # for each EI, get home graph of it and parent and move it..
        while read parent ei;  do
            local hgQuery="SELECT DISTINCT ?pg ?eig WHERE {
                GRAPH ?pg { <$parent> a ?ptype } FILTER(?pg != :NG_Inferred)
                GRAPH ?eig { <$ei> a ?etype } FILTER(?eig != :NG_Inferred)}"
            local graphs=""
            if graphs=`doQueryInternal all text/plain "$hgQuery"` ; then
                ## first sanity-check the home graphs result
                if [ `echo "$graphs"|wc -w` -eq 2 ]; then
                    local pg=`echo "$graphs"|awk '{print $1}'`
                    local eig=`echo "$graphs"|awk '{print $2}'`

                    # move ei to parent's home graph:
                    local eiQuery="CONSTRUCT {<${ei}> ?p ?o} WHERE {
                        GRAPH <${eig}> {<${ei}> ?p ?o}}"
                    if doQueryInternal all text/plain "$eiQuery" > $eiGuts ; then
                        if doGraph "$eig" delete "text/plain" "@$eiGuts"; then
                            if doGraph "$pg" add "text/plain" "@$eiGuts"; then
                                echo "INFO: OK - moved EI=$ei into parent graph $pg"
                            else
                                echo "FAIL: failed @$LINENO to add EI=$ei to parent graph $pg"
                                result=1
                            fi
                        else
                            echo "FAIL: failed @$LINENO to delete EI=$ei from its home graph $eig"
                            result=1
                        fi
                    else
                        echo "FAIL: failed @$LINENO in home graph query for EI=$ei "
                        result=1
                    fi
                else
                    echo "FAIL: Sanity check failed @$LINENO , home graphs not found for parent=$parent EI=$ei line $LINENO"
                    result=1
                fi
            else
                echo "FAIL: home graph query failed, @$LINENO for parent=$parent EI=$ei line $LINENO"
                result=1
            fi
        done <"$toFix"
    else
        echo "ERROR: FAIL: @$LINENO Failed querying for mixed graphs failed, query="
        echo "$workQuery"|cat -n
        result=1
    fi
    return $result
}


# move orphans to Withdrawn graph, cahnge their workflow state too.
# first, get a file of  graph, orphan-URI, and old WFS; use it to
#  construct the insert and delete lists..
special_1_1MS5_rusticate_orphans () {
    local orphanage="${tmpDir}/orphans.txt"
    local deletes="${tmpDir}/deletes1.txt"
    local inserts="${tmpDir}/inserts1.txt"
    local instStms="${tmpDir}/instances.txt"
    local orphanDebug=false
    local result=0
    if doQueryInternal "all" text/plain "SELECT ?hg ?orphan ?wfs $orphanQueryBody ORDER BY ?hg" > $orphanage; then
        local -a graphs=(`awk '{print $1}' < $orphanage | uniq`)
        # DEBUG
        #echo "  Got ${#graphs[*]} graphs = (${graphs[*]})"

        # first, remove old WFS and add Withdrawn one by munging grahps:
        for graph in ${graphs[*]}; do
            awk "\$1 == \"${graph}\" && \$3 != \"\" { printf \"<%s> <${repo}hasWorkflowState> <%s>.\\n\",\$2,\$3}" < $orphanage > $deletes
            awk "\$1 == \"${graph}\" { printf \"<%s> <${repo}hasWorkflowState> <${repo}WFS_Withdrawn> .\\n\",\$2}" < $orphanage > $inserts
            if $orphanDebug; then
                echo "DEBUG: deletes="
                cat -n $deletes
                echo "DEBUG: inserts="
                cat -n $inserts
            else
                if doGraph "${repo}NG_Metadata" delete "text/plain" "@$deletes"; then
                    echo "INFO: OK - Deleted `wc -l < $deletes` MD statements about Orphans."
                else
                    echo "ERROR: FAIL: Metadata statements on Emebedded Instances at $LINENO"
                    result=1
                fi
                if doGraph "${repo}NG_Metadata" add "text/plain" "@$inserts"; then
                    echo "INFO: OK - Inserted `wc -l < $inserts` MD statements about Orphans."
                else
                    echo "ERROR: FAIL: Metadata statements on Emebedded Instances at $LINENO"
                    result=1
                fi
            fi
             
            # now move instance statements in this graph to NG_Withdrawn:
            local orphanInstanceQuery="CONSTRUCT {?orphan ?op ?oo} WHERE {
              ?orphan a ?eiType . ?eiType <${inClassGroup}> <${embeddedClass}>
              optional{ ?parent ?v ?orphan } filter(!BOUND(?parent))
              graph <${graph}> { ?orphan ?op ?oo }}"
            if doQueryInternal "all" text/plain "$orphanInstanceQuery" > $instStms; then
                if $orphanDebug; then
                    echo "====Moving instances for graph=$graph"
                    cat -n $instStms
                else
                    if doGraph "${graph}" delete "text/plain" "@$instStms"; then
                        if doGraph "${repo}NG_Withdrawn" add "text/plain" "@$instStms"; then
                            echo "  OK - moved orphan instances out of $graph"
                        else
                            echo "FAIL: moving into Withdrawn at $LINENO"
                            result=1
                        fi
                    else
                        echo "FAIL: deleting from $graph at $LINENO"
                        result=1
                    fi
                fi
            else
                echo "ERROR: FAIL: Failed querying for orphan instances failed at $LINENO"
                echo "$orphanInstanceQuery" |cat -n
                result=1
            fi
        done

        # final report: record of orphan URIs in log
        if [ "$result" -eq 0 ]; then
            echo "INFO: Record of all orphan URIs:"
            awk '{print "INFO:   MOVED ORPHAN ",$2}' < $orphanage
        else
            echo "FAIL: Errors in orphan processing, must assume some orphans got horribly mutilated at $LINENO"
        fi
    else
        echo "ERROR: FAIL: Failed querying for orphans failed, query="
        echo "SELECT ?hg ?orphan ?wfs $orphanQueryBody ORDER BY ?hg" |cat -n
        result=1
    fi
    return $result
}

# break up shared EIs - get list of home-graph + EI-uri from query
# and iterate over it
special_1_1MS5_fix_all_shared () {
    local sharedQuery="SELECT DISTINCT ?hg ?ei WHERE ${sharedEIQueryBody}"
    local fixme="${tmpDir}/fixme.txt"
    local bad="${tmpDir}/bad.txt"
    local result=0

    if doQueryInternal "all" text/plain "$sharedQuery" > "$fixme"; then
        local ei
        local hg
        while read hg ei;  do
            if special_1_1MS5_fix_one_shared "$hg" "$ei"; then
                true;
            else
                result=1
            fi
        done <"$fixme"
    else
        echo "ERROR: FAIL: Failed querying Metadata statements on Emebedded Instances; query="
        echo "$sharedQuery" |cat -n
        return 1
    fi

    # sanity check: are there now any EIs with statements referring to
    # some OTHER EI's parent as an object prop value?  *should* have
    # eradicated all those with the is_part_construct cleanup.
    local wrongParentQuery="select distinct ?ei ?eip ?parent2 where {
            ?parent ?pp ?ei . ?ei a ?eiType .
            ?eiType <${inClassGroup}> <${embeddedClass}> .
            ?parent2 ?pp2 ?ei2 . ?ei2 a ?eiType . ?ei ?eip ?parent2
            filter(?parent != ?parent2)}"
    if doQueryInternal "all" text/plain "$wrongParentQuery" > "$bad"; then
        if [ -s "$bad" ]; then
            echo "WARNING: There are EIs with links to another EIs parent, (EI-URI, predicate, Parent-URI):"
            cat $bad
        else
            echo ""
            echo "OK - passed sanity check for EIs with link to wrong parent."
        fi
    else
        echo "ERROR: FAIL: Failed querying bad parents on Emebedded Instances; query="
        echo "$wrongParentQuery" |cat -n
        result=1
    fi
    return $result
}

# fix a shared EI:
#  - get list of all its parents
#  - get all properties of EI itself..
#  - for parents 2-N, delete it and then re-add it as a new instance
# usage: $0  eis-homegraph-URI  EI-URI
special_1_1MS5_fix_one_shared () {
    local hg="$1"
    local ei="$2"
    local allParentsQuery="SELECT DISTINCT ?parent WHERE { GRAPH :NG_Metadata {?parent dcterms:modified ?m} ?parent ?pred <${ei}> }"
    local parents=""
    local insertTemplate="${tmpDir}/insertTmpl.txt"
    local result=0

    # get list of all parents of this shared EI, and sanity-check
    if parents=`doQueryInternal "all" text/plain "$allParentsQuery"`; then
        # WARNING: don't want to call "set" on empty string, it's bad..
        set ${parents:-gigo}
        if [ $# -lt 2 ]; then
            echo "FAIL: The EI <${ei}> had fewer than 2 parents: result=$parents"
            echo "parentsQuery="
            echo "$allParentsQuery" |cat -n
            return 1
        fi

        local firstParent="$1"

        # sentinals for later text-replacement
        local infoNew=info:NEW
        local infoResource=info:RESOURCE
        ##local debug=true
        local debug=false

        # make a template of statements to insert, with sentinals to replace:
        # NOTE: get rid of is_part_construct statements since they are
        # artifacts, materialized inverse properties that refer back to the OLD
        # parent and thus cause trouble later..
        local insertQuery="CONSTRUCT {<${infoNew}> ?p ?o . <${infoResource}> ?pp <${infoNew}>} WHERE {
                           { GRAPH <${hg}> { {<${ei}> ?p ?o FILTER(?p != <${isPartConstruct}>)}}}
                            UNION
                           {<${firstParent}> ?pp <${ei}>}}"
        if doQueryInternal "all" text/plain "$insertQuery" > $insertTemplate; then
            if $debug; then
                echo "=== INSERT QUERY:"
                echo "$insertQuery"|cat -n
                echo "=== INSERT TEMPLATE:"
                cat -n $insertTemplate
            fi
        else
            echo "FAIL: query to make insert tempalte failed line=$LINENO"
            return 1
        fi

        # sanity check: do all parents refer to
        # Ei by same pred??  select distinct ...
        local allPreds=""
        local predQuery="SELECT DISTINCT ?pred WHERE { ?any ?pred <${ei}> }"
        if allPreds=`doQueryInternal "all" text/plain "$predQuery"`; then
            if [ `echo "$allPreds"|wc -w` -eq 1 ]; then
                echo "INFO: OK - just one distinct predicate referring to the EI"
            else
                echo "FAIL: Sanity check failed, EI=${ei}, Multiple distinct predicates=${allPreds}"
                result=1
            fi
        else
            echo "FAIL: Failed making Sanity Check query, line $LINENO EI=${ei}"
            result=1
        fi

        # dike the EI's statements separately since it may be in
        # a different graph from parents
        local deleteEI="<${ei}> <${wildcard}> <${wildcard}> ."
        if doUpdate "$login" update "$ei" "" "$deleteEI" "" "200" ""; then
            echo "INFO: OK - deleted statements for EI=$ei"
        else
            echo "FAIL: error deleting statements for EI=$ei"
            result=1
        fi

        # For each parent, delete the reference to EI and
        # add a newly synthesized EI.
        # On FIRST parent only, delete the EI's statements as well.  They'll
        # be gone and woudl break the update on subsequent calls.
        local new
        while [ $# -ge 1 ]; do
            if new=`newURI`; then
                local deletes="<${1}> <${wildcard}> <${ei}>."
                local inserts=`perl -pe "s@${infoNew}@${new}@g; s@${infoResource}@${1}@g;" < $insertTemplate`
                if $debug; then
                   echo "========DEBUG=========="
                   echo "  parent=$1"
                   echo "  deletes=$deletes"
                   echo "  inserts=$inserts"
                else
                    if doUpdate "$login" update "$1" "$inserts" "$deletes" "" "200" "bypassSanity"; then
                        echo "INFO: OK - added new separate EI=$new to $1"
                    else
                        echo "FAIL: error adding separate EI=$new to $1"
                        result=1
                    fi
                fi
            else
                echo "FAIL: newURI failed, skipping."
                result=1
            fi
            shift
        done
    else
        echo "ERROR: FAIL: Failed querying Parents of the Emebedded Instance."
        result=1
    fi
    return $result
}

# delete metadata staetments on EIs
special_1_1MS5_remove_metadata () {
    local mdQuery="CONSTRUCT {?ei ?mdverb ?mdobj} WHERE $mdQuerySuffix"
    local deleteme="${tmpDir}/deleteme.txt"

    if doQueryInternal "all" text/plain "$mdQuery" > "$deleteme"; then
        if doGraph "$NG_Metadata" delete "text/plain" "@$deleteme"; then
            echo "INFO: OK - Deleted `wc -l < $deleteme` MD statements on Emebedded Instances."
        else
            echo "ERROR: FAIL: Metadata statements on Emebedded Instances."
        fi
    else
        echo "ERROR: FAIL: Failed querying Metadata statements on Emebedded Instances."
    fi
}

# delete old FakeWorkflow-style claims."
special_1_1MS4_delete_claims () {
    echo "====>>> Step 4. Deleting old Fake Workflow-style claims."
    local deleteme="${tmpDir}/deleteme.txt"
    local wcQuery='construct {?s :hasWorkflowOwner ?v} where {?s :hasWorkflowOwner ?v OPTIONAL{?s ?g ?r . ?g rdfs:subPropertyOf :hasAnyAccess} FILTER(!bound(?g))}'
    if doQueryInternal "all" text/plain "$wcQuery" > "$deleteme"; then
        if doGraph "$NG_Metadata" delete "text/plain" "@$deleteme"; then
            echo "INFO: OK - deleted old Fake Workflow claims"
        else
            echo "ERROR: FAIL: Failed deleting old Fake Workflow claims"
        fi
    else
        echo "ERROR: FAIL: Failed querying old Fake Workflow claims"
    fi
}


#---------------------------------------------------------------------------
# Consolidated special 1.1MS4 upgrades
special_1_1MS4 () {
    local answer=""

    # Step 1 - add Role descriptions to NG_Internal
    # NOTE: Test query returns FALSE to indicate this is needed
    if answer=`doBooleanQuery all 'ask  where { graph :NG_Internal {?s a :Role}}'`; then
        if [ "$answer" = false ]; then
            special_1_1MS4_add_roles
        elif [ "$answer" = true ]; then
            echo "INFO: Step 1 - Already done: add Role descriptions"
        else
            echo "ERROR: Step 1 got unknown result = \"$answer\""
        fi
    else
        echo "ERROR: FAILED evaluating query in SPECIAL 1.1MS4 step 1"
    fi
    echo ""

    # Step 2 - delete old Workflow States from NG_Internal
    # NOTE: Test query returns FALSE to indicate this is needed
    if answer=`doBooleanQuery all 'ask  where { graph :NG_Internal {?s a :WorkflowState}}'`; then
        if [ "$answer" = true ]; then
            special_1_1MS4_workflowStates
        elif [ "$answer" = false ]; then
            echo "INFO: Step 2 - Already done: delete old workflow states"
        else
            echo "ERROR: Step 2 got unknown result = \"$answer\""
        fi
    else
        echo "ERROR: FAILED evaluating query in SPECIAL 1.1MS4 step 2"
    fi
    echo ""

    # Step 3 - add Workflow Transition descriptions to NG_Internal
    # NOTE: Test query returns FALSE to indicate this is needed
    if answer=`doBooleanQuery all 'ask  where { graph :NG_Internal {?s a :WorkflowTransition}}'`; then
        if [ "$answer" = false ]; then
            special_1_1MS4_add_WFTs
        elif [ "$answer" = true ]; then
            echo "INFO: Step 3 - Already done: add WorkflowTranstion descriptions"
        else
            echo "ERROR: Step 3 got unknown result = \"$answer\""
        fi
    else
        echo "ERROR: FAILED evaluating query in SPECIAL 1.1MS4 step 3"
    fi
    echo ""

    # Step 4 - delete FakeWorkflow-style claims if any found
    #  search for claims w/o any ACL grants on the instance.
    if answer=`doBooleanQuery all 'ask  where {?s :hasWorkflowOwner ?v OPTIONAL{?s ?g ?r . ?g rdfs:subPropertyOf :hasAnyAccess} FILTER(!bound(?g))}'` ; then
        if [ "$answer" = true ]; then
            special_1_1MS4_delete_claims
        elif [ "$answer" = false ]; then
            echo "INFO: Step 4 - Already done: delete old FakeWorkflow-style claims."
        else
            echo "ERROR: Step 4 got unknown result = \"$answer\""
        fi
    else
        echo "ERROR: FAILED evaluating query in SPECIAL 1.1MS4 step 4"
    fi
    echo ""

    # Step 4.1 - delete orphaned and bogus Edit Tokens
    # Delete editToken bindings and their data that DON'T belong
    #  *** Careful not to nuke any that are in legitimate ues ***
    # look for stray edit tokens on wrong graphs (data tools bug)
    #  and edit tokens w/o claims
    local detectEditToken='ask where { {graph ?g {?s :hasEditToken ?t}
      {{?g :ngType :NGType_Workspace} UNION {?g :ngType :NGType_Published}}}
      UNION
      {graph :NG_Internal {?s :hasEditToken ?t}
       optional {?s :hasWorkflowOwner ?claim} filter(!bound(?claim)) }}'
    if answer=`doBooleanQuery all "$detectEditToken"` ; then
        if [ "$answer" = true ]; then
            if special_1_1MS4_delete_editTokens; then
                true
            else
                echo "ERROR: Step 4.1 ***FAILED***. See messages above."
            fi
        elif [ "$answer" = false ]; then
            echo "INFO: Step 4.1 - Already done: delete orphaned and bogus Edit Tokens."
        else
            echo "ERROR: Step 4.1 got unknown result = \"$answer\""
        fi
    else
        echo "ERROR: FAILED evaluating query in SPECIAL 1.1MS4 step 4.1"
    fi
    echo ""


    # Step 5 .. fix grants on named graphs.
    # NOTE: The function both DETECTS and FIXES since it has all the data..
    if special_1_1MS4_NamedGraph_grants; then
        true
    else
        echo "ERROR: FAILED detect or change in SPECIAL 1.1MS4 Step 5"
    fi
    echo ""

    # Step 6 - user admin roles
    if answer=`doBooleanQuery all 'ask where {?s a :Person; :hasPrincipalName ?n; :hasRole :Role_Superuser; :hasRole ?r \
                 FILTER(!(?r = :Role_Superuser || ?r = :Role_Authenticated || ?r = :Role_Anonymous))}'`; then
        if [ "$answer" = true ]; then
            special_1_1MS4_user_roles
        elif [ "$answer" = false ]; then
            echo "INFO: Step 6. - Already done: Removed unnecessary Admin role from users"
        else
            echo "ERROR: Step 6 got unknown result = \"$answer\""
        fi
    else
        echo "ERROR: FAILED evaluating query in step 6, user admin roles"
    fi
    echo ""

    # Setp 7 - implicit roles
    if answer=`doBooleanQuery all 'ask where {?s a :Person. OPTIONAL{?s ?p :Role_Anonymous} FILTER(!bound(?p))}'`; then
        if [ "$answer" = true ]; then
            special_1_1MS4_user_implicit
        elif [ "$answer" = false ]; then
            echo "INFO: Step 7 - Already done: User implicit roles"
        else
            echo "ERROR: Step 7 got unknown result = \"${answer}\""
        fi
    else
        echo "ERROR: FAILED evaluating query in step 7, user admin roles"
    fi
    echo ""

    if doDecache; then
        echo "INFO: Forced Decache in Repo."
    else
        echo "ERROR: FAILED to Force Decache in Repo."
    fi
}

# detect and fix the named graph grants.
# to detect: look for ANY init grant statements missing from current repo:
special_1_1MS4_NamedGraph_grants () {
    local nggQuery='construct {?s ?g ?r} where {?s ?g ?r . ?s a :NamedGraph . ?g rdfs:subPropertyOf :hasAnyAccess} order by ?s ?r ?g'
    local curGrants=""
    local newGrants=""
    local result=0
    if curGrants=`doQueryInternal "all" text/plain "$nggQuery"`; then
        if newGrants=`queryInitializer "$nggQuery" "$NG_Repo" "$NG_Internal" `; then
            local ngg0="${tmpDir}/ngg0"
            local ngg1="${tmpDir}/ngg1"
            echo "$newGrants" > "$ngg0"
            echo "$curGrants" > "$ngg1"
            if diff "$ngg0" "$ngg1" | grep -q '^< '; then
                echo "====>>> Step 5. adding new Named Graph access grants."
                if doGraph "$NG_Internal" delete "text/plain" "$curGrants"; then
                    echo "INFO: OK - deleted current Named Graph access grants."
                else
                    echo "ERROR: FAIL: Failed deleting current Named Graph access grants."
                    result=1
                fi
                if doGraph "$NG_Internal" add "text/plain" "$newGrants"; then
                    echo "INFO: OK - added new Named Graph access grants."
                else
                    echo "ERROR: FAIL: Failed adding new Named Graph access grants."
                    result=1
                fi
            else
                echo "INFO: Step 5. - Already done: Named Graphs have all initial 1.1MS4 grants."
            fi
        else
            echo "ERROR: Step 5 failed to query initializer for NG grants."
            result=1
        fi
    else
        echo "ERROR: Step 5 failed to query exisitng NG grants."
        result=1
    fi
    return $result
}

# delete old FakeWorkflow-style claims."
special_1_1MS4_delete_claims () {
    echo "====>>> Step 4. Deleting old Fake Workflow-style claims."
    local deleteme="${tmpDir}/deleteme.txt"
    local wcQuery='construct {?s :hasWorkflowOwner ?v} where {?s :hasWorkflowOwner ?v OPTIONAL{?s ?g ?r . ?g rdfs:subPropertyOf :hasAnyAccess} FILTER(!bound(?g))}'
    if doQueryInternal "all" text/plain "$wcQuery" > "$deleteme"; then
        if doGraph "$NG_Metadata" delete "text/plain" "@$deleteme"; then
            echo "INFO: OK - deleted old Fake Workflow claims"
        else
            echo "ERROR: FAIL: Failed deleting old Fake Workflow claims"
        fi
    else
        echo "ERROR: FAIL: Failed querying old Fake Workflow claims"
    fi
}

# Step 4.1 - delete orphaned and bogus Edit Tokens
special_1_1MS4_delete_editTokens () {
    echo "====>>> Step 4.1 Delete Orphaned and Bogus EditTokens"
    local deleteme="${tmpDir}/deleteme.txt"
    local result=0

    # 1. easy case first: nuke orphaned edit tokens on NG_Intenral
    local orphanEditTokens='
        construct  {?s :hasEditToken ?t . ?t ?etp ?eto}  where {
         {graph :NG_Internal {?s :hasEditToken ?t} . OPTIONAL{?t ?etp ?eto}
          optional {?s :hasWorkflowOwner ?claim} filter(!bound(?claim)) }}'
    if doQueryInternal "all" text/plain "$orphanEditTokens" > "$deleteme"; then
        if [ -s "$deleteme" ]; then
            echo "INFO: ..Step 4.1.1: Removing orphaned editTokens."
            if doGraph "$NG_Internal" delete "text/plain" "@$deleteme"; then
                echo "INFO: OK - deleted orphaned editTokens."
            else
                echo "ERROR: FAIL: Failed deleting orphaned editTokens."
                result=1
            fi
        else
            echo "INFO: No orphaned editTokens found."
        fi
    else
        echo "ERROR: FAIL: Failed querying for orphaned editTokens."
        result=1
    fi

    # 2. find all graphs with bogus edit tokens, and dig them out.
    local bogusGraphQuery='select distinct ?g where {
        {graph ?g {?s :hasEditToken ?t}
        {{?g :ngType :NGType_Workspace} UNION {?g :ngType :NGType_Published}}}}'
    local graphs=""
    if graphs=`doQueryInternal "all" text/plain "$bogusGraphQuery"`; then
        if [ -n "$graphs" ]; then
            for graph in $graphs ; do
                echo "INFO: ..Step 4.1.2: Removing bogus editTokens in graph ${graph}"
                local bogusTokenQuery="construct  {?s :hasEditToken ?t . ?t ?etp ?eto}  where { \
                          {graph <${graph}> {?s :hasEditToken ?t} OPTIONAL { ?t ?etp ?eto}}}"
                if doQueryInternal "all" text/plain "$bogusTokenQuery" > "$deleteme"; then
                    if [ -s "$deleteme" ]; then
                        if doGraph "$graph" delete "text/plain" "@$deleteme"; then
                            echo "INFO: OK - deleted bogus edit tokens in graph $graph"
                        else
                            echo "ERROR: FAIL: Failed deleted bogus edit tokens in graph $graph"
                            result=1
                        fi
                    else
                        echo "INFO: ..Step 4.1: WARNING: No bogus editTokens found where they SHOULD have been, in graph ${graph}"
                        result=1
                    fi
                else
                    echo "ERROR: FAIL: Failed querying bogus edit tokens graph=$graph"
                    result=1
                fi
            done
        else
            echo "INFO: No bogus edit tokens found."
        fi
    else
        echo "ERROR: FAIL: Failed querying to get graphs of bogus edit tokens"
        result=1
    fi
    return $result
}





# delete OLD workflow state descriptions from NG_Internal
special_1_1MS4_workflowStates () {
    echo "====>>> Step 2. Deleting old WorkflowState descriptions in NG_Internal for 1.1MS4.00"
    local deleteme=""
    local wfQuery='construct {?s ?p ?o} where {graph :NG_Internal {?s a :WorkflowState; ?p ?o}}'
    if deleteme=`doQueryInternal "all" text/plain "$wfQuery"`; then
        if doGraph "$NG_Internal" delete "text/plain" "$deleteme"; then
            echo "INFO: OK - deleted old WorkflowState descriptions"
        else
            echo "ERROR: FAIL: Failed deleting old WorkflowState descriptions"
        fi
    else
        echo "ERROR: FAIL: Failed querying old WorkflowState descriptions"
    fi
}

# add the WFT descriptions to NG_Intenral graph
# do this by querying NG_Internal's initializer
# and then extracting relevant statements with a CONSTRUCT, load them back.
special_1_1MS4_add_WFTs () {
    echo "====>>> Step 3. Adding Role descriptions to NG_Internal for 1.1MS4.00"
    local query='construct {?s ?p ?o} where {?s a :WorkflowTransition; ?p ?o}'
    local wft=""
        if wft=`queryInitializer "$query" "$NG_Internal"`; then
            if doGraph "$NG_Internal" add "text/plain" "$wft"; then
                echo "INFO: OK - added new WorkflowTransition  descriptions"
            else
                echo "ERROR: FAIL: Failed adding new WorkflowTransition  descriptions"
            fi
        else
            echo "ERROR: FAIL: Failed querying new WorkflowTransition  descriptions"
        fi
}

# add the Role descriptions to NG_Intenral graph
# do this by querying NG_Internal's initializer
# and then extracting relevant statements with a CONSTRUCT, load them back.
special_1_1MS4_add_roles () {
    echo "====>>> Step 1. Adding Role descriptions to NG_Internal for 1.1MS4.00"
    local roles=""
    local query='construct {?s ?p ?o} where {?s a :Role; ?p ?o}'
    if roles=`queryInitializer "$query" "$NG_Internal"`; then
        if doGraph "$NG_Internal" add "text/plain" "$roles"; then
            echo "INFO: OK - added new Role descriptions"
        else
            echo "ERROR: FAIL: Failed adding new Role descriptions"
        fi
    else
        echo "ERROR: Failed querying NG_Internal initializer."
    fi
}

#   Special for 1.1MS4.00: transform user account metadata
#     a. remove Superuser (Admin) role from users who have other roles
#     b. add implicit Anonymous and Authenticated roles to all users if needed
#       (this will be done automatically by the Admin UI anyway)
# NOTE: Even though this is a one-off task only required for the 1.1MS2->MS4
# upgrade, keep this code around to serve as an example of how to ues the
# Edit User JSP as a REST service.
special_1_1MS4_user_roles () {
    echo "====>>> Step 6. Removing Unneeded User Administrator Roles for 1.1MS4.00"
    # ------- Part a. Remoev Superuser role from users who don't need it
    # Get list of usernames of such users; for each one get its roles
    # and invoke the Admin UI Edit User page to change them.  This is
    # necessary because removing the Superuser role also changes the RDBMS
    # and only that JSP (or the import-users service) can do both.
    local badAdminQuery="SELECT DISTINCT ?n WHERE
     {?s a <${repo}Person>; <${repo}hasPrincipalName> ?n;
      <${repo}hasRole> <${repo}Role_Superuser> ; <${repo}hasRole> ?r
      FILTER(!(?r = <${repo}Role_Superuser> ||
               ?r = <${repo}Role_Authenticated> ||
               ?r = <${repo}Role_Anonymous>))}"

    local count=0
    local userCount=0
    if fixUsers=`doTupleQuery all "$badAdminQuery"` ; then
        if [ -z "$fixUsers" ]; then
            echo "INFO: All unecessary Superuser roles have already been revoked."
        else
            for rawUser in $fixUsers; do
                userCount=`expr "$userCount" + 1`
                user=`echo "$rawUser" | tr -d '"'`
                # Skip yourself, SHOULD NOT be necessary, but you never know..
                if [ "$user" = "$adminUser" ]; then
                    echo "WARNING: Skipping admin user $user - but IT SHOULD NOT HAVE OTHER ROLES!"
                else
                    # get all non-admin roles for a user
                    getUserRolesQuery="SELECT DISTINCT ?r WHERE
                     {?s <${repo}hasPrincipalName> \"${user}\"; <${repo}hasRole> ?r
                      filter(?r != <${repo}Role_Superuser>)}"
                    if roles=`doTupleQuery all "$getUserRolesQuery"`; then
                        args="-d standalone=true -d username=${user} -d submit=true "
                        for ruri in $roles; do
                            args="${args} --data-urlencode role=${ruri} "
                        done
                        #
                        # Sneaky invocation of Admin UI servlet to mung user,
                        # because ONLY it can turn off Superuser in Derby DB as well:
                        echo "INFO: Revoking Superuser role for user: $user"
                        if hs=`curl -k -s -S -u "$login" -o /dev/null $args --write-out "%{http_code}" \
                             ${repoURL}/repository/admin/editUserPortlet.jsp`; then
                            if [ "$hs" != '200' ]; then
                                echo "FAIL: Failed in Admin UI req to set Roles on user=${user}, status=${hs}, args=$args"
                                commandStatus=1
                            fi
                        else
                            echo "FAIL: HTTP request to Admin UI failed, user=$user"
                            commandStatus=1
                        fi
                    else
                        echo "FAIL: Failed to get roles for user=$user"
                        commandStatus=1
                    fi
                    count=`expr "$count" + 1`
                fi
            done
            echo "INFO: Revoked $count unecessary Superuser roles on $userCount users."
        fi
    else
        echo "FAIL: Failed to get users with Superuser roles to revoke."
        commandStatus=1
    fi
}

# -------  Part b. Add explicit Anonymous and Authenticated roles to each user
special_1_1MS4_user_implicit () {
    echo "====>>> Step 7. Adding \"Implicit\" User Roles for 1.1MS4.00"
    local addRolesQuery="CONSTRUCT {?s <${repo}hasRole> <${repo}Role_Anonymous>,
                                    <${repo}Role_Authenticated>} WHERE
     {GRAPH <${repo}NG_Users> {?s a <${repo}Person>}}"
    local addRoles="${tmpDir}/addme.txt"
    rm -f "$addRoles"

    if doTupleQuery all "$addRolesQuery" > $addRoles; then
        if doGraph "$NG_Internal" add "text/plain" "@$addRoles"; then
            count=`grep -c 'Role_Authenticated' "$addRoles"`
            echo "INFO: Added implicit roles to $count users."
        else
            echo "FAIL: Failed adding the implicit role statements, though they were generated."
        fi
    else
        echo "FAIL: Failed generating the implicit role statements to add."
    fi
}

# usage: $0  expect  fromversion  toversion
# test rig used with Ontology Version test
mvTest () {
    local expect=$1
    shift
    modelVersionNewer "$1" "$2"
    cmd=$?
    echo "  modelVersionNewer=$cmd for \"${1}\" NEWER THAN \"${2}\""
    if [ $cmd -ne $expect ]; then
        echo "FAIL: Last test failed."
        exit 1
    fi
}

# test teh ontology version comparison
testVersionComparison () {
    echo "Testing Version Compare:"
    # these return 1
    mvTest 1 "0.6.4" "0.6.4 5727 2010-11-22 12:52:35"
    mvTest 1 "0.6.4 5727 2010-11-22 12:52:35" "0.6.4 5727 2010-11-22 12:52:35"
    mvTest 1 "0.6.4 5729 2010-11-22 12:52:35" "0.6.4 5727 2010-11-22 12:52:35"
    mvTest 1 "0.6.4 5727 2010-11-23 12:52:35" "0.6.4 5727 2010-11-22 12:52:35"
    mvTest 1 "0.6" "0.6.4 5727 2010-11-22 12:52:35"
    mvTest 1 "0.6.1" "0.6.4 5727 2010-11-22 12:52:35"
    mvTest 1 "0.6.1.2" "0.6.4 5727 2010-11-22 12:52:35"
    mvTest 1 "0.6.4.1" "0.6.4.1 5727 2010-11-22 12:52:35"
    mvTest 1 "GIGO"                           "(not loaded or not in ontology)"
    # these will return 0
    mvTest 0 "0.6.4.2" "0.6.4.1 5727 2010-11-22 12:52:35"
    mvTest 0 "0.6.4.1" "0.6.4 5727 2010-11-22 12:52:35"
    mvTest 0 "0.6.5" "0.6.4 5727 2010-11-22 12:52:35"
    mvTest 0 "0.7" "0.6.4 5727 2010-11-22 12:52:35"
    mvTest 0 "0.7.4" "0.6.4 5727 2010-11-22 12:52:35"
    mvTest 0 "1" "0.6.4 5727 2010-11-22 12:52:35"
    mvTest 0 "1.0.0" "0.6.4 5727 2010-11-22 12:52:35"
    mvTest 0 "0.6.4 5727 2010-11-22 12:52:35" "(not loaded or not in ontology)"
    mvTest 0 "0.1" "(not loaded or not in ontology)"
    mvTest 0 "0" "(not loaded or not in ontology)"
    echo "Test Passed!"
    echo ""
}


# ------------------ Main Thread ---------------------------------

# Sanity-check args
if [ "$1" = '--version' ]; then
    echo "$0 from release 1.2-MS2.02 SCM revision 8583"
    exit 0

# Secret test mode
elif [ "$1" = '--test' ]; then
    testVersionComparison
    exit 0

elif [ $# -lt 3 ]; then
    echo "ERROR: $usage"
    exit 1
fi

adminUser=$1
login="${1}:$2"
repoURL=$3

if echo "$repoURL"|egrep -qv '^https?:'; then
    echo "ERROR: Repository URL must be a valid http(s) URL: \"$repoURL\""
    exit 1
fi

commandStatus=0

# ------------ Upgrade Procedure Steps:

echo ""
echo "***** Checking and Upgrading Repository's Internal Graphs"
echo ""
if internalGraphs=`getInternalGraphs` ; then
    for uri in $internalGraphs ; do
        if internalAvailableVersionIsNewer "$uri" ; then
            if loadInternalGraph "$uri" ; then
                echo "INFO: Upgraded internal graph: $uri"
            else
                echo "WARNING: FAILED while trying to upgrade internal graph: $uri"
                commandStatus=1
            fi
        else
            echo "INFO: Skipping, internal graph is already up to date: $uri"
        fi
    done
else
    echo "WARNING: FAILED to get list of internal graphs, continuing.."
    commandStatus=1
fi

echo ""
echo "***** Checking and Upgrading Data Model Ontology"
echo ""
upgradeDataModelOntology

## Only needed for MS4 data but should be OK to run again since it
## cehcks and skips steps it's already done.

if true; then
  echo ""
  echo "***** *S*P*E*C*I*A*L*  UPGRADES for 1.2MS2.00"
  echo ""
  if special_1_2MS2_all ; then
      echo ""
      echo "OK - 1.2MS2 specials all succeeded."
  else
      echo ""
      echo "FAIL - there were failures in 1.2MS2 specials, YOU MUST RESTORE THE BACKUP."
      commandStatus=1
  fi
fi

exit $commandStatus
