#!/bin/sh

# eagle-i Data Repository Installation - First Step
# Create initial administrator user in the Apache Derby DB
#
#  WARNING: THIS *MUST* be executed with tomcat shut down.
#
#  Usage:  $0  [ --check | --version ]  admin-username passwd  repo-home-dir
#   --check -- ONLY reports current admin users in existing db.
#   --version -- just show versionof this script and exit
#   default action is to create the indicated user in a new DB.
#
#  Follows the DERBY_HOME environment variable, if set, to find system-wide
#  installation of Derby jars, otherwise uses the Derby packaged with repo.
#
# Started Nov 2010
# $Id: prepare-install.sh 6445 2011-01-08 00:01:50Z lcs14 $
# Author: Larry Stone

#-------- Command Args
if [ "$1" = "--version" ]; then
    echo "$0 from release ${project.version} SCM revision ${buildNumber}"
    exit 0
elif [ "$1" = "--check" ]; then
    testMode=true
    shift
fi

if [ "$#" -lt 3 ]; then
    echo "Usage: $0  [--check]  admin-username  admin-password   repo-home-directory"
fi
username="$1"
password="$2"

# repository home dir path - sanity-check it
repoHome="$3"
if [ ! -d "$repoHome" ]; then
    echo "ERROR, Repository home directory does not exist: $repoHome"
    exit 1
fi

#-------- Constants and Configuration

# name of RDBMS subdir of repo home
dbSubdir="db"

# name of expected Derby database
dbName="eagle-i-users.derby"

# Derby SQL listener entry point
derbyMain=org.apache.derby.tools.ij

# derby sysinfo utility
derbySysinfo=org.apache.derby.tools.sysinfo

# DDL to create new tables in Derby
createStms='
  CREATE TABLE Users (
    Username VARCHAR(64) PRIMARY KEY,
    Password VARCHAR(64) NOT NULL );
  CREATE TABLE Roles (
    RoleName VARCHAR(64) NOT NULL,
    Username VARCHAR(64) NOT NULL REFERENCES Users(Username),
    PRIMARY KEY (RoleName, Username));'

# SQL to initalize rows
initStms="
  INSERT INTO Users VALUES ('${username}', '${password}');
  INSERT INTO Roles VALUES ('superuser', '${username}');"

reportStms="SELECT Username FROM Roles WHERE RoleName='superuser';"

#-------- temp directory boilerplate - cleanup on exit
tmpDir="/tmp/ei-create-admin-$$"
mkdir $tmpDir
cleanup () {
  rm -rf $tmpDir
}
trap cleanup EXIT

#------ find Java the same way Tomcat does: check JAVA_HOME, JRE_HOME
if [ -z "$JRE_HOME" -a -n "$JAVA_HOME" ]; then
    JRE_HOME="$JAVA_HOME"
fi
if [ -n "${JRE_HOME}" ]; then
    _RUNJAVA="$JRE_HOME"/bin/java
else
    _RUNJAVA="java"
fi

#------- Setup for Derby - classpath and DERBY_HOME
if [ "${DERBY_HOME:-none}" = "none" ]; then
    DERBY_HOME=$repoHome
    DERBY_CP=`echo ${repoHome}/lib/derby*.jar | tr ' ' :`
    if echo "${DERBY_CP}" | fgrep -q "*"; then
        echo "ERROR, no Derby jars found in ${repoHome}/lib"
        exit 2
    fi
elif [ -d "${DERBY_HOME}" -a -f "${DERBY_HOME}/lib/derby.jar" ]; then
    echo "Using existing Apache Derby installation, DERBY_HOME=${DERBY_HOME}"
    DERBY_CP="${DERBY_HOME}/lib/derby.jar:${DERBY_HOME}/lib/derbytools.jar"
else
    echo "ERROR, DERBY_HOME is set but directory or jars do not exist: DERBY_HOME=${DERBY_HOME}"
    exit 2
fi

#------- Sanity check the derby setup:
#-- 1. only ONE version of derby in the classpath
#   Get count of unique version expressions in [...foo.jar] lines output by sysinfo
#   should be exactly 1.
versions=${tmpDir}/v.txt
if $_RUNJAVA -cp "${DERBY_CP}" $derbySysinfo > $versions ; then
    nversions=`perl -ne 'm/^\[.*\.jar\]\s+([\d\.]+)\s/ && print "$1\n";' "$versions"|uniq|wc -l`
    if echo "$nversions" | grep  -E -q '^[0-9[:space:]]+$'; then
        if [ "$nversions" -lt 1 ]; then
            echo "ERROR: No Derby versions on classpath! (how did sysinfo manage to run?)"
            exit 11
        elif [ "$nversions" -gt 1 ]; then
            echo "ERROR: There are multiple versions of Apache Derby on your classpath.  Please fix and retry:"
            echo "classpath = ${DERBY_CP}"
            exit 12
        fi
    else
        echo "WARNING: Failed getting count of different Apache Derby versions, sysinfo output:"
        cat "$versions"
    fi
else
    echo "ERROR: Failed to run Apache Derby sysinfo."
    exit 13
fi

#--- 2. check that derby embedded and tools code is available
if $_RUNJAVA -cp "${DERBY_CP}" $derbySysinfo -cp embedded tools > "$versions"; then
    if grep -q '^NOT FOUND' "$versions"; then
        echo "ERROR: Your Apache Derby installation is missing necessary components, see this report:"
        cat "$versions"
        exit 14
    fi
else
    echo "ERROR: Failed to run Apache Derby sysinfo -cp embedded tools."
    exit 13
fi

#------- Prepare and sanity check dbPath - it must NOT already exist.
dbDir="${repoHome}/${dbSubdir}"
if test -d "$dbDir" || mkdir -p "$dbDir"; then
    true
else
    echo "ERROR, failed to find or create Derby DB subdirectory: $dbDir"
    exit 3
fi
dbPath="${dbDir}/${dbName}"
jdbcURL="jdbc:derby:${dbPath}"
if [ -x "${dbPath}" -a "$testMode" != true ]; then
    echo "ERROR, Derby database already exists at this path: ${dbPath}"
    echo "       To continue, you must remove it, but BE SURE it does not contain user accounts you need to save."
    exit 10
fi

#------- Create and initialize tables
# Derby ij client options for all cmds
ijOpts="-Dderby.stream.error.file=${tmpDir}/derby.log -Dij.showNoConnectionsAtStart=true -Dij.showNoCountForSelect=true -Dij.showErrorCode=true"

# create and initialize tables:
ddlOut="${tmpDir}/ddl-result.txt"
if [ "$testMode" != true ]; then
    echo "$createStms $initStms" | $_RUNJAVA -cp "${DERBY_CP}" ${ijOpts} "-Dij.database=${jdbcURL};create=true" "-Dij.outfile=${ddlOut}" $derbyMain
    if [ $? -ne 0 ]; then
        echo "Failed invoking java; check JRE_HOME, classpath." ; exit 13
    fi
    if ! test -f "$ddlOut" || egrep -q '(ERROR|WARNING)' "$ddlOut"; then
        echo "ERROR, FAILED to create and initialize DB tables, record follows:"
        test -f "$ddlOut" && cat "$ddlOut"
        exit 11
    fi
fi
   
# check that desired username was created as superuser
reportOut="${tmpDir}/report-out.txt"
echo "$reportStms" | $_RUNJAVA -cp "${DERBY_CP}" ${ijOpts} "-Dij.database=${jdbcURL}" "-Dij.outfile=${reportOut}" $derbyMain
if ! test -f "$reportOut" || egrep -q '(ERROR|WARNING)' "$reportOut"; then
    echo "ERROR, FAILED to query the DB, record follows:"
    test -f "$reportOut" && cat "$reportOut"
    exit 13
elif [ "$testMode" = true ] ||  grep -q "^$username" "$reportOut"; then
    echo "---all superuser logins---"
    tail -n +4 "$reportOut" | grep -v '^ij>' | sed 's/^/  /'
else
    echo "ERROR, FAILED to create administrator login although SQL worked; here is the log: "
    test -f "$ddlOut" && cat "$ddlOut"
    exit 12
fi
