package org.eaglei.repository;

import org.eaglei.repository.vocabulary.REPO;
import org.eaglei.repository.vocabulary.FOAF;
import org.eaglei.repository.util.Utils;
import org.eaglei.repository.util.SPARQL;
import org.eaglei.repository.status.ForbiddenException;
import org.eaglei.repository.status.BadRequestException;
import java.util.Arrays;
import java.util.ArrayList;
import java.util.List;
import java.util.Set;
import java.util.HashSet;
import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;
import org.apache.log4j.Logger;
import org.apache.log4j.LogManager;

import org.openrdf.OpenRDFException;
import org.openrdf.model.URI;
import org.openrdf.model.Literal;
import org.openrdf.model.Value;
import org.openrdf.model.impl.LiteralImpl;
import org.openrdf.repository.RepositoryConnection;
import org.openrdf.repository.RepositoryException;
import org.openrdf.model.ValueFactory;
import org.openrdf.model.vocabulary.RDF;
import org.openrdf.model.vocabulary.RDFS;
import org.openrdf.query.TupleQuery;
import org.openrdf.query.TupleQueryResultHandlerBase;
import org.openrdf.query.TupleQueryResultHandlerException;
import org.openrdf.query.BindingSet;
import org.openrdf.query.QueryLanguage;
import org.openrdf.query.Dataset;
import org.openrdf.query.impl.DatasetImpl;
import org.openrdf.query.MalformedQueryException;

import org.eaglei.repository.servlet.WithRepositoryConnection;
import org.eaglei.repository.rid.RIDGenerator;

/**
 * User object model, reflects the user's properties in the both RDBMS
 * and RDF database.  Also manages the RDF descriptions of users.
 *
 * Named Graph Usage:
 *  1. Almost all statements about a repo User are in the graph named "repo:NG_Users".
 *  2. Ony the :hasPrincipalName statement is on the NG_Internal graph
 *     sicne it should not be exposed; there is no reason to let login
 *     names leak and it's good security practice to keep them hidden,
 *     since that just makes dictionary attacks so much easier.
 *
 * @author Larry Stone
 * Started April 26, 2010
 * @version $Id: $
 */
public class User
{
    private static Logger log = LogManager.getLogger(User.class);

    /** Named graph where we create User objects.  Note taht the
     *  hasPrincipalName is still on the Internal graph, it's confidential.
     */
    private static final URI USER_GRAPH = REPO.NG_USERS;

    /** dataset that includes relevant graphs - internal and user */
    private static DatasetImpl userDataset = SPARQL.copyDataset(SPARQL.InternalGraphs);
    static {
        SPARQL.addGraph(userDataset, USER_GRAPH);
        log.debug("User Dataset = "+Utils.prettyPrint(userDataset));
    }

    private URI uri;  /*  immutable, the subject of this instance */
    private String username; /* immutable, "user" (i.e. Principal) of this instance */
    private String firstName = null;
    private String lastName = null;
    private String mbox = null;
    private Set<Role> roles = new HashSet<Role>();
    private boolean dirty = false;    // anything modified?

    /**
     * @return dataset of graphs needed to query for all repo users' data
     */
    public static Dataset getUserDataset()
    {
        return userDataset;
    }

    private User(URI uri, String username)
    {
        super();
        this.uri = uri;
        this.username = username;
    }

    // note that we NEED the "order by" because the handler expects
    // all the hasRole bindings for a given URI to be clumped together.
    private final static String userForURIQuery =
          "SELECT * WHERE { ?uri a <"+REPO.PERSON+"> ; \n"+
          "<"+REPO.HAS_PRINCIPAL_NAME+"> ?hasPrincipalName .\n"+
          "  OPTIONAL { ?uri <"+FOAF.FIRST_NAME+"> ?firstName }\n"+
          "  OPTIONAL { ?uri <"+FOAF.SURNAME+"> ?surname }\n"+
          "  OPTIONAL { ?uri <"+FOAF.MBOX+"> ?mbox }\n"+
          "  OPTIONAL { ?uri <"+REPO.HAS_ROLE+"> ?hasRole . ?hasRole <"+RDFS.LABEL+"> ?roleLabel }\n"+
          " } ORDER BY ?uri";

    // assumes uri is defined
    // load values from query results.
    private void load(RepositoryConnection rc)
        throws ServletException
    {
        try {
            log.debug("Single user SPARQL query = "+userForURIQuery);
            TupleQuery q = rc.prepareTupleQuery(QueryLanguage.SPARQL, userForURIQuery);
            q.setDataset(userDataset);
            q.clearBindings(); // needed if re-using query
            if (uri != null)
                q.setBinding("uri", uri);
            else if (username != null)
                q.setBinding("hasPrincipalName", new LiteralImpl(username));
            else
                throw new ServletException("Cannot load repository user description, because neither URI nor username was given.");
            // do NOT infer any roles.
            q.setIncludeInferred(false);
            q.evaluate(new oneUserHandler(this));
        } catch (MalformedQueryException e) {
            log.error("Rejecting malformed query:"+e);
            throw new ServletException(e);
        } catch (OpenRDFException e) {
            log.error(e);
            throw new ServletException(e);
        }
    }

    /**
     * <p>Get User object matching subject URI.</p>
     *
     * @param request a {@link javax.servlet.http.HttpServletRequest} object.
     * @param uri subject URI, expected to be of type :Person
     * @return a {@link org.eaglei.repository.User} object.
     * @throws javax.servlet.ServletException if any.
     */
  /*** XXX deprecated
    public static User find(HttpServletRequest request, URI uri)
        throws ServletException
    {
        User result =  new User(uri, null);
        result.load(WithRepositoryConnection.get(request));
        log.debug("User.find("+uri.stringValue()+") => "+result);
        return result;
    }
   ***/

    /**
     * find a record by username, i.e. login principal name
     * Returns null if NOT found..
     * NOTE that user instance gets auto-created for a "new" principal
     * upon login, so this CAN be called (by that code) when checking first
     * before creating a new Person instance.
     *
     * @param request a {@link javax.servlet.http.HttpServletRequest} object.
     * @param pname principal name (from user account RDBMS).
     * @return the User object, or null if none found.
     * @throws javax.servlet.ServletException if any.
     */
    public static User findUsername(HttpServletRequest request, String pname)
        throws ServletException
    {
        User result =  new User(null, pname);
        result.load(WithRepositoryConnection.get(request));
        log.debug("User.findUsername("+pname+") => "+result);
        return result.uri == null ? null : result;
    }

    /**
     * <p>Get all known Users from RDF DB - may not match auth'n DB.
     * <br>XXX TODO should add start and count for pagination
     * </p>
     *
     * @param request a {@link javax.servlet.http.HttpServletRequest} object.
     * @return resulting Users in a {@link java.lang.Iterable} object.
     * @throws javax.servlet.ServletException if any.
     */
    public static Iterable<User> findAll(HttpServletRequest request)
        throws ServletException
    {
        RepositoryConnection rc = WithRepositoryConnection.get(request);
        List<User> result = new ArrayList<User>();

        try {
            log.debug("All user SPARQL query = "+userForURIQuery);
            TupleQuery q = rc.prepareTupleQuery(QueryLanguage.SPARQL, userForURIQuery);
            q.setDataset(userDataset);
            // do NOT infer any roles.
            q.setIncludeInferred(false);
            q.evaluate(new allUserHandler(result));
        } catch (MalformedQueryException e) {
            log.error("Rejecting malformed query:"+e);
            throw new ServletException(e);
        } catch (OpenRDFException e) {
            log.error(e);
            throw new ServletException(e);
        }
        return result;
    }

    /**
     * <p>Getter for the field <code>uri</code>.</p>
     *
     * @return a {@link org.openrdf.model.URI} object.
     */
    public URI getURI()
    {
        return uri;
    }
    /**
     * <p>Getter for the field <code>username</code>.</p>
     *
     * @return a {@link java.lang.String} object.
     */
    public String getUsername()
    {
        return username;
    }
    /**
     * <p>Getter for the field <code>firstName</code>.</p>
     *
     * @return a {@link java.lang.String} object or null if not set.
     */
    public String getFirstName()
    {
        return firstName;
    }
    /**
     * <p>Getter for the field <code>lastName</code>.</p>
     *
     * @return a {@link java.lang.String} object  or null if not set.
     */
    public String getLastName()
    {
        return lastName;
    }
    /**
     * <p>Getter for the field <code>mbox</code>.</p>
     *
     * @return a {@link java.lang.String} object  or null if not set.
     */
    public String getMbox()
    {
        return mbox;
    }

    /**
     * Computes the most informative "label" for a user, to present in UI.
     * Format is, ideally,   "username (firstname lastname)"
     * But it reverts to bare username or even URI if none is available.
     *
     * @return label as a {@link java.lang.String} object.
     */
    public String getTitle()
    {
        if (username != null) {
            if (firstName != null || lastName != null)
                return username+" ("+((firstName == null) ? lastName :
                                       (lastName == null) ? firstName : firstName+" "+lastName)+")";
            else
                return username;
        } else
            return uri.getLocalName();
    }

    /**
     * <p>Get <code>Roles</code>.</p>
     *
     * @return all known Roles as an array of {@link org.eaglei.repository.Role} objects.
     */
    public Role[] getRoles()
    {
        return roles.toArray(new Role[roles.size()]);
    }

    /**
     * has role predicate - the P suffix is a Lisp thing
     *
     * @param r role as a {@link org.eaglei.repository.Role} object.
     * @return a boolean, true if this User has indicated role.
     */
    public boolean hasRoleP(Role r)
    {
        return roles.contains(r);
    }

    /**
     * <p>hasRoleP</p>
     *
     * @param ru role as a {@link org.openrdf.model.URI} object.
     * @return a boolean, true if this User has indicated role.
     */
    public boolean hasRoleP(URI ru)
    {
        for (Role r : roles) {
            if (r.getURI().equals(ru))
                return true;
        }
        return false;
    }


    /**
     * <p>alternate create with access check off</p>
     *
     * @param request a {@link javax.servlet.http.HttpServletRequest} object.
     * @param username principal name, a {@link java.lang.String} object.
     * @return the new {@link org.eaglei.repository.User} object.
     * @throws javax.servlet.ServletException if any.
     */
    public static User create(HttpServletRequest request, String username)
        throws ServletException
    {
        return create(request, username, false);
    }
    /**
     * <p>
     * Create new user instance for given username.
     * Returns User object with URI and username set; nothing else.
     * Fails if there is an existing user with the same principal.
     * Skip access test if skipAccessCheck bit is set -- dangerous, only for
     *  bootstrapping the authentication.
     *  </p>
     *
     * @param request a {@link javax.servlet.http.HttpServletRequest} object.
     * @param username principal name, a {@link java.lang.String} object.
     * @param skipAccessCheck a boolean.
     * @return the new {@link org.eaglei.repository.User} object.
     * @throws javax.servlet.ServletException if any.
     */
    public static User create(HttpServletRequest request, String username, boolean skipAccessCheck)
        throws ServletException
    {
        if (!skipAccessCheck && !Access.hasPermissionOnUser(request, username))
            throw new ForbiddenException("Not allowed to create user: "+username);

        RepositoryConnection rc = WithRepositoryConnection.get(request);
        ValueFactory vf = rc.getValueFactory();
        User result = new User(null, username);

        try {
            // sanity check, ensure there is no user with this principal already
            Literal lpname = vf.createLiteral(username);
            if (rc.hasStatement(null, REPO.HAS_PRINCIPAL_NAME, lpname, false, REPO.NG_INTERNAL))
                throw new BadRequestException("Cannot create user: there is already a repository user with the login principal name (username) \""+username+"\"");

            result.uri = vf.createURI(DataRepository.getInstance().getDefaultNamespace(),
                                      RIDGenerator.getInstance().newID().toString());
            rc.add(result.uri, RDF.TYPE, REPO.PERSON, USER_GRAPH);
            rc.add(result.uri, RDFS.LABEL, lpname, USER_GRAPH);
            rc.add(result.uri, REPO.HAS_PRINCIPAL_NAME, lpname, REPO.NG_INTERNAL);

            log.debug("create: created new User instance, username="+username+", uri="+result.uri);
            result.dirty = true;
            return result;
        } catch (RepositoryException e) {
            log.error("Failed creating user URI: ",e);
            throw new ServletException("Failed creating user URI: ",e);
        }
    }

    // Setters

    /**
     * Change value of first name. Setting it to null clears it.
     *
     * @param request a {@link javax.servlet.http.HttpServletRequest} object.
     * @param name first name a {@link java.lang.String} object.
     * @throws javax.servlet.ServletException if any.
     */
    public void setFirstName(HttpServletRequest request, String name)
        throws ServletException
    {
        firstName = name;
        setMetadataInternal(request, FOAF.FIRST_NAME, name);
    }

    /**
     * <p>Setter for the field <code>lastName</code>.</p>
     *
     * @param request a {@link javax.servlet.http.HttpServletRequest} object.
     * @param name a {@link java.lang.String} object.
     * @throws javax.servlet.ServletException if any.
     */
    public void setLastName(HttpServletRequest request, String name)
        throws ServletException
    {
        lastName = name;
        setMetadataInternal(request, FOAF.SURNAME, name);
    }
    /**
     * <p>Setter for the field <code>mbox</code>.</p>
     *
     * @param request a {@link javax.servlet.http.HttpServletRequest} object.
     * @param mbox a {@link java.lang.String} object.
     * @throws javax.servlet.ServletException if any.
     */
    public void setMbox(HttpServletRequest request, String mbox)
        throws ServletException
    {
        this.mbox = mbox;
        setMetadataInternal(request, FOAF.MBOX, mbox);
    }

    // update one of the metadata properties.
    private void setMetadataInternal(HttpServletRequest request, URI property, String newVal)
        throws ServletException
    {
        if (!Access.hasPermissionOnUser(request, username))
            throw new ForbiddenException("Not allowed to modify user: "+username);
        try {
            RepositoryConnection rc = WithRepositoryConnection.get(request);
            boolean hasProp = rc.hasStatement(uri, property, null, false, USER_GRAPH);
            boolean hasLabel = rc.hasStatement(uri, RDFS.LABEL, null, false, USER_GRAPH);
            ValueFactory vf = rc.getValueFactory();
            if (hasProp)
                rc.remove(uri, property, null, USER_GRAPH);
            if (newVal != null)
                rc.add(uri, property, vf.createLiteral(newVal), USER_GRAPH);
            // recompute label
            String label = getTitle();
            log.debug("Setting User label, uri="+uri+", label="+label);
            if (hasLabel)
                rc.remove(uri, RDFS.LABEL, null, USER_GRAPH);
            rc.add(uri, RDFS.LABEL, vf.createLiteral(label), USER_GRAPH);

            // XXX this may lie if we didn't change anything, but who cares.
            dirty = true;
        } catch (RepositoryException e) {
            throw new ServletException(e);
        }
    }

    /**
     * <p>addRole - add a role</p>
     *
     * @param request a {@link javax.servlet.http.HttpServletRequest} object.
     * @param ru Role to add, as a {@link org.openrdf.model.URI} object.
     * @throws javax.servlet.ServletException if any.
     */
    public void addRole(HttpServletRequest request, URI ru)
        throws ServletException
    {
        addRole(request, Role.find(request, ru));
    }

    /**
     * <p>addRole - add a role</p>
     *
     * @param request a {@link javax.servlet.http.HttpServletRequest} object.
     * @param r role to add as a {@link org.eaglei.repository.Role} object.
     * @throws javax.servlet.ServletException if any.
     */
    public void addRole(HttpServletRequest request, Role r)
        throws ServletException
    {
        addRole(request, r, false);
    }
    /**
     * <p>addRole - add a role</p>
     *
     * @param request a {@link javax.servlet.http.HttpServletRequest} object.
     * @param r role to add as a {@link org.eaglei.repository.Role} object.
     * @param skipAccessCheck a boolean, true to skip access control check.
     * @throws javax.servlet.ServletException if any.
     */
    public void addRole(HttpServletRequest request, Role r, boolean skipAccessCheck )
        throws ServletException
    {
        if (!skipAccessCheck && !Access.isSuperuser(request))
            throw new ForbiddenException("Only the administrator is allowed to modify user roles.");

        if (hasRoleP(r))
            return;
        RepositoryConnection rc = WithRepositoryConnection.get(request);
        try {
            if (roles.add(r)) {
                rc.add(uri, REPO.HAS_ROLE, r.getURI(), REPO.NG_INTERNAL);
                dirty = true;
            }
        } catch (RepositoryException e) {
            log.error("Failed adding role ",e);
            throw new ServletException("Failed adding role ",e);
        }
    }

    /**
     * <p>removeRole</p>
     *
     * @param request a {@link javax.servlet.http.HttpServletRequest} object.
     * @param ru role to remove as a {@link org.openrdf.model.URI} object.
     * @throws javax.servlet.ServletException if any.
     */
    public void removeRole(HttpServletRequest request, URI ru)
        throws ServletException
    {
        removeRole(request, Role.find(request, ru));
    }

    /**
     * <p>removeRole</p>
     *
     * @param request a {@link javax.servlet.http.HttpServletRequest} object.
     * @param r role to remove as a {@link org.eaglei.repository.Role} object.
     * @throws javax.servlet.ServletException if any.
     */
    public void removeRole(HttpServletRequest request, Role r)
        throws ServletException
    {
        if (!Access.isSuperuser(request))
            throw new ForbiddenException("Only the administrator is allowed to modify user roles.");

        if (!hasRoleP(r))
            return;
        RepositoryConnection rc = WithRepositoryConnection.get(request);
        try {
            rc.remove(uri, REPO.HAS_ROLE, r.getURI());
            roles.remove(r);
            dirty = true;
        } catch (RepositoryException e) {
            log.error("Failed adding role ",e);
            throw new ServletException("Failed removing role ",e);
        }
    }


    /**
     * <p>update -
     * push any changes into repository - needs superuser
     * also recomputes rdfs:label
     * Does not need access checks because the only operatiosn taht
     * set "dirty" are access-protected..
     * </p>
     *
     * @param request a {@link javax.servlet.http.HttpServletRequest} object.
     * @throws javax.servlet.ServletException if any.
     */
    public void update(HttpServletRequest request)
        throws ServletException
    {
        RepositoryConnection rc = WithRepositoryConnection.get(request);
        try {

            // label is either full name (if avail.) or principal
            if (dirty) {
                rc.commit();
                Access.decacheUser(request, this);
            }
            dirty = false;
        } catch (RepositoryException e) {
            log.error("Failed updating user, URI="+uri,e);
            throw new ServletException("Failed updating user, URI="+uri,e);
        }
    }

    /**
     * <p>finalize - flag error if GC'ing a dirty instance, its changes are lost. </p>
     */
    protected void finalize()
    {
        if (dirty)
            log.error("finalize: about to destroy a User with dirty flag set, CHANGES WILL BE LOST.  Current state: "+toString());
    }

    /**
     * <p>toString</p>
     *
     * @return a {@link java.lang.String} object.
     */
    public String toString()
    {
        String rs = roles == null ? "{null}" : Arrays.deepToString(getRoles());
        String uu = uri == null ? "(not set)" : uri.toString();

        return "<#User: uri="+uu+
            ", username="+username+", firstName="+firstName+
            ", lastName="+lastName+", mbox="+mbox+", roles="+rs+">";
    }

    // tuple query result handler to gather user contents
    private static class oneUserHandler extends TupleQueryResultHandlerBase
    {
        private User result = null;
        private boolean firstHit = true;

        public oneUserHandler(User u)
        {
            super();
            oneUserHandler.this.result = u;
        }
        protected oneUserHandler()
        {
            super();
        }
        public void endQueryResult()
        {
            // never got any results
            if (firstHit)
                log.debug("Failed to get any query results, User="+result);
        }
        public void handleSolution(BindingSet bs)
            throws TupleQueryResultHandlerException
        {
            firstHit = populateUser(bs, result, firstHit);
        }

        // read the result record into a User - shared with "all" processor
        protected boolean populateUser(BindingSet bs, User u, boolean first)
            throws TupleQueryResultHandlerException
        {
            if (first) {
                Value puri = bs.getValue("uri");
                if (puri != null && puri instanceof URI && u.uri == null)
                    u.uri = (URI)puri;
                Value pn = bs.getValue("hasPrincipalName");
                if (pn != null && u.username == null)
                    u.username = pn instanceof Literal ? ((Literal)pn).getLabel() : pn.stringValue();
                Value firstName =  bs.getValue("firstName");
                if (firstName != null && firstName instanceof Literal)
                    u.firstName = ((Literal)firstName).getLabel();
                Value sur = bs.getValue("surname");
                if (sur != null && sur instanceof Literal)
                    u.lastName = ((Literal)sur).getLabel();
                Value mbox = bs.getValue("mbox");
                if (mbox != null && mbox instanceof Literal)
                    u.mbox = ((Literal)mbox).getLabel();
                Value role = bs.getValue("mbox");
                first = false;
                log.debug("got single User hit, username="+pn);
            }
            Value role = bs.getValue("hasRole");
            Value roleLabel = bs.getValue("roleLabel");
            if (role != null) {
                if (roleLabel == null)
                    log.warn("Got role w/o label, uri="+role.toString());
                else {
                    Role rl = Role.find((URI)role, ((Literal)roleLabel).getLabel());
                    log.debug("Adding role to single User, username="+u.username+", role="+rl );
                    u.roles.add(rl);
                }
            }
            if (u.uri == null || u.username == null)
                throw new TupleQueryResultHandlerException("User description data failed sanity check: Either uri or principal name is null: uri="+u.uri+", username="+u.username);
            return first;
        }
    }

    // tuple query result handler to gather user contents
    private static class allUserHandler extends oneUserHandler
    {
        private List<User> result = null;
        private boolean curFirst = true;
        private User curUser = null;
        private URI curURI = null;

        public allUserHandler(List<User> u)
        {
            super();
            allUserHandler.this.result = u;
        }

        public void endQueryResult()
        {
            finishCurrent();
        }
        public void handleSolution(BindingSet bs)
            throws TupleQueryResultHandlerException
        {
            Value newURI = bs.getValue("uri");
            if (newURI == null || !(newURI instanceof URI))
                log.error("Should not get null or non-URI result in allUserHandler: "+newURI);

            // start a new one
            else if (curURI == null || !newURI.equals(curURI)) {
                finishCurrent();
                curURI = (URI)newURI;
                curUser = new User(curURI, null);
                curFirst = true;
            }
            curFirst = populateUser(bs, curUser, curFirst);
        }

        // close off current User, if any, and push onto result list
        private void finishCurrent()
        {
            if (curUser != null) {
                // never got any results
                if (curFirst)
                    log.warn("Failed to get any query results (all), User="+result);
                result.add(curUser);
                curUser = null;
            }
        }
    }

    /** {@inheritDoc} */
    public boolean equals(Object o)
    {
        return o instanceof User && uri != null && uri.equals(((User)o).uri);
    }
}
