package org.eaglei.repository.servlet;

import java.util.Arrays;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.ServletException;

import org.apache.log4j.Logger;
import org.apache.log4j.LogManager;

import org.openrdf.query.resultio.TupleQueryResultFormat;
import org.openrdf.query.resultio.QueryResultIO;
import org.openrdf.query.TupleQueryResultHandler;
import org.openrdf.query.impl.MapBindingSet;
import org.openrdf.OpenRDFException;
import org.openrdf.model.URI;
import org.openrdf.model.Statement;
import org.openrdf.model.Value;
import org.openrdf.model.ValueFactory;
import org.openrdf.repository.RepositoryConnection;
import org.openrdf.repository.RepositoryException;
import org.openrdf.repository.RepositoryResult;
import org.eaglei.repository.Formats;
import org.eaglei.repository.Access;
import org.eaglei.repository.status.ForbiddenException;

/**
 * Internal Admin-Only servlet to list statements -- used mainly
 * to locate the context (named graph) of a statement.
 *
 * Usage: supply either a value (URI, object may be literal) or null
 * for each arg:  subject, predicate, object
 * Optional format arg is MIME type like the other servlets.
 * Prints a tabular query result of all matching statements.
 *
 * This service is no longer necessary now that Datasets are structured
 * so that the GRAPH keyword works in SPARQL tuple queries to bind a
 * named graph.  There is still no way to get graphs out of CONSTRUCT
 * queries, however.
 *
 * @author Larry Stone
 * @version $Id: $
 */
public class GetStatements extends RepositoryServlet
{
    private static Logger log = LogManager.getLogger(GetStatements.class);

    private static String columnNames[] = { "subject", "predicate", "object", "context" };

    /** {@inheritDoc} */
    protected void doPost(HttpServletRequest request, HttpServletResponse response)
        throws ServletException, java.io.IOException
    {
        doGet(request, response);
    }

    /** {@inheritDoc} */
    protected void doGet(HttpServletRequest request, HttpServletResponse response)
        throws ServletException, java.io.IOException
    {
        String subject = request.getParameter("subject");
        String predicate = request.getParameter("predicate");
        String object = request.getParameter("object");
        String format = request.getParameter("format");

        if (!Access.isSuperuser(request))
            throw new ForbiddenException("This service is only available to administrators.");

        if (format == null)
            format = "text/html";

        // synthesize a SPARQL tuple query response by creating a result
        // writer and driving it directly with a made up binding listing:
        String mimeType = Formats.negotiateTupleContent(request, format, Formats.DEFAULT_TUPLE_MIMETYPE);
        TupleQueryResultFormat tqf = QueryResultIO.getWriterFormatForMIMEType(mimeType);
        if (tqf == null) {
            log.error("Failed to get tuple query format, for mime="+mimeType);
            throw new ServletException("Failed to get tuple query format that SHOULD have been available, for mime="+mimeType);
        } else {
            response.setContentType(mimeType);
            RepositoryResult<Statement> rr = null;
            RepositoryConnection rc = WithRepositoryConnection.get(request);
            ValueFactory vf = rc.getValueFactory();
            try {
                TupleQueryResultHandler h = QueryResultIO.createWriter(tqf, response.getOutputStream());
                h.startQueryResult(Arrays.asList(columnNames));
                MapBindingSet bs = new MapBindingSet(4);

                URI su = (subject == null) ? null : vf.createURI(subject);
                URI pu = (predicate == null) ? null : vf.createURI(predicate);
                Value vu = null;
                if (object != null) {
                    try {
                        vu = vf.createURI(object);
                    } catch (Exception e) {
                        log.debug("Object was NOT a valid URI, try literal: "+object);
                        vu = vf.createLiteral(object);
                    }
                }
                try {
                    // XXX maybe someday make inferred an option?
                    log.debug("Try getStatements("+su+", "+pu+", "+vu+", false)");
                    rr = rc.getStatements(su, pu, vu, false);
                    while (rr.hasNext()) {
                        Statement s = rr.next();
                        bs.clear();
                        bs.addBinding("subject", s.getSubject());
                        bs.addBinding("predicate", s.getPredicate());
                        bs.addBinding("object", s.getObject());
                        bs.addBinding("context", s.getContext());
                        h.handleSolution(bs);
                    }
                    h.endQueryResult();
                } finally {
                    rr.close();
                }
                 
            } catch (RepositoryException e) {
                log.error(e);
                throw new ServletException(e);

            } catch (OpenRDFException e) {
                log.error(e);
                throw new ServletException(e);
            }
        }
    }
}
