package org.eaglei.repository;

import org.eaglei.repository.status.NotAcceptableException;
import java.util.Collection;
import java.util.Set;
import java.util.HashSet;

import org.apache.log4j.Logger;
import org.apache.log4j.LogManager;

import org.openrdf.rio.RDFFormat;
import org.openrdf.rio.RDFWriterFactory;
import org.openrdf.rio.RDFWriterRegistry;
import org.openrdf.query.resultio.BooleanQueryResultFormat;
import org.openrdf.query.resultio.TupleQueryResultFormat;
import org.openrdf.query.resultio.TupleQueryResultWriterFactory;
import org.openrdf.query.resultio.TupleQueryResultWriterRegistry;

import javax.servlet.http.HttpServletRequest;

import org.eaglei.repository.format.SPARQLTextWriterFactory;
import org.eaglei.repository.format.SPARQLHTMLWriterFactory;
import org.eaglei.repository.format.RDFNQuadsWriterFactory;
import org.eaglei.repository.format.RDFHTMLWriterFactory;

/**
 * Constants and utilities related to ALL serialization formats:
 * includes RDF serialization as well as Boolean and Tuple query results.
 *
 * @author Larry Stone
 * @version $Id: $
 */
public class Formats
{
    private static Logger log = LogManager.getLogger(Formats.class);

    /** Constant <code>DEFAULT_RDF_MIMETYPE="application/rdf+xml"</code> */
    public final static String DEFAULT_RDF_MIMETYPE = "application/rdf+xml";

    /** Constant <code>DEFAULT_BOOLEAN_MIMETYPE="application/sparql-results+xml"</code> */
    public final static String DEFAULT_BOOLEAN_MIMETYPE = "application/sparql-results+xml";

    /** Constant <code>DEFAULT_TUPLE_MIMETYPE="application/sparql-results+xml"</code> */
    public final static String DEFAULT_TUPLE_MIMETYPE = "application/sparql-results+xml";

    /** media (MIME) types of acceptable RDF formats (for content negotiation) */
    private static Set<String> mtRDF = new HashSet<String>();
    private static RDFWriterRegistry RDFWregistry = RDFWriterRegistry.getInstance();

    static {

        // Add entries for our new RDF formats, NQuads and HTML
        RDFWregistry.add(new RDFNQuadsWriterFactory());
        RDFWregistry.add(new RDFHTMLWriterFactory());

        for (RDFWriterFactory w : RDFWregistry.getAll()) {
            RDFFormat rf = w.getRDFFormat();
            for (String mt : rf.getMIMETypes()) {
                mtRDF.add(mt);
                log.debug("Adding RDF Serialization: mime="+mt+", format="+rf);
            }
        }
    }

    /** media types of Tuple query result formats */
    private static Set<String> mtTuple = new HashSet<String>();
    static {
        // Add entries for our text and HTML formats:
        TupleQueryResultWriterRegistry registry = TupleQueryResultWriterRegistry.getInstance();
        registry.add(new SPARQLTextWriterFactory());
        registry.add(new SPARQLHTMLWriterFactory());

        for (TupleQueryResultWriterFactory w : registry.getAll()) {
            TupleQueryResultFormat tf = w.getTupleQueryResultFormat();
            for (String mt : tf.getMIMETypes()) {
                mtTuple.add(mt);
                log.debug("Adding Tuple serialization: mime="+mt+", format="+tf);
            }
        }
    }

    /** media types of Boolean query result formats */
    private static Set<String> mtBoolean = new HashSet<String>();
    static {
        for (BooleanQueryResultFormat bff : BooleanQueryResultFormat.SPARQL.values()) {
            for (String mt : bff.getMIMETypes()) {
                mtBoolean.add(mt);
                log.debug("Adding Boolean serialization: mime="+mt+", format="+bff);
            }
        }
    }

    /**
     * Get negotiated MIME type for RDF serialization format
     *
     * @param request a {@link javax.servlet.http.HttpServletRequest} object.
     * @param format a {@link java.lang.String} object.
     * @param dflt a {@link java.lang.String} object.
     * @return mime type of negotated content, never null (unless dflt is null)
     */
    public static String negotiateRDFContent(HttpServletRequest request,
                                             String format, String dflt)
    {
        return negotiateContent(request, format, mtRDF, dflt);
    }

    /**
     * Get negotiated MIME type for Tuple serialization format
     *
     * @param request a {@link javax.servlet.http.HttpServletRequest} object.
     * @param format a {@link java.lang.String} object.
     * @param dflt a {@link java.lang.String} object.
     * @return mime type of negotated content, never null (unless dflt is null)
     */
    public static String negotiateTupleContent(HttpServletRequest request,
                                             String format, String dflt)
    {
        return negotiateContent(request, format, mtTuple, dflt);
    }

    /**
     * Get negotiated MIME type for Boolean serialization format
     *
     * @param request a {@link javax.servlet.http.HttpServletRequest} object.
     * @param format a {@link java.lang.String} object.
     * @param dflt a {@link java.lang.String} object.
     * @return mime type of negotated content, never null (unless dflt is null)
     */
    public static String negotiateBooleanContent(HttpServletRequest request,
                                             String format, String dflt)
    {
        return negotiateContent(request, format, mtBoolean, dflt);
    }

    /**
     * Returns MIME type (or null if none) that satisfied accept
     * from given collection of types.
     */
    // XXX FIX ME!
    // Quick but wrong implementation of Accept MIME type matchign:
    // Choose a resolvable format - must be in map - mabye default if no others match
    // returns chosen format or null if none (that's a bad error, dflt should always work)
    // XXX this SHOULD get more sophisticated, see RFC2068 p. 95 or so
    //    deal with */*, matching the "q" param, etc.. also need to
    //    generate an HTTP 406 (Not Acceptable) error when appropriate.
    // XXX ALSO need to match type/* and return that value..
    // XXX this is WRONG for going through Accept clauses in order instead of by q= number..

    // Smarter revision:
    //   - get the Accept header (and any other context) right from request.
    //   - "format" is explicitly chosen MIME type; it is an error if
    //     this cannot be satisfied, no negotiation.
    //   - it NEVER returns null - will throw NotAcceptable instead.
    // XXX FIX ME!
    // Quick but wrong implementation of Accept MIME type matchign:
    // Choose a resolvable format - must be in map - mabye default if no others match
    // returns chosen format or null if none (that's a bad error, dflt should always work)
    // XXX this SHOULD get more sophisticated, see RFC2068 p. 95 or so
    //    deal with */*, matching the "q" param, etc.. also need to
    //    generate an HTTP 406 (Not Acceptable) error when appropriate.
    // XXX ALSO need to match type/* and return that value..
    // XXX this is WRONG for going through Accept clauses in order instead of by q= number..

    // Smarter revision:
    //   - get the Accept header (and any other context) right from request.
    //   - "format" is explicitly chosen MIME type; it is an error if
    //     this cannot be satisfied, no negotiation.
    //   - it NEVER returns null - will throw NotAcceptable instead.
    private static String negotiateContent(HttpServletRequest request,
                                           String format, Collection<String> formats, String dflt)
    {
        String accept = request.getHeader("Accept");

        // try explicitly chosen format first
        if (format != null) {
            if (formats.contains(format))
                return format;
            throw new NotAcceptableException("The demanded response format is not available: "+format);

        // the "Accept" header is more negotiable..
        } else if (accept != null) {

             // XXX ignoring all params, really should pay attention to q=
             // should sort these by q.
            for (String a : accept.trim().split("\\s*,\\s*")) {
                int semi = a.indexOf(";");
                if (semi >= 0)
                    a = a.substring(0, semi).trim();
                log.debug("Trying accept format mime="+a);
                // XXX this is bogus too, should match e.g. text/* against
                // each format in the set..
                if (formats.contains(a))
                    return a;
            }

        // finally, try default
        } else if (dflt != null)
            return dflt;
        throw new NotAcceptableException("None of the available response formats is acceptable, and there is no default.");
    }

    /**
     * Find RDFFormat which has a writer that generates the given MIME
     * type.  This is distinct (and different) from simply looking up
     * the RDFFormat.forMIMEType() call since we install extra, custom,
     * writers.  Note that a parser is NOT necessarily available for the
     * formats returned by this call.
     *
     * @param mime RDF serialization mime type to look up
     * @return the corresponding {@link org.openrdf.rio.RDFFormat} object, or null if not found
     */
    public static RDFFormat RDFOutputFormatForMIMEType(String mime)
    {
        return RDFWregistry.getFileFormatForMIMEType(mime);
    }
}
