package org.eaglei.repository;

import org.eaglei.repository.vocabulary.REPO;
import org.eaglei.repository.vocabulary.DCTERMS;
import org.eaglei.repository.util.Utils;
import java.net.URLConnection;
import java.util.Date;
import java.util.HashMap;
import java.util.Map;
import java.util.Iterator;
import java.util.Properties;
import java.io.File;
import java.io.FileOutputStream;
import java.io.InputStreamReader;
import java.io.Reader;
import java.io.IOException;
import java.net.InetAddress;

import org.openrdf.repository.Repository;
import org.openrdf.repository.RepositoryConnection;
import org.openrdf.repository.RepositoryException;
import org.openrdf.repository.sail.SailRepository;
import org.openrdf.sail.nativerdf.NativeStore;
import org.openrdf.model.URI;
import org.openrdf.model.Literal;
import org.openrdf.model.Resource;
import org.openrdf.model.ValueFactory;
import org.openrdf.rio.RDFFormat;
import org.openrdf.rio.RDFParseException;
import org.openrdf.OpenRDFException;

import org.apache.log4j.Logger;
import org.apache.log4j.LogManager;
import org.apache.log4j.BasicConfigurator;
import org.apache.log4j.PropertyConfigurator;
import org.apache.log4j.RollingFileAppender;
import org.apache.log4j.Appender;
import org.apache.log4j.PatternLayout;
import org.apache.log4j.Level;

import org.apache.commons.configuration.PropertiesConfiguration;
import org.apache.commons.configuration.Configuration;

import javax.servlet.ServletException;
import javax.servlet.ServletContext;
import javax.servlet.http.HttpSession;

import org.eaglei.repository.inferencer.MinimalInferencer;

/**
 * This is a singleton class that encompasses the application state.
 * It is created and destroyed by Java Servlet Container webapp
 * lifecycle events.
 *
 * Important configuration variables:
 *  - Sesame repository object (not the connection, but what makes connections)
 *  - Default namespace
 *  - Title (in configuration)
 *
 * @author Larry Stone
 * Started April 2010
 * @version $Id: $
 */
public class DataRepository
{
    private static Logger log = LogManager.getLogger(DataRepository.class);

    /** name of Derby database for user authentication */
    /** XXX this needs to move */
    public  static final String AUTH_DB_NAME = "eagle-i-users.derby";

    /** Key to ServletContext attribute for open Sesame repository object */
    private static final String SCTX_REPOSITORY = "org.eaglei.repository.SesameRepository";

    /** Key to generation counter in current app context */
    private static final String SCTX_GENERATION = "org.eaglei.repository.Generation";

    /** Configuration property keys */
    private static final String CONFIG_NAMESPACE = "eaglei.repository.namespace";
    private static final String CONFIG_LOGDIR = "eaglei.repository.log.dir";
    private static final String CONFIG_SESAMEDIR = "eaglei.repository.sesame.dir";

    /** title of repository for UI labels */
    public static final String CONFIG_TITLE = "eaglei.repository.title";

    /** stylesheet for transforming HTML dissemination output, if any */
    public static final String CONFIG_INSTANCE_XSLT = "eaglei.repository.instance.xslt";

    /** CSS stylesheet for HTML dissemination output, if any */
    public static final String CONFIG_INSTANCE_CSS = "eaglei.repository.instance.css";

    // name of system property where home directory is specified (mandatory)
    private static final String REPO_HOME = "org.eaglei.repository.home";

    // name of system property where RDBMS backup directory is specified (optional)
    private static final String BACKUP_HOME = "org.eaglei.repository.backup";

    // map of named graph URI to resource filename from which it is
    // initialized if empty
    private static final Map<URI,String> graphInitFile = new HashMap<URI,String>();
    static {
        graphInitFile.put(REPO.NAMESPACE_URI, "repository-ont.n3");
        graphInitFile.put(REPO.NG_INTERNAL, "repository-internal.n3");
        graphInitFile.put(REPO.NG_QUERY, "query-macros.n3");
    }

    //---------------------- instance variables:

    // cache of the last servlet context
    private ServletContext servletContext = null;

    // cache of default namespace prefix - from congif or defaulted
    private String defaultNamespace = null;

    /** Configuration properties */
    private Configuration configuration = null;

    // cache of properties describing how this software was built.
    private Properties buildProperties = null;

    // timestamp of most recent last-modified setting, useful
    // for optimizing time-bounded metadata harvesting queries.
    private Date lastModified = new Date();


    // the singleton instance
    private static DataRepository instance = null;

    /**
     * Get the singleton instance.  Will return null unless the initialization has run.
     *
     * @return the singleton, a {@link org.eaglei.repository.DataRepository} object.
     */
    public static DataRepository getInstance()
    {
        return instance;
    }

    // handle error before logging is working
    private static void printError(String msg)
    {
        System.err.println("***ERROR*** (at "+(new Date().toString())+
            ") eagle-i DataRepository: "+msg);
    }

    // handle info message before logging is working
    private static void printInfo(String msg)
    {
        System.err.println("***INFO*** (at "+(new Date().toString())+
            ") eagle-i DataRepository: "+msg);
    }

    /**
     * Web app initialization hook:
     *  1. find and load the configuration properties file.
     *  2. configure log4j
     *  3. set up sesame repository based on configs
     *
     * @param sc the {@link javax.servlet.ServletContext} object.
     */
    public static void initialize(ServletContext sc)
    {
        if (instance == null) {
            try {
                instance = new DataRepository(sc);
                instance.finishInitialize();
            } catch (Exception e) {
                printError("Got exception in Webapp context initialization, DO NOT EXPECT ANYTHING TO WORK NOW:"+e.toString());
                log.fatal("Got exception in Webapp context initialization, DO NOT EXPECT ANYTHING TO WORK NOW:", e);
            }
        }
        else
            log.fatal("Initialize was called after DataRepository already initialized!");
    }

    /**
     * <p>Constructor for DataRepository.</p>
     *
     * @param sc a {@link javax.servlet.ServletContext} object.
     */
    public DataRepository(ServletContext sc)
    {
        super();
        servletContext = sc;
    }

    /**
     * <p>finishInitialize - setup at construct time</p>
     *
     * @throws java$io$IOException if any.
     * @throws org.openrdf.OpenRDFException if any.
     */
    public void finishInitialize()
        throws IOException, OpenRDFException
    {
        // 0. Set up home directory, creating it if necessary
        File home = getHomeDirectory();
        printInfo("Configuring with Eagle-I Repository Home dir = "+home.toString());

        if (!home.exists()) {
            if (!home.mkdirs()) {
                printError("Failed to create repository home directory, configured path = "+home);
                printError("THIS WILL CAUSE MANY OTHER ERRORS, DO NOT EXPECT THE REPOSITORY TO WORK!");
            }

        // If homedir path points to something that is not a directory
        // there isn't much we can do but complain about certain doom:
        } else if (!home.isDirectory()) {
            printError("Path configured as repository home is not a directory: "+home);
            printError("THIS WILL CAUSE MANY OTHER ERRORS, DO NOT EXPECT THE REPOSITORY TO WORK!");
        }

        // 1. Setup configuration properties file:
        // If file does not exist, copy in the default template.
        File pf = new File(home,"configuration.properties");
        if (!pf.exists()) {
            try {
                Utils.copyStream(this.getClass().getClassLoader().getResourceAsStream("default.configuration.properties"),
                          new FileOutputStream(pf));

            } catch (IOException e) {
                try {
                    pf.delete();
                } catch (Exception ee) { }
                printError("Failed to copy default configuration file to repository home dir: "+e);
            }
        }

        // 2. load the configuration properties.
        try {
            log.debug("Loading configuration from file="+pf);
            configuration = new PropertiesConfiguration(pf);
        } catch (Exception e) {
            printError("Failed to read configuration, file="+pf+": "+e.toString());
        }

        // 3. set up logging configuration: if it looks like this is
        // the bootstrap sequence or there is no log4j config,
        // set up default logging and then save it out
        // to the (presumably new) properties file for next time
        // and so the admin can edit it.

        // copy over just the log4j properties -- allow these to be
        // set in the main properties because:
        // (1) simpler to have only one config file
        // (2) apache config lets you interpolate variables, would
        //   be confusing to have that in main config but not log4j's..
        Properties lp = new Properties();
        for (Iterator pi = configuration.getKeys("log4j"); pi.hasNext();) {
            String k = (String)pi.next();
            lp.setProperty(k, configuration.getString(k));
        }

        // if there is explicit log4j config, use it:
        if (lp.containsKey("log4j.rootLogger")) {
            printInfo("Setting up log4j with existing configuration properties");
            PropertyConfigurator.configure(lp);

        // otherwise, default log4j config
        } else {
            printInfo("Using default log4j setup, additional property count = "+lp.size());
            setupDefaultLogging(home);
            if (lp.size() > 0)
                PropertyConfigurator.configure(lp);
        }

        // 4. configure the Sesame directory and start up the repository
        String sesPath = configuration.getString(CONFIG_SESAMEDIR);
        File sesDir =  (sesPath == null) ? new File(home, "sesame") : new File(sesPath);
        if (!sesDir.exists())
            sesDir.mkdirs();

        Repository r = new SailRepository(
                           new MinimalInferencer(
                               new NativeStore(sesDir)));
        r.initialize();
        if (!r.isWritable())
            log.fatal("Sesame repo is not writable, this is going to cause trouble soon!");
        servletContext.setAttribute(SCTX_REPOSITORY, r);
        log.info("Sesame Repository open, directory="+sesDir.toString());

        servletContext.setAttribute(SCTX_GENERATION, new Integer(1));
        log.debug("Generation initialized to 1.");

        // 5. Load initial content into graphs, e.g.
        // the repo ontology,  initial local admin MD
        for (Map.Entry<URI,String> me : graphInitFile.entrySet()) {
            loadGraphIfEmpty(r, me.getKey(), me.getValue());
        }
    }


    /**
     * <p>destroy - shut down this application and release resources.</p>
     *
     * @param sc a {@link javax.servlet.ServletContext} object.
     * @throws org.openrdf.repository.RepositoryException if any.
     */
    public void destroy(ServletContext sc)
        throws RepositoryException
    {
        // Shutdown sesame repository - MUST be done to protect
        // the integrity of Native repository files.
        Repository r = (Repository)sc.getAttribute(SCTX_REPOSITORY);
        sc.removeAttribute(SCTX_REPOSITORY);
        if (r != null) {
            r.shutDown();
            log.info("Sesame Repository closed.");
        }
        instance = null;
    }

    // everything to set up logging (log4j) including a default
    // when there is no configuration
    private void setupDefaultLogging(File home)
        throws IOException
    {
        // configure log directory and log4j
        String logPath = configuration.getString(CONFIG_LOGDIR);
        File logDir =  (logPath == null) ? new File(home, "logs") : new File(logPath);
        printInfo("Log file directory configured as: "+logDir.toString());
        if (!logDir.exists())
            logDir.mkdirs();
        Appender a = new RollingFileAppender(new PatternLayout("%d{ISO8601} %p %c - %m%n"),
                                  new File(logDir, "default.log").toString());
        a.setName("default");
        BasicConfigurator.configure(a);
        Logger.getRootLogger().setLevel(Level.INFO);
    }

    // Initialize a named graph if (presumably) this is the first
    // time the repo is starting up and RDF quad-store is empty.
    // If there are no statements in the named graph, load the given
    // resource file into it - it is a resource in this webapp.
    private void loadGraphIfEmpty(Repository r, URI graphURI, String resourcePath)
        throws RepositoryException, IOException, RDFParseException
    {
        RepositoryConnection rc = null;
        try {
            rc = r.getConnection();
            ValueFactory vf = r.getValueFactory();
            if (rc.size((Resource)graphURI) == 0) {
                java.net.URL rurl = this.getClass().getClassLoader().getResource(resourcePath);
                if (rurl == null)
                    log.error("Cannot find webapp resource at path="+resourcePath);
                else {
                    URLConnection ruc = rurl.openConnection();
                    long rdate = ruc.getLastModified();
                    log.debug("Loading graph="+graphURI.toString()+" from URL="+rurl.toString()+", mod-date="+rdate);
                    Reader is = new InputStreamReader(ruc.getInputStream());
                    if (is == null)
                        log.error("Cannot open webapp resource at url="+rurl);
                    else {
                        rc.add(is, graphURI.toString(), RDFFormat.forFileName(resourcePath, RDFFormat.N3), graphURI);

                        // make up some provenance - datestamp should match date of resource in WAR file
                        // which is useful for checking the actual "provenance".
                        Provenance gp = new Provenance(graphURI);
                        gp.setProvenance(rc, DCTERMS.CREATED, Provenance.getDateTime(new Date()));
                        gp.setProvenance(rc, DCTERMS.CREATOR, REPO.ROLE_SUPERUSER);
                        gp.setSource(rc, rc.getValueFactory().createURI(rurl.toString()),
                                             Provenance.getDateTime(new Date(rdate)));
                        rc.commit();
                        log.info("Initialized the empty named graph, name="+graphURI+", from resource path="+resourcePath);
                    }
                }
            }
        } finally {
            if (rc != null)
                rc.close();
        }
    }

    /**
     * <p>getSesameRepository</p>
     *
     * @return the {@link org.openrdf.repository.Repository} object.
     * @throws javax.servlet.ServletException if any.
     */
    public Repository getSesameRepository()
        throws ServletException
    {
        Repository result = (Repository)servletContext.getAttribute(SCTX_REPOSITORY);
        if (result == null)
            throw new ServletException("No RDF database connection, probably because of a failure in initialization.");

        // confirm sesame repo setup in log
        //XXX log.debug("sesame repository type = "+result.getClass().getName());
        //XXX log.debug("sesame repository's top SAIL type = "+((org.openrdf.repository.sail.SailRepository)result).getSail().getClass().getName());
        return result;
    }
    
    /**
     * Get value of default namespace prefix for URIs created by the
     * repository.  This is a critically important value that ought to be
     * configured by the administrator, so that URIs are resolved correctly
     * by the server.  Since it is so essential, we have to provide a
     * usable default.
     *
     * Also, sanity-check the configured value.  It must be a legal absolute
     * URI.
     *
     * @return the default namespace prefix as a {@link java.lang.String} object.
     */
    public String getDefaultNamespace()
    {
        if (defaultNamespace == null) {
            String cfg = configuration.getString(CONFIG_NAMESPACE);
            if (cfg != null) {
                log.debug("Got raw configured default namespace="+cfg);
                try {
                    java.net.URI juri = new java.net.URI(cfg);
                    if (!juri.isAbsolute())
                        log.error("The configured namespace prefix must be an absolute URI: "+CONFIG_NAMESPACE+" = "+cfg);
                    else if (juri.getPath() == null)
                        log.error("The configured namespace prefix must include a path ending with a slash ('/'): "+CONFIG_NAMESPACE+" = "+cfg);
                    else if (!juri.getPath().endsWith("/"))
                        log.error("The configured namespace prefix must end with a slash ('/'): "+CONFIG_NAMESPACE+" = "+cfg);
                    else {
                        String scheme = juri.getScheme();
                        if (scheme.equalsIgnoreCase("http") || scheme.equalsIgnoreCase("https"))
                            defaultNamespace = cfg;
                        else
                            log.error("The configured namespace prefix must be an absolute URI: "+CONFIG_NAMESPACE+" = "+cfg);
                    }
                } catch (java.net.URISyntaxException e) {
                    log.error("The configured namespace prefix has illegal URI syntax: "+CONFIG_NAMESPACE+" = "+cfg+": "+e);
                }
            }

            // Fallback default: construct URI out of hostname.  This is
            // unlikely to be correct since JVM doesn't know the full,
            // actual, domain name for the webserver..
            if (defaultNamespace == null) {
                try {
                    defaultNamespace = "http://"+InetAddress.getLocalHost().getHostName()+"/i/";
                } catch (java.net.UnknownHostException e) {
                    log.error("Failed to get hostname for default URI prefix",e);
                    defaultNamespace = "http://failed-to-get-local-host-name/i/";
                }
                if (cfg == null)
                    log.warn("Default Namespace Prefix URI not configured.  You really should set a configuration value for "+CONFIG_NAMESPACE);
                log.warn("Using emergency default for namespace prefix URI (probably wrong) = "+defaultNamespace);
            }
        }
        return defaultNamespace;
    }

    /**
     * <p>getConfigurationProperty</p>
     *
     * @param key a {@link java.lang.String} object.
     * @return property value or null if not set.
     */
    public String getConfigurationProperty(String key)
    {
        return configuration.getString(key);
    }

    /**
     * return default if there is no value for the indicated property.
     *
     * @param key a {@link java.lang.String} object.
     * @param dflt default value to return if not set a {@link java.lang.String} object.
     * @return the property value or dflt if not set.
     */
    public String getConfigurationProperty(String key, String dflt)
    {
        String result = configuration.getString(key);
        return (result == null) ? dflt : result;
    }

    /**
     * Gets a configuration property with multiple values.
     * Returns null when there is no such key
     *
     * @param key a {@link java.lang.String} object.
     * @return property value as an array of {@link java.lang.String} objects, or null if not set.
     */
    public String[] getConfigurationPropertyArray(String key)
    {
        return configuration.containsKey(key) ? configuration.getStringArray(key) : null;
    }


    /**
     * Bump the generation counter to signal that RDF database contents
     * have been replaced (i.e. a "new generation") and existing sessions
     * should be stale.   This is used to invalidate any cached URIs, e.g.
     * the user's Person-object URI mapped from authenticated principal name.
     */
    public void incrementGeneration()
    {
        int gen = 1 + ((Integer)servletContext.getAttribute(SCTX_GENERATION)).intValue();
        servletContext.setAttribute(SCTX_GENERATION, new Integer(gen));
        log.debug("Generation is incremented to "+gen);
    }

    /**
     * Compare current webapp generation with value (if any) cached
     * in session, also update session's generation to current.
     *
     * @param session a {@link javax.servlet.http.HttpSession} object.
     * @return a boolean, true if the session is "stale" (e.g. after a global restore of RDF db)
     */
    public boolean isSessionStale(HttpSession session)
    {
        Integer sessionGen = (Integer)session.getAttribute(SCTX_GENERATION);
        Integer ctxGen = (Integer)servletContext.getAttribute(SCTX_GENERATION);
        boolean result = sessionGen != null &&
                 sessionGen.intValue() < ctxGen.intValue();
        session.setAttribute(SCTX_GENERATION, ctxGen);
        return result;
    }

    /**
     * Read a resource file out of the webapp, path is relative to webapp root
     * and MUST NOT begin with '/' -- e.g. "repository/style/foo.xsl".
     *
     * @param path relative path to resource file, a {@link java.lang.String} object.
     * @return a {@link java.io.Reader} object open on the resource, or null if not available.
     */
    public Reader getWebappResourceAsReader(String path)
    {
        if (servletContext == null) {
            log.error("getWebappResourceAsStream: currentContext not set!");
            return null;
        }
        return new InputStreamReader(servletContext.getResourceAsStream(path));
    }

    /**
     * Get the Maven project version
     *
     * @return version as an opaque a {@link java.lang.String} object.
     */
    public String getProjectVersion()
    {
        return getBuildProperties().getProperty("version");
    }

    /**
     * Get the source revision, e.g. the subversion revision number
     *
     * @return revision as an opaque a {@link java.lang.String} object.
     */
    public String getRevision()
    {
        return getBuildProperties().getProperty("revision");
    }

    /**
     * Get the source SCM branch, if any.
     *
     * @return branch name or null if on the trunk.
     */
    public String getBranch()
    {
        String branch = getBuildProperties().getProperty("branch");
        if (branch == null) {
            log.warn("Cannot find build property \"branch\".");
            return null;
        } else if (branch.startsWith("branches/")) {
            int nextSlash = branch.indexOf("/", 9);
            return nextSlash < 0 ? branch.substring(9) : branch.substring(9, nextSlash);
        } else {
            if (!branch.equals("trunk"))
                log.warn("Build property \"branch\" has unexpected value: \""+branch+"\"");
            return null;
        }
    }

    /**
     * Get the time when this software version was built
     *
     * @return time as an opaque a {@link java.lang.String} object.
     */
    public String getTimestamp()
    {
        return getBuildProperties().getProperty("timestamp");
    }

    // Read the "build.properties" resource into a cached Properties.
    // It is used to obtain the version and timestamp info
    private Properties getBuildProperties()
    {
        if (buildProperties == null) {
            buildProperties = new Properties();
            try {
                buildProperties.load(this.getClass().getClassLoader().getResourceAsStream("build.properties"));
            } catch (IOException e) {
                log.error("Failed loading build.properties: ",e);
            }
        }
        return buildProperties;
    }


    /*  Find the root of the repository configuration and data hierarchy.
     *  It will be the first one of these that exists:
     *     a. value of system property "org.eaglei.repository.home"
     *     b. otherwise, default is "$HOME/eaglei/repository/"
     *       (where $HOME == System.getProperty("user.home") in Java-land)
     *  ALSO, this sets the system property to the default if it
     *   was unset before, so e.g. Apache config interpolation will work.
     */
    /**
     * <p>getHomeDirectory - get configured repository home directory.</p>
     *
     * @return repository home directory as a {@link java.io.File} object.
     * @throws java$io$IOException if any.
     */
    public File getHomeDirectory()
        throws IOException
    {
        String home = System.getProperty(REPO_HOME);
        File result = null;
        boolean haveSysProp = (home != null);
        if (!haveSysProp)
            result = new File(System.getProperty("user.home"),
                        "eaglei"+File.separator+"repository");
        else
            result = new File(home);
        if (!result.exists() && !result.mkdirs()) {
            log.fatal("The configured home directory does not exist: "+result.toString());
            throw new IOException("The configured home directory does not exist: "+result.toString());
        }
        if (!result.isDirectory()) {
            log.fatal("The configured home directory is not a directory or is protected: "+result.toString());
            throw new IOException("The configured home directory is not a directory or is protected: "+result.toString());
        }
        log.debug("repository home directory = "+result.toString());
        if (!haveSysProp)
            System.setProperty(REPO_HOME, result.toString());
        return result;
    }

    /**
     * <p>getBackupDirectory- get configured repository backup directory</p>
     *
     * @return repository backup directory as a {@link java.io.File} object.
     * @throws java$io$IOException if any.
     */
    public File getBackupDirectory()
       throws IOException
    {
        String home = System.getProperty(BACKUP_HOME);
        File result = null;
        boolean haveProp = (home != null);
        if (!haveProp)
            result = new File(getHomeDirectory(), "backup");
        else
            result = new File(home);
        if (!result.exists() && !result.mkdirs()) {
            log.fatal("The configured home directory does not exist: "+result.toString());
            throw new IOException("The configured home directory does not exist: "+result.toString());
        }
        if (!result.isDirectory()) {
            log.fatal("The configured home directory is not a directory or is protected: "+result.toString());
            throw new IOException("The configured home directory is not a directory or is protected: "+result.toString());
        }
        log.debug("backup home directory = "+result.toString());
        if (!haveProp)
            System.setProperty(BACKUP_HOME, result.toString());
        return result;
    }

    /**
     * Update the most recent "last modified" timestamp if profferred
     * value is newer
     *
     * @param lm date of new most recent dcterms:modified
     */
    public synchronized void setLastModified(Date lm)
    {
        if (lastModified.before(lm))
            lastModified = lm;
    }

    /**
     * Gets the most recent "last modified" timestamp applied to
     * provenance metadata.
     *
     * @return date of most recent dcterms:modified, or repository startup time by default
     */
    public Date getLastModified()
    {
        return lastModified;
    }
}
