package org.eaglei.repository.format;

import org.apache.commons.lang.StringEscapeUtils;

import java.io.PrintStream;
import java.io.OutputStream;
import java.io.UnsupportedEncodingException;
import java.nio.charset.Charset;
import java.util.Arrays;
import java.util.List;

import org.apache.log4j.Logger;
import org.apache.log4j.LogManager;

import org.openrdf.query.BindingSet;
import org.openrdf.query.resultio.TupleQueryResultFormat;
import org.openrdf.model.Value;
import org.openrdf.model.URI;
import org.openrdf.model.Literal;
import org.openrdf.model.vocabulary.XMLSchema;
import org.openrdf.query.resultio.TupleQueryResultWriter;
import org.openrdf.query.TupleQueryResultHandlerException;

import org.eaglei.repository.DataRepository;

/**
 * Very quick-n-dirty HTML query result output, helpful for interactive debugging
 *
 * XXX FIXME low priority, maybe later, add class tags, odd/even rows
 *     and columns, to facilitate better styling.
 *
 * @author lcs
 * @version $Id: $
 */
public class SPARQLHTMLWriter implements TupleQueryResultWriter
{
    private static Logger log = LogManager.getLogger(SPARQLHTMLWriter.class);

    private static final String mimeTypes[] = { "text/html", "application/xhtml+xml" };
    private static final String fileExts[] = { ".html" };

    /** Constant <code>theFormat</code> */
    public static final TupleQueryResultFormat theFormat =
        new TupleQueryResultFormat("Plain Old HTML", Arrays.asList(mimeTypes),
            Charset.defaultCharset(), Arrays.asList(fileExts));

    private PrintStream out = null;

    private static final String localPrefix = DataRepository.getInstance().getDefaultNamespace();

    /** HTML header */
    public final static String Prologue =
"<?xml version=\"1.0\" encoding=\"UTF-8\"?>"+
"<!DOCTYPE html PUBLIC \"-//W3C//DTD XHTML 1.0 Strict//EN\" \"http://www.w3.org/TR/xhtml1/DTD/xhtml1-strict.dtd\">"+
"<html xmlns=\"http://www.w3.org/1999/xhtml\" lang=\"en\" xml:lang=\"en\">"+
  "<head>"+
  "  <meta http-equiv=\"Content-Type\" content=\"text/html; charset=UTF-8\" />"+
  "  <title>SPARQL Query Results</title>"+
  " <link rel=\"stylesheet\" type=\"text/css\" href=\"/repository/styles/sparql.css\" />"+
  " <script type=\"text/javascript\">\r\n"+
  "     var firstSheet = document.styleSheets[0]\r\n"+
  "     var tuRule = null; \r\n"+
  "     // Firefox/NS has cssRules, MSIE has rules \r\n"+
  "     if (firstSheet.cssRules) { \r\n"+
  "         firstSheet.insertRule('.typeURI { display: none; }', 0); \r\n"+
  "         tuRule = firstSheet.cssRules[0]; \r\n"+
  "     } else { \r\n"+
  "         firstSheet.addRule('.typeURI', 'display: none', 0); \r\n"+
  "         tuRule = firstSheet.rules[0]; \r\n"+
  "     } \r\n"+
  " </script> \r\n"+
  "</head><body> \r\n"+
  "<button type=\"button\" onClick=\""+
  " if (tuRule.style.display == 'none') { \r\n"+
  "     tuRule.style.display = 'inline';  \r\n"+
  "     textContent = 'Less Detail';            \r\n"+
  " } else {                              \r\n"+
  "     tuRule.style.display = 'none'     \r\n"+
  "     textContent = 'More Detail';            \r\n"+
  " } \">More Detail</button> \r\n"+
  "<div><table>\r\n";

    /** HTML trailer */
    public final static String Postlogue = "</table></div></body></html>\r\n";

    // cached list of columns to preserve order
    private List<String> columns = null;

    /**
     * <p>Constructor for SPARQLHTMLWriter.</p>
     *
     * @param out a {@link java.io.OutputStream} object.
     */
    public SPARQLHTMLWriter(OutputStream out)
    {
        super();
        try {
            this.out = new PrintStream(out, false, "UTF-8");
        } catch (UnsupportedEncodingException e) {
            log.error("Failed opening PrintStream: ",e);
        }
    }

    /**
     * <p>getTupleQueryResultFormat</p>
     *
     * @return a {@link org.openrdf.query.resultio.TupleQueryResultFormat} object.
     */
    public TupleQueryResultFormat getTupleQueryResultFormat()
    {
        return theFormat;
    }

    /** {@inheritDoc} */
    public void startQueryResult(List<String> bindingNames)
                          throws TupleQueryResultHandlerException
    {
        out.println(Prologue);
        out.print("<tr>");

        columns = bindingNames;
        for (String bn : bindingNames) {
            out.print("<th>"+StringEscapeUtils.escapeHtml(bn)+"</th>");
        }
        out.println("</tr>");
    }

    /**
     * <p>endQueryResult</p>
     *
     * @throws org.openrdf.query.TupleQueryResultHandlerException if any.
     */
    public void endQueryResult()
                        throws TupleQueryResultHandlerException
    {
        out.println(Postlogue);
    }

    /** {@inheritDoc} */
    public void handleSolution(BindingSet bs)
                        throws TupleQueryResultHandlerException
    {
        out.println("<tr>");
        for (String bn : columns) {
            Value v = bs.hasBinding(bn) ? bs.getValue(bn) : null;
            out.print("<td>");

            // empty cell
            if (v == null) {
                out.print("&nbsp;");

            // URI value
            } else if (v instanceof URI) {
                out.print(renderLinkedURI((URI)v));

            // literal
            } else if (v instanceof Literal) {
                URI dt = ((Literal)v).getDatatype();
                String label = ((Literal)v).getLabel();
                if (XMLSchema.ANYURI.equals(dt)) {
                    out.print("<a href=\""+label+"\">"+StringEscapeUtils.escapeHtml(label)+"</a>");
                } else {
                    out.print("<span>\""+StringEscapeUtils.escapeHtml(label)+"\"</span>");
                    if (dt != null) {
                        out.print("<span class=\"typeURI\">^^&lt;"+
                                  StringEscapeUtils.escapeHtml(dt.stringValue())+"&gt;</span>");
                    }
                }

            // blank node, whatever
            } else {
                out.print(StringEscapeUtils.escapeHtml(bs.getValue(bn).toString()));
            }
            out.println("</td>");
        }
        out.println("</tr>");
    }

    /**
     * Produce HTML rendering of a URI that is possibly linked data.
     * This initial version just makes links for resources that this
     * repository is obviously capable of resolving (if correctly configured).
     * Eventually it ought to be extended to include resolution URLs
     * for other links that we expect to be resolvable, e.g. ontology URLs,
     * internal repository objects, etc..
     *
     * @param uri a {@link org.openrdf.model.URI} object.
     * @return a {@link java.lang.String} object.
     */
    protected static String renderLinkedURI(URI uri)
    {
        if (uri.getNamespace().equals(localPrefix))
            return "<a href=\""+uri+"\">"+StringEscapeUtils.escapeHtml(uri.toString())+"</a>";
        else
            return StringEscapeUtils.escapeHtml(uri.toString());
    }
}
