package org.eaglei.repository.servlet;

import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.Reader;
import java.io.StringReader;
import java.io.OutputStreamWriter;
import java.io.File;
import java.io.IOException;
import java.nio.charset.IllegalCharsetNameException;
import java.nio.charset.UnsupportedCharsetException;
import java.nio.charset.Charset;
import java.util.Arrays;
import java.util.Date;
import java.util.ArrayList;
import java.util.List;
import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.apache.commons.fileupload.servlet.ServletFileUpload;
import org.apache.commons.fileupload.FileItemHeaders;
import org.apache.commons.fileupload.FileUploadException;
import org.apache.commons.fileupload.disk.DiskFileItemFactory;
import org.apache.commons.fileupload.disk.DiskFileItem;

import org.apache.log4j.Logger;
import org.apache.log4j.LogManager;

import org.openrdf.OpenRDFException;
import org.openrdf.repository.RepositoryResult;
import org.openrdf.repository.RepositoryConnection;
import org.openrdf.model.Literal;
import org.openrdf.model.Resource;
import org.openrdf.model.URI;
import org.openrdf.model.ValueFactory;
import org.openrdf.model.vocabulary.RDF;
import org.openrdf.rio.RDFHandler;
import org.openrdf.rio.RDFHandlerException;
import org.openrdf.rio.RDFFormat;
import org.openrdf.rio.Rio;
import org.openrdf.rio.RDFParser;
import org.openrdf.rio.helpers.RDFHandlerWrapper;
import org.openrdf.repository.util.RDFRemover;

import org.eaglei.repository.Access;
import org.eaglei.repository.Formats;
import org.eaglei.repository.DataRepository;
import org.eaglei.repository.NamedGraph;
import org.eaglei.repository.Provenance;
import org.eaglei.repository.status.BadRequestException;
import org.eaglei.repository.status.NotFoundException;
import org.eaglei.repository.status.ForbiddenException;
import org.eaglei.repository.status.HttpStatusException;
import org.eaglei.repository.inferencer.TBoxInferencer;
import org.eaglei.repository.util.Utils;
import org.eaglei.repository.vocabulary.REPO;

/**
 * Load and dump named graphs <-> serialized RDF.
 *
 * @author Larry Stone
 * @version $Id: $
 */
public class Graph extends RepositoryServlet
{
    private static Logger log = LogManager.getLogger(Graph.class);

    private enum Action { add, replace, delete };

    /**
     * {@inheritDoc}
     *
     * POST or PUT the contents of a graph
     * Query Args:
     *  - name = URI of graph, required (can be "all")
     *  - action = (add|replace|delete) - required, how to process triples
     *  - format = MIME type (overrides content-type)
     *  - uri = remote URL from which to obtain serialization
     *  - content = immediate serlialized RDF content to load
     *  - type = ngType to record in metadata (ontology, metadata, etc..)
     * Result: HTTP status 200 for success, 400 or 4xx or 5xx otherwise.
     */
    protected void doPost(HttpServletRequest request, HttpServletResponse response)
        throws ServletException, IOException
    {
        String name = null;
        String action = null;
        String format = null;
        String contentType = null;
        String remoteURI = null;
        String source = null;
        String sourceModified = null;
        String type = null;
        InputStream content = null;
        String contentString = null;
        boolean all = false;

        // if POST with multipart, grovel through args
        if (ServletFileUpload.isMultipartContent(request)) {
            try {
                ServletFileUpload upload = new ServletFileUpload();
                upload.setFileItemFactory(new DiskFileItemFactory(100000,
                  (File)getServletConfig().getServletContext().getAttribute("javax.servlet.context.tempdir")));
                for (DiskFileItem item : (List<DiskFileItem>)upload.parseRequest(request)) {
                    String ifn = item.getFieldName();
                    if (ifn.equals("name"))
                        name = item.getString();
                    else if (ifn.equals("all"))
                        all = true;
                    else if (ifn.equals("action"))
                        action = item.getString();
                    else if (ifn.equals("format"))
                        format = item.getString();
                    else if (ifn.equals("type"))
                        type = item.getString();
                    else if (ifn.equals("uri"))
                        remoteURI =  item.getString();
                    else if (ifn.equals("content")) {
                        content = item.getInputStream();
                        contentType = item.getContentType();
                        if (source == null)
                            source = item.getName();
                        FileItemHeaders hd = item.getHeaders();
                        if (hd != null && sourceModified == null)
                            sourceModified = hd.getHeader("Last-Modified");
                        log.debug("Got content stream, MIME type = "+contentType);
                    } else if (ifn.equals("source"))
                        source =  item.getString();
                    else if (ifn.equals("sourceModified"))
                        sourceModified =  item.getString();
                    else
                        log.warn("Unrecoginized request argument: "+ifn);
                }
            } catch  (FileUploadException e) {
                log.error(e);
                throw new BadRequestException("failed parsing multipart request");
            }

        // gather args from input params instead
        } else {
            request.setCharacterEncoding("UTF-8");
            name = request.getParameter("name");
            all = request.getParameter("all") != null;
            action = request.getParameter("action");
            format = request.getParameter("format");
            type = request.getParameter("type");
            remoteURI = request.getParameter("uri");
            contentString = request.getParameter("content");
            source = request.getParameter("source");
            sourceModified = request.getParameter("sourceModified");
        }
        if (format == null || format.length() == 0)
            format = contentType;
        putGraphInternal(request, response, name, all, action, format, remoteURI, content, contentString, source, sourceModified, type);
    }

    /**
     * {@inheritDoc}
     *
     * PUT the contents of a graph
     * Query Args:
     *  - name = URI of graph, required (can be "all")
     *  - action = (add|replace|delete) - required, how to process triples
     *  - format = MIME type (overrides content-type)
     *  - uri = remote URL from which to obtain serialization
     *  - (PUT content is seralized RDF)
     * Result: HTTP status 200 for success, 400 or 4xx or 5xx otherwise.
     */
    protected void doPut(HttpServletRequest request, HttpServletResponse response)
        throws ServletException, IOException
    {
        String name = request.getParameter("name");
        String action = request.getParameter("action");
        String format = request.getParameter("format");
        if (format == null || format.length() == 0)
            format = request.getContentType();
        String remoteURI = request.getParameter("uri");
        boolean all = request.getParameter("all") != null;
        String source = request.getParameter("source");
        String type = request.getParameter("type");

        // last-mod date of source, CAN be expressed as entity header..
        String sourceModified = request.getParameter("sourceModified");
        if (sourceModified == null)
            sourceModified = request.getHeader("Last-Modified");

        putGraphInternal(request, response, name, all, action, format,
            remoteURI, request.getInputStream(), null,
            source, sourceModified, type);
    }


    // do the real work of a PUT or POST
    private void putGraphInternal(HttpServletRequest request, HttpServletResponse response,
            String name, boolean all, String rawAction, String rawFormat,
            String remoteURI, InputStream contentStream, String contentString,
            String source, String sourceModified, String type)
        throws ServletException, IOException
    {
        RepositoryConnection rc = WithRepositoryConnection.get(request);
        ValueFactory vf = rc.getValueFactory();

        // arg sanity checks: action must be keyword
        Action action = null;
        if (rawAction == null)
            throw new BadRequestException("Missing required argument: action");
        try {
            action = Action.valueOf(rawAction);
        } catch (IllegalArgumentException e) {
            throw new BadRequestException("Illegal value for 'action', must be one of: "+Arrays.deepToString(Action.values()));
        }
        log.debug("Argument action = "+action);

        // sanity check format
        if (rawFormat == null) {
            throw new BadRequestException("Missing required argument: format (or content-type)");
        }
        // sanity check: viable graph name or 'all' was specified
        if (all && name != null) {
            throw new BadRequestException("You may not specify both a named graph and the 'all' keyword");
        } else if (!all && name == null) {
            throw new BadRequestException("Missing required argument: name");
        }
        URI graphName = null;
        if (name != null) {
            try {
                graphName = vf.createURI(name);
                log.debug("Argument graph name = "+graphName);
            } catch (IllegalArgumentException e) {
                throw new BadRequestException("Graph name URI was badly formed: "+e.toString());
            }
        }

        // sanity check: source of input
        if (remoteURI == null && contentStream == null && contentString == null) {
            throw new BadRequestException("No input source, either remote URI or content must be specified");
        }

        // XXX temporary - don't implement remote URI yet.
        //  mapping between all the MIME types is a tricky challenge. ugh.
        //  (i guess with format= we can make the humans do it as fallback..)
        if (remoteURI != null) {
            throw new HttpStatusException(HttpServletResponse.SC_NOT_IMPLEMENTED, "Not accepting graphs from remote URLs yet.");
        }

        // sanity check value for graph type
        URI ngType = null;
        if (!all && type != null) {
            NamedGraph.Type t = NamedGraph.Type.parse(type);
            if (t == null) {
                throw new BadRequestException("Named graph type \""+type+"\" does not match any of the defined types: "+Arrays.deepToString(NamedGraph.Type.values()));
            }
            ngType = t.uri;
        }

        // open character-oriented reader to translate according to charset:
        Reader content = null;
        if (contentStream != null) {
            try {
                String csn = Utils.contentTypeGetCharset(rawFormat, "UTF-8");
                log.debug("Reading serialized RDF from stream with format="+rawFormat+", charset="+csn);
                content = new InputStreamReader(contentStream, Charset.forName(csn));
            } catch  (IllegalCharsetNameException e) {
                throw new BadRequestException("Illegal character set name in content-type spec: "+e);
            } catch  (UnsupportedCharsetException e) {
                throw new BadRequestException("Unsupported character set name in content-type spec: "+e);
            }
        } else if (contentString != null) {
            content = new StringReader(contentString);
        } else {
            throw new BadRequestException("Missing value for required arg 'content'");
        }
        String mime = Utils.contentTypeGetMIMEType(rawFormat);
        RDFFormat format = Formats.RDFOutputFormatForMIMEType(mime);
        if (format == null) {
            throw new HttpStatusException(HttpServletResponse.SC_UNSUPPORTED_MEDIA_TYPE, "MIME type of serialized RDF is not supported: \""+mime+"\"");
        }
        log.debug("RDF format derived from arg ("+rawFormat+") = "+format);

        // sanity check that a quad format was chosen if 'all' is on.
        if (all && !format.supportsContexts()) {
            throw new BadRequestException("'All' may not be specified with a format that does not support quads");
        }

        // finally, test access and add or delete the triples
        if ((all && Access.isSuperuser(request)) ||
            (graphName != null &&
             (!(action == Action.add || action == Action.replace) ||
               Access.hasPermission(request, graphName, Access.ADD)) &&
             (!(action == Action.delete || action == Action.replace) ||
               Access.hasPermission(request, graphName, Access.REMOVE)))) {
            try {
                // flag to add provenance metadata for created: if replace or add to empty graph..
                boolean created = action == Action.replace ||
                    (action == Action.add && !all && rc.size(graphName) == 0);
                if (action == Action.add || action == Action.replace) {

                    if (action == Action.replace) {
                        if (all) {
                            rc.clear();

                            // this records that old user records, e.g., are gone
                            DataRepository.getInstance().incrementGeneration();
                        } else
                            rc.clear(graphName);
                    }
                    if (all) {
                        long before = rc.size();
                        rc.add(content, "", format);
                        log.info("Added "+String.valueOf(rc.size()-before)+" statements to all graphs.");
                    } else {
                        long before = rc.size(graphName);
                        rc.add(content, "", format, graphName);
                        log.info("Added "+String.valueOf(rc.size(graphName)-before)+" statements to graph:"+graphName.toString());
                    }

                    // Ensure there is a :NamedGraph node for this graph
                    if (!all && !rc.hasStatement(graphName, RDF.TYPE, REPO.NAMED_GRAPH, true)) {
                        rc.add(graphName, RDF.TYPE, REPO.NAMED_GRAPH, REPO.NG_INTERNAL);
                        log.debug("Declaring rdf:type of named graph URI = "+graphName);
                    }

                    // (re)set the ngType if one was given
                    if (!all && ngType != null)
                    {
                        NamedGraph ng = NamedGraph.find(request, graphName);
                        ng.setType(request, ngType);
                    }

                } else if (action == Action.delete) {
                    long before = rc.size(graphName);
                    RDFParser parser = Rio.createParser(format, vf);
                    RDFRemover rr = new RDFRemover(rc);
                    if (!all)
                        rr.enforceContext(graphName);
                    parser.setRDFHandler(rr);
                    parser.parse(content, "");
                    log.info("Deleted "+String.valueOf(before-rc.size(graphName))+" statements to graph:"+graphName.toString());
                }
             
                // don't add provenance for 'all' since it was most likely
                // used to restore a dump and we want to preserve old provenance
                if (!all) {
                    Date now = new Date();
                    Provenance p = new Provenance(graphName);
                    if (created)
                        p.setCreated(request, now);
                    p.setModified(request, now);
                    if (source != null) {
                        Date smDate = null;
                        if (sourceModified != null) {
                            smDate = Utils.parseXMLDate(sourceModified).toGregorianCalendar().getTime();
                            log.debug("Got sourceModified date, parsed date = "+smDate);
                        }
                        p.setSource(request, source, smDate);
                    }
                }
             
                rc.commit();
                response.setStatus(created ? HttpServletResponse.SC_CREATED : HttpServletResponse.SC_OK);
            } catch (OpenRDFException e) {
                log.error("Failed parsing user-supplied graph content: ",e);
                throw new BadRequestException(e);
            }
        } else {
            throw new ForbiddenException("User is not permitted to "+action+" this graph");
        }
    }

    /**
     * {@inheritDoc}
     *
     * GET the contents of a graph
     * Query Args:
     *  - name = URI of graph, required (can be "all")
     *  - format = MIME type (overrides content-type)
     *  - inferred -- flag, if present, include inferred triples
     *  - force -- secret option, if set, allows use of non-quad format with ALL
     * Result: HTTP status 200 for success, 400 or 4xx or 5xx otherwise.
     */
    protected void doGet(HttpServletRequest request, HttpServletResponse response)
        throws ServletException, IOException
    {
        RepositoryConnection rc = WithRepositoryConnection.get(request);
        ValueFactory vf = rc.getValueFactory();

        // sanity check format
        String mimeType = Formats.negotiateRDFContent(request, request.getParameter("format"));
        RDFFormat format = Formats.RDFOutputFormatForMIMEType(mimeType);
        if (format == null) {
            throw new BadRequestException("Unrecognized MIME type for serialized RDF format: \""+mimeType+"\"");
        }
        log.debug("Negotiated output format = "+format);

        // secret option 'force' allows use of 'all' with non-quad formats
        // ..since this is mainly useful for easy testing using n-triples
        boolean force = request.getParameter("force") != null;

        // include inferred triples
        boolean inferred = request.getParameter("inferred") != null;

        // sanity check: graph name or 'all'
        String name = request.getParameter("name");
        boolean all = request.getParameter("all") != null;
        if (all && name != null) {
            throw new BadRequestException("You may not specify both a named graph and the 'all' keyword");
        } else if (!all && name == null) {
            throw new BadRequestException("Missing required argument: name");
        }
        if (all && !format.supportsContexts() && !force) {
            throw new BadRequestException("'All' may not be specified with a format that does not support quads");
        }
        URI graphName = null;
        if (name != null) {
            try {
                graphName = vf.createURI(name);
            } catch (IllegalArgumentException e) {
                throw new BadRequestException("Graph name URI was badly formed: "+e.toString());
            }
        }
        log.debug("Argument graph name = "+graphName);

        if ((all && Access.isSuperuser(request)) ||
            (graphName != null && Access.hasPermission(request, graphName, Access.READ))) {

            response.setContentType(Utils.makeContentType(format.getDefaultMIMEType(), "UTF-8"));
            try {

                // when dumping all contexts, write all the ontology
                // graphs first so restore won't have to thrash around
                // recomputing inferencing when the ontology changes.
                if (all) {
                    multiWriter writer = new multiWriter(Rio.createWriter(format, new OutputStreamWriter(response.getOutputStream(), "UTF-8")));
                    List<Resource> later = new ArrayList<Resource>();
                    RepositoryResult<Resource> rr = null;
                    try {
                        rr = rc.getContextIDs();
                        while (rr.hasNext()) {
                            Resource ctx = rr.next();
                            if (ctx instanceof URI &&
                                  TBoxInferencer.TBoxGraphs.contains((URI)ctx))
                                rc.exportStatements(null, null, null, inferred, writer, ctx);
                            else
                                later.add(ctx);
                        }
                    } finally {
                        rr.close();
                    }
                    for (Resource ctx : later) {
                        rc.exportStatements(null, null, null, inferred, writer, ctx);
                    }
                    writer.close();

                // just write one named graph
                } else {
                    if (rc.hasStatement(graphName, RDF.TYPE, REPO.NAMED_GRAPH, true))
                        rc.exportStatements(null, null, null, inferred,
                            Rio.createWriter(format, new OutputStreamWriter(response.getOutputStream(), "UTF-8")),
                            graphName);
                    else
                        throw new NotFoundException("This URI is not a named graph: "+graphName);
                }
            } catch (OpenRDFException e) {
                log.error("Failed exporting graph content: ",e);
                throw new ServletException(e);
            }
        } else {
            throw new ForbiddenException("User is not permitted to read this graph");
        }
    }

    // interpose a handler to write RDF serialization in multiple chunks.
    // ignore multiple startRDF and endRDF invocations, add "close".
    private class multiWriter extends RDFHandlerWrapper
    {
        private boolean started = false;

        private multiWriter(RDFHandler h)
        {
            super(h);
        }

        public void startRDF()
            throws RDFHandlerException
        {
            if (!started) {
                super.startRDF();
                started = true;
            }
        }

        public void endRDF()
            throws RDFHandlerException
        {
        }

        public void close()
            throws RDFHandlerException
        {
            super.endRDF();
        }
    }
}
