package org.eaglei.repository.servlet;

import java.util.ArrayList;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.ServletException;

import org.apache.log4j.Logger;
import org.apache.log4j.LogManager;

import org.openrdf.query.BindingSet;
import org.openrdf.query.impl.MapBindingSet;
import org.openrdf.model.impl.URIImpl;
import org.openrdf.model.impl.LiteralImpl;
import org.openrdf.model.URI;
import org.openrdf.repository.RepositoryConnection;
import org.openrdf.repository.RepositoryException;
import org.eaglei.repository.Access;
import org.eaglei.repository.FakeFlow;
import org.eaglei.repository.util.SPARQL;
import org.eaglei.repository.status.BadRequestException;
import org.eaglei.repository.status.InternalServerErrorException;

/**
 * Drive fake workflow - promote to new state, and/or claim by user.
 * No result, just status indicating success (200) or failure
 *
 *  XXX NOTE: This whole servlet is a TEMPORARY KLUDGE to work around
 *   the lack of time to implement full workflow.  It should
 *   go away as soon as is practical.
 *
 * Query Args:
 *   uri - resource instance URI
 *   promote - new workflow state URI
 *   claim - user to claim - 'nobody' to clear, 'self' for auth'd user, or URI
 *
 * @author Larry Stone
 * Started 18-May-10
 * @version $Id: $
 */
public class FakeWorkflow extends RepositoryServlet
{
    private static Logger log = LogManager.getLogger(FakeWorkflow.class);

    // GET only displays workflow
    /** {@inheritDoc} */
    protected void doGet(HttpServletRequest request, HttpServletResponse response)
        throws ServletException, java.io.IOException
    {
        String rawuri = request.getParameter("uri");
        String format = request.getParameter("format");

        // sanity-check required args
        if (rawuri == null)
            throw new BadRequestException("Missing required arg 'uri'.");

        // sanity-check that they didn't specify POST-only args:
        if (request.getParameter("promote") != null || request.getParameter("claim") != null) {
            throw new BadRequestException("The promote and claim args require POST method.");
        }

        FakeFlow ff = getFakeFlow(request, rawuri, false);
        if (ff != null) {
            ArrayList<BindingSet> results = new ArrayList<BindingSet>(1);
            MapBindingSet bs = new MapBindingSet(4);
            bs.addBinding("state", ff.getState());
            if (ff.getStateLabel() != null)
                bs.addBinding("stateLabel", new LiteralImpl(ff.getStateLabel()));
            URI owner = ff.getOwner();
            if (owner != null) {
                bs.addBinding("owner", owner);
                if (ff.getOwnerLabel() != null)
                    bs.addBinding("ownerLabel", new LiteralImpl(ff.getOwnerLabel()));
            }
            results.add(bs);
            SPARQL.sendTupleQueryResults(request, response, format, results);
        }
    }

    /** {@inheritDoc} */
    protected void doPost(HttpServletRequest request, HttpServletResponse response)
        throws ServletException, java.io.IOException
    {
        String rawuri = request.getParameter("uri");
        String promote = request.getParameter("promote");
        String claim = request.getParameter("claim");

        // sanity-check required args
        if (rawuri == null)
            throw new BadRequestException("Missing required arg 'uri'.");
        if (promote == null && claim == null)
            throw new BadRequestException("Missing required arg: at least one of 'promote' or 'claim' must be specified.");

        // sanity-check promote uri
        URI stateURI = null;
        if (promote != null) {
            try {
                stateURI = new URIImpl(promote);
            } catch (IllegalArgumentException e) {
                throw new BadRequestException("Promotion state is malformed: "+promote);
            }
        }

        // sanity-check new owner URI if any
        URI claimURI = null;
        if (!(claim == null || claim.equals("nobody"))) {
            if (claim.equals("self"))
                claimURI = Access.getPrincipalURI(request);
            else {
                try {
                    claimURI = new URIImpl(claim);
                } catch (IllegalArgumentException e) {
                    throw new BadRequestException("Claim user URI is malformed: "+claim);
                }
            }
        }

        // sanity-check instance
        FakeFlow ff = getFakeFlow(request, rawuri, false);
        if (ff == null)
            throw new BadRequestException("Resource instance has no (fake) workflow properties: "+rawuri);

        boolean dirty = false;
        if (stateURI != null) {
            ff.promote(request, stateURI);
            dirty = true;
        }

        if (claim != null && claim.equals("nobody")) {
            ff.claim(request, null);
            dirty = true;
        } else if (claimURI != null) {
            ff.claim(request, claimURI);
            dirty = true;
        }

        if (dirty) {
            try {
                WithRepositoryConnection.get(request).commit();
            } catch (RepositoryException e) {
                String msg = "Failed committing workflow changes, resource inst="+ff.getResource();
                log.error(msg);
                throw new InternalServerErrorException(msg,e);
            }
        }
    }

    private static FakeFlow getFakeFlow(HttpServletRequest request, String rawuri, boolean createp)
    {
        try {
            URI uri = new URIImpl(rawuri);
            FakeFlow result = FakeFlow.find(request, uri);
            if (result == null && createp)
                return FakeFlow.create(request, uri);
            else
                return result;
        } catch (IllegalArgumentException e) {
            throw new BadRequestException("Resource URI is malformed: "+rawuri);
        }
    }
}
