package org.eaglei.repository.auth;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import javax.sql.DataSource;
import javax.naming.Context;
import javax.naming.InitialContext;
import javax.naming.NamingException;

import javax.servlet.ServletException;

import org.apache.log4j.Logger;
import org.apache.log4j.LogManager;

/**
 * Simple lightweight class to manage RDBMS authorization system users
 * for the purpose of importing and exporting auth. users.
 *
 * @author Larry Stone
 * Started June 24 2010
 */
public class AuthUserTomcat implements AuthUser
{
    private static Logger log = LogManager.getLogger(AuthUserTomcat.class);

    // This must match the Datasource defined in webapp context, see context.xml
    private static final String AUTH_DATASOURCE = "jdbc/eaglei";

    private String username;
    private String password;
    private boolean isSuperuser;
    private String oldPassword;
    private boolean oldIsSuperuser;
    private boolean isCreated = false;
    private boolean isDeleted = false;
    private boolean dirty = false;

    public AuthUserTomcat(String u, String p, boolean s)
    {
        super();
        username = u;
        password = p;
        oldPassword = password;
        isSuperuser = s;
        oldIsSuperuser = isSuperuser;
    }

    /**
     * Create a new user entry. The actual creation is deferred until  the
     * commit call.  Just mark this object for creation later.
     *
     */
    @Override
    public void create()
    {
        isCreated = true;
        dirty = true;
    }

    /**
     * Sanity-check content of username and password strings:
     * Username MUST not include ':', and (according to HTTP/1.1 MAY
     * include graphic chars in the ISO-8859-1 codeset, linear whitespace,
     * and other chars WITH special MIME RFC-2047 encoding rules.  Since
     * this is a PITA and is not likely to be implemented right by HTTP
     * clients anyway, we will further restrict usernames and passwords to:
     *  - letter or digit characters in Unicode C0 an C1 (basic latin, latin-1)
     *  - punctuation: ~, @, #, $, %, _, -, . (dot)
     *
     * Throw a runtime error upon failure.
     */
    private static void nameCheck(String s, String field)
    {
        for (int i = 0; i < s.length(); ++i) {
            char c = s.charAt(i);
            if (!((Character.isLetterOrDigit(c) && c < 0x100) ||
                  c == '~' || c == '@' || c == '#' || c == '$' ||
                  c == '%' || c == '_' || c == '-' || c == '.'))
                throw new IllegalArgumentException(field+" contains an illegal character: '"+c+"'");
        }
    }

    /**
     * Create a new auth user in the DB with indicated username, password,
     * and superuser status.  Always add a role of "authenticated", since
     * the web container security has to see *some* role to recognize the user.
     */
    private void doCreate(Connection c)
        throws NamingException, SQLException
    {
        nameCheck(username, "Username");
        if (password != null)
            nameCheck(password, "Password");
        if (username.length() == 0)
            throw new IllegalArgumentException("Username may not be empty.");

        PreparedStatement iu = null;
        PreparedStatement ia = null;
        PreparedStatement ir = null;
        try {
            iu = c.prepareStatement("INSERT INTO Users (Username, Password) VALUES (?, ?)");
            iu.setString(1, username);
            iu.setString(2, password);
            if (iu.executeUpdate() != 1)
                throw new SQLException("INSERT INTO Users failed, row count incorrect.");
            ia = c.prepareStatement("INSERT INTO Roles (RoleName, Username) VALUES ('authenticated', ?)");
            ia.setString(1, username);
            if (ia.executeUpdate() != 1)
                throw new SQLException("INSERT INTO Roles failed, row count incorrect.");
            if (isSuperuser) {
                ir = c.prepareStatement("INSERT INTO Roles (RoleName, Username) VALUES (?, ?)");
                ir.setString(1, Authentication.SUPERUSER_ROLE_NAME);
                ir.setString(2, username);
                if (ir.executeUpdate() != 1)
                    throw new SQLException("INSERT INTO Roles failed, row count incorrect.");
            }
            log.debug("Created Auth DB entries for user="+username);
            isCreated = false;
        } finally {
            if (iu != null)
                iu.close();
            if (ia != null)
                ia.close();
            if (ir != null)
                ir.close();
        }
    }

    /**
     * Update auth DB entries if any values have changed.
     * Pass in a DB connection so it can share an external transaction
     * with other operations - create and update.
     */
    private void doUpdate(Connection c)
        throws NamingException, SQLException
    {
        if (password.equals(oldPassword) && isSuperuser == oldIsSuperuser)
            return;

        PreparedStatement iu = null;
        PreparedStatement ir = null;
        try {
            if (!password.equals(oldPassword)) {
                iu = c.prepareStatement("UPDATE Users SET Password = ? WHERE Username = ?");
                iu.setString(1, password);
                iu.setString(2, username);
                if (iu.executeUpdate() != 1)
                    throw new SQLException("UPDATE Users failed, row count incorrect.");
            }
            if (isSuperuser != oldIsSuperuser) {
                ir = c.prepareStatement(isSuperuser ?
                  "INSERT INTO Roles (Username, RoleName) VALUES (?, ?)" :
                  "DELETE FROM Roles WHERE (Username = ?) AND (rolename = ?)");
                ir.setString(1, username);
                ir.setString(2, Authentication.SUPERUSER_ROLE_NAME);
                if (ir.executeUpdate() != 1)
                    throw new SQLException("Update: Modification of Roles failed, row count incorrect.");
            }
            log.debug("Updated Auth DB entries for user="+username);
        } finally {
            if (iu != null)
                iu.close();
            if (ir != null)
                ir.close();
        }
    }

    /**
     * Flag this user for deletion upon commit.
     */
    @Override
    public void delete()
    {
        isDeleted = true;
        dirty = true;
    }

    /**
     * Remove user from the auth DB.
     */
    private void doDelete(Connection c)
        throws NamingException, SQLException
    {
        PreparedStatement du = null;
        PreparedStatement dr = null;
        try {
            dr = c.prepareStatement("DELETE FROM Roles WHERE Username = ?");
            du = c.prepareStatement("DELETE FROM Users WHERE Username = ?");
            du.setString(1, username);
            dr.setString(1, username);
            dr.executeUpdate();
            if (du.executeUpdate() != 1)
                throw new SQLException("Update: Modification of Users failed, row count incorrect.");
            isDeleted = false;
        } finally {
            if (du != null)
                du.close();
            if (dr != null)
                dr.close();
        }
    }

    /** Getter */
    @Override
    public String getUsername()
    {
        return username;
    }

    /** Getter */
    @Override
    public String getPassword()
    {
        return password;
    }

    /** Getter */
    @Override
    public boolean isSuperuser()
    {
        return isSuperuser;
    }

    /**
     * Separate method to test password, in case it's encrypted or something.
     * returns true when authentication succeeds.
     * @param pw password to try
     * @return true if it would succeed in authenticating this user
     */
    @Override
    public boolean authenticate(String pw)
    {
        return password != null && password.equals(pw);
    }


    /**
     * Setter for password
     * @param pw new password
     */
    @Override
    public void setPassword(String pw)
    {
        nameCheck(pw, "Password");
        password = pw;
        dirty = true;
    }

    /**
     * Setter for superuser role
     * @param su new superuser status
     */
    @Override
    public void setIsSuperuser(boolean su)
    {
        if (isSuperuser != su) {
            isSuperuser = su;
            dirty = true;
        }
    }

    // get a jdbc Connection to the authentication RDBMS
    private static Connection getConnection()
        throws NamingException, SQLException
    {
        Context initContext = new InitialContext();
        Context envContext  = (Context)initContext.lookup("java:/comp/env");
        DataSource ds = (DataSource)envContext.lookup(AUTH_DATASOURCE);
        return ds.getConnection();
    }

    // true if there is a user table entry for this username already
    private boolean exists(Connection c)
        throws ServletException
    {
        try {
            PreparedStatement s = null;
            try {
                s = c.prepareStatement("SELECT Users.Username FROM Users WHERE Users.Username = ?");
                s.setString(1, username);
                ResultSet r = s.executeQuery();
                int count = 0;
                boolean result = r.next();
                log.debug("exists("+username+") => "+result);
                return result;
            } finally {
                if (s != null)
                    s.close();
            }
        } catch (SQLException e) {
            log.error("Failed in query for exists(): ",e);
            throw new ServletException(e);
        }
    }

    /**
     * Update auth DB entries if any values have changed.
     */
    @Override
    public void commit()
        throws ServletException
    {
        if (dirty) {
            try {
                Connection c = null;
                try {
                    c = getConnection();
                    c.setAutoCommit(false);
                    flush(c);
                    c.commit();
                } finally {
                    if (c != null)
                        c.close();
                }
            } catch (NamingException e) {
                log.error("Failed in single commit(): ",e);
                throw new ServletException(e);
            } catch (SQLException e) {
                log.error("Failed in single commit(): ",e);
                throw new ServletException(e);
            }
        }
    }

    /**
     * Do the actual work to update auth DB entries if any values have changed.
     */
    public void flush(Connection c)
        throws ServletException, NamingException, SQLException
    {
        if (dirty) {
            if (isCreated && !exists(c)) {
                doCreate(c);
            } else if (isDeleted) {
                doDelete(c);
             
            // default, covers created but already exists..
            } else {
                doUpdate(c);
            }
            dirty = false;
        }
    }

    /** {@inheritDoc} */
    @Override
    public boolean equals(Object o)
    {
        return o instanceof AuthUserTomcat && username!= null && username.equals(((AuthUserTomcat)o).username);
    }

    /** {@inheritDoc} */
    @Override
    public int hashCode()
    {
        return username.hashCode();
    }
}
