package org.eaglei.repository.model;

import java.util.Set;
import java.io.IOException;
import java.io.OutputStreamWriter;

import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.apache.log4j.Logger;
import org.apache.log4j.LogManager;

import org.openrdf.OpenRDFException;
import org.openrdf.model.URI;
import org.openrdf.model.Literal;
import org.openrdf.model.Value;
import org.openrdf.model.impl.LiteralImpl;
import org.openrdf.model.impl.ContextStatementImpl;
import org.openrdf.repository.RepositoryConnection;
import org.openrdf.model.vocabulary.RDF;
import org.openrdf.model.vocabulary.RDFS;
import org.openrdf.rio.RDFFormat;
import org.openrdf.rio.Rio;
import org.openrdf.rio.RDFHandler;
import org.openrdf.query.TupleQuery;
import org.openrdf.query.TupleQueryResult;
import org.openrdf.query.TupleQueryResultHandlerBase;
import org.openrdf.query.TupleQueryResultHandlerException;
import org.openrdf.query.BindingSet;
import org.openrdf.query.QueryLanguage;

import org.eaglei.repository.auth.Authentication;
import org.eaglei.repository.vocabulary.REPO;
import org.eaglei.repository.util.SPARQL;
import org.eaglei.repository.util.HandlerBadRequest;
import org.eaglei.repository.status.BadRequestException;
import org.eaglei.repository.status.ForbiddenException;
import org.eaglei.repository.status.InternalServerErrorException;
import org.eaglei.repository.servlet.WithRepositoryConnection;
import org.eaglei.repository.servlet.ImportExport.DuplicateArg;
import org.eaglei.repository.util.Utils;

/**
 * Export and import of the Role object.
 *
 * @author Larry Stone
 * Started March, 2011
 */
public class TransportRole implements Transporter
{
    private static Logger log = LogManager.getLogger(TransportRole.class);

    /**
     * Check that current authenticated user is authorized for this
     * operation; some export requests require Superuser privilege.
     *
     * Policy: Any authenticated user can EXPORT roles.
     */
    @Override
    public void authorizeExport(HttpServletRequest request)
        throws ServletException
    {
        if (Authentication.getAuthenticatedUsername(request) == null)
            throw new ForbiddenException("Export of roles requires an authenticated user login.");
    }

    /**
     * Check that current authenticated user is authorized for this
     * operation; some import requests require Superuser privilege.
     *
     * Policy: Only Admin user can IMPORT roles, since they cah
     * change the authorization situation.
     */
    @Override
    public void authorizeImport(HttpServletRequest request)
        throws ServletException
    {
        if (!Authentication.isSuperuser(request))
            throw new ForbiddenException("Import of roles requires administrator privileges.");
    }

    /**
     * Export roles to serialized quad format.
     * Write a stylized representation of essential data, since the
     * actual statements implementing a Role are bound to chagne soon.
     * The export contains these statements for each role:
     *
     * {All statements in :NG_Internal graph}
     *     <role-uri> a :Role .
     *     <role-uri> rdfs:label "label" .
     *     {optional} <role-uri> rdfs:comment  "comment" .
     *
     *  {@inheritDoc}
     */
    @Override
    public void doExport(HttpServletRequest request, HttpServletResponse response,
                       RDFFormat format, Set<String> includes, Set<String> excludes)
        throws ServletException, IOException
    {
        try {
            RDFHandler out = Rio.createWriter(format, new OutputStreamWriter(response.getOutputStream(), "UTF-8"));
            out.startRDF();
            for (Role r : Role.findAll(request)) {
                if (!r.isBuiltin()) {
                    URI ru = r.getURI();
                    String rus = ru.stringValue();
                    String rc = r.getComment();
                    String rl = r.getLabel();
                    if (excludes.contains(rl) || excludes.contains(rus)) {
                        log.debug("SKIP ROLE because of exclude: "+r);
                    } else if (includes.isEmpty() ||
                             includes.contains(rl) || includes.contains(rus)) {
                        // NOTE: synthesize a bogus rdf:type, just to reassure import
                        out.handleStatement(new ContextStatementImpl(
                            ru, RDF.TYPE, REPO.ROLE, REPO.NG_INTERNAL));
                        out.handleStatement(new ContextStatementImpl(
                            ru, RDFS.LABEL, new LiteralImpl(r.getLabel()), REPO.NG_INTERNAL));
                        if (rc != null)
                            out.handleStatement(new ContextStatementImpl(
                                ru, RDFS.COMMENT, new LiteralImpl(rc), REPO.NG_INTERNAL));
                    } else {
                        log.debug("SKIP ROLE because of include: "+r);
                    }
                }
            }
            out.endRDF();
        } catch (OpenRDFException e) {
            throw new InternalServerErrorException(e);
        }
    }

    /**
     * Import description of roles from serialized RDF quads.
     *  {@inheritDoc}
     */
    @Override
    public void doImport(HttpServletRequest request, HttpServletResponse response,
                       RepositoryConnection content,
                       Set<String> includes, Set<String> excludes,
                       DuplicateArg duplicate,
                       boolean transform, boolean ignoreACL)
        throws ServletException, IOException
    {
        try {
            TupleQuery q = content.prepareTupleQuery(QueryLanguage.SPARQL, Role.ROLE_QUERY);
            if (log.isDebugEnabled())
                log.debug("SPARQL query to get IMPORTED ROLES (against internal memory repo) =\n\t"+Role.ROLE_QUERY);
            q.setDataset(SPARQL.InternalGraphs);
            q.setIncludeInferred(false);
            importRoleHandler ih = new importRoleHandler(request, includes, excludes, duplicate, transform);
            q.evaluate(ih);

            // if there are statements but nothing matching roles, flag an
            // error because it may be an invalid export file.
            if (ih.count == 0 && content.size() > 0)
                throw new BadRequestException("Invalid input: None of the statements in the input data are valid Role representations.");
        } catch (HandlerBadRequest e) {
            throw new BadRequestException(e.getMessage(), e);
        } catch (OpenRDFException e) {
            throw new InternalServerErrorException(e);
        }
    }

    // sparql query result handler for import
    private static class importRoleHandler extends TupleQueryResultHandlerBase
    {
        private static final String dupeQueryString =
            "SELECT DISTINCT ?uri WHERE {\n"+
            " ?uri <"+RDFS.SUBCLASSOF+"> <"+REPO.ROLE+">; \n"+
            "  <"+RDFS.LABEL+"> ?label. \n"+
            "FILTER (?uri = ?quri || ?label = ?qlabel)}";

        private HttpServletRequest request;
        private boolean transform;
        private DuplicateArg duplicate;
        private Set<String> includes;
        private Set<String> excludes;
        private TupleQuery dupeQuery = null;
        private int count = 0;

        public importRoleHandler(HttpServletRequest arequest,
                Set<String> aincludes, Set<String> aexcludes,
                DuplicateArg aduplicate,
                boolean atransform)
            throws OpenRDFException
        {
            super();
            request = arequest;
            transform = atransform;
            duplicate = aduplicate;
            includes = aincludes;
            excludes = aexcludes;

            // re-use the same query with different bindings
            RepositoryConnection rc = WithRepositoryConnection.get(request);
            dupeQuery = rc.prepareTupleQuery(QueryLanguage.SPARQL, dupeQueryString);
            dupeQuery.setDataset(SPARQL.InternalGraphs);
            dupeQuery.setIncludeInferred(false);
        }

        // process result of uri, label, comment
        @Override
        public void handleSolution(BindingSet bs)
            throws TupleQueryResultHandlerException
        {
            ++count;
            Value vuri = bs.getValue("uri");
            Value label = bs.getValue("label");
            String us = vuri.stringValue();
            String ls = Utils.valueAsString(label);
            Value comment = bs.getValue("comment");
            boolean implicit = false;
            Value rimplicit = bs.getValue("implicit");
            if (rimplicit instanceof Literal)
                implicit = ((Literal)rimplicit).booleanValue();
            if (log.isDebugEnabled())
                log.debug("importRoleHandler: Got result: uri="+vuri+", label="+label+", implicit="+implicit);

            // first, apply include/exclude filter
            if (excludes.contains(us) || excludes.contains(ls)) {
                log.debug("SKIP IMPORT ROLE because of exclude: uri="+us+", label="+ls);
                return;
            } else if (!includes.isEmpty() && !(includes.contains(us) || includes.contains(ls))) {
                log.debug("SKIP IMPORT ROLE because of include: uri="+us+", label="+ls);
                return;
            }

            // check for duplicate - horrible nested query but necessary
            // if we are to check label AND uri.. at least there are
            // only a few roles expected.
            TupleQueryResult qr = null;
            try {
                dupeQuery.clearBindings();
                dupeQuery.setBinding("quri", vuri);
                dupeQuery.setBinding("qlabel", label);
                qr = dupeQuery.evaluate();
                if (qr.hasNext()) {
                    Value du = qr.next().getValue("uri");
                    log.debug("Found a duplicate Role, uri="+du.stringValue());
                    if (duplicate == DuplicateArg.ignore) {
                        log.debug("SKIPPING duplicate role, input uri="+us);
                        return;
                    } else if (duplicate == DuplicateArg.abort) {
                        throw new HandlerBadRequest("Import contains a duplicate Role for existing Role uri="+du.stringValue());

                    // to replace -- delete old role first.
                    } else if (duplicate == DuplicateArg.replace) {
                        Role r = Role.find(request, (URI)du);
                        r.delete(request);
                    }
                }
                Role nr = Role.create(request, transform ? null : (URI)vuri,
                            ls, comment == null ? null : Utils.valueAsString(comment), implicit);
            } catch (OpenRDFException e) {
                throw new InternalServerErrorException(e);
            } catch (ServletException e) {
                throw new TupleQueryResultHandlerException(e);
            } finally {
                try {
                    if (qr != null)
                        qr.close();
                } catch (OpenRDFException e) {
                    log.warn("Ignoring exception while closing result: "+e);
                }
            }
        }
    }
}
