package org.eaglei.repository.servlet;

import java.io.Reader;
import java.io.FileReader;
import java.io.File;
import java.io.OutputStreamWriter;
import javax.xml.datatype.XMLGregorianCalendar;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.ServletException;
import javax.servlet.ServletConfig;

import org.apache.log4j.Logger;
import org.apache.log4j.LogManager;

import org.openrdf.query.BindingSet;
import org.openrdf.query.impl.DatasetImpl;
import org.openrdf.query.QueryLanguage;
import org.openrdf.query.MalformedQueryException;
import org.openrdf.query.TupleQueryResultHandlerBase;
import org.openrdf.query.TupleQueryResultHandlerException;
import org.openrdf.query.TupleQuery;
import org.openrdf.query.GraphQuery;
import org.openrdf.model.URI;
import org.openrdf.model.Literal;
import org.openrdf.model.Resource;
import org.openrdf.model.Statement;
import org.openrdf.model.Value;
import org.openrdf.model.vocabulary.RDFS;
import org.openrdf.model.vocabulary.RDF;
import org.openrdf.model.vocabulary.XMLSchema;
import org.openrdf.repository.RepositoryConnection;
import org.openrdf.repository.RepositoryResult;
import org.openrdf.OpenRDFException;
import org.openrdf.rio.RDFFormat;
import org.openrdf.rio.Rio;

import org.jdom.Namespace;
import org.jdom.Element;
import org.jdom.Document;
import org.jdom.DocType;
import org.jdom.output.Format;
import org.jdom.output.XMLOutputter;
import org.jdom.transform.XSLTransformer;
import org.jdom.transform.XSLTransformException;

import org.eaglei.repository.model.Access;
import org.eaglei.repository.Configuration;
import org.eaglei.repository.Lifecycle;
import org.eaglei.repository.util.Formats;
import org.eaglei.repository.HasContentCache;
import org.eaglei.repository.model.DataModel;
import org.eaglei.repository.model.NamedGraph;
import org.eaglei.repository.model.NamedGraphType;
import org.eaglei.repository.model.View;
import org.eaglei.repository.auth.Authentication;
import org.eaglei.repository.status.BadRequestException;
import org.eaglei.repository.status.HttpStatusException;
import org.eaglei.repository.status.NotFoundException;
import org.eaglei.repository.util.Utils;
import org.eaglei.repository.vocabulary.DCTERMS;
import org.eaglei.repository.vocabulary.REPO;

/**
 * Dissemination of the content for an Eagle-I Resource representation instance.
 * Exact content varies by the format:  if HTML is requested or
 * negotiated, output only includes properties that should be visible to
 * end users.  It is basic unstyled HTML since styles are applied via UI
 * customization.
 *  When an RDF serialization format is selected, output is a graph of
 * *all* relevant statements that authenticated user is entitled to know.
 *
 * Query Args:
 *   uri - alternate method of specifying URI
 *   format - override content negotiation with this MIME type
 *   view - use this view as source of RDF data; default is 'user'
 *
 * The generated XHTML includes the following classes on relevant tags,
 * to assist in styling and transformation:
 *      eaglei_resourceLabel
 *      eaglei_resourceURI
 *      eaglei_resourceProperties
 *      eaglei_resourceProperty
 *      eaglei_resourcePropertyTerm
 *      eaglei_resourcePropertyValue
 *      eaglei_resourceTypes
 *      eaglei_resourceType
 *      rdf_uriLocalName
 *      rdf_literal
 *      rdf_type_{XSD-type-localname}
 *      rdf_value
 *
 * @author Larry Stone
 * @version $Id: $
 */
@HasContentCache
public class Disseminate extends RepositoryServlet
{
    private static Logger log = LogManager.getLogger(Disseminate.class);

    // used by JDOM to generate XHTML tags
    private static final String XHTML = "http://www.w3.org/1999/xhtml";
    private static final Namespace XHTML_NS = Namespace.getNamespace(XHTML);

    private static final URI labelPredicate[] = DataModel.LABEL_PREDICATE.getArrayOfURI();

    // switch controlling whether contact-hiding behavior is on.
    private static final boolean hideContactsEnabled = Boolean.parseBoolean(Configuration.getInstance().getConfigurationProperty("eaglei.repository.hideContacts", Boolean.FALSE.toString()));

    // URLs of JavaScript files to include in generated HTML.
    // There is a default which can be overridden by a configuration value
    // for Configuration.INSTANCE_JS, which is a comma-separated
    // list of URLs
    private static final String defaultScriptURLs[] = {
        "http://ajax.googleapis.com/ajax/libs/jquery/1.4.1/jquery.min.js",
        "/repository/styles/i.js"
    };
    private static final String scriptURLs[] =
        Configuration.getInstance().getConfigurationPropertyArray(Configuration.INSTANCE_JS, defaultScriptURLs);

    private boolean authenticated;

    /**
     * {@inheritDoc}
     *
     * Configure this servlet instance as "authenticated" or not.
     */
    @Override
    public void init(ServletConfig sc)
        throws ServletException
    {
        super.init(sc);

        // default is non-authenitcated
        String a = sc.getInitParameter("authenticated");
        authenticated = a != null && Boolean.parseBoolean(a);
    }

    /**
     * Invalidate any in-memory cache of RDF data.
     */
    public static void decache()
    {
    }

    /** {@inheritDoc} */
    @Override
    protected void doPost(HttpServletRequest request, HttpServletResponse response)
        throws ServletException, java.io.IOException
    {
        doGet(request, response);
    }

    /** {@inheritDoc} */
    @Override
    protected void doGet(HttpServletRequest request, HttpServletResponse response)
        throws ServletException, java.io.IOException
    {
        request.setCharacterEncoding("UTF-8");
        String format = getParameter(request, "format", false);
        String rawView = getParameter(request, "view", false);
        URI uri = getParameterAsURI(request, "uri", false);
        URI workspace = getParameterAsURI(request, "workspace", false);
        boolean noinferred = isParameterPresent(request, "noinferred");
        boolean forceRDF = isParameterPresent(request, "forceRDF");

        // deduce resource URI from arg or path
        if (uri == null) {
            String pi = request.getPathInfo();
            if (pi == null || pi.length() == 0)
                throw new BadRequestException("Missing required instance URI to disseminate.");
            uri = Utils.parseURI(Configuration.getInstance().getDefaultNamespace() + pi.substring(1),
                                 "Resource URI", true);
        }
        log.debug("Got requested resource uri="+uri.stringValue());

        // sanity check: cannot specify view and workspace
        if (workspace != null && rawView != null)
            throw new BadRequestException("The 'view' and 'workspace' arguments are mutually exclusive.  Choose only one.");

        // more sanity and security checking: to be reasonably sure
        // the URI refers to a resource instance, make sure there is
        // a rdf:type statement about it in a graph of type Public or WOrkspace
        try {
            RepositoryConnection rc = WithRepositoryConnection.get(request);

            // configuration sanity check
            if (labelPredicate == null)
                throw new ServletException("Configuration failure, no label predicates found.");

            // populate dataset from workspace or view, or user's
            // visible graphs by default.
            DatasetImpl ds = new DatasetImpl();
            View view = null;
            if (workspace != null) {
                View.addWorkspaceGraphs(request, ds, workspace);
            } else {
                view = (rawView == null) ? View.USER : View.parseView(rawView);
                if (view == null)
                    throw new BadRequestException("Unknown view: "+rawView);
                View.addGraphs(request, ds, view);
            }

            //  Use prettyPrint because ds.toString() breaks when a default graph is null
            //  which is possible for the 'null' view.  don't ask, it's ugly.
            if (log.isDebugEnabled())
                log.debug("Dataset derived from initial 'view' or 'workspace' args = "+Utils.prettyPrint(ds));

            // Heuristic to check whether URI is subject of a legitimate
            // resource: does it have a rdf:type statement in a named
            // graph that is expected to contain only resources?
            // Unfortunately SPARQL can't act on context URIs so we
            // have to do it the hard way:
            RepositoryResult<Statement> rr = null;
            boolean goodResource = false;
            boolean exists = false;
            boolean visible = false;
            boolean any = false;
            URI contexts[] = View.NULL.equals(view) ? new URI[1] :
                             ds.getDefaultGraphs().toArray(new URI[ds.getDefaultGraphs().size()]);
            URI homeGraph = null;
            try {
                rr = rc.getStatements(uri, RDF.TYPE, null, false, contexts);

                // if no types are visible, check ALL graphs just to be sure:
                if (!rr.hasNext()) {
                    log.debug("Probing for rdf:type in ALL graphs, after failing in allowed ones");
                    exists = rc.hasStatement(uri, RDF.TYPE, null, false);
                    any = rc.hasStatement(uri, null, null, false, contexts);
                }
                while (rr.hasNext()) {
                    Statement s = rr.next();
                    Resource ctx = s.getContext();
                    log.debug("Found statement: "+uri+" rdf:type "+s.getObject()+", in graph "+ctx);
                    exists = true;

                    // shortcut: test for read access by checking if context
                    //  URI is listed in Dataset, instead of slower call
                    //  to  Access.hasPermission(request, (URI)ctx, REPO.NAMED_GRAPH, Access.READ)
                    if (ctx != null && ctx instanceof URI &&
                          // XXX FIXME: temp kludge to accomodate NULL view:
                          (View.NULL.equals(view) ?
                           Access.hasPermission(request, (URI)ctx, Access.READ) :
                           ds.getDefaultGraphs().contains((URI)ctx))) {
                        visible = true;
                        NamedGraph cng = NamedGraph.find(request, (URI)ctx);
                        NamedGraphType cngt = cng == null ? null : cng.getType();
                        if (cngt == NamedGraphType.published || cngt == NamedGraphType.workspace) {
                            log.debug("...approving resource because of rdf:type statement in graph="+ctx);
                            goodResource = true;
                            homeGraph = (URI)ctx;
                            break;
                        }
                    }
                }
            } finally {
                rr.close();
            }
            log.debug("Probe results: exists="+exists+", any="+any+", visible="+visible+", goodResource="+goodResource);

            // NOTE: Deny it "exists" if user does not have read permission so
            //  we do not give away information they would otherwise not know.
            // Leave a log message in case admin wants to diagnose the 404
            //  so they'll know it was caused by access failure
            if (exists && !visible) {
                log.info("Resource exists but is not visible to user, so returning 404.  User="+Authentication.getPrincipalURI(request)+", resource="+uri);
                throw new NotFoundException("Subject not found in this repository: "+uri.toString());
            
            // Give a cryptic hint if there are some statements found but no rdf:type
            // This is usually caused by all rdf:type's being removed when
            // instance is deleted, and metadata persists.
            } else if (!exists) {
                if (any)
                    throw new HttpStatusException(HttpServletResponse.SC_GONE, "Subject was deleted or is not a well formed eagle-i resource: "+uri.toString());
                else
                    throw new NotFoundException("Subject not found in this repository: "+uri.toString());

            // We can read it, but it isn't in a Published or Workspace graph.
            // This tries to protect the URI resolution service from
            // being abused to romp through the triplestore..
            // NOTE: allow superuser to abuse Disseminate since it may
            // help him/her browse the repository.
            } else if (!goodResource || homeGraph == null) {
                if (!Authentication.isSuperuser(request))
                    throw new NotFoundException("Subject exists but is not available as an Eagle-I resource: "+uri.toString());
            }

            // Is HTML an acceptable format? format arg is override, otherwise check accepts
            String mimeType = Formats.negotiateHTMLorRDFContent(request, format);

            // XXX this is a terrible kludge, but that's the nature of MIME..
            // see http://www.w3.org/TR/xhtml-media-types also RFC3236
            boolean html = !forceRDF &&
                           (mimeType.equals("text/html") ||
                            mimeType.equals("application/xhtml+xml"));
            String contentType = Utils.makeContentType(html ? "application/xhtml+xml": mimeType, "UTF-8");

            log.debug("Preparing query for authenticated="+authenticated+", hideContactsEnabled="+hideContactsEnabled);
            StringBuilder query = new StringBuilder();
            if (html) {
                query.append("SELECT DISTINCT ?g ?p ?v");
                for (int i = 0; i < labelPredicate.length; ++i) {
                    query.append(" ?pl").append(String.valueOf(i)).
                          append(" ?vl").append(String.valueOf(i));
                }
            } else {
                query.append("CONSTRUCT { ?subject ?p ?v . "/*}*/);
                for (int i = 0; i < labelPredicate.length; ++i) {
                    query.append(" ?p <").append(labelPredicate[i]).
                      append("> ?pl").append(String.valueOf(i)).append(" .").
                    append(" ?v <").append(labelPredicate[i]).
                      append("> ?vl").append(String.valueOf(i)).append(" .").
                    append(" ?vo <").append(labelPredicate[i]).
                      append("> ?vol").append(String.valueOf(i)).append(" .");
                }
                query.append(/*{*/" ?v ?vp ?vo }\n");
            }
            query.append(" WHERE { \n"/*}*/);
            query.append("{ GRAPH ?g { ?subject ?p ?v } \n"/*}*/);
            for (int i = 0; i < labelPredicate.length; ++i) {
                query.append(" OPTIONAL { ?p <"/*}*/).append(labelPredicate[i]).
                    append("> ?pl").append(String.valueOf(i)).append(/*{*/" } \n").
                  append(" OPTIONAL { ?v <"/*}*/).append(labelPredicate[i]).
                    append("> ?vl").append(String.valueOf(i)).append(/*{*/" } \n");
            }

            // pick up embedded class instances for CONSTRUCT result
            if (!html) {
                query.append("OPTIONAL { ?v a ?vt . ?vt <"/*}*/).
                      append(DataModel.EMBEDDED_INSTANCE_PREDICATE.toString()).append("> <").
                      append(DataModel.EMBEDDED_INSTANCE_OBJECT.toString()).append("> . ");

                if (noinferred) {
                    query.append("GRAPH ?eg ");
                }
                // add all properties of the embedded instance
                query.append("{?v ?vp ?vo}\n");
                if (noinferred) {
                    query.append(" FILTER( ?eg != <"/*)*/).append(REPO.NG_INFERRED).append(/*(*/">)\n");
                }
        // XXX THIS IS BAD, need other metric than just authenticated.
        //  ideally some kind of ACL on props, but that's probly too expensive.
                if (!authenticated) {
                    addPropertyFilters(query, "vp", !authenticated);
                }
                // add labels of object properties of the EI...where does this end?!
                for (int i = 0; i < labelPredicate.length; ++i) {
                    query.append(" OPTIONAL { ?vo <"/*}*/).append(labelPredicate[i]).
                      append("> ?vol").append(String.valueOf(i)).append(/*{*/" } \n");
                }

                // close the OPTIONAL
                query.append(/*{*/"}\n");
            }
        // XXX THIS IS BAD, need other metric than just authenticated.
        //  ideally some kind of ACL on props, but that's probly too expensive.
            if (!authenticated) {
                addPropertyFilters(query, "p", !authenticated);
            }
            query.append(/*{*/" }");
            if (noinferred) {
                query.append(" FILTER( ?g != <"/*)*/).
                      append(REPO.NG_INFERRED).append(/*(*/">)\n");
            }
            query.append(/*{*/"}");
            if (html) {
                query.append(" ORDER BY ?p ?v");
            }
            log.debug("SPARQL Query for Dissemination: ?subject="+uri+", query=\n"+query);

            // Dataset to use in results query: unless there is an explicit
            // view arg, compose it out of metadata + onto + home graph.
            DatasetImpl qds = null;
            if (rawView == null && workspace == null) {
                qds = new DatasetImpl();
                View.addWorkspaceGraphs(request, qds, homeGraph);
            } else
                qds = ds;
            if (log.isDebugEnabled()) {
                log.debug("Home graph = "+homeGraph);
                log.debug("Dataset for QUERY FOR RESOURCE PROPERTIES = "+Utils.prettyPrint(qds));
            }
            // produce tabular output for HTML
            // XXX TODO, maybe sort the results according to annotation props
            if (html) {
                Element root = new Element("html", XHTML_NS);
                Element head = new Element("head", XHTML_NS);
                root.addContent(head);
                head.addContent(new Element("meta", XHTML_NS).
                                setAttribute("http-equiv", "Content-Type").
                                setAttribute("content", contentType));

                // software version identification
                head.addContent(new Element("meta", XHTML_NS).
                                setAttribute("name", "eaglei.version").
                                setAttribute("content", Configuration.getInstance().getProjectVersion()));

                head.addContent(new Element("title", XHTML_NS).addContent("Resource "+uri.toString()));

                // for DC-HTML, see http://dublincore.org/documents/2008/08/04/dc-html/
                head.addContent(new Element("link", XHTML_NS).
                        setAttribute("rel", "schema.DCTERMS").
                        setAttribute("href", DCTERMS.NAMESPACE));
                head.addContent(new Element("meta", XHTML_NS).
                                setAttribute("name", Configuration.TITLE).
                                setAttribute("content",
                                    Configuration.getInstance().getConfigurationProperty(Configuration.TITLE, "(Config property eaglei.repository.title needs to be set)")));

                String cssURL = Configuration.getInstance().getConfigurationProperty(Configuration.INSTANCE_CSS);
                if (cssURL != null)
                    head.addContent(new Element("link", XHTML_NS).
                        setAttribute("rel", "stylesheet").
                        setAttribute("type", "text/css").
                        setAttribute("href", cssURL));

                for (String scriptURL : scriptURLs) {
                    head.addContent(new Element("script", XHTML_NS).
                            setAttribute("language", "javascript").
                            setAttribute("type", "text/javascript").
                            setAttribute("src", scriptURL));
                }
                Element body = new Element("body", XHTML_NS);
                root.addContent(body);

                // add eagle-i standard header:

                // <div id="header" class="container_12">
                Element header = new Element("div", XHTML_NS).
                                   setAttribute("id", "header").
                                   setAttribute("class", "container_12 ");
  
                // <div id="logo" class="grid_6 alpha">
                //   <a href="#" title="eagle-1 consortium">
                //     <img src="images/eagle-i_logo.png" alt=""/>
                String logoURL = Configuration.getInstance().getConfigurationProperty("eaglei.repository.logo",  "/repository/images/eagle-i_logo.png");
                header.addContent(
                      new Element("div", XHTML_NS).
                            setAttribute("logo", "header_container").
                            setAttribute("class", "grid_6 alpha ").
                            addContent(new Element("a", XHTML_NS).
                              setAttribute("href", "#").
                              setAttribute("title", "eagle-1 consortium").
                              addContent(new Element("img", XHTML_NS).
                                setAttribute("src", logoURL).
                                setAttribute("alt", ""))));


                //  <div id="header_container">
                body.addContent(
                      new Element("div", XHTML_NS).
                            setAttribute("id", "header_container").
                            addContent(header));

                Element hlabel = new Element("h2", XHTML_NS).setAttribute("class", "eaglei_resourceLabel ");
                body.addContent(hlabel);
                Element subTitle = new Element("h3", XHTML_NS).
                                     setAttribute("class", "eaglei_resourceURI ").
                                     addContent("Resource "+uri.toString());
                body.addContent(subTitle);

                // setup the contact link
                Element contactLink = null;
                if (hideContactsEnabled) {
                    contactLink = new Element("a", XHTML_NS);
                    body.addContent(contactLink);
                }

                // Direct type list is a <OL> tag
                body.addContent(new Element("h3", XHTML_NS).
                                     setAttribute("class", "eaglei_resourceTypes directTypes ").
                                     addContent("Direct Types"));
                Element directTypes = new Element("ol", XHTML_NS).setAttribute("class", "eaglei_resourceTypes ");
                body.addContent(directTypes);

                body.addContent(new Element("h3", XHTML_NS).
                                     setAttribute("class", "eaglei_resourceTypes inferredTypes").
                                     addContent("Inferred Types"));
                Element inferredTypes = new Element("ol", XHTML_NS).setAttribute("class", "eaglei_resourceTypes ");
                body.addContent(inferredTypes);

                // property list is a <OL> tag
                body.addContent(new Element("h3", XHTML_NS).
                                     setAttribute("class", "eaglei_resourceProperties").
                                     addContent("Properties"));
                Element props = new Element("ol", XHTML_NS).setAttribute("class", "eaglei_resourceProperties ");
                body.addContent(props);

                // provenance property list is a <OL> tag
                body.addContent(new Element("h3", XHTML_NS).
                                     setAttribute("class", "eaglei_provenanceProperties").
                                     addContent("Provenance Metadata"));
                Element provenance = new Element("ol", XHTML_NS).setAttribute("class", "eaglei_provenanceProperties ");
                body.addContent(provenance);

                TupleQuery q = rc.prepareTupleQuery(QueryLanguage.SPARQL, query.toString());
                q.setBinding("subject", uri);
                // if specified view arg was "null", DO NOT set the dataset:
                if (!View.NULL.equals(view))
                    q.setDataset(qds);
                StringBuilder labelBuf = new StringBuilder();
                propsHandler handler = new propsHandler(labelBuf, directTypes, inferredTypes, props, provenance, head);
                q.evaluate(handler);
                String label = labelBuf.toString();
                hlabel.addContent(new Element("span", XHTML_NS).
                                      setAttribute("class", "rdf_literal ").
                                      addContent(label));

                // Finish filling in the contact link if there is one:
                if (contactLink != null) {
                    String emailURL = "/repository/contact?uri=" + Utils.urlEncode(uri.stringValue()) + "&label="+ Utils.urlEncode(label);
                    Element popupDiv = new Element("div", XHTML_NS).
                        setAttribute("id", "popup1").
                        setAttribute("class", "popup_block");
                    /** XXX FIXME width and height should NOT be hardcoded! */
                    /** XXX ideally percent, maybe config.. */
                    Element popupIframe = new Element("iframe", XHTML_NS).
                        setAttribute("width", "590").
                        setAttribute("height", "450").
                        setAttribute("src", emailURL).
                        setAttribute("scrolling", "no").
                        setAttribute("class", "mailframe").
                        setAttribute("frameborder", "0").
                        setAttribute("name","popupframe");
                    popupDiv.addContent(popupIframe);
                    body.addContent(popupDiv);
                    contactLink.addContent("Contact the owner of this resource!").
                        setAttribute("rel", "popup1").
                        setAttribute("class", "poplight").
                        setAttribute("href", "#?w=600").
                        setAttribute("onclick","window.frames[0].location.href='" + emailURL + "'");
                }
                     
                // set last-modified header if available
                if (handler.modified != null) {
                    try {
                        XMLGregorianCalendar mc = handler.modified.calendarValue();
                        log.debug("Got last-modified date = "+mc.toString());
                        response.addDateHeader("Last-Modified",
                                mc.toGregorianCalendar().getTime().getTime());
                    } catch (IllegalArgumentException e) {
                        log.warn("Failed to parse dcterms:created or dcterms:modified as date; value="+handler.modified);
                    }
                }

                // ONLY for HTML, add link to edit ACL if we have Admin access
                if (Access.hasPermission(request, uri, Access.ADMIN)) {
                    Element acl = new Element("form", XHTML_NS).
                        setAttribute("method", "POST").
                        setAttribute("action", "/repository/admin/editGrants.jsp");
                    acl.addContent(new Element("input", XHTML_NS).
                        setAttribute("type", "hidden").
                        setAttribute("name", "uri").
                        setAttribute("value", uri.toString()));
                    acl.addContent(new Element("input", XHTML_NS).
                        setAttribute("type", "hidden").
                        setAttribute("name", "type").
                        setAttribute("value", "Resource Instance"));
                    acl.addContent(new Element("input", XHTML_NS).
                        setAttribute("type", "hidden").
                        setAttribute("name", "label").
                        setAttribute("value", label));
                    acl.addContent(new Element("button", XHTML_NS).
                        setAttribute("type", "submit").
                        addContent("Edit Access Controls"));
                    body.addContent(acl);
                }
                Document result = new Document(root);
                    if (!result.hasRootElement())
                        log.error("No root element in doc, this will be bad.");

                // do we have optional XSL stylesheet to transform result?
                // if path is relative, look in webapp, otherwise, on filesystem
                XSLTransformer xfrm = null;
                String xslPath = Configuration.getInstance().getConfigurationProperty(Configuration.INSTANCE_XSLT);
                if (xslPath != null) {
                    File xslFile = new File(xslPath);
                    Reader xis = xslFile.isAbsolute() ?
                                        new FileReader(xslFile) :
                                        Lifecycle.getInstance().getWebappResourceAsReader(xslPath);
                    xfrm = new XSLTransformer(xis);
                    log.debug("Using XSL transformation stylesheet from: "+xslPath);
                    result = xfrm.transform(result);
                    if (!result.hasRootElement())
                        log.error("No root element in transformed doc, this will be bad.");
                }

                result.setDocType(new DocType("HTML",
                                              "-//W3C//DTD XHTML 1.0 Strict//EN",
                                              "http://www.w3.org/TR/xhtml1/DTD/xhtml1-strict.dtd"));
                XMLOutputter out = new XMLOutputter();
                out.setFormat(Format.getPrettyFormat());
                response.setContentType(contentType);
                out.output(result, response.getOutputStream());

            // normal construct query results.
            } else {
                RDFFormat rf = Formats.RDFOutputFormatForMIMEType(mimeType);
                if (rf == null) {
                    log.error("Failed to get RDF serialization format, for mime="+mimeType);
                    throw new ServletException("Failed to get RDF serialization format that SHOULD have been available, for mime="+mimeType);
                } else {
                    long startMs = System.currentTimeMillis();
                    response.setContentType(contentType);
                    GraphQuery q = rc.prepareGraphQuery(QueryLanguage.SPARQL, query.toString());
                    q.setBinding("subject", uri);
                    q.setDataset(qds);
                    q.evaluate(Rio.createWriter(rf, new OutputStreamWriter(response.getOutputStream(), "UTF-8")));
                    log.debug(String.format("SPARQL query completed in %,d mSec.", System.currentTimeMillis()-startMs));
                }
            }

            
        } catch (XSLTransformException e) {
            log.error("Failed in XSL transformation: ",e);
            throw new ServletException("Failed in XSL transformation: ",e);

        } catch (MalformedQueryException e) {
            log.error("Malformed query generated internally: ",e);
            throw new ServletException("Malformed query generated internally: "+e.toString(), e);

        // this "should never happen", means the SAIL or repo does not support generic prepareQuery()
        } catch (UnsupportedOperationException e) {
            log.error(e);
            throw new ServletException(e);

        } catch (OpenRDFException e) {
            log.error(e);
            throw new ServletException(e);
        }
    }

    // add clauses to eliminate properties that should not be shown
    // call from within a triple pattern.
    private void addPropertyFilters(StringBuilder query, String propName, boolean hide)
    {
        String hideObj = propName + "_hide";
        String contactObj = propName + "_contact";
        if (hide) {
            query.append(" OPTIONAL { ?"/*}*/).append(propName).
             append(" <").append(DataModel.HIDE_PROPERTY_PREDICATE.toString()).append("> ?").
             append(hideObj).append("\n");
            query.append("  FILTER(?"/*)*/).append(hideObj).append(" = <").
             append(DataModel.HIDE_PROPERTY_OBJECT.toString()).append(/*{(*/">)}\n");
        }
        if (hideContactsEnabled) {
            query.append(" OPTIONAL { ?"/*}*/).append(propName).
             append(" <").append(DataModel.CONTACT_PROPERTY_PREDICATE.toString()).append("> ?").
             append(contactObj).append("\n");
            query.append("  FILTER(?"/*)*/).append(contactObj).append(" = <").
             append(DataModel.CONTACT_PROPERTY_OBJECT.toString()).append(/*{(*/">)}\n");
        }
        query.append("FILTER(!(BOUND(?"/*)))*/).append(hideObj).append(/*(*/")");
        if (hideContactsEnabled) {
            query.append(" || BOUND(?"/*)*/).append(contactObj).append(/*(*/")");
        }
        query.append(/*((*/"))\n");
    }

    // add each property-value result to JDOM model of HTML output,
    // except the rdfs:label value becomes instance title.
    private static class propsHandler extends TupleQueryResultHandlerBase
    {
        // resulting list tags
        private Element props;
        private Element provenance;
        private Element head;
        private Element directTypes;
        private Element inferredTypes;

        // previous solution's values with which to remove duplicates
        // caused by duplicate labels..
        private Value lastP = null;
        private Value lastV = null;

        // resource label
        private StringBuilder label;

        // last-modified date
        private Literal modified = null;

        public propsHandler(StringBuilder label, Element directTypes, Element inferredTypes,  Element props, Element provenance, Element head)
        {
            super();
            propsHandler.this.directTypes = directTypes;
            propsHandler.this.inferredTypes = inferredTypes;
            propsHandler.this.props = props;
            propsHandler.this.provenance = provenance;
            propsHandler.this.label = label;
            propsHandler.this.head = head;
        }

        // represent a URI as label - use the local name, tag the class
        private Element makeLabel(URI src)
        {
            return new Element("span", XHTML_NS).
               setAttribute("class", "rdf_uriLocalName ").
               addContent(src.getLocalName());
        }

        // represent a Literal as label - tag type and lang
        private Element makeLabel(Literal src)
        {
            String lclass = "rdf_literal ";
            URI dt = src.getDatatype();
            String lang = src.getLanguage();

            if (dt != null)
                lclass += "rdf_type_"+dt.getLocalName()+" ";
            Element result = new Element("span", XHTML_NS).
               setAttribute("class", lclass).
               addContent(src.getLabel());
            if (lang != null)
                result.setAttribute("lang", lang);
            return result;
        }

        // represent value visually, ideally as a "label"
        private Element showValue(Value v)
        {
            if (v instanceof URI)
                return makeLabel((URI) v);
            else if (v instanceof Literal)
                return makeLabel((Literal) v);
            else
                return new Element("span", XHTML_NS).
                   setAttribute("class", "rdf_value ").
                   addContent(v.toString());
        }

        // Expect vars: ?p ?v ?pl0...n, ?vl0..n
        @Override
        public void handleSolution(BindingSet bs)
            throws TupleQueryResultHandlerException
        {
            Value p = bs.getValue("p");
            Value v = bs.getValue("v");
            if (p == null || v == null)
                return;

            // skip duplicate record - these are very likely caused by
            // values with duplicate labels, e.g.
            //  <http://purl.org/obo/owl/NCBITaxon#NCBITaxon_9031>
            //   <http://purl.obolibrary.org/obo/IAO_0000111>
            //   "chicken" .
            //  <http://purl.org/obo/owl/NCBITaxon#NCBITaxon_9031>
            //   <http://purl.obolibrary.org/obo/IAO_0000111>
            //   "Gallus gallus" .
            // .. makes the sparql query yield 2 tuple results for one prop.
            if (p.equals(lastP) && v.equals(lastV)) {
                log.debug("Skipping duplicate record pred="+p+", value="+v);
                return;
            }
            lastP = p;
            lastV = v;

            log.debug("handleSolution: Gathering labels for predicate="+p+", value="+v);
            Value vl = null;
            for (int i = 0; i < labelPredicate.length; ++i) {
                String key = "vl"+String.valueOf(i);
                if (bs.hasBinding(key)) {
                    vl = bs.getValue(key);
                    if (log.isDebugEnabled())
                        log.debug("handleSolution: "+key+"=\""+vl+"\"");
                    break;
                }
            }
            if (vl == null)
                vl = v;
            Element vlabel = showValue(vl);

            // when predicate is rdfs:label, just set the label
            Value g = bs.getValue("g");
            if (RDFS.LABEL.equals(p)) {
                label.append(vlabel.getText());
                log.debug("Setting instance's rdfs:label = "+vlabel.getText());
            }

            // add to one of the type lists: check if graph indicates inferred
            else if (RDF.TYPE.equals(p)) {
                Element te = (new Element("li", XHTML_NS).
                                   setAttribute("class", "eaglei_resourceType ").
                                   addContent(new Element("a", XHTML_NS).
                                     addContent(vlabel).
                                     setAttribute("href", v.toString())));
                if (REPO.NG_INFERRED.equals(g)) {
                    inferredTypes.addContent(te);
                    log.debug("Adding INFERRED TYPE = "+vlabel.getText());
                } else {
                    directTypes.addContent(te);
                    log.debug("Adding DIRECT TYPE = "+vlabel.getText());
                }
                     
            // add to general properties OR metadata
            } else {
                Value pl = null;
                for (int i = 0; i < labelPredicate.length; ++i) {
                    String key = "pl"+String.valueOf(i);
                    if (bs.hasBinding(key)) {
                        pl = bs.getValue(key);
                        if (log.isDebugEnabled())
                            log.debug("handleSolution: p="+p+", "+key+"=\""+pl+"\"");
                        break;
                    }
                }
                if (pl == null)
                    pl = p;
                Element plabel = showValue(pl);

                Element dl = new Element("dl", XHTML_NS).setAttribute("class", "eaglei_resourceProperty ");
                dl.addContent(new Element("dt", XHTML_NS).
                                setAttribute("class", "eaglei_resourcePropertyTerm ").
                                addContent(new Element("a", XHTML_NS).addContent(plabel).
                                  setAttribute("href", p.toString())));
                Element dd = new Element("dd", XHTML_NS).
                                setAttribute("class", "eaglei_resourcePropertyValue ");
                dl.addContent(dd);
                if (v instanceof URI)
                    dd.addContent(new Element("a", XHTML_NS).addContent(vlabel).
                                    setAttribute("href", v.toString()));
                else {
                    URI vdt = (v instanceof Literal) ? ((Literal)v).getDatatype() : null;
                    if (XMLSchema.ANYURI.equals(vdt)) {
                        dd.addContent(new Element("a", XHTML_NS).addContent(vlabel).
                                        setAttribute("href", ((Literal)v).getLabel()));
                    } else
                        dd.addContent(vlabel);
                }

               // choose either provenance or value properties, by graph:
                if (REPO.NG_METADATA.equals(g)) {
                    provenance.addContent(new Element("li", XHTML_NS).
                            setAttribute("class", "eaglei_provenanceProperty ").
                            addContent(dl));
                    // if it's a DCTERMS statement, add to DC-HTML
                    if (p instanceof URI && DCTERMS.NAMESPACE.equals(((URI)p).getNamespace())) {
                        String pln = ((URI)p).getLocalName();
                        if (v instanceof URI)
                            head.addContent(new Element("link", XHTML_NS).
                                setAttribute("rel", "DCTERMS."+pln).
                                setAttribute("href", v.toString()));
                        else if (v instanceof Literal) {
                            head.addContent(new Element("meta", XHTML_NS).
                                setAttribute("name", "DCTERMS."+pln).
                                setAttribute("content", ((Literal)v).getLabel()));
                            // save last-modified date for later
                            if (DCTERMS.MODIFIED.equals(p) ||
                                (DCTERMS.CREATED.equals(p) && modified == null))
                                modified = (Literal)v;
                        }
                    }
                } else
                    props.addContent(new Element("li", XHTML_NS).
                            setAttribute("class", "eaglei_resourceProperty ").
                            addContent(dl));
            }
        }
    }
}
