package org.eaglei.repository.servlet;

import java.io.Reader;
import java.io.IOException;
import java.util.Arrays;
import java.util.Set;
import java.util.HashSet;

import javax.servlet.ServletConfig;
import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;


import org.apache.log4j.Logger;
import org.apache.log4j.LogManager;

import org.openrdf.OpenRDFException;
import org.openrdf.model.URI;
import org.openrdf.rio.RDFFormat;
import org.openrdf.repository.Repository;
import org.openrdf.repository.RepositoryConnection;
import org.openrdf.repository.sail.SailRepository;
import org.openrdf.sail.memory.MemoryStore;
import org.openrdf.repository.RepositoryException;

import org.eaglei.repository.util.Formats;
import org.eaglei.repository.model.View;
import org.eaglei.repository.model.Transporter;
import org.eaglei.repository.model.TransportUser;
import org.eaglei.repository.model.TransportRole;
import org.eaglei.repository.model.workflow.TransportWorkflowTransition;
import org.eaglei.repository.model.TransportAccessGrant;

import org.eaglei.repository.status.BadRequestException;
import org.eaglei.repository.status.HttpStatusException;
import org.eaglei.repository.util.Utils;

/**
 * Structured export and import of resource instances OR user instances.
 * The same servlet implementation is called by both /export and /import,
 * they should set the init parameter "import" or "export" appropriately.
 * e.g.
 * <pre>
 * <servlet>
 *   <servlet-name>Export</servlet-name>
 *   <servlet-class>org.eaglei.repository.servlet.ImportExport</servlet-class>
 *   <init-param>
 *     <param-name>export</param-name>
 *     <param-value>true</param-value>
 *   </init-param>
 * </servlet>
 * </pre>
 *
 * @author Larry Stone
 * @version $Id: $
 */
public class ImportExport extends RepositoryServlet
{
    private static Logger log = LogManager.getLogger(ImportExport.class);

    /**
     * Enumerated values of 'duplicate' arg - must be public for parseKeyword.
     */
    public enum DuplicateArg { abort, ignore, replace };

    /**
     * Enumerated values of 'graph' arg - must be public for parseKeyword.
     */
    public enum NewGraphArg { abort, create };

    /**
     * Enumerated values of 'type' arg - must be public for parseKeyword.
     * Also includes the Transporter instance to manage the import/export
     */
    public enum TypeArg
    {
        // enum values
        resource (null),
        user (new TransportUser()),
        role (new TransportRole()),
        transition (new TransportWorkflowTransition()),
        grant  (new TransportAccessGrant());

        private Transporter transporter = null;

        private TypeArg(Transporter t)
        {
            transporter = t;
        }

        /** Getter for transporter */
        public Transporter getTransporter()
        {
            return transporter;
        }
    }

    // identify which kind of servlet this is since both can POST
    private boolean isImport = false;
    private boolean isExport = false;

    /**
     * {@inheritDoc}
     *
     * Configure this servlet as either import or export mode.
     */
    @Override
    public void init(ServletConfig sc)
        throws ServletException
    {
        super.init(sc);

        // if this is not the export servlet, don't implement GET:
        isExport = sc.getInitParameter("export") != null;
        isImport = sc.getInitParameter("import") != null;

        if (!(isExport || isImport))
            log.error("Servlet was initialized without either import or export mode set, THIS IS BAD.");
    }

    /**
     * {@inheritDoc}
     *
     * GET the contents of a graph - for EXPORT only
     * Query Args:
     *  - format = MIME type (overrides content-type)
     *  - view = view to query (mutually excl. w/workspace)
     *  - workspace = workspace to query (mutually excl. w/workspace)
     *     (NOTE: default to view = USER for type=resource)
     *  - type=(resource|user) - what to import
     *  - include="URI ..." or "username ..." if type=user
     *  - exclude="URI ..." or "username ..." if type=user
     * Result: HTTP status 200 for success, 400 or 4xx or 5xx otherwise.
     */
    @Override
    protected void doGet(HttpServletRequest request, HttpServletResponse response)
        throws ServletException, IOException
    {
        if (!isExport)
            throw new HttpStatusException(HttpServletResponse.SC_NOT_IMPLEMENTED, "GET is not implemented by this service");

        request.setCharacterEncoding("UTF-8");
        String rawFormat = getParameter(request, "format", false);
        String rawView = getParameter(request, "view", false);
        String rawWorkspace = getParameter(request, "workspace", false);
        String rawType = getParameter(request, "type", true);
        String include = getParameter(request, "include", false);
        String exclude = getParameter(request, "exclude", false);

        // defaulting and sanity-checking
        TypeArg type = (TypeArg)Utils.parseKeywordArg(TypeArg.class, rawType, "type", true, null);
        View view = (View)Utils.parseKeywordArg(View.class, rawView, "view", false, null);
        URI workspace = Utils.parseURI(rawWorkspace, "workspace", false);

        // sanity check, then set default view if needed
        if (workspace != null && view != null)
            throw new BadRequestException("Only one of the 'workspace' or 'view' args may be specified.");
        else if (workspace == null && view == null)
            view = (type == TypeArg.resource ? View.USER_RESOURCES : View.USER);

        Set<String> includes = parseXCludeList(include);
        Set<String> excludes = parseXCludeList(exclude);
        if (log.isDebugEnabled()) {
            log.debug("INCLUDES = "+Arrays.deepToString(includes.toArray()));
            log.debug("EXCLUDES = "+Arrays.deepToString(excludes.toArray()));
        }

        // sanity check format, also must be a quad format
        String mimeType = Formats.negotiateRDFContent(request, rawFormat);
        RDFFormat format = Formats.RDFOutputFormatForMIMEType(mimeType);
        if (format == null) {
            throw new HttpStatusException(HttpServletResponse.SC_UNSUPPORTED_MEDIA_TYPE, "MIME type of serialized RDF is not supported: \""+mimeType+"\"");
        }
        if (!format.supportsContexts()) {
            throw new HttpStatusException(HttpServletResponse.SC_UNSUPPORTED_MEDIA_TYPE, "Format does not support quad (graph) encoding: "+format);
        }
        log.debug("Output serialization format = "+format);

        Transporter tp = type.getTransporter();
        if (tp == null) {
            throw new HttpStatusException(HttpServletResponse.SC_NOT_IMPLEMENTED, "Export of "+type+" is not implemented.");
        } else {
            tp.authorizeExport(request);
            tp.doExport(request, response, format, includes, excludes);
        }
    }

    /**
     * {@inheritDoc}
     *
     * Import a collection of Users *or* Instances from an export serialization
     * generated by this service.
     * POST is for IMPORT only.
     *
     * Args:
     *  - format = MIME type (overrides content-type)
     *  - duplicate=(abort|ignore|replace)
     *  - transform=yes|no - no default
     *  - graph=(abort|create) -- what to do when import would create new graph
     *  - type=(resource|user) - what to import
     *  - ignoreACL=yes|no - default=no, skip access grants on import (transition only)
     * Result: HTTP status 200 for success, 400 or 4xx or 5xx otherwise.
     */
    @Override
    protected void doPost(HttpServletRequest request, HttpServletResponse response)
        throws ServletException, IOException
    {
        // GET implements the export function, even for a POST.
        if (isExport) {
            doGet(request, response);
            return;
        }
        if (!isImport)
            throw new HttpStatusException(HttpServletResponse.SC_NOT_IMPLEMENTED, "Servlet must be configured for import or export.");

        request.setCharacterEncoding("UTF-8");
        TypeArg type = (TypeArg)getParameterAsKeyword(request, "type",
                                                      TypeArg.class, null, true);
        Reader content = getParameterAsReader(request, "content", true);
        String contentType = getParameterContentType(request, "content");
        String rawFormat = getParameter(request, "format", false);
        String include = getParameter(request, "include", false);
        String exclude = getParameter(request, "exclude", false);
        boolean transform = getParameterAsBoolean(request, "transform", false, true);
        boolean ignoreACL = getParameterAsBoolean(request, "ignoreACL", false, false);
        DuplicateArg duplicate = (DuplicateArg)getParameterAsKeyword(request,
            "duplicate", DuplicateArg.class, DuplicateArg.abort, false);

        // defaulting and sanity-checking
        // XXX NOTE:  newGraph is currently not used, will be needed
        // later when newgraph arg is implemented, to choose whether import
        // is allowed to create new graphs.
        //String  rawNewGraph = getParameter(request, "newgraph", false);
        //NewGraphArg newGraph = (NewGraphArg)Utils.parseKeywordArg(NewGraphArg.class, rawNewGraph, "newgraph", false, NewGraphArg.abort);
        Set<String> includes = parseXCludeList(include);
        Set<String> excludes = parseXCludeList(exclude);

        if (log.isDebugEnabled()) {
            log.debug("INCLUDES = "+Arrays.deepToString(includes.toArray()));
            log.debug("EXCLUDES = "+Arrays.deepToString(excludes.toArray()));
            log.debug("duplicate = "+duplicate);
        }

        // sanity check format
        if (rawFormat == null)
            rawFormat = contentType;
        if (rawFormat == null) {
            throw new BadRequestException("Missing required argument: format (or content-type on input)");
        }

        String mime = Utils.contentTypeGetMIMEType(rawFormat);
        RDFFormat format = Formats.RDFOutputFormatForMIMEType(mime);
        if (format == null) {
            throw new HttpStatusException(HttpServletResponse.SC_UNSUPPORTED_MEDIA_TYPE, "MIME type of serialized RDF is not supported: \""+mime+"\"");
        }
        if (!format.supportsContexts()) {
            throw new HttpStatusException(HttpServletResponse.SC_UNSUPPORTED_MEDIA_TYPE, "Format does not support quad (graph) encoding: "+format);
        }
        // read document into memory repo so we can query it without polluting main repo
        log.debug("Input serialization format = "+format);
        Repository mr = null;
        RepositoryConnection mrc = null;
        try {
            mr = new SailRepository(new MemoryStore());
            mr.initialize();
            mrc = mr. getConnection();
            mrc.add(content, "", format);

            Transporter tp = type.getTransporter();
            if (tp == null) {
                throw new HttpStatusException(HttpServletResponse.SC_NOT_IMPLEMENTED, "Import of "+type+" is not implemented.");
            } else {
                tp.authorizeImport(request);
                tp.doImport(request, response, mrc, includes, excludes,
                            duplicate, transform, ignoreACL);
            }
            WithRepositoryConnection.get(request).commit();
        } catch (OpenRDFException e) {
            log.error("Failed loading import content or calling import: ",e);
            throw new ServletException(e);
        } finally {
            try {
                if (mrc != null && mrc.isOpen()) {
                    mrc.close();
                }
                if (mr != null)
                    mr.shutDown();
            } catch (RepositoryException e) {
                log.error("Failed while closing temporary repo of import content: ",e);
                throw new ServletException(e);
            }
        }
    }


    // parse excludes or includes list - space separated URI or string values
    private Set<String> parseXCludeList(String s)
    {
        Set<String> result = new HashSet<String>();
        if (s != null) {
            for (String e : s.split("\\s*,\\s*"))
                result.add(e);
        }
        return result;
   }
}
