package org.eaglei.repository.servlet;

import java.util.ArrayList;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.ServletException;

import org.apache.log4j.Logger;
import org.apache.log4j.LogManager;

import org.openrdf.query.impl.MapBindingSet;
import org.openrdf.query.BindingSet;
import org.openrdf.model.URI;
import org.openrdf.model.impl.LiteralImpl;
import org.openrdf.model.impl.BooleanLiteralImpl;

import org.eaglei.repository.model.Access;
import org.eaglei.repository.model.workflow.WorkflowTransition;
import org.eaglei.repository.util.SPARQL;

/**
 * Returns a tabular listing of available workflow transitions.
 * The 'format' arg selects one of the tuple query result formats.
 * Optional 'workspace' arg selects only transitions that apply to a given
 * workspace (which includes wildcards).
 * @see WorkflowTransition
 * @author Larry Stone
 * @version $Id: $
 * Started October 20, 2010
 */
public class WorkflowTransitions extends RepositoryServlet
{
    private static Logger log = LogManager.getLogger(WorkflowTransitions.class);

    /** {@inheritDoc} */
    @Override
    protected void doPost(HttpServletRequest request, HttpServletResponse response)
        throws ServletException, java.io.IOException
    {
        doGet(request, response);
    }

    /** {@inheritDoc} */
    @Override
    protected void doGet(HttpServletRequest request, HttpServletResponse response)
        throws ServletException, java.io.IOException
    {
        request.setCharacterEncoding("UTF-8");
        String format = getParameter(request, "format", false);
        URI workspace = getParameterAsURI(request, "workspace", false);
        log.debug("workspace parameter = "+workspace);

        ArrayList<BindingSet> result = new ArrayList<BindingSet>();
        for (WorkflowTransition t : WorkflowTransition.findAll(request)) {
            URI tw = t.getWorkspace();
            if (workspace == null || tw == null || tw.equals(workspace)) {
                URI subject = t.getURI();
                boolean allowed = Access.hasPermission(request, subject, Access.READ);
                MapBindingSet nbs = new MapBindingSet(10);
                nbs.addBinding("subject", subject);
                nbs.addBinding("label", new LiteralImpl(t.getLabel()));
                nbs.addBinding("initial", t.getInitial());
                nbs.addBinding("initialLabel", new LiteralImpl(t.getInitialLabel()));
                nbs.addBinding("final", t.getFinal());
                nbs.addBinding("finalLabel", new LiteralImpl(t.getFinalLabel()));
                String comment = t.getComment();
                nbs.addBinding("description", comment == null ? null : new LiteralImpl(comment));
                nbs.addBinding("workspace", tw);
                String wl = t.getWorkspaceLabel();
                nbs.addBinding("workspaceLabel", wl == null ? null : new LiteralImpl(wl));
                nbs.addBinding("allowed", BooleanLiteralImpl.valueOf(allowed));
                result.add(nbs);
            } else {
                log.debug("Skipping because of WORKSPACE restriction: "+t);
            }
        }
        SPARQL.sendTupleQueryResults(request, response, format, result);
    }
}
