package org.eaglei.repository.util;

import java.io.IOException;
import java.io.OutputStream;
import java.io.InputStream;
import java.io.UnsupportedEncodingException;
import java.util.Arrays;
import java.util.GregorianCalendar;
import java.util.Collection;
import java.util.Date;
import java.util.Iterator;
import java.util.Locale;
import java.util.SimpleTimeZone;
import java.util.TimeZone;
import javax.xml.datatype.XMLGregorianCalendar;
import javax.xml.datatype.DatatypeConstants;
import java.net.URL;
import java.net.MalformedURLException;
import java.util.Properties;
import java.util.jar.JarInputStream;
import java.util.jar.JarEntry;
import java.text.SimpleDateFormat;

import org.apache.log4j.Logger;
import org.apache.log4j.LogManager;

import org.openrdf.query.Dataset;
import org.openrdf.model.Literal;
import org.openrdf.model.Resource;
import org.openrdf.model.Statement;
import org.openrdf.model.Value;
import org.openrdf.model.URI;
import org.openrdf.model.impl.URIImpl;
import org.openrdf.model.datatypes.XMLDatatypeUtil;
import org.openrdf.model.vocabulary.OWL;
import org.openrdf.model.vocabulary.RDF;
import org.openrdf.repository.RepositoryConnection;
import org.openrdf.repository.RepositoryException;
import org.openrdf.repository.RepositoryResult;

import org.eaglei.repository.status.BadRequestException;
import org.eaglei.repository.status.InternalServerErrorException;
import org.eaglei.repository.vocabulary.REPO;

/**
 * Utility methods
 *
 * @author Larry Stone
 * @version $Id: $
 */
public class Utils
{
    private static Logger log = LogManager.getLogger(Utils.class);

    private static final int BUFSIZ = 8192;

    private static final int MILLISEC_MINUTE = 1000 * 60;

    private static final int BYTE_MASK = 0xff;

    // Similar to standard HTTP date format but with millisecond precision
    // Initialize it to use the GMT (UTC) timezone
    private static final SimpleDateFormat preciseHTTPDate =
        new SimpleDateFormat("EEE, dd MMM yyyy HH:mm:ss.SSS zzz");
    static {
        preciseHTTPDate.setCalendar(new GregorianCalendar(new SimpleTimeZone(0, "GMT"), Locale.getDefault()));
    }

    private Utils()
    {
    }

    /**
     * The canonical file copy loop.
     * Closes streams when done.
     *
     * @param in source, a {@link java.io.InputStream} object.
     * @param out destination, a {@link java.io.OutputStream} object.
     * @throws java$io$IOException if anything goes wrong.
     */
    public static void copyStream(InputStream in, OutputStream out)
        throws IOException
    {
        byte[] buf = new byte[BUFSIZ];
        int len;
        while ((len = in.read(buf)) > 0) {
            out.write(buf, 0, len);
        }
        in.close();
        out.close();
    }

    /**
     * Formatted human-readable view of Dataset contents which works
     * even when some members are null.  Needed because Sesame's toString()
     * breaks on a null URI, even though that is allowed and necessary(!)
     * for temporary fix of inferencing context problem..
     *
     * @param ds a {@link org.openrdf.query.Dataset} object.
     * @return multi-line prettyprint in a {@link java.lang.String} object.
     */
    public static String prettyPrint(Dataset ds)
    {
        StringBuilder result = new StringBuilder();

        for (URI u : ds.getDefaultGraphs())
            result.append("FROM ").append(u == null ? "{null}" : "<"+u.toString()+">").append("\n");
        for (URI u : ds.getNamedGraphs())
            result.append("FROM NAMED ").append(u == null ? "{null}" : "<"+u.toString()+">").append("\n");
        if (result.length() > 0)
            result.deleteCharAt(result.length()-1);
        return result.toString();
    }

    /**
     * Test whether a string is a well-formed-enough absolute URI that
     * it will not cause createURI to fail.
     *
     * @param s possible URI
     * @return a boolean, true if s is a valid URI.
     */
    public static boolean isValidURI(String s)
    {
        try {
            new URIImpl(s);
            return true;
        } catch (IllegalArgumentException e) {
            return false;
        }
    }

    /**
     * Translates string to URI with sanity check.
     *
     * @param s possible URI
     * @param argName name of parameter for error messages
     * @param required true if arg is required
     * @return a URI
     */
    public static URI parseURI(String s, String argName, boolean required)
    {
        if (s == null || s.trim().length() == 0) {
            if (required)
                throw new BadRequestException("Missing required argument: "+argName);
            else
                return null;
        }
        try {
            return new URIImpl(s);
        } catch (IllegalArgumentException e) {
            throw new BadRequestException("Argument '"+argName+"' is not a valid URI: "+e);
        }
    }

    /**
     * Parse a query arg value that is supposed to belong to an Enum
     * set.  If a value is required, throw the BadRequest exception if
     * not present, and in any case throw BadRequest if it is not
     * a valid enumerated value.
     * NOTE: This _ought_ to be shared in utils or some such but then all
     * the arg enums would haev to be public, and they shouldn't be.
     * @param et the enumerated type class
     * @param name value of the argument
     * @param argName name of the argument, e.g. parameter name
     * @param required when true this arg MUST have a value.
     * @param defaultValue default to subsitute when arg is not required and missing
     * @return the enum or null only if required is false and there is no value.
     * @throws BadRequestException if arg is missing or illegal
     */
    public static Enum parseKeywordArg(Class et, String name, String argName, boolean required, Enum defaultValue)
    {
        if (name == null) {
            if (required)
                throw new BadRequestException("Missing required argument: "+argName);
            else
                return defaultValue;
        }
        try {
            return Enum.valueOf(et, name);
        } catch (IllegalArgumentException e) {
            throw new BadRequestException("Illegal value for '"+argName+"', must be one of: "+Arrays.deepToString(getKeywordValues(et)), e);
        }
    }

    // get the acceptable values for given keyword enum, with own error handling
    private static Enum[] getKeywordValues(Class et)
    {
        try {
              return (Enum[])et.getMethod("values").invoke(null);
        } catch (NoSuchMethodException e) {
            log.error("Failed generating list of acceptable values for keyword: "+et.getName(), e);
            throw new InternalServerErrorException("Failed generating list of acceptable values for keyword: "+et.getName(), e);
        } catch (IllegalAccessException e) {
            log.error("Failed generating list of acceptable values for keyword: "+et.getName(), e);
            throw new InternalServerErrorException("Failed generating list of acceptable values for keyword: "+et.getName(), e);
        } catch (java.lang.reflect.InvocationTargetException e) {
            log.error("Failed generating list of acceptable values for keyword: "+et.getName(), e);
            throw new InternalServerErrorException("Failed generating list of acceptable values for keyword: "+et.getName(), e);
        }
    }

    // values of a boolean arg for use by special keyword parser
    private enum BooleanArg
    {
        true1 ("yes", true),
        true2 ("true", true),
        true3 ("t", true),
        false1 ("no", false),
        false2 ("false", false),
        false3 ("nil", false);

        private String label = null;
        private boolean truth = false;

        private BooleanArg(String nm, boolean v)
        {
            label = nm;
            truth = v;
        }
        @Override
        public String toString()
        {
            return label;
        }
        public boolean getBooleanValue()
        {
            return truth;
        }
    }

    /**
     * Special query arg parser for args with boolean values.
     * If a value is "required", throw the BadRequest exception if
     * not present, and in any case throw BadRequest if it is not
     * a valid enumerated value.  Use the enum as a convenient table
     * of strings and values.
     * @param name value of the argument
     * @param argName name of the argument, e.g. parameter name
     * @param required when true this arg MUST have a value.
     * @param defaultValue default to subsitute when arg is not required and missing
     * @return the boolean result of parsed or defaulted arg
     * @throws BadRequestException if arg is missing or illegal
     */
    public static boolean parseBooleanParameter(String name, String argName, boolean required, boolean defaultValue)
    {
        BooleanArg b = null;
        if (name == null) {
            if (required)
                throw new BadRequestException("Missing required argument: "+argName);
            else
                b = defaultValue ? BooleanArg.true1 : BooleanArg.false1;
        } else {
            for (BooleanArg ba : BooleanArg.values()) {
                if (ba.label.equals(name)) {
                    b = ba;
                    break;
                }
            }
            if (b == null)
                throw new BadRequestException("Illegal value for '"+argName+"', must be one of: "+
                  Arrays.deepToString(BooleanArg.values()));
        }
        return b.getBooleanValue();
    }

    public static XMLGregorianCalendar parseXMLDate(String raw)
    {
        XMLGregorianCalendar result = null;
        try {
            result = XMLDatatypeUtil.parseCalendar(raw);
            if (result.getDay() == DatatypeConstants.FIELD_UNDEFINED)
                throw new BadRequestException("This date/time argument is incomplete, it must include at least complete date: "+raw);
            if (result.getHour() == DatatypeConstants.FIELD_UNDEFINED)
                result.setHour(0);
            if (result.getMinute() == DatatypeConstants.FIELD_UNDEFINED)
                result.setMinute(0);
            if (result.getSecond() == DatatypeConstants.FIELD_UNDEFINED)
                result.setSecond(0);
            if (result.getTimezone() == DatatypeConstants.FIELD_UNDEFINED) {
                TimeZone td = TimeZone.getDefault();
                int tzMinutes = td.getRawOffset()/(MILLISEC_MINUTE);
                if (td.inDaylightTime(new Date()))
                    tzMinutes += td.getDSTSavings()/(MILLISEC_MINUTE);
                // XXX only for TZ debugging
                // XXX too noisy: log.debug("Tweaking by default timezone minutes = "+tzMinutes);
                result.setTimezone(tzMinutes);
            }
        } catch (IllegalArgumentException e) {
            throw new BadRequestException("Illegal date format in this date/time argument: "+e.toString(), e);
        }
        if (log.isDebugEnabled())
            log.debug("parseXMLDate: given string = \""+raw+"\", interpreted as = "+result+
                      ", as java.util.Date = "+result.toGregorianCalendar().getTime().toString());
        return result;
    }

    /**
     * Combine MIME type expression with charset string (IFF not already
     * present), and return the result.
     */
    public static String makeContentType(String mimeType, String charset)
    {
        if (mimeType.indexOf("charset=") < 0) {
            return mimeType + "; charset=" + charset;
        } else
            return mimeType;
    }

    /**
     * Extracts the character set specified with the content-type value.
     * E.g. ``text/plain; charset="UTF-8"'' => UTF-8
     * @return the charset (without quotes) or null if none specified
     */
    public static String contentTypeGetCharset(String ct, String dflt)
    {
        if (ct == null)
            return dflt;
        String e[] = ct.split(";\\s*[cC][hH][aA][rR][sS][eE][tT]=", 2);
        if (e.length < 2)
            return dflt;
        String result = e[1];
        // the charset may be enclosed in ""'s:
        if (result.startsWith("\""))
            result = result.substring(1);
        int lastQuote = result.indexOf('"');
        if (lastQuote >= 0)
            result = result.substring(0, lastQuote);
        return result;
    }

    /**
     * Extracts the MIME type specified with the content-type value.
     * E.g. ``text/plain; charset="UTF-8"'' => text/plain
     * @return the MIME type portion of the content-type string.
     */
    public static String contentTypeGetMIMEType(String ct)
    {
        String e[] = ct.split(";\\s*", 2);
        return e[0];
    }

    /**
     * Like Arrays.toString() but prints bytes in hex instead of
     * signed decimal for easier debugging of char strings.
     */
    public static String toString(byte ba[])
    {
        StringBuilder result = new StringBuilder("[" /*]*/ );
        boolean first = true;
        for (byte b : ba) {
            if (first)
                first = false;
            else
                result.append(", ");
            String h = Integer.toHexString(((int)b)& BYTE_MASK);
            if (h.length() < 2)
                result.append("0");
            result.append(h);
        }
        result.append(/*[*/ "]");
        return result.toString();
    }

    /**
     * Return the "most readable" string representation of a Sesame
     * RDF Value object - Value can be URI, BNode, or Literal; in the
     * case of a Literal it just returns hte string content, no
     * datatype or language annotations.
     * @param val incoming Value object
     * @return String rendition of the value optimized for clarity
     */
    public static String valueAsString(Value val)
    {
        return val == null ? "{null}" :
               (val instanceof Literal ? ((Literal)val).getLabel() : val.stringValue());
    }

    /**
     * Returns an URL-encoded UTF8 encoding of the string.
     * Beware, many web servers (incl. Tomcat) default to assuming
     * URLs are ISO-8859-1
     */
    public static String urlEncode(String in)
        throws IOException
    {
        try {
            byte bb[] = in.getBytes("UTF-8");
            StringBuilder result = new StringBuilder();

            for (byte b : bb) {
                char c = (char)b;
                if ((c >= 'A' && c <= 'Z') ||
                    (c >= 'a' && c <= 'z') ||
                    (c >= '0' && c <= '9') ||
                      c == '-' || c == '_' || c == '.' || c == '~') {
                    result.append(c);
                } else {
                    String hex = Integer.toHexString(b & BYTE_MASK);
                    // log.debug("Converting reserved char "+String.valueOf(b)+" => "+hex);
                    result.append((hex.length() < 2) ? "%0" : "%");
                    result.append(hex);
                }
            }
            return result.toString();
        } catch (UnsupportedEncodingException e) {
            log.error(e);
            throw e;
        }
    }

    /**
     * Escape a character string for HTML or XML attribute value:
     * (assumes you're using double-quotes to delineate the value)
     * 1. Escape the ampersands '&' to '&amp;'
     * 2. Escape double-quotes '"' to '&quot;'
     *
     * @param in the source string
     * @return escaped string
     */
    public static String escapeHTMLAttribute(String in)
    {
        return in.replace("&","&amp;").replace("\"", "&quot;");
    }


    /**
     * Get the Maven version string, if available, from a resource
     * URL containing a Jar archive.  Depends on the Maven conventions
     * that: (a) the jar contains a file named along the pattenr
     *  "META-INF/maven/GROUPID/ARTIFACTID/pom.properties", and (b) this
     * file contains a serialized Properties list, which (c) has a property
     * named "version".
     *
     * Note that upon failure, there will be an explanation in the log
     * (level WARN) of why it failed.
     *
     * @param jarURLspec the URL where the jar may be found
     * @return version string or null if not found for various reasons.
     */
    public static String getMavenVersionFromJar(String jarURLspec)
        throws IOException
    {
        JarInputStream jis = null;
        try {
            URL jarURL = new URL(jarURLspec);
            Properties pp = new Properties();
            jis = new JarInputStream(jarURL.openStream());
            JarEntry je;
            while ((je = jis.getNextJarEntry()) != null) {
                if (je.isDirectory())
                    continue;
                String fn = je.getName();
                if (fn.startsWith("META-INF/maven/") && fn.endsWith("/pom.properties")) {
                    log.debug("..found POM properties: "+fn);
                    pp.load(jis);
                    if (pp.containsKey("version"))
                        return pp.getProperty("version");
                    else {
                        log.warn("Cannot find 'version' property in jar's POM properties, jar URL="+jarURL+", entry="+fn);
                        return null;
                    }
                }
            }
        } catch (MalformedURLException e) {
            log.error(e);
            throw new IOException(e);
        } finally {
            if (jis != null)
                jis.close();
        }
        log.warn("Cannot find Maven POM properties file in jar, resource URL="+jarURLspec);
        return null;
    }

    /**
     * Get the owl:versionInfo value (as String) for the given subject and
     * named graph.  Named graph can be null to search all graphs.
     * Try searching on both the OWL and repo-internal version predicates.
     * @return string value of owl:versionInfo statement or null if none found.
     */
    public static String getVersionInfo(RepositoryConnection rc, URI subject, URI graph)
        throws RepositoryException
    {
        String result = getVersionInfoInternal(rc, subject, OWL.VERSIONINFO, graph);
        if (result == null)
            result = getVersionInfoInternal(rc, subject, REPO.VERSION_INFO, graph);
        return result;
    }

    // search for value of a version predicate.
    private static String getVersionInfoInternal(RepositoryConnection rc, URI subject, URI predicate, URI graph)
        throws RepositoryException
    {
        RepositoryResult<Statement> rr =
                rc.getStatements(subject, predicate, null, false, graph);
        try {
            if (rr.hasNext()) {
                Value version = rr.next().getObject();
                if (version instanceof Literal)
                    return ((Literal)version).getLabel();
            }
        } finally {
            rr.close();
        }
        return null;
    }

    /**
     * Get a resource instance's home graph, ideally where its *asserted*
     * rdf:type statements live - inferred types don't count.
     * NOte this does NOT impose access controls.
     * @param request
     * @param resource - the URI of the resorce
     * @return URI of home graph or null if not found for any reason
     */
    public static URI getHomeGraph(RepositoryConnection rc, URI resource)
        throws RepositoryException
    {
        RepositoryResult<Statement> rr = null;
        try {
            rr = rc.getStatements(resource, RDF.TYPE, null, false);
            while (rr.hasNext()) {
                Statement s = rr.next();
                Resource ctx = s.getContext();
                log.debug("Found statement: "+resource+" rdf:type "+s.getObject()+", in graph "+ctx);
                if (ctx instanceof URI) {
                    return (URI)ctx;
                }
            }
        } finally {
            rr.close();
        }
        return null;
    }

    /**
     * Returns a string representation of the "deep contents" of the specified
     * Collection, rendering each element with its toString().  String
     * elements are enclosed in double-quotes.  The whole collection is
     * surrounded in curly braces and elements are separated by comma.
     * @param c the collection to print
     * @return string representation of the collection.
     */
    public static String collectionDeepToString(Collection<? extends Object> c)
    {
        StringBuilder result = new StringBuilder("{"/*}*/);
        boolean first = true;
        Iterator<? extends Object> ci = c.iterator();
        while (ci.hasNext()) {
            if (first)
                first = false;
            else
                result.append(", ");
            Object cin = ci.next();
            if (cin instanceof String)
                result.append("\"").append(cin).append("\"");
            else
                result.append(cin);
        }
        result.append(/*{*/"}");
        return result.toString();
    }

    /**
     * Returns a Date formatted in a more precise version of the standard
     * HTTP date format - the same as the conventional standard format but
     * with milliseconds added to the time.
     *
     * @param d the Date to format
     * @return formatted string representation of the date
     */
    public static String makePreciseHTTPDate(Date d)
    {
        return preciseHTTPDate.format(d);
    }

    // like perl join()
    public static String join(String glue, String as[])
    {
        StringBuilder result = new StringBuilder();
        boolean first = true;
        for (String s : as) {
            if (first) {
                first = false;
            } else {
                result.append(glue);
            }
            result.append(s);
        }
        return result.toString();
    }
}
