package org.eaglei.repository;

import java.io.File;
import java.io.InputStreamReader;
import java.io.Reader;
import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.net.URL;
import java.net.URLConnection;
import java.nio.charset.IllegalCharsetNameException;
import java.nio.charset.UnsupportedCharsetException;
import java.util.Date;
import java.util.HashMap;
import java.util.Map;

import javax.servlet.ServletException;
import javax.servlet.ServletContext;
import javax.servlet.http.HttpSession;

import org.apache.log4j.Logger;
import org.apache.log4j.LogManager;

import org.reflections.Reflections;

import org.openrdf.repository.Repository;
import org.openrdf.repository.RepositoryConnection;
import org.openrdf.repository.RepositoryException;
import org.openrdf.repository.sail.SailRepository;
import org.openrdf.sail.nativerdf.NativeStore;
import org.openrdf.model.URI;
import org.openrdf.model.Resource;
import org.openrdf.model.Statement;
import org.openrdf.model.vocabulary.OWL;
import org.openrdf.rio.RDFFormat;
import org.openrdf.rio.RDFParseException;
import org.openrdf.rio.Rio;
import org.openrdf.rio.RDFHandlerException;
import org.openrdf.rio.helpers.RDFHandlerBase;
import org.openrdf.rio.RDFParser;
import org.openrdf.OpenRDFException;

import org.eaglei.repository.inferencer.MinimalInferencer;
import org.eaglei.repository.model.Provenance;
import org.eaglei.repository.vocabulary.REPO;
import org.eaglei.repository.vocabulary.DCTERMS;
import org.eaglei.repository.util.Utils;

/**
 * This is a singleton class that encompasses the application state.
 * It is created and destroyed by Java Servlet Container webapp
 * lifecycle events, through initialize() and destroy() hooks.
 *
 * @author Larry Stone
 * Started April 2010
 * @version $Id: $
 */
public class Lifecycle
{
    private static Logger log = LogManager.getLogger(Lifecycle.class);

    /** Key to ServletContext attribute for open Sesame repository object */
    private static final String SCTX_REPOSITORY = "org.eaglei.repository.SesameRepository";

    /** Key to generation counter in current app context */
    private static final String SCTX_GENERATION = "org.eaglei.repository.Generation";

    // map of named graph URI to resource filename from which it is
    // initialized if empty
    private static final Map<URI,String> graphInitFile = new HashMap<URI,String>();
    static {
        graphInitFile.put(REPO.NAMESPACE_URI, "repository-ont.n3");
        graphInitFile.put(REPO.NG_INTERNAL, "repository-internal.n3");
        graphInitFile.put(REPO.NG_QUERY, "query-macros.n3");
    }

    /** Mark the startup time for display in admin page, etc. */
    public static final Date STARTUP = new Date();

    //---------------------- instance variables:

    // cache of the servlet context given to initialize(), mainly
    // used to store things in attributes.
    private ServletContext servletContext = null;

    // timestamp of most recent last-modified setting, useful
    // for optimizing time-bounded metadata harvesting queries.
    private Date lastModified = new Date();

    // the singleton instance
    private static Lifecycle instance = null;

    /**
     * <p>Constructor for Lifecycle.</p>
     *
     * @param sc a {@link javax.servlet.ServletContext} object.
     */
    public Lifecycle(ServletContext sc)
    {
        super();
        servletContext = sc;
    }

    /**
     * Get the singleton instance.  Will return null unless the initialization has run.
     *
     * @return the singleton, a {@link org.eaglei.repository.Lifecycle} object.
     */
    public static Lifecycle getInstance()
    {
        return instance;
    }

    /**
     * Web app initialization hook that initiates the bootstrap process:
     *  1. find and load the configuration properties file.
     *  2. configure log4j
     *  3. set up sesame repository based on configs
     *
     * @param sc the {@link javax.servlet.ServletContext} object.
     */
    public static void initialize(ServletContext sc)
    {
        if (instance == null) {
            try {
                Configuration.getInstance().initialize();
                instance = new Lifecycle(sc);
                instance.finishInitialize();
            } catch (Exception e) {
                log.fatal("Got exception in Webapp context initialization, DO NOT EXPECT ANYTHING TO WORK NOW:", e);
            }
        }
        else
            log.fatal("Initialize was called after Lifecycle already initialized!");
    }



    /**
     * <p>finishInitialize - setup at construct time</p>
     *
     * @throws java$io$IOException if any.
     * @throws org.openrdf.OpenRDFException if any.
     */
    public void finishInitialize()
        throws IOException, OpenRDFException
    {
        // 1. Start up Sesame repository in the configured directory,
        //    with configured indexes.
        File sesDir = Configuration.getInstance().getSesameDirectory();
        NativeStore ns = new NativeStore(sesDir);
        Repository r = new SailRepository(new MinimalInferencer(ns));
        String indexes[] = Configuration.getInstance().getConfigurationPropertyArray(Configuration.SESAMEINDEXES, null);
        if (indexes != null) {
            String si = Utils.join(", ", indexes);
            log.debug("Setting Sesame NativeStore indexes to: \""+si+"\"");
            ns.setTripleIndexes(si);
        }
        r.initialize();
        if (!r.isWritable())
            log.fatal("Sesame repo is not writable, this is going to cause trouble soon!");
        servletContext.setAttribute(SCTX_REPOSITORY, r);
        log.info("Sesame Repository open, directory="+sesDir.toString());

        servletContext.setAttribute(SCTX_GENERATION, Integer.valueOf(1));
        log.debug("Generation initialized to 1.");

        // 2. Load initial content into Sesame graphs if necessary, e.g.
        // the repo ontology,  initial local admin MD
        for (URI key : graphInitFile.keySet()) {
            loadEmptyGraphFromInit(r, key, true);
        }
    }

    /**
     * <p>destroy - shut down this application and release resources.</p>
     *
     * @param sc a {@link javax.servlet.ServletContext} object.
     * @throws org.openrdf.repository.RepositoryException if any.
     */
    public void destroy(ServletContext sc)
        throws RepositoryException
    {
        // Shutdown sesame repository - MUST be done to protect
        // the integrity of Native repository files.
        Repository r = (Repository)sc.getAttribute(SCTX_REPOSITORY);
        sc.removeAttribute(SCTX_REPOSITORY);
        if (r != null) {
            r.shutDown();
            log.info("Sesame Repository closed.");
        }
        instance = null;
    }

    /**
     * Gets the version that would be set by loading the RDF from given
     * init file; this is the value of owl:versionInfo on the given URI,
     * if any, in the serialized RDF fiel found on resourcePath.
     *
     * @param uri the uri of the graph, subject of owl:versionInfo
     * @return the exact content of the versionInfo literal or null if none found.
     */
    public static String getInitFileVersion(URI uri)
        throws IOException, RDFParseException, RDFHandlerException
    {
        String resourcePath = graphInitFile.get(uri);
        if (resourcePath != null) {
            RDFParser parser = Rio.createParser(RDFFormat.forFileName(resourcePath, RDFFormat.N3));
            VersionFinder vf = new VersionFinder(uri);
            parser.setRDFHandler(vf);
            parser.parse(new InputStreamReader(Lifecycle.class.getClassLoader().getResourceAsStream(resourcePath), "UTF-8"), "");
            return vf.result;
        } else {
            log.warn("No resource path found for uri="+uri);
            return null;
        }
    }

    // handler for parser to extract the specific owl:versionInfo value
    // used by getInitFileVersion()
    private static class VersionFinder extends RDFHandlerBase
    {
        private String result = null;
        private URI name = null;

        private VersionFinder(URI uri)
        {
            super();
            name = uri;
        }

        @Override
        public void handleStatement(Statement s)
            throws RDFHandlerException
        {
            Resource subject = s.getSubject();
            if (subject instanceof URI && name.equals(subject) &&
                  (OWL.VERSIONINFO.equals(s.getPredicate()) ||
                   REPO.VERSION_INFO.equals(s.getPredicate()))) {
                result = Utils.valueAsString(s.getObject());
            }
        }
    }

    /**
     * Get the init file (relative resource path) for given graph URI.
     * @return relative path or null if there is none for this URI.
     */
    public static String getGraphInitFile(URI uri)
    {
        return graphInitFile.get(uri);
    }


    /**
     * Initialize an empty named graph from its registered "initial
     * contents" file (a webapp resource).  Typically this only actually
     * loads the graph during the initial bootstrap of a new repo when
     * the entire Sesame repository is empty.
     */
    private void loadEmptyGraphFromInit(Repository r, URI graphURI, boolean ifEmpty)
        throws RepositoryException, IOException, RDFParseException
    {
        RepositoryConnection rc = null;
        try {
            rc = r.getConnection();
            loadEmptyGraphFromInit(rc, graphURI, ifEmpty);
        } finally {
            if (rc != null)
                rc.close();
        }
    }

    /**
     * Initialize an empty named graph from its registered "initial
     * contents" file (a webapp resource).  Typically this only actually
     * loads the graph during the initial bootstrap of a new repo when
     * the entire Sesame repository is empty, although it is also used
     * by some upgrade procedures.
     * @param rc the sesame repo
     * @param graphURI name of the graph
     * @param ifEmpty when true only load an empty graph
     */
    public void loadEmptyGraphFromInit(RepositoryConnection rc, URI graphURI, boolean ifEmpty)
        throws RepositoryException, IOException, RDFParseException
    {
        if (!ifEmpty || rc.size((Resource)graphURI) == 0) {
            String resourcePath = graphInitFile.get(graphURI);
            if (resourcePath == null) {
                log.error("No local resource path found (NOT loading anything) for URI="+graphURI);
                return;
            }
            URL rurl = this.getClass().getClassLoader().getResource(resourcePath);
            if (rurl == null)
                log.error("Cannot find webapp resource at path="+resourcePath);
            else {
                URLConnection ruc = rurl.openConnection();
                long rdate = ruc.getLastModified();
                log.debug("Loading graph="+graphURI.toString()+" from URL="+rurl.toString()+", mod-date="+rdate);
                Reader is = new InputStreamReader(ruc.getInputStream(), "UTF-8");
                if (is == null)
                    log.error("Cannot open webapp resource at url="+rurl);
                else {
                    rc.add(is, graphURI.toString(), RDFFormat.forFileName(resourcePath, RDFFormat.N3), graphURI);

                    // make up some provenance - datestamp should match date of resource in WAR file
                    // which is useful for checking the actual "provenance".
                    Provenance gp = new Provenance(graphURI);
                    gp.setProvenanceStatement(rc, DCTERMS.CREATED, Provenance.makeDateTime(new Date()));
                    gp.setProvenanceStatement(rc, DCTERMS.CREATOR, REPO.ROLE_SUPERUSER);
                    gp.setSourceStatements(rc, rc.getValueFactory().createURI(rurl.toString()),
                                         Provenance.makeDateTime(new Date(rdate)));
                    rc.commit();
                    log.info("Initialized the empty named graph, name="+graphURI+", from resource path="+resourcePath);
                }
            }
        }
    }

    /**
     * <p>getSesameRepository</p>
     *
     * @return the {@link org.openrdf.repository.Repository} object.
     * @throws javax.servlet.ServletException if any.
     */
    public Repository getSesameRepository()
        throws ServletException
    {
        Repository result = (Repository)servletContext.getAttribute(SCTX_REPOSITORY);
        if (result == null)
            throw new ServletException("No RDF database connection, probably because of a failure in initialization.");
        return result;
    }

    /**
     * Hook to signal to the application that all RDF data has been replaced,
     * e.g. after restoring a complete backup of all graphs.  This is
     * responsible for updating or invalidating all memory caches that depend
     * on the RDF contents.
     */
    public void notifyDataReplaced()
    {
        incrementGeneration();
        decacheAll();
    }

    /**
     * Bump the generation counter to signal that RDF database contents
     * have been replaced (i.e. a "new generation") and existing sessions
     * should be stale.   This is used to invalidate any cached URIs, e.g.
     * the user's Person-object URI mapped from authenticated principal name.
     */
    private void incrementGeneration()
    {
        int gen = 1 + ((Integer)servletContext.getAttribute(SCTX_GENERATION)).intValue();
        servletContext.setAttribute(SCTX_GENERATION, Integer.valueOf(gen));
        log.debug("Generation is incremented to "+gen);
    }

    /**
     * Compare current webapp generation with value (if any) cached
     * in session, also update session's generation to current.
     *
     * @param session a {@link javax.servlet.http.HttpSession} object.
     * @return a boolean, true if the session is "stale" (e.g. after a global restore of RDF db)
     */
    public boolean isSessionStale(HttpSession session)
    {
        Integer sessionGen = (Integer)session.getAttribute(SCTX_GENERATION);
        Integer ctxGen = (Integer)servletContext.getAttribute(SCTX_GENERATION);
        boolean result = sessionGen != null &&
                 sessionGen.intValue() < ctxGen.intValue();
        session.setAttribute(SCTX_GENERATION, ctxGen);
        return result;
    }

    /**
     * Read a resource file out of the webapp, path is relative to webapp root
     * and MUST NOT begin with '/' -- e.g. "repository/styles/foo.xsl".
     *
     * @param path relative path to resource file, a {@link java.lang.String} object.
     * @return a {@link java.io.Reader} object open on the resource, or null if not available.
     */
    public Reader getWebappResourceAsReader(String path)
        throws ServletException
    {
        if (servletContext == null) {
            log.error("getWebappResourceAsStream: currentContext not set!");
            return null;
        }
        try {
            return new InputStreamReader(servletContext.getResourceAsStream(path), "UTF-8");
        } catch  (UnsupportedEncodingException e) {
            throw new ServletException("Unsupported encoding content-type spec: "+e);
        } catch  (IllegalCharsetNameException e) {
            throw new ServletException("Illegal character set name in content-type spec: "+e);
        } catch  (UnsupportedCharsetException e) {
            throw new ServletException("Unsupported character set name in content-type spec: "+e);
        }
    }

    /**
     * Update the repo's global "last modified" timestamp, changing it
     * ONLY if the profferred value is later.  It should reflect the
     * latest last-modified time on ANY resource instance.
     *
     * @param lm date of new most recent dcterms:modified
     */
    public synchronized void updateLastModified(Date lm)
    {
        if (lastModified.before(lm))
            lastModified = lm;
    }

    /**
     * Gets the most recent "last modified" timestamp applied to
     * provenance metadata.
     *
     * @return date of most recent dcterms:modified, or repository startup time by default
     */
    public Date getLastModified()
    {
        return lastModified;
    }

    /**
     * Call the decache() method of all classes with the HasContentCache
     * annotation.  This gets called when the RDF database was replaced
     * so in-memory caches must be invalidated.
     */
    public static void decacheAll()
    {
        log.info("Decaching all content caches..");
        Reflections r = new Reflections("org.eaglei.repository");
            for (Class c : r.getTypesAnnotatedWith(HasContentCache.class)) {
                log.debug("  Decaching "+c.getName());
            try {
                c.getMethod("decache").invoke(null);
            } catch (NoSuchMethodException e) {
                log.error("Class is marked as content cacher but has no static decache() method - "+c.getName()+": "+e);
            } catch (IllegalAccessException e) {
                log.error("Failed calling decache() on content cacher class="+c.getName()+": "+e);
            } catch (java.lang.reflect.InvocationTargetException e) {
                log.error("Failed calling decache() on content cacher class="+c.getName()+": "+e);
            }
        }
    }
}
