package org.eaglei.repository.model;

import java.util.List;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.ServletException;

import org.openrdf.model.URI;

/**
 * <pre>
 * Outline of all "Object Model" classes - even the writable ones are a
 * subclass of this.  Object Models are Java objects that represent an
 * instance or layer -- i.e. a body of properties on an instance -- in the
 * RDF database.  They exist to cache the information and make it easier
 * to manipulate in Java.  This class describes a common interface to
 * be implemented by all object models.
 *
 * This is implemented as an abstract class instead of an interface for
 * several reasons:
 *  1. implement and/or require static methods
 *  2. allow subclasses to skip implementing irrelevant methods so long
 *     as they never get called.
 *  3. consistency with the WritableObjectModel class, which had to be
 *     abstract since it includes its own common logic.
 * </pre>
 * @see WritableObjectModel
 * @see HasContentCache
 * @author Larry Stone
 * Started March 2011
 */
public abstract class ImmutableObjectModel
{
    /**
     * Get an instance of the object model whose subject is the given URI.
     * This is the canonical way to look up or discover an object.
     * @param request the HTTP request object from the servlet
     * @param subject URI of hte subject of the RDF object or layer
     * @return an instance of the object model class filled in for subject, or null if not found.
     */
    public static ImmutableObjectModel find(HttpServletRequest request, URI subject)
        throws ServletException
    {
        throw new IllegalStateException("find() was not implemented for this ImmutableObjectModel subclass.");
    }

    /**
     * Gets List of object models for _all_ instances of the class found in the
     * repository.  Should be cached, perhaps statically, for infrequently-
     * changning items.  See the HasContentCache class for clues about decaching.
     * @param request the HTTP request object from the servlet
     * @return List of all instances of the object model class in the repository
     */
    public static List<? extends ImmutableObjectModel> findAll(HttpServletRequest request)
        throws ServletException
    {
        throw new IllegalStateException("findAll() was not implemented for this ImmutableObjectModel subclass.");
    }

    /**
     * Get the subject URI taht created this instance.
     * @return the subject URI
     */
    public abstract URI getURI();

    /**
     * Get a human-readable text string appropriate to label this instance
     * in e.g. a UI display or printed document.
     * @return the text label
     */
    public abstract String getLabel();
}
