package org.eaglei.repository.model;

import java.io.IOException;
import java.io.OutputStreamWriter;
import java.util.Map;
import java.util.HashMap;
import java.util.Set;
import java.util.HashSet;

import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.apache.log4j.Logger;
import org.apache.log4j.LogManager;

import org.openrdf.OpenRDFException;
import org.openrdf.model.URI;
import org.openrdf.model.impl.URIImpl;
import org.openrdf.model.impl.ContextStatementImpl;
import org.openrdf.model.BNode;
import org.openrdf.model.Resource;
import org.openrdf.model.Value;
import org.openrdf.model.impl.LiteralImpl;
import org.openrdf.model.ValueFactory;
import org.openrdf.model.vocabulary.XMLSchema;
import org.openrdf.model.vocabulary.RDF;
import org.openrdf.repository.RepositoryConnection;
import org.openrdf.query.TupleQuery;
import org.openrdf.query.TupleQueryResultHandlerBase;
import org.openrdf.query.TupleQueryResultHandlerException;
import org.openrdf.query.BindingSet;
import org.openrdf.query.QueryLanguage;
import org.openrdf.rio.RDFFormat;
import org.openrdf.rio.Rio;

import org.eaglei.repository.auth.Authentication;
import org.eaglei.repository.servlet.WithRepositoryConnection;
import org.eaglei.repository.vocabulary.REPO;
import org.eaglei.repository.status.BadRequestException;
import org.eaglei.repository.status.InternalServerErrorException;
import org.eaglei.repository.servlet.ImportExport.DuplicateArg;
import org.eaglei.repository.status.ForbiddenException;
import org.eaglei.repository.util.HandlerBadRequest;
import org.eaglei.repository.util.AppendingRDFHandler;
import org.eaglei.repository.util.Utils;

/**
 * Export and import of the User object (and authentication credentials).
 *
 * @author Larry Stone
 * Started March, 2011
 */
public class TransportUser implements Transporter
{
    private static Logger log = LogManager.getLogger(TransportUser.class);

    // special property only used in export and import of users to hold
    // RDBMS password entry
    private static final URI EXPORT_AUTH_PASSWORD = new URIImpl(REPO.NAMESPACE + "exportAuthPassword");

    // Special property only used in export and import of users to
    // indicate which type of user authentication implements this user
    private static final URI EXPORT_AUTH_TYPE = new URIImpl(REPO.NAMESPACE + "exportAuthType");

    // interpret the statements in user import graph
    private static final String importUserPass1Query =
        "SELECT DISTINCT ?subject ?username ?password ?userProp ?userPropValue WHERE { \n"+
        " GRAPH <"+User.USER_GRAPH+"> { ?subject a <"+REPO.PERSON+"> }\n"+
        " GRAPH <"+REPO.NG_INTERNAL+"> { ?subject <"+REPO.HAS_PRINCIPAL_NAME+"> ?username}\n"+
        " OPTIONAL { GRAPH <"+User.USER_GRAPH+"> { ?subject ?userProp ?userPropValue }}\n"+
        " OPTIONAL { GRAPH <"+REPO.NG_INTERNAL+"> { ?subject <"+EXPORT_AUTH_PASSWORD+"> ?password }}}\n"+
        "ORDER BY ?subject";

    // just get Roles
    private static final String importUserPass2Query =
        "SELECT DISTINCT ?subject ?username ?role WHERE { \n"+
        " GRAPH <"+User.USER_GRAPH+"> { ?subject a <"+REPO.PERSON+"> }\n"+
        " GRAPH <"+REPO.NG_INTERNAL+"> { ?subject <"+REPO.HAS_PRINCIPAL_NAME+"> ?username ; <"+REPO.HAS_ROLE+"> ?role}}\n"+
        "ORDER BY ?subject";

    /**
     * Check that current authenticated user is authorized for this
     * operation.
     *
     * Policy: user export requires Superuser privilege because  it
     * can display login credentials.
     */
    @Override
    public void authorizeExport(HttpServletRequest request)
        throws ServletException
    {
        if (!Authentication.isSuperuser(request))
            throw new ForbiddenException("Export of Users requires administrator privileges.");
    }

    /**
     * Check that current authenticated user is authorized for this
     * operation.
     *
     * Policy: user import requires Superuser privilege because  it
     * can change login credentials.
     */
    @Override
    public void authorizeImport(HttpServletRequest request)
        throws ServletException
    {
        if (!Authentication.isSuperuser(request))
            throw new ForbiddenException("Import of Users requires administrator privileges.");
    }

    /**
     * Export description of user accounts to serialized RDF quads.
     *
     * {@inheritDoc}
     */
    @Override
    public void doExport(HttpServletRequest request, HttpServletResponse response,
                       RDFFormat format, Set<String> includes, Set<String> excludes)
        throws ServletException, IOException
    {
        try {
            AppendingRDFHandler out = new AppendingRDFHandler(Rio.createWriter(format, new OutputStreamWriter(response.getOutputStream(), "UTF-8")));
            RepositoryConnection rc = WithRepositoryConnection.get(request);
            ValueFactory vf = rc.getValueFactory();

            out.startRDF();
            for (User u : User.findAll(request)) {
                String un = u.getUsername();
                URI uu = u.getURI();
                String uus = uu == null ? null : uu.toString();
                if (excludes.contains(un) || (uus != null && excludes.contains(uus))) {
                    log.debug("SKIP USER because of exclude: "+u);
                } else if (includes.isEmpty() ||
                         includes.contains(un) || (uus != null && includes.contains(uus))) {
                    Resource subject = null;

                    // undocumented user
                    if (uu == null) {
                        log.debug("EXPORT UNDOCUMENTED USER: \""+un+"\"");
                        subject = vf.createBNode();
                        out.handleStatement(new ContextStatementImpl(subject,
                            RDF.TYPE,
                            REPO.PERSON,
                            User.USER_GRAPH));
                        out.handleStatement(new ContextStatementImpl(subject,
                            REPO.HAS_PRINCIPAL_NAME,
                            new LiteralImpl(un, XMLSchema.STRING),
                            REPO.NG_INTERNAL));
                        if (u.isSuperuser()) {
                            out.handleStatement(new ContextStatementImpl(subject,
                                REPO.HAS_ROLE,
                                REPO.ROLE_SUPERUSER,
                                REPO.NG_INTERNAL));
                        }

                    // documented, but may not be in RDBMS (i.e. no passwd)
                    } else {
                        log.debug("EXPORT DOCUMENTED USER: "+uu);
                        subject = uu;
                        // XXX maybe add NG_METADATA graph here? nothing there YET..
                        rc.exportStatements(uu, null, null, false, out,
                            User.USER_GRAPH, REPO.NG_INTERNAL);
                    }
                     
                    String pw = u.getPassword();
                    if (pw == null) {
                        log.debug("User has no password or credentials: username="+un);
                    } else {
                        // synthetic statement: identify user type
                        out.handleStatement(new ContextStatementImpl(subject,
                            EXPORT_AUTH_TYPE,
                            u.getAuthType(),
                            REPO.NG_INTERNAL));
                        // synthetic statement: add the password from RDBMS
                        out.handleStatement(new ContextStatementImpl(subject,
                            EXPORT_AUTH_PASSWORD,
                            new LiteralImpl(pw, XMLSchema.STRING),
                            REPO.NG_INTERNAL));
                    }
                } else {
                    log.debug("SKIP USER because of include: "+u);
                }
            }
            out.reallyEndRDF();
        } catch (OpenRDFException e) {
            throw new InternalServerErrorException(e);
        }
    }


    /**
     * Import description of user accounts from serialized RDF quads.
     *
     * {@inheritDoc}
     */
    @Override
    public void doImport(HttpServletRequest request, HttpServletResponse response,
                       RepositoryConnection content,
                       Set<String> includes, Set<String> excludes,
                       DuplicateArg duplicate,
                       boolean transform, boolean ignoreACL)
        throws ServletException, IOException
    {
        try {
            // pass 1 - basic description and user metadata
            TupleQuery q = content.prepareTupleQuery(QueryLanguage.SPARQL, importUserPass1Query);
            if (log.isDebugEnabled())
                log.debug("SPARQL query PASS1 against internal memory repo = "+importUserPass1Query);
            q.setDataset(User.USER_DATASET);
            q.setIncludeInferred(false);
            UserHandler uh = new UserHandler(request, transform, duplicate, includes, excludes);
            q.evaluate(uh);

            // pass 2 - add roles
            q = content.prepareTupleQuery(QueryLanguage.SPARQL, importUserPass2Query);
            if (log.isDebugEnabled())
                log.debug("SPARQL query PASS2 against internal memory repo = "+importUserPass2Query);
            q.setDataset(User.USER_DATASET);
            q.setIncludeInferred(false);
            q.evaluate(uh);

            // commit all the new and/or modified Users at once
            User.commitMultiple(request, uh.result.values());
        } catch (HandlerBadRequest e) {
            log.error("Failed in query result handler: ",e);
            throw new BadRequestException(e.getMessage(), e);
        } catch (OpenRDFException e) {
            log.error("Failed in sesame: ",e);
            throw new InternalServerErrorException(e);
        }
    }

    // Pass 1 of user import:
    //  - filter users to import
    //  - create RDBMS entry for selected users
    //  - prepare map of URI to username for second pass
    private static class UserHandler extends TupleQueryResultHandlerBase
    {
        // map of user URI to principal username
        private HttpServletRequest request;
        private boolean transform;
        private DuplicateArg duplicate;
        private Set<String> includes;
        private Set<String> excludes;

        // results, indexed by username, to find users to add to
        private Map<String,User> result = new HashMap<String,User>();

        // Users to skip as determined by include/exclude, index by username
        private Set<String> ignore = new HashSet<String>();

        public UserHandler(HttpServletRequest request, boolean transform,
                DuplicateArg duplicate, Set<String> includes, Set<String> excludes)
        {
            super();
            UserHandler.this.request = request;
            UserHandler.this.transform = transform;
            UserHandler.this.duplicate = duplicate;
            UserHandler.this.includes = includes;
            UserHandler.this.excludes = excludes;
        }

        @Override
        public void handleSolution(BindingSet bs)
            throws TupleQueryResultHandlerException
        {
            try {
                // sanity check - must have a username
                if (bs.hasBinding("username")) {
                    String username = Utils.valueAsString(bs.getValue("username"));
                 
                    // subject may be URI or BNode
                    Value vsubject = bs.getValue("subject");
                    URI uri = null;
                    if (vsubject instanceof URI) {
                        uri = (URI)vsubject;
                    } else if (!(vsubject instanceof BNode)) {
                        log.error("UserHandler: Got broken result with invalid subject, not URI or BNode; Subject="+vsubject.stringValue());
                        return;
                    }
                 
                    // if role is bound, this must be pass2:
                    // NOTE: nonexistent role will result in failure
                    if (bs.hasBinding("role")) {
                        if (result.containsKey(username)) {
                            User u = result.get(username);
                            if (u == null) {
                                log.error("UserHandler: got Role for nonexistent user="+username);
                            } else {
                                Value vrole = bs.getValue("role");
                                if (vrole instanceof URI) {
                                    if (log.isDebugEnabled())
                                        log.debug("Adding role: username="+username+", role="+vrole.stringValue());
                                    u.addRole(request, (URI)vrole);
                                } else {
                                    log.error("UserHandler: Got broken Role with invalid role URI="+vrole.stringValue());
                                }
                            }
                        } else if (!ignore.contains(username)) {
                            log.error("UserHandler: Got bad Role result for unknown uri="+uri);
                        }
                     
                    // this is a pass1 result
                    // ?subject ?username ?password ?userProp ?userPropValue
                    // XXX NOTE: Should really check the EXPORT_AUTH_TYPE
                    // XXX  property as well and call User.setAuthType with it..
                    // XXX  can get away without this since there's no real pluggable auth
                    } else {
                            User u = result.get(username);
                     
                            // ...this is first time we've seen this user URI, so
                            // check for exclude/include and create it
                            if (u == null && !ignore.contains(username)) {
                                String suri = uri == null ? "" : uri.stringValue();
                                String pw = (bs.hasBinding("password")) ? Utils.valueAsString(bs.getValue("password")) : null;
                                if (excludes.contains(username) || excludes.contains(suri)) {
                                    log.debug("PASS1: SKIP USER import because of exclude: username="+username+", or subject="+suri);
                                    ignore.add(username);
                                } else if (!(includes.isEmpty() ||
                                           includes.contains(username) ||
                                           includes.contains(suri))) {
                                    log.debug("PASS1: SKIP USER import because of include: username="+username+", or subject="+suri);
                                    ignore.add(username);
                                } else {
                                    // check for duplicate before creating:
                                    if ((u = User.findByUsername(request, username)) != null ||
                                         (uri != null && (u = User.find(request, uri)) != null)) {
                                        log.debug("PASS1: FOUND DUPLICATE SUBJECT, username="+username+", subject="+uri);
                                        if (duplicate == DuplicateArg.ignore) {
                                            ignore.add(username);
                                            return;
                                        } else if (duplicate == DuplicateArg.abort) {
                                            throw new HandlerBadRequest("Import contains a duplicate user for username="+username+", URI="+uri);
                                        } else if (duplicate == DuplicateArg.replace) {
                                            if (transform) {
                                                u = User.create(request, username, pw);
                                            } else {
                                                u.recreate(request, uri, username, pw);
                                            }
                                        }

                                    // create a new User with either explicit URI or new local one
                                    } else {
                                        // when import has no URI, create  an
                                        // undocumented user with uri=null.
                                        if (uri == null || !transform)
                                            u = User.create(request, uri, username, pw);
                                        else
                                            u = User.create(request, username, pw);
                                        log.debug("PASS1: Created new User, username="+username+", subject="+u.getURI());
                                    }
                                    result.put(username, u);
                                }
                            }
                     
                            // handle any metadata statements
                            if (u != null && bs.hasBinding("userProp")) {
                                Value vprop = bs.getValue("userProp");
                                if (vprop instanceof URI) {
                                    Value vval = bs.getValue("userPropValue");
                                    if (vval == null) {
                                        log.error("UserHandler: bad result record, user="+username+", NO userPropValue for userProp="+vprop);
                                    } else {
                                        u.setProperty(request, (URI)vprop, Utils.valueAsString(vval));
                                    }
                                } else {
                                    log.error("UserHandler: bad result record, user="+username+", userProp is not a URI, userProp="+vprop);
                                }
                            }
                     
                        }
                 
                // sanity check fail, username not bound
                } else {
                    log.error("UserHandler: bad result record, username not bound, bindings="+bs);
                }
            } catch (ServletException e) {
                log.error("Failed in UserHandler: ",e);
                throw new TupleQueryResultHandlerException(e);
            }
        }
    }
}
