package org.eaglei.repository.model;

import java.util.Arrays;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.HashSet;
import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;

import org.apache.log4j.Logger;
import org.apache.log4j.LogManager;
import org.eaglei.repository.Configuration;

import org.openrdf.OpenRDFException;
import org.openrdf.model.URI;
import org.openrdf.model.Literal;
import org.openrdf.model.Value;
import org.openrdf.model.impl.LiteralImpl;
import org.openrdf.model.impl.URIImpl;
import org.openrdf.model.ValueFactory;
import org.openrdf.model.vocabulary.RDF;
import org.openrdf.model.vocabulary.RDFS;
import org.openrdf.repository.RepositoryException;
import org.openrdf.repository.RepositoryConnection;
import org.openrdf.query.TupleQuery;
import org.openrdf.query.TupleQueryResultHandlerBase;
import org.openrdf.query.TupleQueryResultHandlerException;
import org.openrdf.query.BindingSet;
import org.openrdf.query.QueryLanguage;
import org.openrdf.query.impl.DatasetImpl;
import org.openrdf.query.MalformedQueryException;

import org.eaglei.repository.auth.Authentication;
import org.eaglei.repository.auth.AuthUser;
import org.eaglei.repository.auth.AuthUserFactory;
import org.eaglei.repository.auth.AuthUserTomcatFactory;
import org.eaglei.repository.servlet.WithRepositoryConnection;
import org.eaglei.repository.rid.RIDGenerator;
import org.eaglei.repository.vocabulary.REPO;
import org.eaglei.repository.vocabulary.FOAF;
import org.eaglei.repository.util.Utils;
import org.eaglei.repository.util.SPARQL;
import org.eaglei.repository.status.ForbiddenException;
import org.eaglei.repository.status.BadRequestException;
import org.eaglei.repository.status.InternalServerErrorException;

/**
 * User object model, reflects the user's properties in the both RDBMS
 * and RDF database.  Also manages the RDF descriptions of users.
 *
 * Named Graph Usage:
 *  1. Almost all statements about a repo User are in the graph named "repo:NG_Users".
 *  2. Ony the :hasPrincipalName statement is on the NG_Internal graph
 *     sicne it should not be exposed; there is no reason to let login
 *     names leak and it's good security practice to keep them hidden,
 *     since that just makes dictionary attacks so much easier.
 *
 * @see AuthUser, ImportExport
 *
 * @author Larry Stone
 * Started April 26, 2010
 * @version $Id: $
 */
public class User extends WritableObjectModel
{
    private static final Logger log = LogManager.getLogger(User.class);

    /** Named graph where we create User objects.  Note taht the
     *  hasPrincipalName is still on the Internal graph, it's confidential.
     */
    public static final URI USER_GRAPH = REPO.NG_USERS;

    /** dataset that includes relevant graphs - internal and user */
    public static final DatasetImpl USER_DATASET = SPARQL.copyDataset(SPARQL.InternalGraphs);
    static {
        SPARQL.addGraph(USER_DATASET, USER_GRAPH);
        if (log.isDebugEnabled())
            log.debug("User Dataset = "+Utils.prettyPrint(USER_DATASET));
    }

    // hardwired source of AuthUsers..
    private static final AuthUserFactory authFactory = AuthUserTomcatFactory.getInstance();

    /** instance variables of the User object */
    private URI uri;  /* immutable (usually), the subject of this instance */
    private String username; /* immutable (usually), "user" (i.e. Principal) of this instance */
    private String firstName = null;
    private String lastName = null;
    private String mbox = null;
    private Set<URI> roles = new HashSet<URI>();
    private AuthUser auth = null;
    private Boolean hasAuthUser = null;

    // Find users query - get all or a specific one by binding 'uri'
    // note that we NEED the "order by" because the handler expects
    // all the hasRole bindings for a given URI to be clumped together.
    private static final String userForURIQuery =
          "SELECT * WHERE { ?uri a <"+REPO.PERSON+"> ; \n"+
          "<"+REPO.HAS_PRINCIPAL_NAME+"> ?hasPrincipalName .\n"+
          "  OPTIONAL { ?uri <"+FOAF.FIRST_NAME+"> ?firstName }\n"+
          "  OPTIONAL { ?uri <"+FOAF.SURNAME+"> ?surname }\n"+
          "  OPTIONAL { ?uri <"+FOAF.MBOX+"> ?mbox }\n"+
          "  OPTIONAL { ?uri <"+REPO.HAS_ROLE+"> ?hasRole . ?hasRole <"+RDFS.LABEL+"> ?roleLabel }\n"+
          " } ORDER BY ?hasPrincipalName";

    /**
     * Constructor, for principal.  Also initialize default implicit
     * roles in case they aren't in the RDF (or there IS no RDF).
     */
    private User(URI uri, String username)
    {
        super();
        if (username == null)
            throw new IllegalArgumentException("Cannot create User without username");
        this.uri = uri;
        this.username = username;
        roles.addAll(REPO.IMPLICIT_ROLES);
    }

    /**
     * find a record by username, i.e. login principal name
     * Returns null if NOT found.
     *
     * @param request a {@link javax.servlet.http.HttpServletRequest} object.
     * @param pname principal name (from user account RDBMS).
     * @return the User object, or null if none found.
     * @throws javax.servlet.ServletException if any.
     */
    public static User findByUsername(HttpServletRequest request, String pname)
        throws ServletException
    {
        return findOneUserInternal(request, pname, null);
    }

    /**
     * Find a user record by URI;  Returns null if NOT found..
     *
     * @param request a {@link javax.servlet.http.HttpServletRequest} object.
     * @param uri the URI of a person object
     * @return the User object, or null if none found.
     * @throws javax.servlet.ServletException if any.
     */
    public static User find(HttpServletRequest request, URI uri)
        throws ServletException
    {
        return findOneUserInternal(request, null, uri);
    }

    /**
     * Internal find() implementation.
     * @param request a {@link javax.servlet.http.HttpServletRequest} object.
     * @param pname principal name (from user account RDBMS).
     * @param uri the URI of a person object
     * @return the User object, or null if none found.
     */
    private static User findOneUserInternal(HttpServletRequest request, String pname, URI uri)
        throws ServletException
    {
        RepositoryConnection rc = WithRepositoryConnection.get(request);
        try {
            log.debug("Single user SPARQL query = "+userForURIQuery);
            TupleQuery q = rc.prepareTupleQuery(QueryLanguage.SPARQL, userForURIQuery);
            q.setDataset(USER_DATASET);
            q.clearBindings(); // needed if re-using query
            if (pname != null)
                q.setBinding("hasPrincipalName", new LiteralImpl(pname));
            else if (uri != null)
                q.setBinding("uri", uri);
            else
                throw new ServletException("sanity check: findOneUserInternal called without either username OR uri");

            // do NOT infer any roles.
            q.setIncludeInferred(false);
            UserHandler h = new UserHandler(request, null);
            q.evaluate(h);
            User result = h.lastUser;
            if (log.isDebugEnabled())
                log.debug("User.findOneUserInternal(pname=\""+pname+"\", uri="+uri+"), SPARQL result => "+result);

            // check whether Superuser role is expressed in AuthUser
            // if user isn't in RDF, create one with null URI just to
            // query for authUser..
            if (pname != null) {
                User temp = result;
                if (temp == null) {
                    temp = new User(null, pname);
                }
                AuthUser au = temp.getAuthUser();
                if (au != null) {
                    if (au.isSuperuser())
                        temp.roles.add(REPO.ROLE_SUPERUSER);
                    return temp;
                }
            }
            return result;
        } catch (MalformedQueryException e) {
            log.error("Rejecting malformed query:"+e);
            throw new ServletException(e);
        } catch (OpenRDFException e) {
            log.error(e);
            throw new ServletException(e);
        }
    }

    /**
     * <p>Get all known Users from RDF DB - may not match auth'n DB.
     * <br>XXX TODO should add start and count for pagination
     * </p>
     *
     * @param request a {@link javax.servlet.http.HttpServletRequest} object.
     * @return resulting Users in a {@link java.lang.Iterable} object.
     * @throws javax.servlet.ServletException if any.
     */
    public static List<User> findAll(HttpServletRequest request)
        throws ServletException
    {
        RepositoryConnection rc = WithRepositoryConnection.get(request);
        List<User> result = new ArrayList<User>();

        try {
            if (log.isDebugEnabled())
                log.debug("All user SPARQL query = "+userForURIQuery);
            TupleQuery q = rc.prepareTupleQuery(QueryLanguage.SPARQL, userForURIQuery);
            q.setDataset(USER_DATASET);
            // do NOT infer any roles.
            q.setIncludeInferred(false);
            q.evaluate(new UserHandler(request, result));

            // make a pass over authUsers for 2 reasons:
            // 1. fixup superuser role if it's in authUser but not RDF
            // 2. check off authUsers found in RDF to isolate undocumented ones
            Map<String,? extends AuthUser> authUsers = authFactory.findAllAsMap();
            for (User ru : result) {
                log.debug("findAll: Found documented user: "+ru.username);
                AuthUser au = authUsers.get(ru.username);
                if (au != null) {
                    if (au.isSuperuser())
                        ru.roles.add(REPO.ROLE_SUPERUSER);
                    authUsers.remove(ru.username);
                }
            }

            // add any undocumented AuthUsers not found in the RDF..
            for (AuthUser au : authUsers.values()) {
                log.debug("findAll: Found undocumented user: "+au.getUsername());
                User ru = new User(null, au.getUsername());
                if (au.isSuperuser())
                    ru.roles.add(REPO.ROLE_SUPERUSER);
                result.add(ru);
            }
        } catch (MalformedQueryException e) {
            log.error("Rejecting malformed query:"+e);
            throw new ServletException(e);
        } catch (OpenRDFException e) {
            log.error(e);
            throw new ServletException(e);
        }
        return result;
    }

    /**
     * <p>
     * Create new user instance for given username.
     * Returns User object with URI and username set; nothing else.
     * Fails if there is an existing user with the same principal.
     * Requires Administrator role.
     *  </p>
     *
     * @param request a {@link javax.servlet.http.HttpServletRequest} object.
     * @param username principal name, a {@link java.lang.String} object.
     * @param password the password or null if not login-able
     * @return the new {@link org.eaglei.repository.User} object.
     * @throws javax.servlet.ServletException if any.
     */
    public static User create(HttpServletRequest request, String username, String password)
        throws ServletException
    {
        return createInternal(request, newURI(), username, password, false);
    }

    /**
     * <p>
     * Create new user instance for given username.
     * Returns User object with URI and username set; nothing else.
     * Fails if there is an existing user with the same principal.
     * Skip access test -- THIS IS VERY DANGEROUS, only for
     *  bootstrapping the authentication.
     *  </p>
     *
     * @param request a {@link javax.servlet.http.HttpServletRequest} object.
     * @param username principal name, a {@link java.lang.String} object.
     * @param password the password or null if not login-able
     * @return the new {@link org.eaglei.repository.User} object.
     * @throws javax.servlet.ServletException if any.
     */
    public static User createAsAdministrator(HttpServletRequest request, String username, String password)
        throws ServletException
    {
        return createInternal(request, newURI(), username, password, true);
    }

    /**
     * <p>
     * Create new user instance for given username AND URI.
     * Returns User object with URI and username set; nothing else.
     * Fails if there is an existing user with the same principal.
     * Requires Administrator role.
     *  </p>
     *
     * @param request a {@link javax.servlet.http.HttpServletRequest} object.
     * @param username principal name, a {@link java.lang.String} object.
     * @param password the password or null if not login-able
     * @return the new {@link org.eaglei.repository.User} object.
     * @throws javax.servlet.ServletException if any.
     */
    public static User create(HttpServletRequest request, URI uri, String username, String password)
        throws ServletException
    {
        return createInternal(request, uri, username, password, false);
    }

    /**
     * <p>
     * Create new user instance for given username.
     * Returns User object with URI and username set; nothing else.
     * Fails if there is an existing user with the same principal.
     * Skip access test if skipAccessCheck bit is set -- dangerous, only for
     *  bootstrapping the authentication.
     *  </p>
     *
     * @param request a {@link javax.servlet.http.HttpServletRequest} object.
     * @param uri the subject URI for the user's RDF entry, may be null for an authentication-only user
     * @param username principal name, required, even if there is no authenticaiton record
     * @param password the password or null if not login-able
     * @param skipAccessCheck a boolean.
     * @return the new {@link org.eaglei.repository.User} object.
     * @throws javax.servlet.ServletException if any.
     */
    private static User createInternal(HttpServletRequest request, URI uri, String username, String password, boolean skipAccessCheck)
        throws ServletException
    {
        if (!skipAccessCheck && !Access.hasPermissionOnUser(request, username))
            throw new ForbiddenException("Not allowed to create user: "+username);

        RepositoryConnection rc = WithRepositoryConnection.get(request);
        ValueFactory vf = rc.getValueFactory();
        User result = null;

        try {
            // sanity check, ensure there is no user with this principal already
            Literal lpname = vf.createLiteral(username);
            if (rc.hasStatement(null, REPO.HAS_PRINCIPAL_NAME, lpname, false, REPO.NG_INTERNAL)) {
                throw new BadRequestException("Cannot create user: there is already a repository user with the login principal name (username) \""+username+"\"");
            }
            result = new User(uri, username);
            if (uri != null) {
                result.initialStatements(rc);
            }
            if (password != null)
                result.createAuthUser(password);
            if (log.isDebugEnabled())
                log.debug("create: created new User instance, username="+username+", uri="+result.uri);
            return result;
        } catch (RepositoryException e) {
            log.error("Failed creating user URI: ",e);
            throw new ServletException("Failed creating user URI: ",e);
        }
    }

    /**
     * <p>Getter for the field <code>uri</code>.</p>
     *
     * @return a {@link org.openrdf.model.URI} object.
     */
    @Override
    public URI getURI()
    {
        return uri;
    }
    /**
     * <p>Getter for the field <code>username</code>.</p>
     *
     * @return a {@link java.lang.String} object.
     */
    public String getUsername()
    {
        return username;
    }
    /**
     * <p>Getter for the field <code>firstName</code>.</p>
     *
     * @return a {@link java.lang.String} object or null if not set.
     */
    public String getFirstName()
    {
        return firstName;
    }
    /**
     * <p>Getter for the field <code>lastName</code>.</p>
     *
     * @return a {@link java.lang.String} object  or null if not set.
     */
    public String getLastName()
    {
        return lastName;
    }
    /**
     * <p>Getter for the field <code>mbox</code>.</p>
     *
     * @return a {@link java.lang.String} object  or null if not set.
     */
    public String getMbox()
    {
        return mbox;
    }

    /**
     * Computes the most informative "label" for a user, to present in UI.
     * Format is, ideally,   "username (firstname lastname)"
     * But it reverts to bare username or even URI if none is available.
     *
     * @return label as a {@link java.lang.String} object.
     */
    @Override
    public String getLabel()
    {
        if (username != null) {
            String pn = getPersonalName();
            return pn.length() > 0 ? username+" ("+pn+")" : username;
        } else
            return uri == null ? "(no username or uri)" : uri.getLocalName();
    }

    /**
     * <p>Get personal name, if available, as "firstname lastname". </p>
     *
     * @return name in String or else empty string if there is none; never null.
     */
    public String getPersonalName()
    {
        return (firstName == null) ?
                   ((lastName == null) ? "" : lastName) :
                   ((lastName == null) ? firstName : firstName+" "+lastName);
    }

    /**
     * <p>Get <code>Roles</code>.</p>
     *
     * @return all known Roles as an array of {@link org.eaglei.repository.Role} objects.
     */
    private Role[] getRoles(HttpServletRequest request)
        throws ServletException
    {
        Role result[] = new Role[roles.size()];
        int i = 0;
        for (URI ru : roles) {
            result[i] = Role.find(request, ru);
            ++i;
        }
        return result;
    }

    /**
     * has role predicate - the P suffix is a Lisp thing
     *
     * @param r role as a {@link org.eaglei.repository.Role} object.
     * @return a boolean, true if this User has indicated role.
     */
    public boolean hasRoleP(Role r)
    {
        return roles.contains(r.getURI());
    }

    /**
     * <p>hasRoleP</p>
     *
     * @param ru role as a {@link org.openrdf.model.URI} object.
     * @return a boolean, true if this User has indicated role.
     */
    public boolean hasRoleP(URI ru)
    {
        return roles.contains(ru);
    }

    /**
     * <p>isSuperuser</p> - convenience method to test for superuser (administrator) role
     *
     * @return a boolean, true if this User has Superuser (Admin) role.
     */
    public boolean isSuperuser()
    {
        return hasRoleP(REPO.ROLE_SUPERUSER);
    }

    /**
     * <p>setIsSuperuser</p> - change local value of Superuser bit WITHOUT
     *  updating the RDF description, ONLY for cached local User object.
     *
     * @param su a boolean, true if this User has Superuser (Admin) role.
     */
    public void setIsSuperuser(boolean su)
    {
        if (su != isSuperuser()) {
            if (su)
                roles.add(REPO.ROLE_SUPERUSER);
            else
                roles.remove(REPO.ROLE_SUPERUSER);
        }
    }

    /**
     * Change value of first name. Setting it to null clears it.
     *
     * @param request a {@link javax.servlet.http.HttpServletRequest} object.
     * @param name first name a {@link java.lang.String} object.
     * @throws javax.servlet.ServletException if any.
     */
    public void setFirstName(HttpServletRequest request, String name)
        throws ServletException
    {
        firstName = name;
        setMetadataInternal(request, FOAF.FIRST_NAME, name);
    }

    /**
     * <p>Setter for the field <code>lastName</code>.</p>
     *
     * @param request a {@link javax.servlet.http.HttpServletRequest} object.
     * @param name a {@link java.lang.String} object.
     * @throws javax.servlet.ServletException if any.
     */
    public void setLastName(HttpServletRequest request, String name)
        throws ServletException
    {
        lastName = name;
        setMetadataInternal(request, FOAF.SURNAME, name);
    }

    /**
     * <p>Setter for the field <code>mbox</code>.</p>
     *
     * @param request a {@link javax.servlet.http.HttpServletRequest} object.
     * @param mbox a {@link java.lang.String} object.
     * @throws javax.servlet.ServletException if any.
     */
    public void setMbox(HttpServletRequest request, String mbox)
        throws ServletException
    {
        this.mbox = mbox;
        setMetadataInternal(request, FOAF.MBOX, mbox);
    }

    /**
     * Set a random property on the user (for RDF import).
     * @return true if it was a valid property, false if ignored.
     */
    public boolean setProperty(HttpServletRequest request, URI property, String value)
        throws ServletException
    {
        if (FOAF.MBOX.equals(property)) {
            setMbox(request, value);
        } else if (FOAF.FIRST_NAME.equals(property)) {
            setFirstName(request, value);
        } else if (FOAF.SURNAME.equals(property)) {
            setLastName(request, value);
        } else {
            return false;
        }
        return true;
    }

    // ensure there is a valid URI - e.g. if this was created
    // only in auth'n DB, there won't be one, so we create a new URI.
    private void ensureURI(RepositoryConnection rc)
        throws RepositoryException
    {
        if (uri == null) {
            uri = newURI();
            log.debug("Created new URI for User username="+username+", uri="+uri);
            initialStatements(rc);
        }
    }

    // create the initial RDF statements, assumes uri and username are set.
    // this is common code shared by initial create, recreate, etc.
    private void initialStatements(RepositoryConnection rc)
        throws RepositoryException
    {
        Literal luname = rc.getValueFactory().createLiteral(username);
        rc.add(uri, RDF.TYPE, REPO.PERSON, USER_GRAPH);
        rc.add(uri, RDFS.LABEL, luname, USER_GRAPH);
        rc.add(uri, REPO.HAS_PRINCIPAL_NAME, luname, REPO.NG_INTERNAL);

        // KLUDGE: Assert the "implicit" roles..
        for (URI ruri : REPO.IMPLICIT_ROLES) {
            rc.add(uri, REPO.HAS_ROLE, ruri, REPO.NG_INTERNAL);
        }

        // if authentication thinks we're superuser, reflect that role
        AuthUser au = getAuthUser();
        if (au != null && au.isSuperuser()) {
            rc.add(uri, REPO.HAS_ROLE, REPO.ROLE_SUPERUSER, REPO.NG_INTERNAL);
        }
        setDirty(true);
    }

    // update one of the metadata properties.
    private void setMetadataInternal(HttpServletRequest request, URI property, String newVal)
        throws ServletException
    {
        if (!Access.hasPermissionOnUser(request, username))
            throw new ForbiddenException("Not allowed to modify user: "+username);
        try {
            RepositoryConnection rc = WithRepositoryConnection.get(request);
            ensureURI(rc);
            boolean hasProp = rc.hasStatement(uri, property, null, false, USER_GRAPH);
            boolean hasLabel = rc.hasStatement(uri, RDFS.LABEL, null, false, USER_GRAPH);
            ValueFactory vf = rc.getValueFactory();
            if (hasProp)
                rc.remove(uri, property, null, USER_GRAPH);
            if (newVal != null)
                rc.add(uri, property, vf.createLiteral(newVal), USER_GRAPH);
            // recompute label
            String label = getLabel();
            if (log.isDebugEnabled())
                log.debug("Setting User label, uri="+uri+", label="+label);
            if (hasLabel)
                rc.remove(uri, RDFS.LABEL, null, USER_GRAPH);
            rc.add(uri, RDFS.LABEL, vf.createLiteral(label), USER_GRAPH);
            setDirty(true);
        } catch (RepositoryException e) {
            throw new ServletException(e);
        }
    }

    /**
     * Create a new user graph to replace the existing user, for import.
     * Clear all properties and roles, add back username, and credentials
     * if given.
     */
    public void recreate(HttpServletRequest request, URI newURI, String newUsername, String pw)
        throws ServletException
    {
        if (!Access.hasPermissionOnUser(request, username))
            throw new ForbiddenException("Not allowed to modify user: "+username);
        try {
            RepositoryConnection rc = WithRepositoryConnection.get(request);
            if (uri != null) {
                rc.remove(uri, null, null, USER_GRAPH);
                rc.remove(uri, null, null, REPO.NG_INTERNAL);
            }
            if (log.isDebugEnabled())
                log.debug("recreate: oldURI="+uri+", newURI="+newURI+", username="+username);
            uri = newURI;
            username = newUsername;
            roles.clear();
            roles.addAll(REPO.IMPLICIT_ROLES);
            firstName = null;
            lastName = null;
            mbox = null;
            if (uri != null) {
                initialStatements(rc);
            }
            if (pw != null) {
                if (isLoginable()) {
                    setPassword(request, pw);
                } else {
                    reinstate(pw);
                }
            }
            setDirty(true);
        } catch (RepositoryException e) {
            throw new ServletException(e);
        }
    }

    /**
     * <p>addRole - add a role</p>
     *
     * @param request a {@link javax.servlet.http.HttpServletRequest} object.
     * @param ru Role to add, as a {@link org.openrdf.model.URI} object.
     * @throws javax.servlet.ServletException if any.
     */
    public void addRole(HttpServletRequest request, URI ru)
        throws ServletException
    {
        addRole(request, Role.find(request, ru));
    }

    /**
     * <p>addRole - add a role</p>
     *
     * @param request a {@link javax.servlet.http.HttpServletRequest} object.
     * @param r role to add as a {@link org.eaglei.repository.Role} object.
     * @throws javax.servlet.ServletException if any.
     */
    public void addRole(HttpServletRequest request, Role r)
        throws ServletException
    {
        addRoleInternal(request, r, false);
    }

    /**
     * <p>addRoleAsAdministrator - add a role WITHOUT access control checks</p>
     *
     * @param request a {@link javax.servlet.http.HttpServletRequest} object.
     * @param r role to add as a {@link org.eaglei.repository.Role} object.
     * @throws javax.servlet.ServletException if any.
     */
    public void addRoleAsAdministrator(HttpServletRequest request, Role r)
        throws ServletException
    {
        addRoleInternal(request, r, true);
    }

    private void addRoleInternal(HttpServletRequest request, Role r, boolean skipAccessCheck )
        throws ServletException
    {
        if (!skipAccessCheck && !Authentication.isSuperuser(request))
            throw new ForbiddenException("Only the administrator is allowed to modify user roles.");

        if (hasRoleP(r)) {
            if (log.isDebugEnabled())
                log.debug("addRole: Already have role="+r+", skipping.");
        } else {
            RepositoryConnection rc = WithRepositoryConnection.get(request);
            try {
                if (roles.add(r.getURI())) {
                    // only create a URI if necessary for this role
                    if (!r.isSuperuser())
                        ensureURI(rc);
                    if (uri != null) {
                        rc.add(uri, REPO.HAS_ROLE, r.getURI(), REPO.NG_INTERNAL);
                        if (log.isDebugEnabled())
                            log.debug("Added hasRole statement: user="+this+" => role="+r);
                    }
                    if (r.isSuperuser()) {
                        AuthUser au = getAuthUser();
                        if (au != null)
                            au.setIsSuperuser(true);
                    }
                    setDirty(true);
                    log.debug("Added role: "+r);
                } else {
                    if (log.isDebugEnabled())
                        log.debug("Role "+r+" was already in Set.");
                }
            } catch (RepositoryException e) {
                log.error("Failed adding role ",e);
                throw new ServletException("Failed adding role ",e);
            }
        }
    }

    /**
     * <p>removeRole</p>
     *
     * @param request a {@link javax.servlet.http.HttpServletRequest} object.
     * @param ru role to remove as a {@link org.openrdf.model.URI} object.
     * @throws javax.servlet.ServletException if any.
     */
    public void removeRole(HttpServletRequest request, URI ru)
        throws ServletException
    {
        removeRole(request, Role.find(request, ru));
    }

    /**
     * <p>removeRole</p>
     *
     * @param request a {@link javax.servlet.http.HttpServletRequest} object.
     * @param r role to remove as a {@link org.eaglei.repository.Role} object.
     * @throws javax.servlet.ServletException if any.
     */
    public void removeRole(HttpServletRequest request, Role r)
        throws ServletException
    {
        if (!Authentication.isSuperuser(request))
            throw new ForbiddenException("Only the administrator is allowed to modify user roles.");
        if (!hasRoleP(r))
            return;
        RepositoryConnection rc = WithRepositoryConnection.get(request);
        try {
            // only create a URI if necessary for this role
            if (!r.isSuperuser())
                ensureURI(rc);
            if (uri != null)
                rc.remove(uri, REPO.HAS_ROLE, r.getURI());
            roles.remove(r.getURI());
            if (r.isSuperuser()) {
                AuthUser au = getAuthUser();
                if (au != null)
                    au.setIsSuperuser(false);
            }
            setDirty(true);
            log.debug("Removed role: "+r);
        } catch (RepositoryException e) {
            log.error("Failed adding role ",e);
            throw new ServletException("Failed removing role ",e);
        }
    }

    /**
     * <p>setRoles</p>  make the user's set of roles equal to contents of this array
     *
     * @param request a {@link javax.servlet.http.HttpServletRequest} object.
     * @param role array of Role objects which should be the new role set
     * @return true if any roles were actually changed.
     * @throws javax.servlet.ServletException if any.
     */
    public boolean setRoles(HttpServletRequest request, Role newRoles[])
        throws ServletException
    {
        List<Role> addRoles = new ArrayList<Role>();
        Set<Role> deleteRoles = new HashSet<Role>();
        for (Role r : getRoles(request)) {
            if (!r.isImplicit())
                deleteRoles.add(r);
        }

        // remove matches from existing set so any remaining ones must be deleted
        for (Role r : newRoles) {
            if (deleteRoles.contains(r))
                deleteRoles.remove(r);
            else
                addRoles.add(r);
        }
        if (deleteRoles.isEmpty() && addRoles.isEmpty())
            return false;
        for (Role r : deleteRoles)
            removeRole(request, r);
        for (Role r : addRoles)
            addRole(request, r);
        return true;
    }

    /**
     * <p>
     * Commits any changes to RDF database *and* RDBMS.
     * Also recomputes rdfs:label, passes superuser bit into RDBMS.
     * Does *not* need to check access again here because the only
     * operatiosn taht set "dirty" are already access-protected..
     * </p>
     *
     * @param request a {@link javax.servlet.http.HttpServletRequest} object.
     * @throws javax.servlet.ServletException if any.
     */
    @Override
    public void commit(HttpServletRequest request)
        throws ServletException
    {
        RepositoryConnection rc = WithRepositoryConnection.get(request);
        if (dirty) {
            if (auth != null)
                auth.commit();
            super.commit(request);
            Authentication.decacheAuthentication(request, this);
        }
        setDirty(false);
    }

    /**
     * Runs commit() on a batch of users at once for
     * greater efficiency in imports.
     */
    public static void commitMultiple(HttpServletRequest request, Collection<User> users)
        throws ServletException
    {
        try {
            List<AuthUser> aus = new ArrayList<AuthUser>();
            for (User u : users) {
                if (u.auth != null)
                    aus.add(u.auth);
            }
            authFactory.commitMultiple(request, aus);
            WithRepositoryConnection.get(request).commit();
            if (log.isDebugEnabled())
                log.debug("commitMultiple: committed "+String.valueOf(users.size())+" Users.");
            for (User u : users) {
                u.setDirty(false);
                if (log.isDebugEnabled())
                    log.debug("commitMultiple: clear dirty on "+u);
            }
        } catch (RepositoryException e) {
            log.error("Failed in multiple commit: ",e);
            throw new ServletException("Failed in multiple commit ",e);
        }
    }

    /**
     * <p>finalize - flag error if GC'ing a dirty instance, its changes are lost. </p>
     */
    @Override
    protected void finalize()
        throws Throwable
    {
        if (dirty)
            log.error("finalize: about to destroy a User with dirty flag set, CHANGES WILL BE LOST.  Current state: "+toString());
        super.finalize();
    }

    /**
     * <p>toString</p>
     *
     * @return a {@link java.lang.String} object.
     */
    @Override
    public String toString()
    {
        String rs = roles == null ? "{null}" :
            Arrays.deepToString(roles.toArray(new URI[roles.size()]));
        String uu = uri == null ? "(not set)" : uri.toString();

        return "<#User: uri="+uu+
            ", username="+username+", firstName="+firstName+
            ", lastName="+lastName+", mbox="+mbox+", roles="+rs+">";
    }

    /** {@inheritDoc}
     * Equality is defined as EITHER URI or uesrname equal -since some
     * instances may not have a URI at first but always(?) have a username
     */
    @Override
    public boolean equals(Object o)
    {
        return o instanceof User &&
            ((uri != null && uri.equals(((User)o).uri)) ||
             (username != null&& username.equals(((User)o).username)));
    }

    /** {@inheritDoc} */
    @Override
    public int hashCode()
    {
        return uri.hashCode();
    }

    // tuple query result handler to gather user contents
    private static class UserHandler extends TupleQueryResultHandlerBase
    {
        private List<User> result = null;
        private User lastUser = null;
        protected HttpServletRequest request = null;

        UserHandler(HttpServletRequest request, List<User> lu)
        {
            super();
            UserHandler.this.result = lu;
            UserHandler.this.request = request;
        }

        /** {@inheritDoc} */
        @Override
        public void endQueryResult()
        {
            finishCurrent();
            if (lastUser == null)
                log.debug("Failed to get any query results, ");
        }

        /** {@inheritDoc} */
        @Override
        public void handleSolution(BindingSet bs)
            throws TupleQueryResultHandlerException
        {
            Value curURI = bs.getValue("uri");
            if (curURI == null || !(curURI instanceof URI))
                log.error("Should not get null or non-URI result in UserHandler: "+curURI);

            // if there is no current user, or this result is for a different
            // user, start a new User
            else if (lastUser == null || !curURI.equals(lastUser.getURI())) {
                finishCurrent();
                lastUser = populateUser(request, bs, null);

            // this must be more results (e.g. another Role) for current user
            } else {
                populateUser(request, bs, lastUser);
            }
        }

        /**
         * read the result record into a User - shared with "all" processor
         * if u == null, create a new one
         */
        protected User populateUser(HttpServletRequest request, BindingSet bs, User u)
            throws TupleQueryResultHandlerException
        {
            User ret = u;
            if (ret == null) {
                Value puri = bs.getValue("uri");
                if (puri == null || !(puri instanceof URI))
                    throw new TupleQueryResultHandlerException("Missing valid result for required column 'uri'");
                Value pn = bs.getValue("hasPrincipalName");
                if (pn == null)
                    throw new TupleQueryResultHandlerException("Missing valid result for required column 'hasPrincipalName'");
                ret = new User((URI)puri, Utils.valueAsString(pn));

                Value firstName =  bs.getValue("firstName");
                if (firstName != null && firstName instanceof Literal)
                    ret.firstName = ((Literal)firstName).getLabel();
                Value sur = bs.getValue("surname");
                if (sur != null && sur instanceof Literal)
                    ret.lastName = ((Literal)sur).getLabel();
                Value mbox = bs.getValue("mbox");
                if (mbox != null && mbox instanceof Literal)
                    ret.mbox = ((Literal)mbox).getLabel();
            }
            Value role = bs.getValue("hasRole");
            // Value roleLabel = bs.getValue("roleLabel");
            if (role != null) {
                if (log.isDebugEnabled())
                    log.debug("Adding role to single User, username="+ret.username+", role="+role.stringValue());
                ret.roles.add((URI)role);
            }
            return ret;
        }

        // if there's a User in progress, put it on the list.
        private void finishCurrent()
        {
            if (lastUser != null && result != null) {
                result.add(lastUser);
            }
        }
    }

    private AuthUser getAuthUser()
    {
        // haven't tried yet
        if (hasAuthUser == null) {
            try {
                auth = authFactory.find(username);
            } catch (ServletException e) {
                log.error("failed getting authUser: ",e);
                throw new InternalServerErrorException(e);
            }
            hasAuthUser = auth == null ? Boolean.FALSE : Boolean.TRUE;
        }
        return auth;
    }

    // create an AuthUser and attach it, set status
    private AuthUser createAuthUser(String pw)
    {
        auth = authFactory.create(username);
        auth.setPassword(pw);
        auth.setIsSuperuser(isSuperuser());
        hasAuthUser = Boolean.TRUE;
        return auth;
    }

    /**
     * does this user have credentials to be able to login.
     * @return true if user can login (i.e. is not disabled)
     */
    public boolean isLoginable()
    {
        if (hasAuthUser == null)
            getAuthUser();
        return hasAuthUser.booleanValue();
    }

    /**
     * compare password (or credientals) without revealing anythign
     * @return true if given credential would let user login
     */
    public boolean authenticate(String pw)
    {
        AuthUser au = getAuthUser();
        return au != null && au.authenticate(pw);
    }

    /**
     * Getter for password.
     */
    public String getPassword()
    {
        AuthUser au = getAuthUser();
        return au == null ? null : au.getPassword();
    }

    /**
     * Ignores request but arg is there for uniformity.
     */
    public void setPassword(HttpServletRequest request, String pw)
    {
        AuthUser au = getAuthUser();
        if (au == null) {
            log.warn("Cannot set password on user without authUser: "+this);
        } else {
            au.setPassword(pw);
            setDirty(true);
        }
    }

    /**
     * Remove ability to login with credentials
     */
    public void disable()
    {
        AuthUser au = getAuthUser();
        if (au == null) {
            throw new IllegalStateException("Cannot disable a user who is already disabled.");
        } else {
            au.delete();
            setDirty(true);
        }
    }

    /**
     * Restore ability to login with credentials
     */
    public void reinstate(String pw)
    {
        if (pw == null)
            throw new BadRequestException("Password is required to reinstate a user.");
        AuthUser au = getAuthUser();
        if (au == null) {
            au = createAuthUser(pw);
            setDirty(true);
        } else {
            throw new IllegalStateException("Cannot reinstate a user who has not been disabled.");
        }
    }

    /**
     * Get the URI identifying the kind of authorization behind this
     * user, needed for export and import.
     * XXX waiting for a pluggable auth interface
     */
    public URI getAuthType()
    {
        return authFactory.getType();
    }

    /**
     * Ignores request but arg is there for uniformity.
     * XXX Not really implemented, just a sanity check for now.
     * XXX waiting for a pluggable auth interface
     */
    public void setAuthType(HttpServletRequest request, URI uri)
    {
        if (!authFactory.getType().equals(uri))
            throw new IllegalArgumentException("Cannot accept any auth type but the built-in one, this is unacceptable: "+ uri);
    }

    // return a newly minted URI in the local default namespace
    private static URI newURI()
    {
        return new URIImpl(Configuration.getInstance().getDefaultNamespace()+
                           RIDGenerator.getInstance().newID().toString());
    }
}
