package org.eaglei.repository.model.workflow;

import java.util.List;
import java.util.Set;
import java.util.HashSet;
import javax.servlet.http.HttpServletRequest;

import org.apache.log4j.Logger;
import org.apache.log4j.LogManager;

import org.openrdf.model.URI;
import org.openrdf.model.Resource;
import org.openrdf.model.Statement;
import org.openrdf.model.Value;
import org.openrdf.query.impl.DatasetImpl;
import org.openrdf.query.GraphQuery;
import org.openrdf.query.GraphQueryResult;
import org.openrdf.query.QueryLanguage;
import org.openrdf.repository.RepositoryConnection;
import org.openrdf.model.vocabulary.RDF;
import org.openrdf.repository.RepositoryResult;

import org.eaglei.repository.model.DataModel;
import org.eaglei.repository.model.View;
import org.eaglei.repository.servlet.WithRepositoryConnection;
import org.eaglei.repository.status.BadRequestException;
import org.eaglei.repository.status.InternalServerErrorException;
import org.eaglei.repository.util.Utils;

public class ActionMoveGraph implements WorkflowAction
{
    private static Logger log = LogManager.getLogger(ActionMoveGraph.class);

    // Query to find all instance statements, including embedded instances
    // Bind ?subject, ?homeGraph  first
    private static final String statementsQuery =
        "CONSTRUCT { ?subject ?pred ?obj . ?obj ?eip ?eio }  WHERE { \n"+
        " GRAPH ?homeGraph { ?subject ?pred ?obj } \n"+
        " OPTIONAL { ?obj a ?eit . \n"+
        "   ?eit <"+DataModel.EMBEDDED_INSTANCE_PREDICATE.toString()+"> <"+
                    DataModel.EMBEDDED_INSTANCE_OBJECT.toString()+"> . \n"+
        "   GRAPH ?homeGraph { ?obj ?eip ?eio }}}\n";

    // move resource's properties to new named graph
    // (i.e. to implement different access control and query response)
    public void onTransition(HttpServletRequest request, URI resource, Value param)
        throws Exception
    {
        if (param == null || !(param instanceof URI))
            throw new BadRequestException("ActionMoveGraph requires a valid URI parameter, naming the destination graph.");
        URI newGraph = (URI)param;
        URI homeGraph = null;
        RepositoryConnection rc = WithRepositoryConnection.get(request);

        // 1. get resource's home graph, ideally where its *asserted*
        //    rdf:type statements live - avoid inferred types.
        homeGraph = Utils.getHomeGraph(rc, resource);
        if (homeGraph == null) {
            log.error("Failed to find home graph for resource instance="+resource);
            throw new InternalServerErrorException("Failed to find home graph for resource instance="+resource);
        } else if (homeGraph.equals(newGraph)) {
            log.warn("Tried to move resource inst="+resource+" to the same named graph: "+homeGraph);
            return;
        }

        // 2. Move all statements to new graph.
        // Save a Set of unique subjects to delete in case
        // removing them screws up the query result iterator.
        Set<URI> subjects = new HashSet<URI>();
        DatasetImpl qds = new DatasetImpl();
        View.addWorkspaceGraphs(request, qds, homeGraph);
        GraphQuery sq = rc.prepareGraphQuery(QueryLanguage.SPARQL, statementsQuery);
        sq.setIncludeInferred(true);
        sq.setDataset(qds);
        sq.setBinding("subject", resource);
        sq.setBinding("homeGraph", homeGraph);
        GraphQueryResult sqr = sq.evaluate();
        if (log.isDebugEnabled()) {
            log.debug("SPARQL Query to get statements, ?subject="+resource+", ?homeGraph="+homeGraph+"\n"+statementsQuery);
            log.debug("Moving statements from "+homeGraph+" to "+newGraph);
        }
        try {
            while (sqr.hasNext()) {
                Statement s = sqr.next();
                subjects.add((URI)s.getSubject());
                rc.add(s, newGraph);
                if (log.isDebugEnabled())
                    log.debug("Moving statement: { "+s.getSubject()+" "+s.getPredicate()+" "+s.getObject()+" }");
            }
        } finally {
            sqr.close();
        }
        for (URI subject : subjects) {
            rc.remove(subject, null, null, homeGraph);
        }
    }
}
