package org.eaglei.repository.model.workflow;

import java.util.Collection;
import java.util.Date;

import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;

import org.apache.log4j.Logger;
import org.apache.log4j.LogManager;

import org.openrdf.OpenRDFException;
import org.openrdf.model.URI;
import org.openrdf.model.Value;
import org.openrdf.repository.RepositoryConnection;
import org.openrdf.repository.RepositoryException;
import org.openrdf.query.impl.DatasetImpl;
import org.openrdf.query.TupleQuery;
import org.openrdf.query.TupleQueryResult;
import org.openrdf.query.BindingSet;
import org.openrdf.query.QueryLanguage;

import org.eaglei.repository.model.Access;
import org.eaglei.repository.model.AccessGrant;
import org.eaglei.repository.model.View;
import org.eaglei.repository.model.Provenance;
import org.eaglei.repository.auth.Authentication;
import org.eaglei.repository.vocabulary.REPO;
import org.eaglei.repository.status.NotFoundException;
import org.eaglei.repository.status.ForbiddenException;
import org.eaglei.repository.status.InternalServerErrorException;
import org.eaglei.repository.status.ConflictException;
import org.eaglei.repository.servlet.WithRepositoryConnection;

/**
 * Collect and manage the workflow properties on an instance.
 *
 * @author Larry Stone
 * Started October, 2010
 * @version $Id: $
 */
public class Workflow
{
    private static Logger log = LogManager.getLogger(Workflow.class);

    private URI uri;                  // subject URI: required
    private URI homeGraph = null;
    private URI state = null;
    private URI owner = null;

    // query to get the workflow-related properties of a resource instance.
    // you MUST bind the ?resource variable before querying.
    private static final String wfQuery =
        "SELECT ?owner ?state ?home WHERE {\n"+
        " GRAPH ?home { ?resource a ?type }\n"+
        " ?resource <"+REPO.HAS_WORKFLOW_STATE+"> ?state\n"+
        " OPTIONAL{ ?resource <"+REPO.HAS_WORKFLOW_OWNER+"> ?owner }}";
         
    private Workflow(URI u)
    {
        super();
        uri = u;
    }

    /** sort of constructor. */
    public static Workflow find(HttpServletRequest request, URI resource)
    {
        Workflow result = new Workflow(resource);
        log.debug("SPARQL QUERY to get owner, state for WORKFLOW STATUS = \n"+wfQuery);
        try {
            RepositoryConnection rc = WithRepositoryConnection.get(request);
            TupleQuery q = rc.prepareTupleQuery(QueryLanguage.SPARQL, wfQuery);
            DatasetImpl queryDS = new DatasetImpl();
            View.addGraphs(request, queryDS, View.USER_RESOURCES);
            q.setDataset(queryDS);
            q.setIncludeInferred(false);
            q.setBinding("resource", resource);
            TupleQueryResult qr = null;
            try {
                qr = q.evaluate();
                if (qr.hasNext()) {
                    BindingSet bs = qr.next();
                    Value vowner = bs.getValue("owner");
                    if (vowner == null) {
                        log.debug("OK, Resource has no owner: "+resource.stringValue());
                    } else {
                        if (vowner instanceof URI) {
                            result.owner = (URI)vowner;
                        } else {
                            throw new InternalServerErrorException("There is an error in the :hasWorkflowOwner data for resource="+resource.stringValue());
                        }
                    }
                    Value vstate= bs.getValue("state");
                    if (vstate == null)
                        throw new InternalServerErrorException("This resource has no Workflow State: "+resource.stringValue());
                    result.state = (URI)vstate;
                    result.homeGraph = (URI)bs.getValue("home");
                } else {
                    throw new NotFoundException("There is no resource for this URI, or it is not visible to you: "+resource.stringValue());
                }
            } finally {
                if (qr != null)
                    qr.close();
            }
        } catch (RepositoryException e) {
            throw new InternalServerErrorException(e);
        } catch (OpenRDFException e) {
            throw new InternalServerErrorException(e);
        }
        return result;
    }

    /** Getter. */
    public URI getURI()
    {
        return uri;
    }
    /** Getter. */
    public URI getHomeGraph()
    {
        return homeGraph;
    }
    /** Getter. */
    public URI getState()
    {
        return state;
    }
    /** Getter. */
    public URI getOwner()
    {
        return owner;
    }

    /**
     * Create the workflow properties on a new resource instance.  First,
     * check that user has an allowed transition out of NEW state.  Then,
     * set the state to the first-found transition's final.  Also, assert
     * a claim so the user will be able to edit the instance he's creating.
     *
     * XXX FIXME: there is currently no path in the API to supply a choice of
     * transitions on creation, nor is there any use case as yet.  There
     * SHOULD only be ONE transition with all of:
     *  1. Starts in NEW state
     *  2. Matches given workspace
     *  3. Accessible to current user.
     * ...but if there are multiple results, we choose the first.
     * WARNING: transitions with specific matching workspace do NOT get prioity
     * although logically they should.
     */
    public static Workflow create(HttpServletRequest request, URI resource, URI workspace)
        throws ServletException
    {
        log.debug("Creating new resource="+resource.stringValue());

        // 1. do we have a transition out of the NEW state?
        URI myself = Authentication.getPrincipalURI(request);
        Collection<WorkflowTransition> ts = WorkflowTransition.findAccessibleByAttributes(request, null, REPO.WFS_NEW,
                workspace, myself);
        if (ts.isEmpty())
            throw new ForbiddenException("Permission denied, you must have access to a transition from the New workflow state to create a new resource instance.");
        else if (ts.size() > 1)
            log.warn("create() found multiple transitions from NEW state, choosing one arbitrarily. User="+myself+", workspace="+workspace);

        WorkflowTransition t = ts.iterator().next();
        Workflow result = new Workflow(resource);
        result.homeGraph = workspace;
        result.state = t.getFinal();
        result.setWorkflowState(request, t.getFinal());
        result.assertClaimInternal(request, myself);
        result.owner = myself;
        return result;
    }

    /**
     * Establish a claim if allowed.
     * Validate resource URI and check access
     *  - asserted rdf:type and :hasWorkflowState properties in readable graph
     *  - no existing claim
     * NOTE that caller still needs to commit the changes.
     */
    public void assertClaim(HttpServletRequest request, URI claimer)
        throws ServletException
    {
        log.debug("Asserting a claim on resource="+uri.stringValue()+", by user="+claimer.stringValue());

        // 1. must not be claimed already
        if (owner != null)
            throw new ConflictException("Resource is already claimed by "+
                (claimer.equals(owner) ? "the proposed owner.":" a different user."));

        // 2. do we have a transition out of current state?
        if (WorkflowTransition.findAccessibleByAttributes(request, null, state,
                homeGraph, claimer).isEmpty())
            throw new ForbiddenException("Permission denied, you do not have access to any of the transitions from the resource instance's current workflow state.");

        assertClaimInternal(request, claimer);
    }

    // execute the actual claim assertion, code shared by by create()
    private void assertClaimInternal(HttpServletRequest request, URI claimer)
        throws ServletException
    {
        // add claim properties and ACL changes
        try {
            RepositoryConnection rc = WithRepositoryConnection.get(request);
            if (log.isDebugEnabled())
                log.debug("ASSERTING CLAIM, owner="+claimer.stringValue()+", resource="+uri.stringValue());
            rc.add(uri, REPO.HAS_WORKFLOW_OWNER, claimer, REPO.NG_METADATA);
            AccessGrant.addGrantAsAdministrator(request, uri, claimer, Access.ADD.getURI());
            AccessGrant.addGrantAsAdministrator(request, uri, claimer, Access.REMOVE.getURI());
        } catch (RepositoryException e) {
            throw new ServletException(e);
        }
    }

    /**
     * Release an existing claim, if allowed.
     * Validate resource URI and check access:
     *  - there must be a claim (obviously)
     *  - user must be owner of the claim or Administrator
     * NOTE that caller still needs to commit the changes.
     */
    public void releaseClaim(HttpServletRequest request)
        throws ServletException
    {
        // 1. must be claimed
        if (owner == null)
            throw new ConflictException("Resource is not claimed.");

        // 2. user must be owner or administrator
        if (!(owner.equals(Authentication.getPrincipalURI(request)) ||
              Authentication.isSuperuser(request)))
            throw new ForbiddenException("Permission denied, only the owner of a claim or the Administrator may release it.");

        releaseClaimInternal(request);
    }

    // actual mechanism to release claim - shared by invokeTransition
    private void releaseClaimInternal(HttpServletRequest request)
        throws ServletException
    {
        try {
            RepositoryConnection rc = WithRepositoryConnection.get(request);
            if (log.isDebugEnabled())
                log.debug("RELEASING CLAIM, resource="+uri.stringValue());
            rc.remove(uri, REPO.HAS_WORKFLOW_OWNER, null, REPO.NG_METADATA);
            if (!AccessGrant.removeGrantAsAdministrator(request, uri, owner, Access.ADD.getURI()))
                log.warn("Releasing workflow claim but WF Owner did NOT have ADD access to instance! resource="+uri.stringValue());
            if (!AccessGrant.removeGrantAsAdministrator(request, uri, owner, Access.REMOVE.getURI()))
                log.warn("Releasing workflow claim but WF Owner did NOT have REMOVE access to instance! resource="+uri.stringValue());
        } catch (RepositoryException e) {
            throw new ServletException(e);
        }
    }

    /**
     * Invoke (follow) a workflow transition on an instance, if allowed.
     * Validate resource URI and check access:
     *  - user has READ permission on transition
     *  - there must be a claim (obviously)
     *  - user must be owner of the claim or Administrator
     * NOTE that caller still needs to commit the changes.
     */
    public void invokeTransition(HttpServletRequest request, URI transitionURI)
        throws ServletException
    {
        // this will fail if transition doesn't exist.
        WorkflowTransition t = WorkflowTransition.find(request, transitionURI);

        // access checks
        if (owner == null)
            throw new ConflictException("Resource must be claimed to invoke a transition.");
        URI myself = Authentication.getPrincipalURI(request);
        if (!(myself.equals(owner) || Authentication.isSuperuser(request)))
            throw new ForbiddenException("You must own the claim on the instance or be Administrator to invoke a transition.");
        if (!(Access.hasPermission(request, transitionURI, Access.READ)))
            throw new ForbiddenException("You are not allowed to invoke this transition:"+transitionURI);

        // final sanity check - is instance in the correct initial state?
        if (!t.getInitial().equals(state))
            throw new ConflictException("Resource's workflow state differs from expected initial state of this transition.");

        releaseClaimInternal(request);
        setWorkflowState(request, t.getFinal());

        // invoke action
        WorkflowAction action = t.getAction();
        if (action != null) {
            try {
                action.onTransition(request, uri, t.getActionParameter());
            } catch (Exception e) {
                log.error("Failed in workflow action: ",e);
                throw new ServletException("Failed in workflow action: ",e);
            }
        }
        new Provenance(uri).setModified(request, new Date());
    }

    // change resource's workflow state value
    private void setWorkflowState(HttpServletRequest request, URI newState)
        throws ServletException
    {
        try {
            RepositoryConnection rc = WithRepositoryConnection.get(request);
            rc.remove(uri, REPO.HAS_WORKFLOW_STATE, null, REPO.NG_METADATA);
            rc.add(uri, REPO.HAS_WORKFLOW_STATE, newState, REPO.NG_METADATA);
        } catch (RepositoryException e) {
            throw new ServletException(e);
        }
    }
    /**
     * Predicate that is true if this is a predicate managed by workflow
     * so e.g. /update can protect against users changing it even in a
     * different graph.
     *
     * @param uri a {@link org.openrdf.model.URI} object.
     * @return a boolean, true if URI is one of the fake workflow predicates.
     */
    public static boolean isWorkflowPredicate(URI uri)
    {
        return REPO.HAS_WORKFLOW_STATE.equals(uri) ||
               REPO.HAS_WORKFLOW_OWNER.equals(uri);
    }
}
