package org.eaglei.repository.servlet;

import java.io.InputStreamReader;
import java.io.Reader;
import java.io.OutputStreamWriter;
import java.io.IOException;
import java.nio.charset.IllegalCharsetNameException;
import java.nio.charset.UnsupportedCharsetException;
import java.nio.charset.Charset;
import java.util.Date;
import java.util.ArrayList;
import java.util.List;
import java.util.Set;
import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;


import org.apache.log4j.Logger;
import org.apache.log4j.LogManager;

import org.openrdf.OpenRDFException;
import org.openrdf.repository.RepositoryResult;
import org.openrdf.repository.RepositoryConnection;
import org.openrdf.model.Resource;
import org.openrdf.model.URI;
import org.openrdf.model.ValueFactory;
import org.openrdf.model.vocabulary.RDF;
import org.openrdf.rio.RDFHandler;
import org.openrdf.rio.RDFHandlerException;
import org.openrdf.rio.RDFFormat;
import org.openrdf.rio.Rio;
import org.openrdf.rio.RDFParser;
import org.openrdf.repository.util.RDFRemover;

import org.eaglei.repository.model.Access;
import org.eaglei.repository.util.Formats;
import org.eaglei.repository.Lifecycle;
import org.eaglei.repository.model.NamedGraph;
import org.eaglei.repository.model.NamedGraphType;
import org.eaglei.repository.model.Provenance;
import org.eaglei.repository.auth.Authentication;
import org.eaglei.repository.status.BadRequestException;
import org.eaglei.repository.status.NotFoundException;
import org.eaglei.repository.status.ForbiddenException;
import org.eaglei.repository.status.HttpStatusException;
import org.eaglei.repository.inferencer.TBoxInferencer;
import org.eaglei.repository.util.Utils;
import org.eaglei.repository.util.AppendingRDFHandler;
import org.eaglei.repository.vocabulary.REPO;

/**
 * Load and dump named graphs <-> serialized RDF.
 *
 * @author Larry Stone
 * @version $Id: $
 */
public class Graph extends RepositoryServlet
{
    private static Logger log = LogManager.getLogger(Graph.class);

    public enum Action { add, replace, delete };

    /**
     * {@inheritDoc}
     *
     * POST or PUT the contents of a graph
     * Query Args:
     *  - name = URI of graph, required (can be "all")
     *  - action = (add|replace|delete) - required, how to process triples
     *  - format = MIME type (overrides content-type)
     *  - uri = remote URL from which to obtain serialization
     *  - content = immediate serlialized RDF content to load
     *  - label = new value for rdfs:label property
     *  - type = NamedGraphType to record in metadata (ontology, metadata, etc..)
     * Result: HTTP status 200 for success, 400 or 4xx or 5xx otherwise.
     */
    @Override
    protected void doPost(HttpServletRequest request, HttpServletResponse response)
        throws ServletException, IOException
    {
        request.setCharacterEncoding("UTF-8");
        URI name = getParameterAsURI(request, "name",false);
        boolean all = isParameterPresent(request, "all");
        Action action = (Action)getParameterAsKeyword(request, "action", Action.class, null, true);
        String format = getParameter(request, "format", false);
        String source = getParameter(request, "source", false);
        String sourceModified = getParameter(request, "sourceModified", false);
        String label = getParameter(request, "label", false);
        NamedGraphType type = (NamedGraphType)getParameterAsKeyword(request, "type", NamedGraphType.class, null, false);
        Reader content = getParameterAsReader(request, "content", true);
        String contentType = getParameterContentType(request, "content");
        if (format == null || format.length() == 0)
            format = contentType;
        putGraphInternal(request, response, name, all, action, format,
                         content, source, sourceModified, type, label);

    }

    /**
     * {@inheritDoc}
     *
     * PUT the contents of a graph
     * Query Args:
     *  - name = URI of graph, required (can be "all")
     *  - action = (add|replace|delete) - required, how to process triples
     *  - format = MIME type (overrides content-type)
     *  - uri = remote URL from which to obtain serialization
     *  - (PUT content is seralized RDF)
     * Result: HTTP status 200 for success, 400 or 4xx or 5xx otherwise.
     */
    @Override
    protected void doPut(HttpServletRequest request, HttpServletResponse response)
        throws ServletException, IOException
    {
        URI name = getParameterAsURI(request, "name",false);
        boolean all = isParameterPresent(request, "all");
        Action action = (Action)getParameterAsKeyword(request, "action", Action.class, null, true);
        String source = getParameter(request, "source", false);
        String label = getParameter(request, "label", false);
        NamedGraphType type = (NamedGraphType)getParameterAsKeyword(request, "type", NamedGraphType.class, null, false);
        String format = getParameter(request, "format", false);
        if (format == null || format.length() == 0)
            format = request.getContentType();

        // last-mod date of source, CAN be expressed as entity header..
        String sourceModified = getParameter(request, "sourceModified", false);
        if (sourceModified == null)
            sourceModified = request.getHeader("Last-Modified");

        String encoding = request.getCharacterEncoding();
        if (encoding == null)
            encoding = "UTF-8";

        try {
            Reader content = new InputStreamReader(request.getInputStream(), Charset.forName(encoding));
            putGraphInternal(request, response, name, all, action, format,
                content, source, sourceModified, type, label);
        } catch  (IllegalCharsetNameException e) {
            throw new BadRequestException("Illegal character set name in content-type spec: "+e);
        } catch  (UnsupportedCharsetException e) {
            throw new BadRequestException("Unsupported character set name in content-type spec: "+e);
        }
    }


    // do the real work of a PUT or POST
    private void putGraphInternal(HttpServletRequest request, HttpServletResponse response,
            URI name, boolean all, Action action, String rawFormat,
            Reader content, String source, String sourceModified, NamedGraphType type, String label)
        throws ServletException, IOException
    {
        RepositoryConnection rc = WithRepositoryConnection.get(request);
        ValueFactory vf = rc.getValueFactory();

        // sanity check format
        if (rawFormat == null) {
            throw new BadRequestException("Missing required argument: format (or content-type)");
        }
        // sanity check: viable graph name or 'all' was specified
        if (all && name != null) {
            throw new BadRequestException("You may not specify both a named graph and the 'all' keyword");
        } else if (!all && name == null) {
            throw new BadRequestException("Missing required argument: name");
        }

        // sanity check: source of input
        if (content == null) {
            throw new BadRequestException("No input source, either remote URI or content must be specified");
        }

        String mime = Utils.contentTypeGetMIMEType(rawFormat);
        RDFFormat format = Formats.RDFOutputFormatForMIMEType(mime);
        if (format == null) {
            throw new HttpStatusException(HttpServletResponse.SC_UNSUPPORTED_MEDIA_TYPE, "MIME type of serialized RDF is not supported: \""+mime+"\"");
        }
        log.debug("RDF format derived from arg ("+rawFormat+") = "+format);

        // sanity check that a quad format was chosen if 'all' is on.
        if (all && !format.supportsContexts()) {
            throw new BadRequestException("'All' may not be specified with a format that does not support quads");
        }

        // finally, test access and add or delete the triples
        if ((all && Authentication.isSuperuser(request)) ||
            (name != null &&
             (!(action == Action.add || action == Action.replace) ||
               Access.hasPermission(request, name, Access.ADD)) &&
             (!(action == Action.delete || action == Action.replace) ||
               Access.hasPermission(request, name, Access.REMOVE)))) {
            try {
                // flag to add provenance metadata for created: if replace or add to empty graph..
                boolean created = action == Action.replace ||
                    (action == Action.add && !all && rc.size(name) == 0);
                if (action == Action.add || action == Action.replace) {

                    if (action == Action.replace) {
                        if (all) {
                            rc.clear();
                        } else {
                            rc.clear(name);
                        }
                    }
                    if (all) {
                        long before = rc.size();
                        rc.add(content, "", format);
                        log.info("Added "+String.valueOf(rc.size()-before)+" statements to all graphs.");

                        // Reset any application-wide status that depends on RDF database
                        Lifecycle.getInstance().notifyDataReplaced();
                    } else {
                        long before = rc.size(name);
                        rc.add(content, "", format, name);
                        log.info("Added "+String.valueOf(rc.size(name)-before)+" statements to graph:"+name.toString());
                    }

                    // Ensure there is a :NamedGraph node for this graph
                    if (!all && !rc.hasStatement(name, RDF.TYPE, REPO.NAMED_GRAPH, true)) {
                        rc.add(name, RDF.TYPE, REPO.NAMED_GRAPH, REPO.NG_INTERNAL);
                        log.debug("Declaring rdf:type of named graph URI = "+name);
                    }

                    // (re)set graph's descriptive metadata
                    if (!all && (type != null || label != null))
                    {
                        NamedGraph ng = NamedGraph.findOrCreate(request, name, true);
                        if (type != null)
                            ng.setType(request, type.getURI());
                        if (label != null)
                            ng.setLabel(request, label);
                    }

                } else if (action == Action.delete) {
                    long before = rc.size(name);
                    RDFParser parser = Rio.createParser(format, vf);
                    RDFRemover rr = new RDFRemover(rc);
                    if (!all)
                        rr.enforceContext(name);
                    parser.setRDFHandler(rr);
                    parser.parse(content, "");
                    log.info("Deleted "+String.valueOf(before-rc.size(name))+" statements to graph:"+name.toString());
                }
             
                // don't add provenance for 'all' since it was most likely
                // used to restore a dump and we want to preserve old provenance
                if (!all) {
                    Date now = new Date();
                    Provenance p = new Provenance(name);
                    if (created)
                        p.setCreated(request, now);
                    p.setModified(request, now);
                    if (source != null) {
                        Date smDate = null;
                        if (sourceModified != null) {
                            smDate = Utils.parseXMLDate(sourceModified).toGregorianCalendar().getTime();
                            log.debug("Got sourceModified date, parsed date = "+smDate);
                        }
                        p.setSource(request, source, smDate);
                    }
                }
             
                rc.commit();
                response.setStatus(created ? HttpServletResponse.SC_CREATED : HttpServletResponse.SC_OK);
            } catch (OpenRDFException e) {
                log.error("Failed parsing user-supplied graph content: ",e);
                throw new BadRequestException(e);
            }
        } else {
            throw new ForbiddenException("User is not permitted to "+action+" this graph");
        }
    }

    /**
     * {@inheritDoc}
     *
     * GET the contents of a graph
     * Query Args:
     *  - name = URI of graph, required (can be "all")
     *  - format = MIME type (overrides content-type)
     *  - inferred -- flag, if present, include inferred triples
     *  - force -- secret option, if set, allows use of non-quad format with ALL
     * Result: HTTP status 200 for success, 400 or 4xx or 5xx otherwise.
     */
    @Override
    protected void doGet(HttpServletRequest request, HttpServletResponse response)
        throws ServletException, IOException
    {
        request.setCharacterEncoding("UTF-8");
        String rawFormat = getParameter(request, "format", false);
        String mimeType = Formats.negotiateRDFContent(request, rawFormat);
        RDFFormat format = Formats.RDFOutputFormatForMIMEType(mimeType);
        if (format == null) {
            throw new BadRequestException("Unrecognized MIME type for serialized RDF format: \""+mimeType+"\"");
        }
        log.debug("Negotiated output format = "+format);

        // secret option 'force' allows use of 'all' with non-quad formats
        // ..since this is mainly useful for easy testing using n-triples
        boolean force = isParameterPresent(request, "force");
        // include inferred triples
        boolean inferred = isParameterPresent(request, "inferred");
        URI name = getParameterAsURI(request, "name",false);
        boolean all = isParameterPresent(request, "all");

        // sanity checks for all: no name, quad output format
        if (all)  {
            if (name != null)
                throw new BadRequestException("You may not specify both a named graph and the 'all' keyword");
            if (!format.supportsContexts() && !force)
                throw new BadRequestException("'All' may not be specified with a format that does not support quads");

        // !all sanity check: require graph name
        } else {
            if (name == null)
                throw new BadRequestException("Missing required argument, either 'name' or 'arg' is required.");
        }

        RepositoryConnection rc = WithRepositoryConnection.get(request);
        try {
            // when dumping all contexts, write all the ontology
            // graphs first so restore won't have to thrash around
            // recomputing inferencing when the ontology changes.
            if (all) {
                List<Resource> graphs = new ArrayList<Resource>();
                List<Resource> later = new ArrayList<Resource>();
                RepositoryResult<Resource> rr = null;
                Set<URI> tboxGraphs = TBoxInferencer.getInstance().getTBoxGraphs();

                // compose graphs list startign with TBox graphs
                try {
                    rr = rc.getContextIDs();
                    while (rr.hasNext()) {
                        Resource ctx = rr.next();
                        if (ctx instanceof URI &&
                              tboxGraphs.contains((URI)ctx))
                            graphs.add(ctx);
                        else
                            later.add(ctx);
                    }
                } finally {
                    rr.close();
                }
                graphs.addAll(later);

                // check all access first, then write output
                for (Resource ctx : graphs) {
                    if (!(ctx instanceof URI &&
                          Access.hasPermission(request, ctx, Access.READ)))
                        throw new ForbiddenException("You are not allowed to read graph "+ctx.stringValue());
                }
                response.setContentType(Utils.makeContentType(format.getDefaultMIMEType(), "UTF-8"));
                AppendingRDFHandler writer = new AppendingRDFHandler(Rio.createWriter(format, new OutputStreamWriter(response.getOutputStream(), "UTF-8")));
                try {
                    for (Resource ctx : graphs) {
                        rc.exportStatements(null, null, null, inferred, writer, ctx);
                    }
                } finally {
                    writer.reallyEndRDF();
                }

            // just write one named graph
            } else {
                if (!Access.hasPermission(request, name, Access.READ))
                    throw new ForbiddenException("You are not allowed to read this graph: "+name.stringValue());

                if (rc.hasStatement(name, RDF.TYPE, REPO.NAMED_GRAPH, true)) {
                    response.setContentType(Utils.makeContentType(format.getDefaultMIMEType(), "UTF-8"));
                    rc.exportStatements(null, null, null, inferred,
                        Rio.createWriter(format, new OutputStreamWriter(response.getOutputStream(), "UTF-8")),
                        name);
                } else
                    throw new NotFoundException("This URI is not a named graph: "+name);
            }
        } catch (OpenRDFException e) {
            log.error("Failed exporting graph content: ",e);
            throw new ServletException(e);
        }
    }

}
